#ifndef CALIBRATION_MODULE_H
#define CALIBRATION_MODULE_H

/////////////////////////////////////////////////////////////////
//
// calibration_module
// 
// handles calibration data
//
// daniel.joseph.antrim@cern.ch
// November 2016
//
//////////////////////////////////////////////////////////////////

// vmm
#include "message_handler.h"
#include "map_handler.h"
#include "calibration_state.h"
#include "calibration_type.h"
#include "vmm_decoder.h"
#include "trigger_event.h"
#include "pulser_calibration.h"
class SocketHandler;
class GlobalSetting;
class VMMMap;
class Channel;
class FPGAClocks;
class TriggerDAQ;

// qt
#include <QObject>
class QUdpSocket;

//boost
#include <boost/shared_ptr.hpp>
#include <boost/array.hpp>
#include <boost/thread.hpp>
#include <boost/asio.hpp>

//std/stl
#include <sstream>

//ROOT

class TFile;
class TTree;
class TBranch;

class MapHandler;
class MessageHandler;

#define MAXBUFLEN 65507

class CalibModule : public QObject
{
    Q_OBJECT

    public:
        explicit CalibModule(QObject *parent=0);
        virtual ~CalibModule(){};

        void LoadModulesForCalibration(Configuration& config, RunModule& run);//, SocketHandler& socket);
        bool load_pulser_calib(PulserCalibDef def);
        bool load_pulser_calib_run_properties(PulserCalibRunProperties props);
        void load_pulser_calib_flags(bool* accept_data);
        void send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& VMMMap,
                std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq);

        void setDebug(bool doit) { m_dbg = doit; }
        bool dbg() { return m_dbg; }

        void set_vmm_type(int type);
        int vmm_type() { return m_vmm_type; }

        void LoadMessageHandler(MessageHandler& msg);
        MessageHandler& msg() { return *m_msg; }
        bool initializeSocket(boost::shared_ptr<boost::asio::io_service> service);
        void toggleSocket();
        void closeCalibrationSocket();

        bool setOutputFile(std::string output_filename, int run_number, bool is_xadc=false);

        void decode_data(std::string& ip_string, std::vector<uint32_t>& datagram, uint32_t length, int& counter);
        bool valid_xadc_sample(std::vector<uint32_t>& datagram);
        void decode_xadc_sampling(std::string& ip_string, std::vector<uint32_t>& datagram,
                                                int& counter);
        void decode_calib_data(std::string& ip_string, std::vector<uint32_t>& datagram, int& counter);
        void decode_calib_data_trig(std::string& ip_string, std::vector<uint32_t>&datagram, int& counter);
        void fill_trigger_event(int& counter);

        void setup_decoding(bool is_xadc = false, bool do_l0 = false);

        void setupOutputTrees(bool do_l0_decoding = false);
        CalibrationState& calibrationState() { return m_calib_state; }

        void fill_event(uint32_t vmm_id, int32_t board_id, uint32_t xadc_samples);
        void fill_event();

        uint32_t decodeGray(uint32_t gray);
        void fillRunComment(std::string comment) { m_run_comment = comment; }
        
        void write_output();

        bool do_xadc();
        CalibrationType& calibType() { return m_calibration_type; }
        
        void clear_data();

        MapHandler& mapHandler(){ return *m_mapHandler; }
        bool mapping(){ return m_mapping_setup; }
        bool writeNtuple() { return m_writeNtuple; }

    private :
        bool m_dbg;
        int m_vmm_type;
        bool m_do_l0;
        std::stringstream m_sx;

        // messaging
        MessageHandler* m_msg;

        // PulserCalib
        PulserCalib* m_pulser_calib;

        // counters
        int n_push_back;
        int n_empty_events_received;
        int n_warnings;
        int n_send_period;

        // meta data
        std::string m_run_comment;
        CalibrationType m_calibration_type;

        // if receive too many empty events, skip that channel in the calibraiton loop
        bool m_skip_dead_channels;

        // xadc configuration params
        uint32_t m_vmm_id; // VMM to sample
        uint32_t n_samples; // number of samples to collect
        uint32_t m_sample_period; // period of sampling

        // UDP socket communication
        int m_bind_port; // port socket will bind to
        int m_xadc_port; // port socket will listen to
        boost::asio::ip::udp::endpoint m_remote_endpoint; // remote endpoint for listening for data
        boost::shared_ptr<boost::asio::ip::udp::socket> m_socket;
        boost::shared_ptr<boost::asio::io_service> m_io_service;

        // decoder
        VMMDecoder m_decoder;

        // mapping added for readout
        MapHandler* m_mapHandler;
        bool m_mapping_setup;
        bool m_writeNtuple;

        // CalibrationState
        CalibrationState m_calib_state;

        //////////////////////////////////////////////
        // variables for output ntuples
        //////////////////////////////////////////////
        // xadc sampling
        uint32_t m_boardid;
        uint32_t m_chipid;
        uint32_t m_xadc_samples;
        int32_t  m_channel_trim;
        uint32_t m_channel;

        // non-xadc sampling
        std::vector<int> m_boardId;
        std::vector<int> m_chipId;
        std::vector< std::vector<int> > m_pdo;
        std::vector< std::vector<int> > m_tdo;
        std::vector< std::vector<int> > m_channelId;
        std::vector< std::vector<int> > m_threshold;
        std::vector<int> m_triggerCounter;
        std::vector<int> m_eventSize;
        std::vector< std::vector<int> > m_flag;
        std::vector< std::vector<int> > m_bcid;
        std::vector< std::vector<int> > m_grayDecoded;
        std::vector< std::vector<int> > m_neighbor_calib;
        std::vector< std::vector<int> > m_overflow;
        std::vector< std::vector<int> > m_orbit_count;
        std::vector< std::vector<int> > m_relbcid;
        
        //////////////////////////////////////////////
        // branches for output ntuples
        //////////////////////////////////////////////

        // this is the TTree object
        TTree* m_calib_tree;

        // these are "calibration state" branches and are for both xadc and non-xadc
        TBranch *br_runNumber;
        TBranch *br_gain;
        TBranch *br_tacSlope;
        TBranch *br_peakTime;
        TBranch *br_dacCounts; // TP threshold
        TBranch *br_tpSkew; // CKTP skew
        TBranch *br_pulserCounts; // TP amplitude
        TBranch *br_subhysteresis;
        TBranch *br_neighbortrigger;
        TBranch *br_nExpected;
        TBranch *br_channelTrims;
        TBranch *br_neighborCalib;

        // branches for xadc sampling
        TBranch *br_boardid;
        TBranch *br_xadc_samples;
        TBranch *br_chipid;
        TBranch *br_boardId;
        TBranch *br_vmm_channel;

        // branches for non-xadc calibration
        TBranch *br_chipId;
        TBranch *br_channelId;
        TBranch *br_tdo;
        TBranch *br_pdo;
        TBranch *br_thresh;
        TBranch *br_flag;
        TBranch *br_grayDecoded;
        TBranch *br_bcid;
        TBranch *br_evSize;

        // output file
        int m_run_number;
        std::string m_output_rootfilename;
        TFile* m_calibRootFile; /// this is the TFile in which we will store the data

        // flags
        bool m_trees_setup;

        // trigger events
        TriggerEvent m_trigger_events;
        uint32_t m_last_trigger_read;

    signals :
        void pulser_calib_complete_signal();
    
    public slots :
        void start_xadc_sampling(QString); // send the command to tell xADC to start sending samples
        void start_xadc_sampling_VMM2(QString);
        void initialize_calibration(int, int, int, int, bool, bool); // load the parameters for configuraing xADC sampling 
        void load_calibration_state(int, int, int,
                                    int, int, int,
                                    int, int, int,
                                    std::vector<int>, double, int);
        void reset_counts();

        void begin_pulser_calibration();
        void pulser_calib_complete();

}; // class CalibModule

#endif
