#include "calibration_threshold_loader.h"

// std/stl
#include <iostream>
#include <sstream>
#include <exception>
using namespace std;

// boost
//#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/format.hpp>
#include <boost/foreach.hpp>
#include <boost/algorithm/string.hpp>


//////////////////////////////////////////////////////////////////////////////
CalibrationThresholdLoader::CalibrationThresholdLoader() :
    m_filename(""),
    m_current_board_ip(""),
    m_current_board_id(-1),
    m_current_chip_id(-1),
    m_current_channel(-1),
    m_current_trim(0),
    m_thresholds_loaded(false)
{
}
//////////////////////////////////////////////////////////////////////////////
CalibrationThresholdLoader::~CalibrationThresholdLoader()
{
}
//////////////////////////////////////////////////////////////////////////////
void CalibrationThresholdLoader::clear()
{
    m_holder.clear();
    clear_current();
    m_thresholds_loaded = false;
}
//////////////////////////////////////////////////////////////////////////////
void CalibrationThresholdLoader::clear_current()
{
    m_current_board_ip = "";
    m_current_board_id = -1;
    m_current_chip_id = -1;
    m_current_channel = -1;
    m_current_trim = 0;
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::thresholds_loaded()
{
    return m_thresholds_loaded;
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::load_file(std::string calib_file_to_load)
{
    cout << "CalibrationThresholdLoader::load_file    Loading file: " << calib_file_to_load << endl;

    bool ok = true;

    using boost::property_tree::ptree;
    using namespace boost::property_tree::xml_parser;
    ptree pt;
    try {
        read_xml(calib_file_to_load, pt, trim_whitespace | no_comments);
    }
    catch(std::exception &e) {
        ok = false;
        cout << "CalibrationThresholdLoader::load_file    ERROR Unable to open provided file: "
            << calib_file_to_load <<  endl;
    } // catch

    if(ok) {

        try {
            for(const auto& node : pt.get_child("threshold_calibration")) {
                if(node.first=="board") {
                    ok = load_board(node);
                    if(!ok) break;
                }
            } // node
        } // try
        catch(std::exception &e)
        {
            cout << "CalibrationThresholdLoader::load_file    ERROR loading threshold calibration: "
                << e.what() << endl;
            ok = false;
        } // catch

        if(ok) {
            sort_data();
            //cout << "CalibrationThresholdLoader::load_file    Calibration data loaded successfully!" << endl;
            //cout << summary_table();
            //cout << "loaded " << m_holder.size() << " channels' thresholds" << endl;
            /*
            for(auto holder : m_holder) {
                cout << "(board ip, board id, chip, channel, trim)=("
                    << holder.board_ip << ","
                    << holder.board_id << ","
                    << holder.chip_id << ","
                    << holder.channel << ","
                    << holder.trim_to_set << ")" << endl;
            }
            */
        }
    }
    m_thresholds_loaded = ok;

    return ok;
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::load_board(const boost::property_tree::ptree::value_type node)
{
    bool ok = true;
    using boost::property_tree::ptree;
    using namespace boost;

    try {
        BOOST_FOREACH(const ptree::value_type &v, node.second) {
            ////////////////////////////////////////
            // board id
            ////////////////////////////////////////
            if(v.first=="id") {
                stringstream id;
                id << v.second.data();
                string id_str = id.str();
                trim(id_str);
                if(id_str=="") {
                    cout << "CalibrationThresholdLoader::load_board    "
                        << "ERROR board id node is \"\"" << endl;
                    ok = false;
                    break;
                }
                trim(id_str);
                m_current_board_id = std::stoi(id_str);
            }
            ////////////////////////////////////////
            // board ip
            ////////////////////////////////////////
            else if(v.first=="ip") {
                m_current_board_ip = v.second.data();
                trim(m_current_board_ip);
            } // ip
            ////////////////////////////////////////
            // chip
            ////////////////////////////////////////
            else if(v.first=="chip") {
                ok = load_chip(v);
            } // chip

        } // foreach

    } // try
    catch(std::exception& e) {
        cout << "CalibrationThresholdLoader::load_board    ERROR loading board info: " << e.what() << endl;
        ok = false;
    } // catch

    return ok;
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::load_chip(const boost::property_tree::ptree::value_type node)
{
    bool ok = true;
    using boost::property_tree::ptree;
    using namespace boost;

    try {
        BOOST_FOREACH(const ptree::value_type &v, node.second) {
            ///////////////////////////////////
            // id
            ///////////////////////////////////
            if(v.first=="id") {
                stringstream id;
                id << v.second.data();
                string id_str = id.str();
                trim(id_str);
                if(id_str=="") {
                    cout << "CalibrationThresholdLoader::load_chip    ERROR getting chip id,"
                        << " chip id is \"\"" << endl;
                    ok = false;
                    break;
                }
                trim(id_str);
                m_current_chip_id = std::stoi(id_str);
            }
            ///////////////////////////////////
            // channel trim
            ///////////////////////////////////
            else if(v.first=="channel_trim") {
                ok = load_trim(v);
                if(!ok) break;
            } // channel_trim

        } // foreach

    } // try
    catch(std::exception& e) {
        cout << "CalibrationThresholdLoader::load_chip    ERROR loading chip info: " << e.what() << endl;
        ok = false;
    }

    return ok;
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::load_trim(const boost::property_tree::ptree::value_type node)
{
    bool ok = true;
    using boost::property_tree::ptree;
    using namespace boost;

    try {
        BOOST_FOREACH(const ptree::value_type &v, node.second) {
            ///////////////////////////////////////
            // channel trims
            ///////////////////////////////////////
            //cout << "trim for channel: " << v.first << " = " << v.second.data() << endl;
            stringstream sx;
            sx << v.first;
            string channel_id = sx.str();
            trim(channel_id);
            m_current_channel = std::stoi(channel_id);

            sx.str("");
            sx << v.second.data();
            string trim_value = sx.str();
            trim(trim_value);
            m_current_trim = std::stoi(trim_value);

            ThresholdHolder current_data;
            current_data.board_ip = m_current_board_ip;
            current_data.board_id = m_current_board_id;
            current_data.chip_id  = m_current_chip_id;
            current_data.channel = m_current_channel;
            current_data.trim_to_set = m_current_trim;
            m_holder.push_back(current_data);

        } // foreach

    } // try
    catch(std::exception& e)
    {
        cout << "CalibrationThresholdLoader::load_trim    ERROR loading trim for "
            << "(board,chip)=("<<m_current_board_id<<","<<m_current_chip_id<<") : "
                << e.what() << endl;
        ok = false;
    } // catch

    return ok;
}
//////////////////////////////////////////////////////////////////////////////
std::vector<ThresholdHolder> CalibrationThresholdLoader::thresholds_for_board(std::string ip)
{
    return m_ip_holder[ip];
}
//////////////////////////////////////////////////////////////////////////////
bool CalibrationThresholdLoader::is_valid_ip(std::string ip)
{
    bool is_valid = false;
    for(auto h : m_ip_holder) {
        if(h.first == ip) {
            is_valid = true;
            break;
        }
    }
    return is_valid;
}
//////////////////////////////////////////////////////////////////////////////
void CalibrationThresholdLoader::sort_data()
{
    ///////////////////////////////////////
    // sort by IP
    ///////////////////////////////////////
    m_ip_holder.clear();

    // gather all IP's present
    vector<string> unique_ips;
    for(const auto  holder : m_holder) {
        string ip = holder.board_ip;
        if(!(std::find(unique_ips.begin(), unique_ips.end(), ip) != unique_ips.end())) {
            unique_ips.push_back(ip);
        }
    }

    for(const auto ip : unique_ips) {
        for(const auto holder : m_holder) {
            if(!(holder.board_ip==ip)) continue;
            m_ip_holder[ip].push_back(holder);
        }
    }
    m_unique_ips = unique_ips;

    ///////////////////////////////////////
    // sort by ID
    ///////////////////////////////////////
    m_id_holder.clear();

    // gather all ID's present
    vector<int> unique_ids;
    for(const auto holder : m_holder) {
        int id = holder.board_id;
        if(!(std::find(unique_ids.begin(), unique_ids.end(), id) != unique_ids.end())) {
            unique_ids.push_back(id);
        }
    }
    std::sort(unique_ids.begin(), unique_ids.end());

    for(const auto id : unique_ids) {
        for(const auto holder : m_holder) {
            if(!(holder.board_id==id)) continue;
            m_id_holder[id].push_back(holder);
        }
    }
    m_unique_ids = unique_ids;


    /*
    cout << " * sorting by IP * " << endl;
    for(auto ip : unique_ips) {
        cout << "IP: " << ip << endl;
        std::vector<ThresholdHolder> current = m_ip_holder[ip];
        for(auto holder : current) {
            cout << "    (id,chip,channel,trim)=("
                    << holder.board_id<<","
                    << holder.chip_id<<","
                    << holder.channel<<","
                    << holder.trim_to_set<<")"<<endl;
        }
    }
    cout << " * sorting by ID * " << endl;
    for(auto id : unique_ids) {
        cout << "ID: " << id << endl;
        std::vector<ThresholdHolder> current = m_id_holder[id];
        for(auto holder : current) {
            cout << "    (ip,chip,channel,trim)=("
                    << holder.board_ip<<","
                    << holder.chip_id<<","
                    << holder.channel<<","
                    << holder.trim_to_set<<")"<<endl;
        }
    }
    */
    
}
//////////////////////////////////////////////////////////////////////////////
int CalibrationThresholdLoader::trim_value(std::string input_ip, int chip, int channel)
{
    bool ip_valid = false;
    int out_trim = -1;
    for(auto ip : m_ip_holder) {
        if(ip.first == input_ip) { ip_valid = true; break; }
    }
    if(!ip_valid) {
        cout << "CalibrationThresholdLoader::trim_value    No trim value stored for"
            << " board with requested IP '" << input_ip << "'!" << endl;
        return -1;
    }
    std::vector<ThresholdHolder> holder = m_ip_holder[input_ip];
    for(auto h : holder) {
        if( (h.chip_id == chip) && (h.channel == channel) ) {
            out_trim = h.trim_to_set;
            break;
        }
    }
    return out_trim;
}
//////////////////////////////////////////////////////////////////////////////
std::string CalibrationThresholdLoader::summary_table()
{
  
    vector<string> ids;
    stringstream sx;
    for(auto id : m_unique_ids) {
        sx.str("");
        sx << id;
        ids.push_back(sx.str());
    }

    ostringstream oss;
     struct shorter { bool operator()(const string &a, const string &b)
         { return a.size() < b.size(); } };

    size_t max_ip_length = max_element(m_unique_ips.begin(), m_unique_ips.end(), shorter())->size();
    size_t element_length = 3;
    //for(int i = 0; i < 144; i++) oss << "=";
    for(int i = 0; i < 40; i++) oss << "=";
    oss << " Loaded Threshold Settings ";
    for(int i = 0; i < 42; i++) oss << "=";
    //oss << "\n";
    
    for(auto ip : m_unique_ips) {

        vector<ThresholdHolder> holder = m_ip_holder[ip];

        vector<int> chips;
        for(auto h : holder) {
            if(!(std::find(chips.begin(),chips.end(),h.chip_id)!=chips.end())) chips.push_back(h.chip_id);
        }
        std::sort(chips.begin(),chips.end());

        oss << "\n";
        for(int i = 0; i < 109; i++) oss << "-";
        oss << "\n";
        oss<< "BOARD IP: " << ip; //<< setw(max_ip_length+2);
        for(auto chip : chips) {
            //oss << setw(max_ip_length+2) << chip;
            oss << "\n" << setw(max_ip_length-2) << "CHIP  " << chip << " | ";
            //oss << "\n" << setw(max_ip_length+2) << "CHIP  " << chip << " | ";
            //oss << " " << chip << " | ";
            vector<int> channels;
            for(auto h : holder) {
                if(!(std::find(channels.begin(), channels.end(), h.channel)!=channels.end())) channels.push_back(h.channel);
            }
            std::sort(channels.begin(),channels.end());
            //for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
            //    oss << setw(element_length) << trim_value(ip, chip, channels.at(ichan));;
            //}
            vector<int> first_32;
            vector<int> second_32;
            for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
                if(ichan<32) first_32.push_back(channels.at(ichan));
                else if(ichan >= 32) { second_32.push_back(channels.at(ichan)); }
            }

            // first 32
            for(int ichan = 0; ichan < (int)first_32.size(); ichan++) {
                oss << setw(element_length) << first_32.at(ichan);
            }
            oss << "\n";
            oss << setw(max_ip_length+2) << "             ";
            for(int ichan = 0; ichan < (int)first_32.size(); ichan++) {
                oss << setw(element_length) << trim_value(ip, chip, first_32.at(ichan));
            }

            oss << "\n";
            for(int i = 0; i < 12; i++) oss << " ";
            for(int i = 0; i < 49; i++) oss << "- ";

            // second 32
            if(channels.size()>32) {
                oss << "\n";
                oss << setw(max_ip_length+2) << "             ";
                for(int ichan = 0; ichan < (int)second_32.size(); ichan++) {
                    oss << setw(element_length) << second_32.at(ichan);
                }
                oss << "\n";
                oss << setw(max_ip_length+2) << "             ";
                for(int ichan = 0; ichan < (int)second_32.size(); ichan++) {
                    oss << setw(element_length) << trim_value(ip, chip, second_32.at(ichan));
                }

                oss << "\n";
                for(int i = 0; i < 12; i++) oss << " ";
                for(int i = 0; i < 49; i++) oss << "- ";

            }

            //for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
            //    oss << setw(element_length) << channels.at(ichan);
            //}
            //oss << "\n";
            //oss << setw(max_ip_length+2) << "              ";
            //for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
            //    oss << setw(element_length) << trim_value(ip, chip, channels.at(ichan));
            //}

            //oss << "\n";
            //for(int i = 0; i < 12; i++) oss << " ";
            //for(int i = 0; i < 66; i++) oss << "- ";
        }
    } // ip
    oss << "\n";
    for(int i = 0; i < 109; i++) oss << "=";
    oss << endl;
    return oss.str();
}
