/////////////////////////////////////////
//
// mainwindow
//
// Main GUI for VMM DAQ/DCS
//
// daniel.joseph.antrim@cern.ch
// March 2016
//
//////////////////////////////////////////

#include "mainwindow.h"
#include "ui_mainwindow.h"

//dataflow
#include "dataflowwindow.h"

// qt
#include <QDir>
#include <QFileInfoList>
#include <QFileInfo>
#include <QFileDialog>
#include <QTime>
#include <QScrollBar>
#include <QInputDialog>
#include <QMessageBox>
// multiselect
#include <QListWidget>

// vmm
#include "calibration_state.h"
#include "pulser_calibration.h"
//#include "calibration_threshold_loader.h"

// std/stl
#include <iostream>
#include <sstream>
#include <cstdlib>
#include <iomanip>
#include <map>

using namespace std;


// ------------------------------------------------------------------------- //
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    m_cmdline_vmm_type(3),
    m_cmdline_config_xml(""),
    m_cmdline_daq_config_xml(""),
    m_cmdline_channel_trims_xml(""),
    m_cmdline_output_directory(""),
    m_dbg(false),
    m_timer(0),
    ignore_channel_update(false),
    FECPORT(6007),
    DAQPORT(6006),
    VMMASICPORT(6603),
    VMMAPPPORT(6600),
    CLOCKSPORT(6602),
    vmmConfigHandler(0),
    vmmSocketHandler(0),
    vmmConfigModule(0),
    vmmRunModule(0),
    vmmDataHandler(0),
    vmmCalibModule(0),
    vmmMessageHandler(0),
    m_in_hard_reset(false),
    m_calculate_total_steps(true),
    n_total_steps_in_loop(0),
    n_current_calibration_step(0),
    n_calib_update_progress(0),
    n_times_checked_progress(0),
    m_calib_threshold_loader(0),
    m_load_thresholds(false),
    m_current_chip_threshold_loaded(false),
    m_threshold_loading_finished(false),
    m_load_threshold_ip(""),
    m_load_threshold_chip(0),
    m_commOK(false),
    m_febSetOK(true),
    m_configOK(false),
    m_tdaqOK(false),
    m_runModeOK(false),
    m_acqMode("OFF"),
    m_daqInProgress(false),
    m_hdmiMaskON(false),
    m_runDirectoryOK(true),
    m_readyToRead(false),
    m_inCalibrationLoop(false),
    m_mapping_loaded_ok(false),
    m_have_loaded_pulser(false),
    m_dataflow_window(0)
{
    ui->setupUi(this);

    m_timer = new QTimer(this);
    m_timer->setInterval(200);
    m_timer->setSingleShot(true);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(activateRC()));

    vmmMessageHandler = new MessageHandler();
    vmmMessageHandler->setMessageSize(116);
    vmmMessageHandler->setGUI(true);
    connect(vmmMessageHandler, SIGNAL(logReady()), this, SLOT(readLog()));

    const char* dir = std::getenv("HOME");
    if(dir) {
        try{ 
            std::string home_dir(dir);
            if(home_dir!="") {
                cout << "MainWindow    Setting default output directory to HOME: " << home_dir << endl;
                ui->runDirectoryField->setText(QString::fromStdString(home_dir));
            }
        }
        catch(std::exception& e) {
            cout << "MainWindow    ERROR Setting output directory: " << e.what() << endl;
        }
    }

    // remove the last tab because QtCreator is terrible
    ui->tabWidget->removeTab(3);
    ui->tabWidget->removeTab(6);

    // disable groups that are not available in MMFE8 FW (yet)
    ui->groupBox_13->setEnabled(false);
    ui->groupBox_7->setEnabled(false);
    ui->groupBox_8->setEnabled(false);


    // group together useful (repeated) items
    m_vmm_mask.push_back(ui->vmm_select_1);
    m_vmm_mask.push_back(ui->vmm_select_2);
    m_vmm_mask.push_back(ui->vmm_select_3);
    m_vmm_mask.push_back(ui->vmm_select_4);
    m_vmm_mask.push_back(ui->vmm_select_5);
    m_vmm_mask.push_back(ui->vmm_select_6);
    m_vmm_mask.push_back(ui->vmm_select_7);
    m_vmm_mask.push_back(ui->vmm_select_8);


    this->setFixedSize(1200*1.03,720); // 1400, 725
    //this->setFixedSize(1200,720); // 1400, 725
    //this->setMinimumHeight(720);
    //this->setMinimumWidth(1200);
    Font.setFamily("Arial");
    ui->loggingScreen->setReadOnly(true);
    ui->debugScreen->setReadOnly(true);
    ui->runStatusField->setReadOnly(true);

    ui->tabWidget->setStyleSheet("QTabBar::tab { height: 25px; width: 156px; }");
    //ui->tabWidget->setStyleSheet("QTabBar::tab { height: 18px; width: 100px; }");
    //ui->tabWidget->setTabText(0,"Information");
    //ui->tabWidget->setTabText(1, "Channels");
    //ui->tabWidget->setTabText(2, "Calibration");

    //msg()("Handling of XML configuration files for VMM3 configuration not yet established");
    //ui->load_config->setEnabled(false);
    //ui->write_config->setEnabled(false);

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // VMM handles
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    vmmConfigHandler = new ConfigHandler();
    vmmSocketHandler = new SocketHandler();
    vmmConfigModule  = new Configuration();
    vmmRunModule     = new RunModule();
    vmmDataHandler   = new DataHandler();
    // initialize the data_handler object
    vmmDataHandler->initialize();
    //vmmCalibModule = new CalibModule();
    // give the Run and Configuration moduel to DataHandler to pass to calibration module

    vmmConfigHandler->LoadMessageHandler(msg());
    vmmSocketHandler->LoadMessageHandler(msg());
    vmmConfigModule ->LoadMessageHandler(msg());
    vmmDataHandler  ->LoadMessageHandler(msg());
    //vmmCalibModule  ->LoadMessageHandler(msg());
    vmmRunModule    ->LoadMessageHandler(msg());


    m_dbg = false;
    vmmConfigHandler ->setDebug(false);
    vmmSocketHandler ->setDebug(false);
    vmmConfigModule  ->setDebug(false);
    vmmRunModule     ->setDebug(false);
    vmmDataHandler   ->setDebug(false);
    //vmmCalibModule   ->setDebug(false);

    // set dry run for testing
    //msg()(" !! SOCKETHANDLER SET FOR DRY RUN !! ");
    //vmmSocketHandler->setDryRun();

    // propagate command counter
    connect(vmmSocketHandler, SIGNAL(commandCounterUpdated()),
                                            this, SLOT(updateCounter()));

    // load the things
    vmmConfigModule->LoadConfig(*vmmConfigHandler);
    vmmConfigModule->LoadSocket(*vmmSocketHandler);

    vmmRunModule->LoadConfig(*vmmConfigHandler);
    vmmRunModule->LoadSocket(*vmmSocketHandler);

    vmmDataHandler->LoadModulesForCalibration(configModule(), runModule());//, socketHandle());
    connect(vmmDataHandler, SIGNAL(stop_pulser_calib_run()), this, SLOT(stop_pulser_calib_run()));

    // testing shared memory
    // make it direct connection (which means the daq thread will not be the thread in which it is processed
    // but this will not be needed when actually processing shared memory)
    //connect(ui->testSharedMemory, SIGNAL(pressed()), vmmDataHandler, SLOT(testSharedMem()), Qt::DirectConnection);


    //////////////////////////////////////////////////////////////////////
    //------------------------------------------------------------------//
    // connect DataHandler functions
    //------------------------------------------------------------------//
    //////////////////////////////////////////////////////////////////////

    connect(vmmDataHandler, SIGNAL(updateRunNumber(int)), this, SLOT(updateRunNumber(int)));
    connect(vmmDataHandler, SIGNAL(badOutputDir()), this, SLOT(badRunDirectory()));
    connect(vmmDataHandler, SIGNAL(updateCountsSend(int, int, unsigned int, unsigned int, bool)),
                    this, SLOT(updateTriggerCount(int, int, unsigned int, unsigned int, bool)));
    connect(vmmDataHandler, SIGNAL(eventCountReachedSend()), this, SLOT(eventCountReached()));
    connect(vmmDataHandler, SIGNAL(effCountReachedSend()), this, SLOT(effCountReached()));
    connect(vmmDataHandler, SIGNAL(updateCalibration()), this, SLOT(updateCalibrationState()), Qt::DirectConnection);
    connect(vmmDataHandler, SIGNAL(daqHangObserved()), this, SLOT(send_test_xadc_configuration()));
    connect(ui->do_calibration_button, SIGNAL(clicked()), this, SLOT(set_calibration()));
    connect(ui->calibration_type_charge, SIGNAL(clicked()), this, SLOT(set_calibration()));
    connect(ui->calibration_type_threshold, SIGNAL(clicked()), this, SLOT(set_calibration()));

    // calibration loop parameters
    //connect(ui->xadccalibration_global_threshold_checkbox, SIGNAL(clicked()),
    //                        this, SLOT(set_calibration_loop_params()));
    //connect(ui->xadccalibration_global_testpulse_checkbox, SIGNAL(clicked()),
    //                        this, SLOT(set_calibration_loop_params()));
    //connect(ui->xadccalibration_channel_trims_checkbox, SIGNAL(clicked()),
    //                        this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_custom_loop_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_baselines_extrapolation_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_baselines_neighbor_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_time_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_efficiency_threshold_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_efficiency_amplitude_checkbox, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));
    connect(ui->calibration_scan_do_analog, SIGNAL(clicked()),
                            this, SLOT(set_calibration_loop_params()));

/*
    connect(this, SIGNAL(monitorDataSignal(bool)), vmmDataHandler, SLOT(set_monitorData(bool)), Qt::DirectConnection);
    connect(this, SIGNAL(clearSharedMemory(int)), vmmDataHandler, SLOT(clearSharedMemory(int)), Qt::DirectConnection);
    connect(this, SIGNAL(setWriteNtuple(bool)), vmmDataHandler, SLOT(setWriteNtuple(bool)));
    connect(this, SIGNAL(setCalibrationRun(bool)), vmmDataHandler, SLOT(setCalibrationRun(bool)));
    connect(this, SIGNAL(setupOutputFiles(QString,QString)),
                    vmmDataHandler, SLOT(setupOutputFiles(QString,QString)));
    connect(vmmDataHandler, SIGNAL(setRunDirOK(bool)), this, SLOT(setRunDirOK(bool)));

    connect(this, SIGNAL(setupOutputTrees()), vmmDataHandler, SLOT(setupOutputTrees()));
    connect(this, SIGNAL(startDAQSocket()), vmmDataHandler,
                                SLOT(connectDAQSocket()));
    connect(this, SIGNAL(closeDAQSocket()), vmmDataHandler,
                                SLOT(closeDAQSocket()));

*/

    channelGridLayout = new QGridLayout(this);
    dummy = new QWidget(ui->groupBox_22);
    //dummy = new QWidget(this);
    //CreateChannelsFields();
    // multi
    CreateChannelsFieldsMulti();

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // configure some of the widgets' styles and attributes
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // make widget connections
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////

    // ~FSM preliminary
    connect(this, SIGNAL(checkFSM()), this, SLOT(updateFSM()));

    // ------------------------------------------------------ //
    // ------- ~roughly in order of typical operation ------- //
    // ------------------------------------------------------ //

    // set number of FECs
    connect(ui->numberOfFecs, SIGNAL(valueChanged(int)),
                                            this, SLOT(setNumberOfFecs(int)));
    // emit this signal to load the (if any) intial ip value
    int current_fec_idx = ui->numberOfFecs->value();
    emit ui->numberOfFecs->valueChanged(current_fec_idx);

    // changing board selection
    connect(ui->setVMMs, SIGNAL(currentIndexChanged(int)),
                                            this, SLOT(updateBoardSelection(int)));

    // ping the boards and connect sockets

    connect(ui->openConnection, SIGNAL(pressed()),
                                            this, SLOT(Connect()));
    // if the IP changes in anyway, we must re-connect
    connect(ui->ip1, SIGNAL(textChanged(QString)),
                                            this, SLOT(IPchanged(QString)));
    connect(ui->ip2, SIGNAL(textChanged(QString)),
                                            this, SLOT(IPchanged(QString)));
    connect(ui->ip3, SIGNAL(textChanged(QString)),
                                            this, SLOT(IPchanged(QString)));
    connect(ui->ip4, SIGNAL(textChanged(QString)),
                                            this, SLOT(IPchanged(QString)));
    connect(ui->numberOfFecs, SIGNAL(valueChanged(int)),
                                            this, SLOT(IPchanged(int)));

    // VMM mask button
    connect(ui->vmm_select_clear, SIGNAL(pressed()),
                                              this, SLOT(updateVMMselection()));
    connect(ui->vmm_select_add, SIGNAL(pressed()),
                                              this, SLOT(updateVMMselection()));
    connect(ui->vmm_select_subtract, SIGNAL(pressed()),
                                              this, SLOT(updateVMMselection()));

    // prepare the board configuration and send
    connect(ui->SendConfiguration, SIGNAL(pressed()),
                                    this, SLOT(prepareAndSendBoardConfig()));

    // set the run mode and send
    connect(ui->trgPulser, SIGNAL(clicked()),
                                    this, SLOT(setTriggerMode()));
    connect(ui->trgExternal, SIGNAL(clicked()),
                                    this, SLOT(setTriggerMode()));
    connect(ui->trgExternalBC, SIGNAL(clicked()),
                                    this, SLOT(setTriggerMode()));

    // send all TDAQ settings
    connect(ui->send_tdaq_settings_button, SIGNAL(clicked()),
                                    this, SLOT(send_tdaq_settings_command()));

    // set trigger settings and mode
    connect(ui->set_trigger_settings_button, SIGNAL(clicked(bool)),
                                    this, SLOT(set_tdaq_configuration(bool)));
    connect(ui->trigger_settings_defaults_button, SIGNAL(clicked()),
                                    this, SLOT(set_trigger_settings_defaults()));

    // set data acquisition on/off
    connect(ui->onACQ, SIGNAL(clicked()),
                                    this, SLOT(setACQMode()));
    connect(ui->offACQ, SIGNAL(clicked()),
                                    this, SLOT(setACQMode()));

    // set data acquisition on/off
    connect(ui->vmmPowerOn, SIGNAL(clicked()),
                                    this, SLOT(setVMMPower()));
    connect(ui->vmmPowerOff, SIGNAL(clicked()),
                                    this, SLOT(setVMMPower()));

    // start run
    connect(ui->checkTriggers, SIGNAL(clicked()),
                                    this, SLOT(triggerHandler()));
    // stop run
    connect(ui->stopTriggerCnt, SIGNAL(clicked()),
                                    this, SLOT(triggerHandler()));
    // clear trigger count
    connect(ui->clearTriggerCnt, SIGNAL(clicked()),
                                    this, SLOT(triggerHandler()));

    // update trigger count
  //  connect(vmmDataHandler, SIGNAL(checkDAQCount(bool)),
  //                                  this, SLOT(updateTriggerCount(bool)));

    /////////////////////////////////////////////////////////////
    // dataflow
    /////////////////////////////////////////////////////////////
    m_dataflow_window = new DataFlowWindow(this);
    m_dataflow_window->setWindowTitle("VERSO DataFlow");
    connect(ui->dataflow_enable_button, SIGNAL(clicked()),
                                    this, SLOT(enable_dataflow_monitor()));
    connect(vmmDataHandler, SIGNAL(update_empty_count_send(unsigned int)),
                    m_dataflow_window, SLOT(update_empty_count(unsigned int)));
    connect(vmmDataHandler, SIGNAL(occupancy_check_send(QString, int, int)),
                    m_dataflow_window, SLOT(perform_occupancy_check(QString, int, int)));
    connect(vmmDataHandler, SIGNAL(trigger_delta_check_send(int, int)),
                    m_dataflow_window, SLOT(perform_trigger_delta_check(int, int)));
    connect(m_dataflow_window, SIGNAL(close_dataflow()), this, SLOT(close_dataflow_window()));

    // ------------------------------------------------------ //
    // ------------- remaining buttons/widgets -------------- //
    // ------------------------------------------------------ //

    // direct timing mode
    connect(ui->global_direct_time_mode, SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(setDirectTimingMode(int)));
    connect(ui->global_stpp, SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(setDirectTimingMode(int)));
    connect(ui->global_stot, SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(setDirectTimingMode(int)));

    // monitoring related
    connect(ui->monitoring_enable_button, SIGNAL(clicked()),
                                    this, SLOT(updateMonitoringState()));
    connect(ui->monitoring_send_config_button, SIGNAL(clicked()),
                                    this, SLOT(updateMonitoringState()));

    ui->testMonButton->setEnabled(false);
    //connect(ui->testMonButton, SIGNAL(clicked()),
    //                                vmmDataHandler, SLOT(testMon()), Qt::DirectConnection);
    //connect(this, SIGNAL(testMonitoringSignal()), vmmDataHandler, SLOT(testMonitoring()));


    // select the output directory for files
    connect(ui->selectDir_output, SIGNAL(clicked()),
                                    this, SLOT(selectOutputDirectory()));

    // select the DAQ configuration file (needed for mapping, etc...)
    connect(ui->selectDir_daqSetup, SIGNAL(clicked()),
                                    this, SLOT(selectDAQSetupFile()));

    connect(ui->daqSetupFilenameField, SIGNAL(returnPressed()),
                                    this, SLOT(checkRequestedFile()));

    connect(ui->load_daqSetup, SIGNAL(clicked()),
                                    this, SLOT(loadDAQSetup()));
    connect(ui->disable_mapping, SIGNAL(clicked()),
                                    this, SLOT(disableMapping()));

    // select the configuration xml file (if you want to load a pre-existing
    // configuration)
    connect(ui->selectDir_config, SIGNAL(clicked()),
                                    this, SLOT(selectConfigXmlFile()));

    connect(ui->configXmlFilenameField, SIGNAL(returnPressed()),
                                    this, SLOT(checkRequestedFile()));

    connect(ui->load_config, SIGNAL(clicked()),
                                    this, SLOT(loadConfigurationFromFile()));

    // clear the user comments
    connect(ui->userComments_clear, SIGNAL(clicked()),
                                    this, SLOT(clearUserComments()));

   // // load the board/global configuration from the XML
   // connect(ui->loadConfigXMLFileButton, SIGNAL(clicked()),
   //                                 this, SLOT(loadConfigurationFromFile()));
    // write the total configuration to an XML file
    connect(ui->write_config, SIGNAL(clicked()),
                                    this, SLOT(writeConfigurationToFile()));
    //connect(ui->writeConfigXMLFileButton, SIGNAL(clicked()),
    //                                this, SLOT(writeConfigurationToFile()));

    // set event headers
    //connect(ui->setEvbld, SIGNAL(clicked()),
    //                                this, SLOT(setAndSendEventHeaders()));

    // reset ASICs
    connect(ui->vmm3_hard_reset_button, SIGNAL(clicked()),
                                    this, SLOT(performHardReset()));

    // reset FPGA
    connect(ui->fpga_reset_button, SIGNAL(clicked()),
                                    vmmRunModule, SLOT(resetFPGA()));

    //connect(ui->asic_reset, SIGNAL(clicked()),
    //                                this, SLOT(resetASICs()));
    //// warm init or reboot FEC
    //connect(ui->fec_WarmInit, SIGNAL(clicked()),
    //                                this, SLOT(resetFEC()));
    //connect(ui->fec_reset, SIGNAL(clicked()),
    //                                this, SLOT(resetFEC()));

    //// include the ART bits in the channel mask
    //connect(ui->enableART, SIGNAL(clicked()),
    //                                this, SLOT(setART()));

    //// link status
    //connect(ui->linkPB, SIGNAL(clicked()),
    //                                this, SLOT(checkLinkStatus()));

    //// reset links
    //connect(ui->resetLinks, SIGNAL(clicked()),
    //                                this, SLOT(resetLinks()));

    // daq ~conversion to mV
    connect(ui->sdt, SIGNAL(valueChanged(int)),
                                    this, SLOT(changeDACtoMVs(int)));
    connect(ui->sdp_2, SIGNAL(valueChanged(int)),
                                    this, SLOT(changeDACtoMVs(int)));

    ///////////////////////////////////////////////////////////////
    // trigger settings
    ///////////////////////////////////////////////////////////////

    ///////////////////////////////////////////////////////////////
    // configuration of clocks
    ///////////////////////////////////////////////////////////////
    connect(ui->cktk_ckbt_set_defaults_button, SIGNAL(clicked()),
                                    this, SLOT(set_clock_defaults()));
    connect(ui->cktp_set_defaults_button, SIGNAL(clicked()),
                                    this, SLOT(set_clock_defaults()));
    connect(ui->clocks_set_defaults_button, SIGNAL(clicked()),
                                    this, SLOT(set_clock_defaults()));
    connect(ui->clocks_send_settings_button, SIGNAL(clicked()),
                                    this, SLOT(send_clock_configuration()));
    connect(ui->ckbc_frequency, SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(update_tp_skew_label(int)));

    // adjusting the monitorign rate
    connect(ui->monitoring_rate, SIGNAL(valueChanged(int)),
                                    vmmDataHandler, SLOT(update_monitoring_rate(int)));

    // test pulse config
    //connect(ui->setTp_s6, SIGNAL(clicked()),
    //                                this, SLOT(configureTP()));

    // move to bottom of log screen
    connect(ui->logBottom, SIGNAL(clicked()),
                                    this, SLOT(updateLogScreen()));

    // set debug and propagate to the tools
    //connect(ui->enableDebugPB, SIGNAL(stateChanged(int)),
    connect(ui->enableDebugPB, SIGNAL(clicked()),
                                    this, SLOT(toggleDebug()));

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    //  calibration
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    // test
    connect(ui->send_xadc_configuration_button, SIGNAL(pressed()),
                                this, SLOT(send_test_xadc_configuration())); 

    // load a thresholds xml calibration file
    connect(ui->selectDir_thresholdCalibration, SIGNAL(clicked()),
                                    this, SLOT(selectCalibrationFile()));
    connect(ui->thresholdCalibrationFileName, SIGNAL(returnPressed()),
                                    this, SLOT(checkRequestedFile()));
    connect(ui->load_thresholdCalibration, SIGNAL(clicked()),
                                    this, SLOT(loadThresholdCalibrationFromFile()));
    connect(ui->disable_thresholdCalibration, SIGNAL(clicked()),
                                    this, SLOT(disableThresholdCalibration()));
    connect(ui->show_thresholdCalibration, SIGNAL(clicked()),
                                    this, SLOT(printThresholdCalibration()));

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // MMFE8 IP configuration
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    connect(ui->ipset_begin_button, SIGNAL(clicked()),
                        this, SLOT(setIPConfigurationState()));
    connect(ui->ipset_done_button, SIGNAL(clicked()),
                        this, SLOT(setIPConfigurationState()));
    connect(ui->ipset_send_button, SIGNAL(clicked()),
                        this, SLOT(sendIPConfiguration()));
    connect(ui->ipset_default_daqIP_button, SIGNAL(clicked()),
                        this, SLOT(setIPConfigurationState()));

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // monitoring
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    connect(this, SIGNAL(set_monitoring(bool, bool)),
                        vmmDataHandler, SLOT(recv_setup_monitoring(bool, bool)));
    connect(this, SIGNAL(send_monitor_config()),
                        vmmDataHandler, SLOT(recv_send_monitor_config()), Qt::DirectConnection);

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // initial state
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    SetInitialState();
    InitializeChannelArrayValues();
    //qDebug() << "MainWindow thread: " << QThread::currentThreadId();

    // for now disable the ability to set the online monitoring ON
    // will wait until this is robus
    //msg()(" *** Disabling online monitoring *** ");
    //ui->monitoring_enable_button->setEnabled(false);

    /////////////////////////////////////////////////////////////////////
    //-----------------------------------------------------------------//
    // VMM TYPE
    //-----------------------------------------------------------------//
    /////////////////////////////////////////////////////////////////////
    set_interface_for_vmm_type(m_cmdline_vmm_type);
    connect(ui->set_vmm2_button, SIGNAL(clicked()), this, SLOT(set_interface_for_vmm_type()));
    connect(ui->set_vmm3_button, SIGNAL(clicked()), this, SLOT(set_interface_for_vmm_type()));
    connect(ui->set_l0_button, SIGNAL(clicked()), this, SLOT(set_l0_mode()));
    

}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_vmm_type(int type)
{
    m_cmdline_vmm_type = type;
    if(type==2) ui->set_vmm2_button->click();
    else if(type==3) ui->set_vmm3_button->click();
}
// ------------------------------------------------------------------------- //
int MainWindow::vmm_type()
{
    if(ui->set_vmm2_button->isChecked()) {
        return 2;
    }
    else if(ui->set_vmm3_button->isChecked()) {
        return 3;
    }
    return -1;
}
// ------------------------------------------------------------------------- //
void MainWindow::set_interface_for_vmm_type()
{
    if(QObject::sender() == ui->set_vmm2_button) {
        set_interface_for_vmm_type(2);
    }
    else if(QObject::sender() == ui->set_vmm3_button) {
        set_interface_for_vmm_type(3);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::set_l0_mode()
{
    if(ui->set_vmm2_button->isChecked()) {
        stringstream sx;
        sx << "ERROR Cannot set L0 readout mode when set for VMM2 readout";
        msg()(sx);sx.str("");
        ui->global_sL0ena->setCurrentIndex(0);
        ui->global_sL0enaV->setCurrentIndex(0);
        ui->set_l0_button->setChecked(0);
        return;
    }
    else if(ui->set_l0_button->isChecked()) {
        ui->global_sL0ena->setCurrentIndex(1);
        ui->global_sL0enaV->setCurrentIndex(1);
        ui->trigExternalBCLatency->setValue(143);
    }
    else if(!ui->set_l0_button->isChecked()) {
        ui->global_sL0ena->setCurrentIndex(0);
        ui->global_sL0enaV->setCurrentIndex(0);
        ui->trigExternalBCLatency->setValue(0);
    }
        
}
// ------------------------------------------------------------------------- //
void MainWindow::set_interface_for_vmm_type(int type)
{

    stringstream sx;
    if(dbg()) {
        sx << "Setting DAQ for VMM" << type;
        msg()(sx);sx.str("");
    }

    ////////////////////////////////////////////////////
    // propagate this to whatever needs it
    ////////////////////////////////////////////////////
    configModule().set_vmm_type(type);
    runModule().set_vmm_type(type);
    dataHandle().set_vmm_type(type);

    ////////////////////////////////////////////////////
    // now update the interface
    ////////////////////////////////////////////////////

    bool keep_stuff = (type==3);

    // disable trigger settings
    ui->frame_6->setEnabled(keep_stuff);

    // disable fixed window TAC readout
    if(m_commOK)
        ui->trgExternalBC->setEnabled(keep_stuff);

    // CKTP configuration
    ui->frame_11->setEnabled(keep_stuff);
    if(m_commOK)
    ui->clocks_send_settings_button->setEnabled(keep_stuff);

    // CKBC configuration
    ui->frame_10->setEnabled(keep_stuff);

    // STH is now SP for channel registers if VMM2
    if(keep_stuff) {
        ui->STHLabel_2->setText("STH");
        ui->STHLabel2_2->setText("STH");
    }
    else {
        ui->STHLabel_2->setText("SP");
        ui->STHLabel2_2->setText("SP");
    }

    // channel trim for calibration range boundaries
    int max_trim = (keep_stuff ? 31 : 15);
    ui->xadccalibration_channel_trim_start->setMaximum(max_trim);
    ui->xadccalibration_channel_trim_end->setMaximum(max_trim);
    ui->xadccalibration_channel_trim_end->setValue(max_trim);
    //ui->xadccalibration_channel_trim_start->setValue(0);

    if(ui->set_l0_button->isChecked() && !keep_stuff) {
        ui->set_l0_button->setChecked(false);
        ui->global_sL0ena->setCurrentIndex(0);
        ui->global_sL0enaV->setCurrentIndex(0);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_config_xml(string filename)
{
    stringstream sx;
    if(filename=="") return;

    bool exists = std::ifstream(filename).good();
    if(!exists) {
        sx << "ERROR Configuration XML file provided at the command line could not be found!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) XML file: " << filename;
        msg()(sx); sx.str("");
        ui->load_config->setEnabled(false);
        return;
    }

    bool is_xml = filename.find(".xml") != std::string::npos;
    if(!is_xml) {
        sx << "ERROR Configuration file provided at the command line is not an XML file!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) file: " << filename;
        msg()(sx);sx.str("");
        ui->load_config->setEnabled(false);
        return;
    }

    ui->configXmlFilenameField->setText(QString::fromStdString(filename));
    ui->load_config->setEnabled(true);
    sx << "Loading configuration XML file provided at the command line";
    msg()(sx); sx.str("");
    sx << " > config: " << filename;
    msg()(sx); sx.str("");
    ui->load_config->click();

    m_cmdline_config_xml = filename;
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_daq_config_xml(string filename)
{
    stringstream sx;
    if(filename=="") return;
    bool exists = std::ifstream(filename).good();
    if(!exists) {
        sx << "ERROR Detector configuration XML file provided at the command line could not be found!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) detector XML file: " << filename;
        msg()(sx); sx.str("");
        ui->load_daqSetup->setEnabled(false);
        return;
    }

    bool is_xml = filename.find(".xml") != std::string::npos;
    if(!is_xml) {
        sx << "ERROR DAQ setup file provided at the command line is not an XML file!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) file: " << filename;
        msg()(sx); sx.str("");
        ui->load_daqSetup->setEnabled(false);
        return;
    }

    ui->daqSetupFilenameField->setText(QString::fromStdString(filename));
    ui->load_daqSetup->setEnabled(true);
    sx << "Loading detector configuration XML file provided at the command line";
    msg()(sx); sx.str("");
    sx << " > det config: " << filename;
    msg()(sx); sx.str("");
    ui->load_daqSetup->click();

    m_cmdline_daq_config_xml = filename;
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_output_directory(string directory)
{
    stringstream sx;
    if(directory=="") return;
    bool exists = std::ifstream(directory).good();
    if(!exists) {
        sx << "ERROR Output directory provided at the command line could not be found!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) output directory: " << directory;
        msg()(sx); sx.str("");
        return;
    }
    sx << "Setting output directory to that proviced at the command line";
    msg()(sx); sx.str("");
    sx << " > output directory: " << directory;
    msg()(sx); sx.str("");
    ui->runDirectoryField->setText(QString::fromStdString(directory));
    m_cmdline_output_directory = directory;
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_channel_trims_xml(std::string file)
{
    stringstream sx;
    if(file=="") return;
    bool exists = std::ifstream(file).good();
    if(!exists) {
        sx << "ERROR Channel threshold trim XML file provided at the command line could not be found!";
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) channel threshold trim XML file: " << file;
        msg()(sx); sx.str("");
        ui->load_thresholdCalibration->setEnabled(false);
        return;
    }
    bool is_xml = file.find(".xml") != std::string::npos;
    if(!is_xml) {
        sx << "ERROR Channel threshold trims file provided at the command line is not an XML file!"; 
        msg()(sx); sx.str("");
        sx << "ERROR Provided (bad) channel trims file: " << file;
        msg()(sx); sx.str("");
        ui->load_thresholdCalibration->setEnabled(false);
        return;
    }


    sx << "Setting channel threshold trim XML to that provided at the command line";
    msg()(sx); sx.str("");
    sx << " > channel trims file: " << file << endl;
    msg()(sx); sx.str("");
    ui->thresholdCalibrationFileName->setText(QString::fromStdString(file));
    ui->load_thresholdCalibration->setEnabled(true);
    ui->load_thresholdCalibration->click();

    vmmDataHandler->toggle_occupancy_check(false);
    m_dataflow_window->stop_dataflow_monitor();
    m_dataflow_window->close();

    m_cmdline_channel_trims_xml = file;
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::load_cmdline_comms(int n_feb, int n_vmm, int base_ip)
{
    stringstream sx;

    if(n_feb>0) {
        ui->numberOfFecs->setValue(n_feb);
    }
    if(n_vmm>0) {
        for(int i = 0; i < n_vmm; i++) {
            m_vmm_mask.at(i)->setChecked(true);
        }
        for(int i = n_vmm; i < 8; i++) {
            m_vmm_mask.at(i)->setChecked(false);
        }
    }
    if(base_ip>0) {
        sx.str(""); sx << base_ip;
        ui->ip4->setText(QString::fromStdString(sx.str()));
    }
}
// ------------------------------------------------------------------------- //
MainWindow::~MainWindow()
{
    delete channelGridLayout;
    delete ui;
}
// ------------------------------------------------------------------------- //
void MainWindow::setDirectTimingMode(int idx)
{
    if(QObject::sender() == ui->global_direct_time_mode) {
        int stot_tmp = (idx & 1);
        int stpp_tmp = ((idx & 2) >> 1);

        ui->global_stpp->setCurrentIndex(stpp_tmp);
        ui->global_stot->setCurrentIndex(stot_tmp);
    }
    else if(QObject::sender() == ui->global_stpp || QObject::sender() == ui->global_stot) {
        int stot_tmp = ui->global_stot->currentIndex();
        int stpp_tmp = ui->global_stpp->currentIndex();
        int mode = 0;
        mode |= stot_tmp;
        mode |= (stpp_tmp << 1);
        ui->global_direct_time_mode->setCurrentIndex(mode);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::enable_dataflow_monitor()
{

    stringstream m;
    if(ui->dataflow_enable_button->isChecked()) {
        m << "Enabling dataflow monitor";
        m_dataflow_window->show();
        if(m_daqInProgress) {
            m << " " << (m_start_run_time.elapsed()/1000.0) << " seconds after run start";
            vmmDataHandler->toggle_occupancy_check(true);
            m_dataflow_window->start_dataflow_monitor(ui->runNumber->value(), m_start_run_time);
        }
        m_dataflow_window->initialize_occupancy_plots(ui->ip4->text().toInt(), ui->numberOfFecs->value());
        m_dataflow_window->initialize_trig_delta_plots(ui->ip4->text().toInt(), ui->numberOfFecs->value());
        msg()(m);
        //m_dataflow_window->start_dataflow_monitor();
    }
    else {
        m << "Disabling dataflow monitor";
        msg()(m);
        m_dataflow_window->stop_dataflow_monitor();
        vmmDataHandler->toggle_occupancy_check(false);
        m_dataflow_window->close();
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::close_dataflow_window()
{
    ui->dataflow_enable_button->setChecked(false);
    m_dataflow_window->stop_dataflow_monitor();
    vmmDataHandler->toggle_occupancy_check(false);
    //msg()("Disabling dataflow monitor");
}
// ------------------------------------------------------------------------- //
void MainWindow::updateMonitoringState()
{

    ////////////////////////////////////////////////////////////
    // user is toggling on/off monitoring
    ////////////////////////////////////////////////////////////
    if(QObject::sender() == ui->monitoring_enable_button) {
        bool already_running = m_daqInProgress;
        if(ui->monitoring_enable_button->isChecked() && m_mapping_loaded_ok) {
            //dataHandle().set_monitoring(true);
            emit set_monitoring(true, already_running);
            ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: rgb(15, 147, 255);}");
            /*

            bool setup_is_ok = true;

            stringstream sx;

            ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: rgb(15, 147, 255);}");
            bool ok;

            QHostAddress address = QHostAddress::LocalHost;
            string ip_address = address.toString().toStdString();
            stringstream ip_default;
            int port_default = 2224;
            ip_default << address.toString().toStdString() << ":" << port_default;

            QStringList ipSettings = QString::fromStdString(ip_default.str()).split(":");

            //QString text = QInputDialog::getText(this, tr("Monitoring Address"), tr("Provide Monitoring Address [<IP>:<PORT>]"),  QLineEdit::Normal, QString::fromStdString(ip_default.str()), &ok);

            //QStringList ipSettings = text.split(":");
            //if(ipSettings.size()!=2) {
            //    setup_is_ok = false;

            //    sx << "ERROR Input monitoring settings has bad form. Must be <IP>:<PORT>";
            //    msg()(sx); sx.str("");
            //    sx << "ERROR You have provided: " << text.toStdString();
            //    msg()(sx); sx.str("");
            //    sx << "ERROR Monitoring will not be initialized";
            //    msg()(sx); sx.str("");
            //}
            //else if(ipSettings.size()==2) {
            //    QStringList ip_split = ipSettings[0].split(".");
            //    if(ip_split.size()!=4) {

            //        setup_is_ok = false;

            //        sx << "ERROR Provided IP address has bad form. Must be IPv4.";
            //        msg()(sx); sx.str("");
            //        sx << "ERROR You have provided: " << ipSettings[0].toStdString();
            //        msg()(sx); sx.str("");
            //        sx << "ERROR Monitoring will not be initialized";
            //        msg()(sx); sx.str("");
            //    }
            //    if(setup_is_ok && (ipSettings[1].length()!=4)) {
            //        setup_is_ok = false;
            //        sx << "ERROR Provided port (" << ipSettings[1].toStdString() << ") must be 4 digits";
            //        msg()(sx); sx.str("");
            //        sx << "ERROR Monitoring will not be initialized";
            //        msg()(sx); sx.str("");
            //    }
            //}

            if(setup_is_ok)
                ip_address = ipSettings[0].toStdString();

            int port = port_default;
            if(setup_is_ok) {
                try {
                    port = std::stoi(ipSettings[1].toStdString());
                }
                catch(std::exception &e) {
                    setup_is_ok = false;
                    sx << "ERROR Provided port (" << ipSettings[1].toStdString() << ") must be an integer!";
                    msg()(sx); sx.str("");
                    sx << "ERROR Monitoring will not be initialized";
                    msg()(sx); sx.str("");
                }
            }

            if(setup_is_ok)  {
                dataHandle().setupMonitoring(setup_is_ok, ip_address, port, already_running);
                ui->monitoring_send_config_button->setEnabled(true);
            }
            else {
                ui->monitoring_send_config_button->setEnabled(true);
                //ui->monitoring_send_config_button->setEnabled(false);
                dataHandle().setupMonitoring(false, "", 0, already_running);
                ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: white;}");
                ui->monitoring_enable_button->setChecked(false);
            }
            */
        }
        else {
            if(!m_mapping_loaded_ok) {
                msg()("Cannot enable monitoring, DAQ configuration is not loaded");
            }
            ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: white;}");
            //dataHandle().setupMonitoring(false, "", 0);
            emit set_monitoring(false, false);
            ui->monitoring_send_config_button->setEnabled(true);
            //ui->monitoring_send_config_button->setEnabled(false);
        }
    } // monitoring enable button press
    ////////////////////////////////////////////////////////////
    // user is sending the board/chamber configuration to monitoring
    ////////////////////////////////////////////////////////////
    else if(QObject::sender() == ui->monitoring_send_config_button) {
        stringstream sx;
        if(ui->monitoring_enable_button->isChecked()) {
            emit send_monitor_config();
        }
        /*
            bool send_ok = dataHandle().sendMonitoringConfiguration();
            if(send_ok) {
                if(dbg())
                    msg()("Monitoring configuration sent successfully");
                cout << "Monitoring configuration sent" << endl;
                //if(ui->checkTriggers->isEnabled()) {
                //    ui->checkTriggers->setEnabled(false);
                //    delay();
                //    delay();
                //    delay();
                //    delay();
                //    ui->checkTriggers->setEnabled(true);
                //}
            }
            else {
                msg()("WARNING Monitoring configuration failed to send");

                ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: white;}");
                dataHandle().setupMonitoring(false, "", 0);
                //ui->monitoring_send_config_button->setEnabled(false);
                ui->monitoring_send_config_button->setEnabled(true);
                ui->monitoring_enable_button->setChecked(false);
            }
        }
        else {
            sx << "ERROR Unable to send monitoring configuration while online monitoring is not enabled!";
            msg()(sx);sx.str("");
        }
        */


    } // send monitor config
}
// ------------------------------------------------------------------------- //
void MainWindow::send_test_xadc_configuration()
{
    qDebug() << "Sending xADC configuration";
    bool ok;
    //int vmm_id = ui->calibration_xadc_vmm_select->value();
    int vmm_id = 1; 
    for(int i = 0; i < 8; i++) {
        if(m_vmm_mask.at(i)->isChecked()) {
            vmm_id = (i+1); // if multiply selected, for now configure only the first (until we implement chip loop)
            break;
        }
    }
    int n_samples = ui->calibration_n_samples_select->text().toInt(&ok,10);
    int sample_period = ui->xadccalibration_sample_period->value();
    ui->SendConfiguration->click();
    CalibrationType calib_type = get_calibration_type();
    dataHandle().initializeCalibration(vmm_id, n_samples, sample_period, calib_type, false, do_l0_decoding());
    //QString ip_to_sample = ips[0];
    QString ip_to_sample = ips[ui->setVMMs->currentIndex()];
    dataHandle().start_xadc_sampling(ip_to_sample.toStdString());

}
// ------------------------------------------------------------------------- //
void MainWindow::testMon()
{
    emit testMonitoringSignal();
    //dataHandle().testMonitoring();
}
// ------------------------------------------------------------------------- //
void MainWindow::selectOutputDirectory()
{
    stringstream sx;

    QFileDialog getdir;
    QString dirStr = QFileDialog::getExistingDirectory(this,
                        tr("Select Output Directory"), ".",
                        QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks); 
    if(dirStr=="") return;

    msg()("Setting output directory to:\n    " + dirStr.toStdString());
    ui->runDirectoryField->setText(dirStr);
    QString check = dirStr;
    if(!ui->runDirectoryField->text().endsWith("/")) check = check + "/";

    dataHandle().setOutputDir(check.toStdString());

//    int run_number = DataHandler::checkForExistingFiles(dirStr.toStdString(),
//                                            ui->runNumber->value());
//    if( !(run_number == ui->runNumber->value()) ) {
//        sx << "Re-setting run number to: " << run_number;    
//        msg()(sx); sx.str("");
//    }
//    ui->runNumber->setValue(run_number);
}
// ------------------------------------------------------------------------- //
void MainWindow::checkRequestedFile()
{
    stringstream sx;
    if(QObject::sender() == ui->daqSetupFilenameField) {
        bool daq_file_bad = false;


        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white; }");
        QString current_text = ui->daqSetupFilenameField->text();
        if(current_text=="") { ui->load_daqSetup->setEnabled(false); daq_file_bad = true; }
        else {
            QFile ftest(current_text);
            if(!ftest.exists()) {
                sx << "DAQ setup file (" << current_text.toStdString() << ") bad";
                msg()(sx); sx.str("");
                ui->load_daqSetup->setEnabled(false);

                ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); }"); 
                delay();
                delay();
                ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white; }");
                delay();

                daq_file_bad = true;
            }
            else {
                ui->load_daqSetup->setEnabled(true);
                ui->load_daqSetup->click();
            }
        }

        // reset the mapping configuration regardless
        dataHandle().resetMonitoringConfiguration();
        if(daq_file_bad) {
            if(ui->monitoring_enable_button->isChecked()) {
                ui->monitoring_enable_button->click();
            }
            ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: white;}");
            ui->monitoring_enable_button->setEnabled(false);
            ui->monitoring_send_config_button->setEnabled(true);
            //ui->monitoring_send_config_button->setEnabled(false);
            m_mapping_loaded_ok = false;
        }

    }
    else if(QObject::sender() == ui->configXmlFilenameField) {
        ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: white; }");
        QString current_text = ui->configXmlFilenameField->text();
        if(current_text=="") ui->load_config->setEnabled(false);
        else {
            QFile ftest(current_text);
            if(!ftest.exists()) {
                sx << "Configuration file (" << current_text.toStdString() << ") bad";
                msg()(sx); sx.str("");
                ui->load_config->setEnabled(false);

                ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); }"); 
                delay();
                delay();
                ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: white; }");
                delay();
            }
            else {
                ui->load_config->setEnabled(true);
                ui->load_config->click();
            }
        }
    }
    ////////////////////////////////////////////////////////////
    // threshold calibration loading
    ////////////////////////////////////////////////////////////
    else if(QObject::sender() == ui->thresholdCalibrationFileName) {
        ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: white; }");
        QString current_text = ui->thresholdCalibrationFileName->text();
        if(current_text=="") ui->load_thresholdCalibration->setEnabled(false);
        else {
            QFile ftest(current_text);
            if(!ftest.exists()) {
                sx.str("");
                sx << "Threshold calibration file (" << current_text.toStdString() << ") bad";
                msg()(sx); sx.str("");
                ui->load_thresholdCalibration->setEnabled(false);

                ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); }"); 
                delay();
                delay();
                ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: white; }");
                delay();

                ui->disable_thresholdCalibration->click();
                
            } // doesn't exist
            else {
                ui->load_thresholdCalibration->setEnabled(true);
                ui->load_thresholdCalibration->click();
            }
        }
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::selectDAQSetupFile()
{
    ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white;}");
    stringstream sx;
    // get the file from the user
    QString filename = QFileDialog::getOpenFileName(this,
        tr("Load DAQ configuration XML"), "../readout_configuration/",
        tr("XML Files (*.xml)"));
    if(filename.isNull()) return;

    QFile ftest(filename);
    //ftest.setFileName(filename);
    if(!ftest.exists()) {
    //if(!ftest.open(QIODevice::ReadWrite)) {
        sx << "DAQ setup file (" << filename.toStdString() << ") unable to be opened";
        msg()(sx); sx.str("");
        ui->daqSetupFilenameField->setText("");

        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0);}"); 
        delay();
        delay();
        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white;}");
        delay();
    }
    else {
        ui->daqSetupFilenameField->setText(filename);
        sx << "Setting DAQ setup file to:\n    " << filename.toStdString();
        msg()(sx); sx.str("");
        ui->load_daqSetup->setEnabled(true);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::loadConfigurationFromFile()
{
    bool ok = false;
    if(ui->configXmlFilenameField->text().toStdString() != "")
        ok = configHandle().LoadConfig(ui->configXmlFilenameField->text());

    if(ok) {
        msg()("Configuration loaded successfully");
        ui->configXmlFilenameField->setStyleSheet("background-color: green");
        updateConfigState();
    }
    else {
        msg()("ERROR Configuration unable to be loaded from requested file");
        ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0);}");
        delay();
        delay();
        ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: white; }");
        delay();
    }

//    // Load the file from the user
//    QString filename = QFileDialog::getOpenFileName(this,
//        tr("Load Configuration XML File"), "../configs/",
//        tr("XML Files (*.xml)"));
//    if(filename.isNull()) return;
//
//    stringstream sx;
//    sx << "Loading configuration from file: " << filename.toStdString();
//    msg()(sx);
//
//    // pass the file to ConfigHandler to parse and load
//    configHandle().LoadConfig(filename);
//    updateConfigState();
}
// ------------------------------------------------------------------------- //
void MainWindow::disableMapping()
{
    if(m_mapping_loaded_ok) {
        msg()("Disabling DAQ configuration mapping");
        msg()(" > No longer providing FEB channel or detector element mapping");

        // toggle mapping flags
        dataHandle().disableMapping();
        m_mapping_loaded_ok = false;
        // prevent attempts to send monitoring data, which needs mapped data
        dataHandle().resetMonitoringConfiguration();
        if(ui->monitoring_enable_button->isChecked()) {
            ui->monitoring_enable_button->click();
        }


        // just set background to white no matter what
        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white;}");
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::loadDAQSetup()
{
    bool ok = false;
    if(ui->daqSetupFilenameField->text().toStdString() != "")
        ok = dataHandle().loadMapping(ui->daqSetupFilenameField->text().toStdString());

    if(ok) {
        msg()("DAQ setup loaded successfully");
        ui->daqSetupFilenameField->setStyleSheet("background-color: green");
        updateIPs();
        dataHandle().resetMonitoringConfiguration();
        ui->monitoring_enable_button->setEnabled(true);
        m_mapping_loaded_ok = true;
    }
    else {
        msg()("DAQ setup unable to be loaded");
        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0);}"); 
        delay();
        delay();
        ui->daqSetupFilenameField->setStyleSheet("QLineEdit { background: white;}");
        delay();
        dataHandle().resetMonitoringConfiguration();
        if(ui->monitoring_enable_button->isChecked()) {
            ui->monitoring_enable_button->click();
        }
        ui->monitoring_enable_button->setEnabled(false);
        ui->monitoring_send_config_button->setEnabled(true);
        //ui->monitoring_send_config_button->setEnabled(false);
        ui->monitoring_enable_button->setStyleSheet("QPushButton {background-color: white;}");
        m_mapping_loaded_ok = false;
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::clearUserComments()
{
    if(ui->userComments->isEnabled()) {
        ui->userComments->setText("");
    }
    else {
        QString new_text = ui->userComments->text();
        new_text = new_text.split("comment: ")[1];
        ui->userComments->setText(new_text);
        ui->userComments->setEnabled(true);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::updateIPs()
{
    std::string ipstring = dataHandle().getFirstIP();
    int nfecs = dataHandle().getNumberOfFecs();

    if(ipstring != "") {
        QString ip = QString::fromStdString(ipstring);
        QStringList ip_split = ip.split(".");
        ui->ip1->setText(ip_split[0]);
        ui->ip2->setText(ip_split[1]);
        ui->ip3->setText(ip_split[2]);
        ui->ip4->setText(ip_split[3]);
    }
    else {
        ui->ip1->setText("10");
        ui->ip2->setText("0");
        ui->ip3->setText("0");
        ui->ip4->setText("2");
    }
    ui->numberOfFecs->setValue(nfecs);
    
}
// ------------------------------------------------------------------------- //
void MainWindow::selectConfigXmlFile()
{
    stringstream sx;
    QString filename = QFileDialog::getOpenFileName(this,
        tr("Load board configuration XML"), "../readout_configuration/configuration/",
        tr("XML Files (*.xml)"));
    if(filename.isNull()) return;

    QFile ftest(filename);
    //ftest.setFileName(filename);
    if(!ftest.exists()) {
    //if(!ftest.open(QIODevice::ReadWrite)) {
        sx << "Configuration file (" << filename.toStdString() << ") unable to be opened";
        msg()(sx); sx.str("");
        ui->configXmlFilenameField->setText("");

        ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); selection-background-color: rgb(233, 99, 0); }"); 
        delay();
        delay();
        ui->configXmlFilenameField->setStyleSheet("QLineEdit { background: white; selection-background-color: rgb(233, 99, 0); }");
    }
    else {
        ui->configXmlFilenameField->setText(filename);
        sx << "Setting configuration file to:\n    " << filename.toStdString();
        msg()(sx); sx.str("");
        ui->load_config->setEnabled(true);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::selectCalibrationFile()
{
    stringstream sx;
    QString select_text = "";
    QString default_lookup_dir = "./";
    QString file_type = "";

    QLineEdit* field = nullptr;
    QPushButton* load_button = nullptr;

    if(QObject::sender() == ui->selectDir_thresholdCalibration) {
        field = ui->thresholdCalibrationFileName;
        load_button = ui->load_thresholdCalibration;

        select_text = "Select threshold calibration file";
        default_lookup_dir = "./";
        file_type = "threshold calibration";
    }

    QString filename = QFileDialog::getOpenFileName(this,
                tr(select_text.toStdString().c_str()), default_lookup_dir, tr("XML Files (*.xml)"));
    if(filename.isNull()) return;

    QFile ftest(filename);
    if(!ftest.exists()) {
        sx.str("");
        sx << "File for " << file_type.toStdString() << " (" << filename.toStdString() << ") unable to be opend!";
        msg()(sx); sx.str("");
        field->setText("");
        field->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); selection-background-color: rgb(233, 99, 0); }"); 
        delay();
        delay();
        field->setStyleSheet("QLineEdit { background: white; selection-background-color: rgb(233, 99, 0); }");
    }
    else {
        field->setText(filename);
        sx.str("");
        sx << "Setting " << file_type.toStdString() << " file to:\n    " << filename.toStdString();
        msg()(sx); sx.str("");
        load_button->setEnabled(true);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::loadThresholdCalibrationFromFile()
{
    msg()("Loading threshold calibration");

    //////////////////////////////////////////////
    // initialize and/or clear
    //////////////////////////////////////////////
    if(!m_calib_threshold_loader) {
        m_calib_threshold_loader = new CalibrationThresholdLoader();
    }
    else {
        m_calib_threshold_loader->clear();
    }
    //////////////////////////////////////////////
    // load the file
    //////////////////////////////////////////////
    string calib_file_to_load = ui->thresholdCalibrationFileName->text().toStdString();
    if(!m_calib_threshold_loader->load_file(calib_file_to_load)) {
        stringstream sx;
        sx << "ERROR Unable to load threshold calibration(s) from file: " << calib_file_to_load;
        msg()(sx); sx.str("");
        m_calib_threshold_loader->clear();

        ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: rgb(220, 0, 0);}");
        delay();
        ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: white; }");
        m_load_thresholds = false;

        ui->disable_thresholdCalibration->setEnabled(false);
        ui->show_thresholdCalibration->setEnabled(false);

        return;
    }
    else {
        msg()("Threshold calibration loaded successfully!");
        ui->thresholdCalibrationFileName->setStyleSheet("background-color: green");
        m_load_thresholds = true;
        ui->disable_thresholdCalibration->setEnabled(true);
        ui->show_thresholdCalibration->setEnabled(true);
        
    }
    vmmMessageHandler->standard_message(m_calib_threshold_loader->summary_table());

}
// ------------------------------------------------------------------------- //
void MainWindow::disableThresholdCalibration()
{
    if(m_calib_threshold_loader && m_load_thresholds) {
        msg()("Loading of thresholds disabled, will take those directly from the user interface");
        m_calib_threshold_loader->clear();
        m_load_thresholds = false;

        ui->disable_thresholdCalibration->setEnabled(false);
        ui->show_thresholdCalibration->setEnabled(false);
        ui->thresholdCalibrationFileName->setStyleSheet("QLineEdit { background: white; }");
    }
    

}
// ------------------------------------------------------------------------- //
void MainWindow::printThresholdCalibration()
{
    if(m_calib_threshold_loader && m_calib_threshold_loader->thresholds_loaded()) {
        vmmMessageHandler->standard_message(m_calib_threshold_loader->summary_table());
    }
    else {
        msg()("No loaded thresholds to print!");
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::toggleDebug()
{
    if(ui->enableDebugPB->isChecked()) {
        ui->enableDebugPB->setStyleSheet("QPushButton {background-color: rgb(15, 147, 255);}");
        m_dbg = true;
        vmmConfigHandler ->setDebug(true);
        vmmSocketHandler ->setDebug(true);
        vmmConfigModule  ->setDebug(true);
        vmmRunModule     ->setDebug(true);
        vmmDataHandler   ->setDebug(true);
        //vmmCalibModule   ->setDebug(true);
    }
    else {
        m_dbg = false;
        vmmConfigHandler ->setDebug(false);
        vmmSocketHandler ->setDebug(false);
        vmmConfigModule  ->setDebug(false);
        vmmRunModule     ->setDebug(false);
        vmmDataHandler   ->setDebug(false);
        //vmmCalibModule   ->setDebug(false);
        ui->enableDebugPB->setStyleSheet("QPushButton {background-color: white;}");
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::changeDACtoMVs(int)
{
    QString tmp;
    if(QObject::sender() == ui->sdt)
        ui->dacmvLabel->setText(tmp.number((0.6862*ui->sdt->value()+63.478), 'f', 2) + " mV");    
    else if(QObject::sender() == ui->sdp_2)
        ui->dacmvLabel_TP->setText(tmp.number((0.6862*ui->sdp_2->value()+63.478), 'f', 2) + " mV");    
}
// ------------------------------------------------------------------------- //
void MainWindow::readLog()
{
    string buff = msg().buffer();
    ui->loggingScreen->append(QString::fromStdString(buff));
    //ui->loggingScreen->moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
    msg().clear(); 
}
// ------------------------------------------------------------------------- //
void MainWindow::updateLogScreen()
{
    if(QObject::sender() == ui->logBottom) {
        ui->loggingScreen->moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
        ui->loggingScreen->verticalScrollBar()->setValue(ui->loggingScreen->verticalScrollBar()->maximum());
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::updateFSM()
{
    if(m_commOK && !m_configOK && !m_tdaqOK && !m_runModeOK && m_acqMode=="OFF"){
        ui->SendConfiguration->setEnabled(true);
        ui->cmdlabel->setText("No Config");

        ui->fec_WarmInit->setEnabled(true);
        ui->fec_reset->setEnabled(true);

        ui->setck_s6->setEnabled(true);
        ui->setck_s6_resets->setEnabled(true);
        ui->setTp_s6->setEnabled(true);

        ui->asic_reset->setEnabled(true);

        ui->setMask->setEnabled(true);
        ui->enableART->setEnabled(true);
        ui->linkPB->setEnabled(true);
        ui->resetLinks->setEnabled(true);

 //       msg()("Waiting for configuration to be sent...");
 //   }
 //   else if(m_commOK && m_configOK && !m_tdaqOK && !m_runModeOK && m_acqMode=="") {
        ui->setEvbld->setEnabled(true);

 //       msg()("Waiting for T/DAQ constants to be sent...");
 //   }
 //   else if(m_commOK && m_configOK && m_tdaqOK && !m_runModeOK && m_acqMode=="") {
        ui->trgPulser->setEnabled(true);
        ui->trgExternal->setEnabled(true);
        ui->trgExternalBC->setEnabled(true);

 //       msg()("Waiting for trigger mode to be established...");
 //   }
 //   else if(m_commOK && m_configOK && m_tdaqOK && m_runModeOK && m_acqMode=="") {
        ui->onACQ->setEnabled(true);
        ui->clearTriggerCnt->setEnabled(true);
        ui->send_tdaq_settings_button->setEnabled(true);

 //       msg()("Ready for DAQ...");
 //   }
 //   else if(m_commOK && m_configOK && m_tdaqOK && m_runModeOK &&
 //               (m_acqMode=="ON")) {
        ui->offACQ->setEnabled(true);
        ui->checkTriggers->setEnabled(true);
        ui->clearTriggerCnt->setEnabled(true);

        // trigger settings
        ui->set_trigger_settings_button->setEnabled(true);
        // clocks
        ui->clocks_send_settings_button->setEnabled(true);
        // resets
        ui->vmm3_hard_reset_button->setEnabled(true);
        ui->fpga_reset_button->setEnabled(true);
    }
    if(ui->set_vmm2_button->isChecked()) {
        ui->set_vmm2_button->click();
    }
    else if(ui->set_vmm3_button->isChecked()) {
        ui->set_vmm3_button->click();
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::updateCounter()
{
    ui->cmdlabel->setText(QString::number(socketHandle().commandCounter()));
}
// ------------------------------------------------------------------------- //
void MainWindow::IPaddressChange()
{
    QString message = "Re-establish Connection";
    QString initial = "Establish Comms";
    if(ui->connectionLabel->text()!=message && ui->connectionLabel->text()!=initial) {
        msg()("IP address change signaled. Re-establish connection to ensure proper connectivity.");
        ui->connectionLabel->setText(message);
        ui->connectionLabel->setStyleSheet("background-color: lightGray");
    }

    m_clocks_set.clear();
}
// ------------------------------------------------------------------------- //
void MainWindow::IPchanged(QString /*string*/)
{
    IPaddressChange();
}
// ------------------------------------------------------------------------- //
void MainWindow::IPchanged(int /*value*/)
{
    IPaddressChange();
}
// ------------------------------------------------------------------------- //
void MainWindow::buildIPs()
{
    QString counter;
    QString empty_ip = "0.0.0.0";
    int n_ips_built = 0;
    for(int i = 0; i < MAX_N_IP; i++) ips[i] = empty_ip;
    for(int i = 0; i < ui->numberOfFecs->value(); i++) {
        ips[i] = ui->ip1->text() + "." + ui->ip2->text() + "." + ui->ip3->text()
                  + "." + counter.setNum(ui->ip4->text().toInt()+i);
        n_ips_built++;
    } // i
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::Connect()
{
    if(ui->numberOfFecs->value() == 0) {
        ui->connectionLabel->setText("Select number of FEBs");
        ui->connectionLabel->setStyleSheet("background-color: light");
        return;
    }

    // ------------------------------------------------- //
    //  "comm info"
    // ------------------------------------------------- //
    CommInfo commInfo;

    commInfo.fec_port       = FECPORT;
    commInfo.daq_port       = DAQPORT;
    commInfo.vmmasic_port   = VMMASICPORT;
    commInfo.vmmapp_port    = VMMAPPPORT;
    commInfo.clocks_port    = CLOCKSPORT;

    buildIPs();

    QString iplist = "";
    QString separator = "";
    for(int i = 0; i < ui->numberOfFecs->value(); i++) {
        if(i==(ui->numberOfFecs->value()-1)) {
            separator = "";
        } else {
            separator = ",";
        }
        iplist += ips[i] + separator;
    }
    commInfo.ip_list = iplist;

    //comment
    commInfo.comment = ui->userComments->text();

    configHandle().LoadCommInfo(commInfo);
    bool pingOK = socketHandle().loadIPList(iplist).ping();

    if(pingOK) {
        msg()("Ping successful");
        ui->connectionLabel->setText("All Boards Alive");
        ui->connectionLabel->setStyleSheet("background-color: green");

        ///////////////////////////////////////////////////////
        // now that we know we are connected OK, add and bind
        // the sockets
        ///////////////////////////////////////////////////////
        socketHandle().addSocket("FEC", FECPORT, QUdpSocket::ShareAddress);
        //connect(vmmRunModule, SIGNAL(checkLinks()), this, SLOT(writeFECStatus()));
        //socketHandle().addSocket("DAQ", DAQPORT); 

        

    } else {
        ui->connectionLabel->setText("Ping Failed"); 
        ui->connectionLabel->setStyleSheet("background-color: lightGray");
    }

    m_clocks_set.clear();
    for(int i = 0; i < (int)ui->numberOfFecs->value(); i++) {
        m_clocks_set.push_back(0);
    }

    m_commOK = pingOK;

    emit checkFSM();

}
void MainWindow::updateVMMselection()
{
    // clear the VMM mask button
    if(QObject::sender() == ui->vmm_select_clear) {
        for(auto& mask : m_vmm_mask) {
            mask->setChecked(false);
        }
    }
    // add a VMM
    else if(QObject::sender() == ui->vmm_select_add) {
        // get the maximum
        int max_vmm_id = -1;
        for(int i = 0; i < 8; i++) {
            if(m_vmm_mask.at(i)->isChecked()) max_vmm_id = i;
        }
        if(max_vmm_id==7) return;
        else {
            int value_to_add = max_vmm_id + 1;
            m_vmm_mask.at(value_to_add)->setChecked(true);
        }
    }
    // subtract a VMM
    else if(QObject::sender() == ui->vmm_select_subtract) {
        // get the maximum
        int max_vmm_id = -1;
        for(int i = 0; i < 8; i++) {
            if(m_vmm_mask.at(i)->isChecked()) max_vmm_id = i;
        }
        if(max_vmm_id<0) return;
        else {
            m_vmm_mask.at(max_vmm_id)->setChecked(false);
        }
    }

}
// ------------------------------------------------------------------------- //
bool MainWindow::performing_hard_reset()
{
    bool is_hard_reset = false;

    bool global_reset_enabled = ui->global_reset->currentIndex()==1;

    if(global_reset_enabled || m_in_hard_reset) {
        is_hard_reset = true;
    }
    return is_hard_reset;
}
// ------------------------------------------------------------------------- //
void MainWindow::set_thresholds_for_current_chip()
{
    vector<string> ips_to_set;
    int number_of_boards = ui->numberOfFecs->value();
    for(int i = 0; i < number_of_boards; i++)
        ips_to_set.push_back(ips[i].toStdString());


    // only configure the selected boards
    int which_to_configure = -1;
    if(!(ui->setVMMs->currentIndex()==number_of_boards)) {
        which_to_configure = ui->setVMMs->currentIndex();
        vector<string> tmp_ip;
        tmp_ip.push_back(ips_to_set.at(which_to_configure));
        ips_to_set.clear();
        ips_to_set = tmp_ip;
    }
    int initial_board_select = ui->setVMMs->currentIndex();

    vector<string> valid_ips;
    for(auto ipstring : ips_to_set) {
        if(!(m_calib_threshold_loader->is_valid_ip(ipstring))) {
            stringstream sx;
            sx << "\nWARNING No thresholds stored for board at IP: " << ipstring
                    << ", will not configure this board\n";
            msg()(sx);
        }
        else {
            valid_ips.push_back(ipstring);
        }
    }
    ips_to_set = valid_ips;

    if(ips_to_set.size()==0) {
        stringstream sx;
        sx << "ERROR No boards have thresholds loaded, cannot configure boards\n";
        sx << "ERROR Check that your threshold data file has appropriate board IPs and IDs";
        msg()(sx);
        m_threshold_loading_finished = true;
        return;
    }

    string last_ip_to_configure = ips_to_set.at(ips_to_set.size()-1);

    for(int iip = 0; iip < (int)ips_to_set.size(); iip++) {

        // select the chosen board(s)
        ui->setVMMs->setCurrentIndex(( (which_to_configure < 0) ? iip : which_to_configure));
        
        // get this board's threshold data
        vector<ThresholdHolder> thresholds = m_calib_threshold_loader->thresholds_for_board(ips_to_set[iip]);

        // get list of chips to configure for this board
        vector<int> chips_to_configure;
        for(auto h : thresholds) {
            int ch = h.chip_id;
            if(!(std::find(chips_to_configure.begin(), chips_to_configure.end(), ch)!=chips_to_configure.end()))
                chips_to_configure.push_back(ch);
        } // h
        std::sort(chips_to_configure.begin(), chips_to_configure.end());
        int last_chip_to_configure = chips_to_configure.at(chips_to_configure.size()-1);

        // now configure each of the chips on this board individually
        for(auto chip : chips_to_configure) {
            vector<int> channels_set;
            for(auto th : thresholds) {
                if(th.chip_id != chip) continue;
                VMMSDVoltage[th.channel]->setCurrentIndex(th.trim_to_set);
                channels_set.push_back(th.channel);
            } // th
            // set channels not in xml to have default trim of 0
            for(int ich = 0; ich < 64; ich++) {
                if((std::find(channels_set.begin(), channels_set.end(), ich)!=channels_set.end())) continue;
                VMMSDVoltage[ich]->setCurrentIndex(0);
            }

            for(int ivmm = 0; ivmm < (int)m_vmm_mask.size(); ivmm++) {
                if(ivmm == chip) m_vmm_mask.at(ivmm)->setChecked(true);
                else { m_vmm_mask.at(ivmm)->setChecked(false); }
            } // ivmm

            m_current_chip_threshold_loaded = true;


            //cout << "Sending config w/ thresholds for board with IP: " << ips_to_set[iip] << "  VMM: " << chip << endl;
            prepareAndSendBoardConfig();
            delay();
            delay();
            //delay();
            //delay();
            //delay();
            //delay();
            if((ips_to_set[iip] == last_ip_to_configure) && (chip == last_chip_to_configure)) {
                m_threshold_loading_finished = true;
                ui->setVMMs->setCurrentIndex(initial_board_select);
            }
            
        } // chip

    } // iip
}
// ------------------------------------------------------------------------- //
bool MainWindow::clocks_have_been_set()
{
    string current_board_selection = ui->setVMMs->currentText().toStdString();

    //cout << "MW::clocks_have_been_set   current_board_selection = " << current_board_selection << endl;

    if(current_board_selection == "None") return false;

    if(current_board_selection=="All") {
        if((int)m_clocks_set.size()!=(int)ui->numberOfFecs->value()) {
            msg()("ERROR clocks set vector not numberOfFecs in size");
            return false;
        }

        bool all_set = true;
        for(int i = 0; i < (int)ui->numberOfFecs->value(); i++) {
            if(m_clocks_set.at(i) == 0) all_set = false;
        }

        //cout << "MW::clocks_have_been_set  all set? " << all_set << endl;

        return all_set;
    }
    else {
        int board_no = -1;
        try {
            board_no = std::stoi(current_board_selection);
        }
        catch(std::exception &e) {
            cout << "MainWindow::clocks_have_been_set    Unable to get selected board ID, return true" << endl;
            return true;
        }

        //cout << "MW::clocks_have_been_set  board no " << board_no << endl;
        bool clocks_set_for_board = (m_clocks_set.at(board_no - 1) == 1 ? true : false);
        //cout << "MW::clocks_have_been_set  clocks_set_for_board? " << clocks_set_for_board << endl;
        return clocks_set_for_board;
    }

    cout << "MainWindow::clocks_have_been_set    Unexpected case, return true" << endl;
    return true;
}
// ------------------------------------------------------------------------- //
void MainWindow::channel_trim_error_VMM2()
{
    stringstream sx;
    sx << "ERROR Attempting to set a channel threshold trim to invalid value for VMM2 (>15), will not configure VMMs";
    msg()(sx);

    for(int i = 0; i < 3; i++) {
        for(int i = 0; i < 32; i++) {
            ui->sd_list_widget->item(i)->setBackground(Qt::red);
            ui->sd_list_widget_2->item(i)->setBackground(Qt::red);
        }
        delay();
        delay();
        for(int i = 0; i < 32; i++) {
            ui->sd_list_widget->item(i)->setBackground(Qt::white);
            ui->sd_list_widget_2->item(i)->setBackground(Qt::white);
        }
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::prepareAndSendBoardConfig()
{

    //if(m_have_loaded_pulser) {
    //    m_have_loaded_pulser = false;
        vmmConfigModule->LoadConfig(*vmmConfigHandler);
    //}

    if(!clocks_have_been_set())
        ui->clocks_send_settings_button->click(); 


    ///////////////////////////////////////////////////////
    // if loading thresholds for each VMM
    // override default behavior here
    ///////////////////////////////////////////////////////
    if(!performing_hard_reset()) {
        if(load_thresholds() && !threshold_for_current_chip_loaded() && !done_loading_thresholds()) {
            msg()("Loading thresholds");
            set_thresholds_for_current_chip();
        }
        if(done_loading_thresholds()) {
            m_threshold_loading_finished = false;
            return;
        }
    }
    //if(!ui->do_calibration_button->isChecked())
    //    msg()("Sending configuration...");

    ///////////////////////////////////////////////////////////
    // global registers
    ///////////////////////////////////////////////////////////
    GlobalSetting global;

    global.sp               = ui->global_sp->currentIndex();
    global.sdp              = ui->global_sdp->currentIndex();
    global.sbmx             = ui->global_sbmx->currentIndex();
    global.sbft             = ui->global_sbft->currentIndex();
    global.sbfp             = ui->global_sbfp->currentIndex();
    global.sbfm             = ui->global_sbfm->currentIndex();
    global.slg              = ui->global_slg->currentIndex();
    global.sm5              = ui->global_sm5->currentIndex();
    global.scmx             = ui->global_scmx->currentIndex();
    global.sfa              = ui->global_sfa->currentIndex();
    global.sfam             = ui->global_sfam->currentIndex();
    global.st               = ui->global_st->currentIndex();
    global.sfm              = ui->global_sfm->currentIndex();
    global.sg               = ui->global_sg->currentIndex();
    global.sng              = ui->global_sng->currentIndex();
    global.stot             = ui->global_stot->currentIndex();
    global.stpp             = ui->global_stpp->currentIndex();
    global.sttt             = ui->global_sttt->currentIndex();
    global.ssh              = ui->global_ssh->currentIndex();
    global.stc              = ui->global_stc->currentIndex();
    global.sdt_dac          = ui->global_sdt_dac->value();
    global.sdp_dac          = ui->global_sdp_dac->value();
    global.sc10b            = ui->global_sc10b->currentIndex();
    global.sc8b             = ui->global_sc8b->currentIndex();
    global.sc6b             = ui->global_sc6b->currentIndex();
    global.s8b              = ui->global_s8b->currentIndex();
    global.s6b              = ui->global_s6b->currentIndex();
    global.s10b             = ui->global_s10b->currentIndex();
    global.sdcks            = ui->global_sdcks->currentIndex();
    global.sdcka            = ui->global_sdcka->currentIndex();
    global.sdck6b           = ui->global_sdck6b->currentIndex();
    global.sdrv             = ui->global_sdrv->currentIndex();
    global.slvs             = ui->global_slvs->currentIndex();
    global.stcr             = ui->global_stcr->currentIndex();
    global.ssart            = ui->global_ssart->currentIndex();
    global.s32              = ui->global_s32->currentIndex();
    global.stlc             = ui->global_stlc->currentIndex();
    global.srec             = ui->global_srec->currentIndex();
    global.sbip             = ui->global_sbip->currentIndex();
    global.srat             = ui->global_srat->currentIndex();
    global.sfrst            = ui->global_sfrst->currentIndex();
    global.slvsbc           = ui->global_slvsbc->currentIndex();
    global.slvstp           = ui->global_slvstp->currentIndex();
    global.slvstk           = ui->global_slvstk->currentIndex();
    global.slvsdt           = ui->global_slvsdt->currentIndex();
    global.slvsart          = ui->global_slvsart->currentIndex();
    global.slvstki          = ui->global_slvstki->currentIndex();
    global.slvsena          = ui->global_slvsena->currentIndex();
    global.slvs6b           = ui->global_slvs6b->currentIndex();
    global.sL0enaV          = ui->global_sL0enaV->currentIndex();
    global.slh              = ui->global_slh->currentIndex();
    global.slxh             = ui->global_slxh->currentIndex();
    global.stgc             = ui->global_stgc->currentIndex();
    global.reset            = ui->global_reset->currentIndex();
    global.sL0ena           = ui->global_sL0ena->currentIndex();
    global.l0offset         = ui->global_l0offset->value();
    global.offset           = ui->global_offset->value();
    global.rollover         = ui->global_rollover->value();
    global.window           = ui->global_window->value();
    global.truncate         = ui->global_truncate->value();
    global.nskip            = ui->global_nskip->value();
    global.sL0cktest        = ui->global_sL0cktest->currentIndex();
    global.sL0ckinv         = ui->global_sL0ckinv->currentIndex();
    global.sL0dckinv        = ui->global_sL0dckinv->currentIndex();
    global.nskipm           = ui->global_nskipm->currentIndex();
    
    ///////////////////////////////////////////////////////////
    // VMM mask
    ///////////////////////////////////////////////////////////
    VMMMap vmmMap;
    vmmMap.mask = 0;
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        if(m_vmm_mask.at(i)->isChecked()) {
            tmp = (tmp << i);
            vmmMap.mask |= tmp;
        }
    }

    ///////////////////////////////////////////////////////////
    // channel registers
    ///////////////////////////////////////////////////////////
    vector<Channel> channels;
    for(int i = 0; i < 64; i++) {
        Channel ch;
        ch.number           = i;
        ch.sc               = (VMMSCBool[i] ? 1 : 0);
        ch.sl               = (VMMSLBool[i] ? 1 : 0);
        ch.st               = (VMMSTBool[i] ? 1 : 0);
        ch.sth              = (VMMSTHBool[i] ? 1 : 0);
        ch.sm               = (VMMSMBool[i] ? 1 : 0);
        ch.sd               = VMMSDValue[i];
        if(ch.sd>15 && vmm_type()==2) {
            channel_trim_error_VMM2();
            return;
        }
        ch.smx              = (VMMSMXBool[i] ? 1 : 0);
        ch.sz10b            = VMMSZ010bValue[i];
        ch.sz8b             = VMMSZ08bValue[i];
        ch.sz6b             = VMMSZ06bValue[i];

        //cout << "ch["<<ch.number<<"] st = " << ch.st << "  sm = " << ch.sm << endl;

        channels.push_back(ch);
    } // i

    bool ok;

    // send
    configHandle().LoadBoardConfiguration(global, vmmMap, channels);
    if(vmmMap.mask != 0) {
        stringstream sx;
        if(ui->setVMMs->currentText().toStdString()=="All") {
            sx << "Sending SPI packet to all connected boards, VMM mask: [" << std::bitset<8>(vmmMap.mask) << "]";
        } 
        else {
            stringstream ip;
            int which_ip = ui->setVMMs->currentText().toInt(&ok, 10) - 1;
            int last = ui->ip4->text().toInt(&ok, 10);
            ip << ui->ip1->text().toStdString() << "." << ui->ip2->text().toStdString() << "."
               << ui->ip3->text().toStdString() << ".";
            ip << (last + which_ip);

            sx << "Sending SPI packet to board #" << ui->setVMMs->currentText().toStdString()
                    << " (IP: " << ip.str() << "),"
                    << " VMM mask [" << std::bitset<8>(vmmMap.mask) << "]";
        }
        //msg()(sx);
        //sx << "Sending configuration packet [" << std::bitset<8>(vmmMap.mask) << "]";
        configModule().SendConfig();
    }
    else {
        msg()("No VMM selected. Not sending configuration.");
    }


    if(!performing_hard_reset()) {
        if(load_thresholds() && threshold_for_current_chip_loaded()) {
            m_current_chip_threshold_loaded = false;
        }
        if(done_loading_thresholds()) m_threshold_loading_finished = false;
    }

    // toggle/status update
    m_configOK = true;
    emit checkFSM();
}
// ------------------------------------------------------------------------- //
void MainWindow::set_trigger_settings_defaults()
{
    if(QObject::sender() == ui->trigger_settings_defaults_button) {
        ui->trigExternalBCLatency->setValue(122);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::set_tdaq_configuration(bool load_only)
{
    TriggerDAQ daq;

    //int latency_value_in_steps = ui->trigExternalBCLatency->value() / 8;
    int latency_value_in_steps = ui->trigExternalBCLatency->value();
    daq.trigger_latency = latency_value_in_steps;

    string run_mode = "pulser";
    if(ui->trgPulser->isChecked()) { run_mode = "pulser"; }
    else if(ui->trgExternal->isChecked()) { run_mode = "external"; }
    daq.run_mode = run_mode;

    if(ui->trgExternalBC->isChecked()) daq.fixed_tac = 1;
    else { daq.fixed_tac = 0; } 

    string mapping_file = "";
    if(ui->daqSetupFilenameField->text()!="") {
        mapping_file = ui->daqSetupFilenameField->text().toStdString();
    }
    daq.mapping_file = mapping_file;

    string output_path = "";
    if(ui->runDirectoryField->text()!="") {
        output_path = ui->runDirectoryField->text().toStdString();
    }
    daq.output_path = output_path;

    // set the global trigger_daq object
    configHandle().LoadTDAQConfiguration(daq);

    // send the TDAQ configuration command for the trigger settings and mode
    if(!load_only) {
        runModule().set_trigger_settings();
    }

    // toggle/status update
    m_tdaqOK = true;
    emit checkFSM();
}
// ------------------------------------------------------------------------- //
void MainWindow::setTriggerMode()
{
    if(QObject::sender() == ui->trgPulser) {
        ui->trgPulser->setChecked(true);
        ui->trgExternal->setChecked(false);
        
    }
    else if(QObject::sender() == ui->trgExternal) {
        ui->trgExternal->setChecked(true);
        ui->trgPulser->setChecked(false);
    }
    else if(QObject::sender() == ui->trgExternalBC) {
        // make it so that the user knows they must re-send these commands
        ui->trgPulser->setChecked(false);
        ui->trgExternal->setChecked(false);
        return;
    }

    // everything for setting up the global trigger_daq object is done here
    // this will also send the trigger settings (e.g. latency) command as well
    set_tdaq_configuration(true);

    // send the command packet
    // dantrim June 14 2017 -- now have the single "SET" button that sends all TDAQ settings
    //bool enable_fixed_latency_run_mode = ui->trgExternalBC->isChecked();
    //runModule().setTriggerMode(enable_fixed_latency_run_mode);

    // update status
    m_runModeOK = true;
    emit checkFSM();

}
// ------------------------------------------------------------------------- //
void MainWindow::send_tdaq_settings_command()
{
    int latency_steps = ui->trigExternalBCLatency->value();
    bool mode_internal = ui->trgPulser->isChecked(); 
    bool mode_external = ui->trgExternal->isChecked();
    bool external_bc = ui->trgExternalBC->isChecked();
    int trig_deadtime = ui->trigDeadTime->value();
    int n_ckbc = ui->nCKBC_fixedTAC->value();
    int art_timeout = ui->art_timeout->value(); // steps of 6.25 ns
    int trig_extra_ckbc_latency = ui->trigLatencyExtraCKBC->value(); // steps of 6.25 ns

    // load the current settings into the config_handle
    set_tdaq_configuration(true);

    if(mode_internal && mode_external) {
        // this shouldn't be possible, but...
        msg()("ERROR Both External and Internal run modes are set, cannot send TDAQ settings!");
        return;
    }

    runModule().send_tdaq_settings(latency_steps, mode_internal,
                    mode_external, external_bc, trig_deadtime, n_ckbc, art_timeout, trig_extra_ckbc_latency);
    
}
// ------------------------------------------------------------------------- //
void MainWindow::set_acq_mode_state(int on_or_off)
{
    // on_or_off: 0 for ACQ on, 1 for ACQ off
    ui->vmm3_hard_reset_button->setEnabled(on_or_off);
    ui->clocks_send_settings_button->setEnabled(on_or_off);
    ui->set_trigger_settings_button->setEnabled(on_or_off);
    ui->trgExternal->setEnabled(on_or_off);
    ui->trgPulser->setEnabled(on_or_off);
    ui->trgExternalBC->setEnabled(on_or_off);
    ui->send_tdaq_settings_button->setEnabled(on_or_off);

    // don't allow changing values from default settings buttons
    ui->trigger_settings_defaults_button->setEnabled(on_or_off);
    ui->cktk_ckbt_set_defaults_button->setEnabled(on_or_off);
    ui->cktp_set_defaults_button->setEnabled(on_or_off);
    ui->clocks_set_defaults_button->setEnabled(on_or_off);

    // changing the monitoring rate
    ui->monitoring_rate->setEnabled(on_or_off);
}
// ------------------------------------------------------------------------- //
void MainWindow::setACQMode()
{
    // acquisition ON
    if(QObject::sender() == ui->onACQ) {
        
        // don't re-configure the board for a setting which it is
        // already in
        if(m_acqMode=="ON") {
            ui->onACQ->setDown(true);
            //return;
        }
        ui->onACQ->setDown(true);
        ui->offACQ->setChecked(false);
        ui->offACQ->setDown(false);
        runModule().ACQon();
        m_acqMode = "ON";

        if(ui->monitoring_enable_button->isChecked() && m_mapping_loaded_ok && !m_daqInProgress) {
            // send the mon configuration
            // this button is off-screen
            ui->monitoring_send_config_button->click();
        }

        // put things down
        set_acq_mode_state(0);
    }
    // acquisition OFF
    else if(QObject::sender() == ui->offACQ) {
        if(m_acqMode=="OFF") {
            ui->offACQ->setDown(true);
            //return;
        }
        ui->offACQ->setDown(true);
        ui->onACQ->setChecked(false);
        ui->onACQ->setDown(false);
        runModule().ACQoff();
        m_acqMode = "OFF";
        ui->monitoring_rate->setEnabled(false);

        // put things back up
        set_acq_mode_state(1);
    }

    emit checkFSM();
}
// ------------------------------------------------------------------------- //
void MainWindow::setVMMPower()
{
    // power ON
    if(QObject::sender() == ui->vmmPowerOn) {
        runModule().setVMMPower(1);
    }
    // power OFF
    else if(QObject::sender() == ui->vmmPowerOff) {
        runModule().setVMMPower(0);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::setNumberOfFecs(int)
{
    // communication board select
    ui->setVMMs->clear();

    // calibration board select
    ui->calibration_board_select->clear();

    // clear that clocks have been set for the boards
    m_clocks_set.clear();

    QString counter;
    for(int i = 0; i < ui->numberOfFecs->value(); i++){
        ips[i] = ui->ip1->text() + "." + ui->ip2->text() + "." + ui->ip3->text()
                    + "." + counter.setNum(ui->ip4->text().toInt()+i);

        // communication board select
        ui->setVMMs->addItem(counter.setNum(i+1));
        ui->setVMMs->setCurrentIndex(i);

        // calibration board select
        ui->calibration_board_select->addItem(counter.setNum(i+1));
        ui->calibration_board_select->setCurrentIndex(0);

        // set false that clocks have been set for all boards
        m_clocks_set.push_back(0);
    }

    if(ui->numberOfFecs->value()>0) {
        // communication board select
        ui->setVMMs->addItem("All");
        ui->setVMMs->setCurrentIndex(ui->setVMMs->currentIndex()+1);

        // calibration board select
        ui->calibration_board_select->addItem("All");
        ui->calibration_board_select->setCurrentIndex(0); // set default board to "1"
    }
    else {
        // communication board select
        ui->setVMMs->addItem("None");
        ui->setVMMs->setCurrentIndex(0);

        // calibration board select
        ui->calibration_board_select->addItem("None");
        ui->calibration_board_select->setCurrentIndex(0);
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::updateBoardSelection(int /*current_selection*/)
{
    std::string board_selection_str = ui->setVMMs->currentText().toStdString();
    bool ok;
    int board_selection = -1;
    if(board_selection_str!="All") {
        board_selection = ui->setVMMs->currentText().toInt(&ok, 10) - 1;
    }
    configHandle().setSelectedBoards(board_selection);
}
// ------------------------------------------------------------------------- //
void MainWindow::InitializeChannelArrayValues()
{
    for(unsigned int i = 0; i < 64; i++) {
        VMMSCBool[i] = false;
        VMMSLBool[i] = false;
        VMMSTBool[i] = false;
        VMMSTHBool[i] = false;
        VMMSMBool[i] = false;
        VMMSMXBool[i] = false;
        VMMSPBool[i] = false;
        VMMSDValue[i] = 0;
        VMMSZ010bValue[i] = 0;
        VMMSZ08bValue[i] = 0;
        VMMSZ06bValue[i] = 0;
    }

    VMMSPBoolAll = false;
    VMMSCBoolAll = false;
    VMMSLBoolAll = false;
    VMMSTBoolAll = false;
    VMMSTHBoolAll = false;
    VMMSMBoolAll = false;
    VMMSMXBoolAll = false;
    VMMSZ010bBoolAll = false;
    VMMSZ08bBoolAll = false;
    VMMSZ06bBoolAll = false;

    VMMSPBoolAll2 = false;
    VMMSCBoolAll2 = false;
    VMMSLBoolAll2 = false;
    VMMSTBoolAll2 = false;
    VMMSTHBoolAll2 = false;
    VMMSMBoolAll2 = false;
    VMMSMXBoolAll2 = false;
    VMMSZ010bBoolAll2 = false;
    VMMSZ08bBoolAll2 = false;
    VMMSZ06bBoolAll2 = false;

}
// ------------------------------------------------------------------------- //
void MainWindow::SetInitialState()
{

    // calibration
    ui->xadccalibration_channel_trims_checkbox->click();
    ui->calibration_custom_loop_checkbox->click();
    ui->calibration_threshold_box->setEnabled(false);
    ui->calibration_charge_box->setEnabled(false);

    // trigger settings
    ui->set_trigger_settings_button->setEnabled(false);
    // clocks
    ui->clocks_send_settings_button->setEnabled(false);
    // resets
    ui->vmm3_hard_reset_button->setEnabled(false);
    ui->fpga_reset_button->setEnabled(false);

    // prevent the sending of the monitoring configuration until monitoring has been initialized
    ui->monitoring_enable_button->setEnabled(false);
    ui->monitoring_send_config_button->setEnabled(true);
    //ui->monitoring_send_config_button->setEnabled(false);

    // can't load anything unless the files have been set
    ui->load_daqSetup->setEnabled(false);
    ui->load_config->setEnabled(false);

    // calibration stuff
    ui->load_thresholdCalibration->setEnabled(false);
    ui->disable_thresholdCalibration->setEnabled(false);
    ui->show_thresholdCalibration->setEnabled(false);

    ui->openConnection->setEnabled(true);
    ui->connectionLabel->setText("Establish Comms");
    msg()("Waiting for open communication with FEB...");

    // disable ability to send config
    ui->SendConfiguration->setEnabled(false);
    ui->cmdlabel->setText("No Comm.");

    // disable reset of FEC
    ui->fec_WarmInit->setEnabled(false);
    ui->fec_reset->setEnabled(false);

    // disable sending of T/DAQ settings and constants
    ui->trgPulser->setEnabled(false);
    ui->trgExternalBC->setEnabled(false);
    ui->trgExternal->setEnabled(false);
    ui->onACQ->setEnabled(false);
    ui->offACQ->setEnabled(false);
    ui->send_tdaq_settings_button->setEnabled(false);

    // disable clock settings
    ui->setck_s6->setEnabled(false);
    ui->setck_s6_resets->setEnabled(false);
    ui->setTp_s6->setEnabled(false);

    // disable start/stop run
    //ui->clearTriggerCnt->setEnabled(false);
    ui->checkTriggers->setEnabled(false);
    ui->stopTriggerCnt->setEnabled(false);

    // misc DAQ buttons
    ui->setEvbld->setEnabled(false);
    ui->asic_reset->setEnabled(false);

    // hdmi mask
    ui->setMask->setChecked(false);
    ui->setMask->setEnabled(false);
    ui->enableART->setEnabled(false);
    ui->linkPB->setEnabled(false);
    ui->resetLinks->setEnabled(false);

    // dac to mV
    QString tmp;
    ui->dacmvLabel->setText(tmp.number((0.6862*ui->sdt->value()+63.478), 'f', 2) + " mV");
    ui->dacmvLabel_TP->setText(tmp.number((0.6862*ui->sdp_2->value()+63.478), 'f', 2) + " mV");

    // disable calibration
    //ui->calibration->setChecked(false);
    ui->doCalib->setChecked(false);

}
// ------------------------------------------------------------------------- //
void MainWindow::CreateChannelsFields()
{
    Font.setPointSize(10);
    int margin = 8;
    int channel_square_size = 16;



    //////////////////////////////////////////////////////
    // add the "global" labels
    //////////////////////////////////////////////////////
    //ui->channelLabelLayout->setContentsMargins(margin*0.75, margin*0.25, margin*2*10, margin);
    //ui->channelLabelLayout->setHorizontalSpacing(4);
    //ui->channelLabelLayout->setVerticalSpacing(1);

    ui->VMMChanLabel->setFixedSize(channel_square_size*1.22, channel_square_size);
    ui->VMMChanLabel->setEnabled(false);

    ui->SCLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SLLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->STLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->STHLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SMLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SDLabel->setFixedSize(55, channel_square_size);
    ui->SMXLabel->setFixedSize(31, channel_square_size);
    ui->SZ010bLabel->setFixedSize(55, channel_square_size);
    ui->SZ08bLabel->setFixedSize(55, channel_square_size);
    ui->SZ06bLabel->setFixedSize(55, channel_square_size);
    
    ui->VMMChanLabel2->setFixedSize(channel_square_size*1.22, channel_square_size);
    ui->VMMChanLabel2->setEnabled(false);
    ui->SCLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SLLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->STLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->STHLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SMLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SDLabel2->setFixedSize(55, channel_square_size);
    ui->SMXLabel2->setFixedSize(31, channel_square_size);
    ui->SZ010bLabel2->setFixedSize(55, channel_square_size);
    ui->SZ08bLabel2->setFixedSize(55, channel_square_size);
    ui->SZ06bLabel2->setFixedSize(55, channel_square_size);
    

    //QLabel* spacer0 = new QLabel("    ");
    //QLabel* spacer1 = new QLabel("      ");
    //ui->channelLabelLayout->addWidget(spacer0,                1,  1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->VMMChanLabel,       1,  1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SCLabel,            1,  3, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SLLabel,            1,  4, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->STLabel,            1,  5, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SMLabel,            1,  6, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SDLabel,            1,  7, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SMXLabel,           1,  8, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ010bLabel,        1,  9, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ08bLabel,         1, 10, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ06bLabel,         1, 11, Qt::AlignCenter); 
    //
    //ui->channelLabelLayout->addWidget(spacer1,                1,   1+11+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->VMMChanLabel2,      1,   1+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SCLabel2,           1,   3+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SLLabel2,           1,   4+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->STLabel2,           1,   5+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SMLabel2,           1,   6+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SDLabel2,           1,   7+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SMXLabel2,          1,   8+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ010bLabel2,       1,   9+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ08bLabel2,        1,  10+11+1+1, Qt::AlignCenter);
    //ui->channelLabelLayout->addWidget(ui->SZ06bLabel2,        1,  11+11+1+1, Qt::AlignCenter); 


    
    //////////////////////////////////////////////////////
    // add the channel stuff
    //////////////////////////////////////////////////////

    channel_square_size = 15;
    //void  setContentsMargins(int left, int top, int right, int bottom)
    channelGridLayout->setContentsMargins(margin*0.75, margin*0.25, margin*2*10, margin);
    channelGridLayout->setHorizontalSpacing(4);
    channelGridLayout->setVerticalSpacing(1);

    QString initialValueRadio = "";
    QString counter;

    // add "global" channel trim labels
    for(int i = 0; i < 32; i++) {
        ui->SDLabel->addItem(counter.setNum(i)+" mV");
        ui->SDLabel2->addItem(counter.setNum(i)+" mV");
    }
    // add "global" 10-bit zero values
    for(int i = 0; i < 32; i++) {
        ui->SZ010bLabel->addItem(counter.setNum(i)+"");
        ui->SZ010bLabel2->addItem(counter.setNum(i)+"");
    }
    // add "global" 8-bit zero values
    for(int i = 0; i < 16; i++) {
        ui->SZ08bLabel->addItem(counter.setNum(i)+"");
        ui->SZ08bLabel2->addItem(counter.setNum(i)+"");
    }
    // add "global" 6-bit zero values
    for(int i = 0; i < 8; i++) {
        ui->SZ06bLabel->addItem(counter.setNum(i)+"");
        ui->SZ06bLabel2->addItem(counter.setNum(i)+"");
    }


    // set initial values for "global" channel bools
    VMMSPBoolAll=0;
    VMMSCBoolAll=0;
    VMMSLBoolAll=0;
    VMMSTBoolAll=0;
    VMMSTHBoolAll=0;
    VMMSMBoolAll=0;
    VMMSMXBoolAll=0;
    VMMSZ010bBoolAll=0;
    VMMSZ08bBoolAll=0;
    VMMSZ06bBoolAll=0;
    
    VMMSPBoolAll2=0;
    VMMSCBoolAll2=0;
    VMMSLBoolAll2=0;
    VMMSTBoolAll2=0;
    VMMSTHBoolAll2=0;
    VMMSMBoolAll2=0;
    VMMSMXBoolAll2=0;
    VMMSZ010bBoolAll2=0;
    VMMSZ08bBoolAll2=0;
    VMMSZ06bBoolAll2=0;

    // prevent scrolling (mouse wheel) from changing these widgets
    ui->SDLabel     ->installEventFilter(this);
    ui->SZ010bLabel ->installEventFilter(this);
    ui->SZ08bLabel  ->installEventFilter(this);
    ui->SZ06bLabel  ->installEventFilter(this);


    ////////////////////////////////////////////////////////////////////
    // add left-hand column
    ////////////////////////////////////////////////////////////////////
    for(int i = 0; i < 32; i++) {

        // this is the fixed channel label for each row on the LHS
        VMMChannel[i] = new QLineEdit(counter.setNum(i), ui->tab_3);
        VMMChannel[i]->setAlignment(Qt::AlignHCenter);

        VMMChannel[i]->setEnabled(false);
        VMMChannel[i]->setFixedSize(channel_square_size*1.22, channel_square_size);

        // these are the per-channel buttons
        VMMSC[i]    = new QPushButton(initialValueRadio,ui->tab_3); // cap
        VMMSL[i]    = new QPushButton(initialValueRadio,ui->tab_3); // leakage
        VMMST[i]    = new QPushButton(initialValueRadio,ui->tab_3); // test-cap
        VMMSTH[i]   = new QPushButton(initialValueRadio,ui->tab_3); // x10
        VMMSM[i]    = new QPushButton(initialValueRadio,ui->tab_3); // mask
        VMMSMX[i]   = new QPushButton(initialValueRadio,ui->tab_3); // mon trim
        VMMSDVoltage[i] = new QComboBox(ui->tab_3);

        // ADC zeros
        VMMSZ010bCBox[i] = new QComboBox(ui->tab_3);
        VMMSZ08bCBox[i] = new QComboBox(ui->tab_3);
        VMMSZ06bCBox[i] = new QComboBox(ui->tab_3);


        VMMSC[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSL[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMST[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSTH[i]   ->setFixedSize(channel_square_size, channel_square_size);
        VMMSM[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSMX[i]   ->setFixedSize(31, channel_square_size);
        VMMSDVoltage[i]->setFixedSize(55, channel_square_size);
        VMMSDVoltage[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSDVoltage[i]->addItem(counter.setNum(j)+" mV");

        VMMSZ010bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ08bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ06bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ010bCBox[i]->setFont(Font);
        VMMSZ08bCBox[i]->setFont(Font);
        VMMSZ06bCBox[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSZ010bCBox[i]->addItem(counter.setNum(j)+"");
        for(int j = 0; j < 16; j++) VMMSZ08bCBox[i]->addItem(counter.setNum(j)+"");
        for(int j = 0; j < 8; j++)  VMMSZ06bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ010bCBox[i]->setCurrentIndex(0);
        VMMSZ08bCBox[i]->setCurrentIndex(0);
        VMMSZ06bCBox[i]->setCurrentIndex(0);
        

        VMMSC[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSL[i]    ->setStyleSheet("background-color:  lightGray");
        VMMST[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSTH[i]   ->setStyleSheet("background-color:  lightGray");
        VMMSM[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSMX[i]   ->setStyleSheet("background-color:  lightGray");

        // these are the boolean values associated with the buttons
        VMMSPBool[i]    = 0;
        VMMSCBool[i]    = 0;
        VMMSMBool[i]    = 0;
        VMMSTBool[i]    = 0;
        VMMSTHBool[i]   = 0;
        VMMSLBool[i]    = 0;
        VMMSMXBool[i]   = 0;

        VMMSZ010bValue[i] = 0;
        VMMSZ08bValue[i]  = 0;
        VMMSZ06bValue[i]  = 0;


        // now put these guys on the grid
        if(i==0) {
            channelGridLayout->addWidget(ui->VMMChanLabel,          i+1, 1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SCLabel,               i+1, 3, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SLLabel,               i+1, 4, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->STHLabel,              i+1, 5, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->STLabel,               i+1, 6, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SMLabel,               i+1, 6+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SDLabel,               i+1, 7+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SMXLabel,              i+1, 8+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ010bLabel,           i+1, 9+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ08bLabel,            i+1, 10+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ06bLabel,            i+1, 11+1, Qt::AlignCenter);
        }
        // multiselect
        channelGridLayout->addWidget(VMMChannel[i],             i+1+1, 1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSC[i],                  i+1+1, 3, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSL[i],                  i+1+1, 4, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSTH[i],                 i+1+1, 5, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMST[i],                  i+1+1, 6, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSM[i],                  i+1+1, 6+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSDVoltage[i],           i+1+1, 7+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSMX[i],                 i+1+1, 8+1, Qt::AlignCenter); 
        channelGridLayout->addWidget(VMMSZ010bCBox[i],          i+1+1, 9+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSZ08bCBox[i],           i+1+1,10+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSZ06bCBox[i],           i+1+1,11+1, Qt::AlignCenter);

        // prevent mouse scrolling from manipulating these widgets
        VMMSDVoltage[i]->installEventFilter(this);
        VMMSDVoltage[i]->setFocusPolicy(Qt::ClickFocus);
        VMMSZ010bCBox[i]->installEventFilter(this);
        VMMSZ08bCBox[i]->installEventFilter(this);
        VMMSZ06bCBox[i]->installEventFilter(this);

    } // i (LHS column)
    ////////////////////////////////////////////////////////////////////
    // add righ-hand column
    ////////////////////////////////////////////////////////////////////
    for(int i = 32; i < 64; i++) {

        // this is the fixed channel label for each row on the LHS
        VMMChannel[i] = new QLineEdit(counter.setNum(i), ui->tab_3);
        VMMChannel[i]->setAlignment(Qt::AlignHCenter);

        VMMChannel[i]->setEnabled(false);
        VMMChannel[i]->setFixedSize(channel_square_size*1.22, channel_square_size);

        // these are the per-channel buttons
        VMMSC[i]    = new QPushButton(initialValueRadio,ui->tab_3); // cap
        VMMSL[i]    = new QPushButton(initialValueRadio,ui->tab_3); // leakage
        VMMST[i]    = new QPushButton(initialValueRadio,ui->tab_3); // test-cap
        VMMSTH[i]   = new QPushButton(initialValueRadio,ui->tab_3); // x10
        VMMSM[i]    = new QPushButton(initialValueRadio,ui->tab_3); // mask
        VMMSMX[i]   = new QPushButton(initialValueRadio,ui->tab_3); // mon trim
        VMMSDVoltage[i] = new QComboBox(ui->tab_3);

        // ADC zeros
        VMMSZ010bCBox[i] = new QComboBox(ui->tab_3);
        VMMSZ08bCBox[i] = new QComboBox(ui->tab_3);
        VMMSZ06bCBox[i] = new QComboBox(ui->tab_3);


        VMMSC[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSL[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMST[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSTH[i]   ->setFixedSize(channel_square_size, channel_square_size);
        VMMSM[i]    ->setFixedSize(channel_square_size, channel_square_size);
        VMMSMX[i]   ->setFixedSize(31, channel_square_size);
        VMMSDVoltage[i]->setFixedSize(55, channel_square_size);
        VMMSDVoltage[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSDVoltage[i]->addItem(counter.setNum(j)+" mV");

        VMMSZ010bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ08bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ06bCBox[i]->setFixedSize(55, channel_square_size);
        VMMSZ010bCBox[i]->setFont(Font);
        VMMSZ08bCBox[i]->setFont(Font);
        VMMSZ06bCBox[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSZ010bCBox[i]->addItem(counter.setNum(j)+"");
        for(int j = 0; j < 16; j++) VMMSZ08bCBox[i]->addItem(counter.setNum(j)+"");
        for(int j = 0; j < 8; j++)  VMMSZ06bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ010bCBox[i]->setCurrentIndex(0);
        VMMSZ08bCBox[i]->setCurrentIndex(0);
        VMMSZ06bCBox[i]->setCurrentIndex(0);
        

        VMMSC[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSL[i]    ->setStyleSheet("background-color:  lightGray");
        VMMST[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSTH[i]   ->setStyleSheet("background-color:  lightGray");
        VMMSM[i]    ->setStyleSheet("background-color:  lightGray");
        VMMSMX[i]   ->setStyleSheet("background-color:  lightGray");

        // these are the boolean values associated with the buttons
        VMMSPBool[i]    = 0;
        VMMSCBool[i]    = 0;
        VMMSMBool[i]    = 0;
        VMMSTBool[i]    = 0;
        VMMSTHBool[i]   = 0;
        VMMSLBool[i]    = 0;
        VMMSMXBool[i]   = 0;

        VMMSZ010bValue[i] = 0;
        VMMSZ08bValue[i]  = 0;
        VMMSZ06bValue[i]  = 0;

        QLabel *spacer = new QLabel("  ");


        // now put these guys on the grid
        // void     addLayout(QLayout * layout, int row, int column, Qt::Alignment alignment = 0)
        if(i==32) {
            channelGridLayout->addWidget(spacer,                     i+1-32, 1+12, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->VMMChanLabel2,          i+1-32, 1+12+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SCLabel2,               i+1-32, 3+12+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SLLabel2,               i+1-32, 4+12+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->STHLabel2,              i+1-32, 5+12+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->STLabel2,               i+1-32, 6+12+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SMLabel2,               i+1-32, 6+12+1+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SDLabel2,               i+1-32, 7+12+1+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SMXLabel2,              i+1-32, 8+12+1+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ010bLabel2,           i+1-32, 9+12+1+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ08bLabel2,            i+1-32, 10+12+1+1, Qt::AlignCenter);
            channelGridLayout->addWidget(ui->SZ06bLabel2,            i+1-32, 11+12+1+1, Qt::AlignCenter);
        }

        channelGridLayout->addWidget(spacer,                    i+1-32+1, 1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMChannel[i],             i+1-32+1, 1+1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSC[i],                  i+1-32+1, 3+1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSL[i],                  i+1-32+1, 4+1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSTH[i],                 i+1-32+1, 5+1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMST[i],                  i+1-32+1, 6+1+12, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSM[i],                  i+1-32+1, 6+1+12+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSDVoltage[i],           i+1-32+1, 7+1+12+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSMX[i],                 i+1-32+1, 8+1+12+1, Qt::AlignCenter); 
        channelGridLayout->addWidget(VMMSZ010bCBox[i],          i+1-32+1, 9+1+12+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSZ08bCBox[i],           i+1-32+1,10+1+12+1, Qt::AlignCenter);
        channelGridLayout->addWidget(VMMSZ06bCBox[i],           i+1-32+1,11+1+12+1, Qt::AlignCenter);

        // prevent mouse scrolling from manipulating these widgets
        VMMSDVoltage[i]->installEventFilter(this);
        VMMSDVoltage[i]->setFocusPolicy(Qt::ClickFocus);
        VMMSZ010bCBox[i]->installEventFilter(this);
        VMMSZ08bCBox[i]->installEventFilter(this);
        VMMSZ06bCBox[i]->installEventFilter(this);

    } // i (RHS column)

    // set the grid layout geometry
    // QRect(int x, int y, int width, int height)
    channelGridLayout->setGeometry(QRect(620,12,100,2000));
    //channelGridLayout->setGeometry(QRect(620,12,380,2000));
    dummy->setLayout(channelGridLayout);


    // attach scroll areay to the widget that the channel grid is attached to
    ui->scrollArea->setWidget(dummy);
    ui->scrollArea->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);


    ///////////////////////////////////////////////////////////////
    // widget connections for "global" channel buttons
    ///////////////////////////////////////////////////////////////
    // LHS globals
    connect(ui->SCLabel,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SLLabel,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->STLabel,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->STHLabel,   SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SMLabel,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SMXLabel,   SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SDLabel,    SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelVoltages(int)));
    connect(ui->SZ010bLabel, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ08bLabel, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ06bLabel, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));

    // RHS globals
    connect(ui->SCLabel2,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SLLabel2,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->STLabel2,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->STHLabel2,   SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SMLabel2,    SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SMXLabel2,   SIGNAL(pressed()), this, SLOT(updateChannelState()));
    connect(ui->SDLabel2,    SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelVoltages(int)));
    connect(ui->SZ010bLabel2, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ08bLabel2, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ06bLabel2, SIGNAL(currentIndexChanged(int)),
                            this, SLOT(updateChannelADCs(int)));


    ///////////////////////////////////////////////////////////////
    // widget connections for individual channel buttons
    ///////////////////////////////////////////////////////////////
    for(int i = 0; i < 64; i++) {

        // channel states
        connect(VMMSC[i],  SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMSM[i],  SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMST[i],  SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMSTH[i], SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMSL[i],  SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMSMX[i], SIGNAL(pressed()), this, SLOT(updateChannelState()));
        connect(VMMSDVoltage[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelVoltages(int)));
        connect(VMMSZ010bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
        connect(VMMSZ08bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
        connect(VMMSZ06bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::CreateChannelsFieldsMulti()
{
    Font.setPointSize(10);
    int margin = 8;
    int channel_square_size = 16;

    //////////////////////////////////////////////////////
    // add the "global" labels
    //////////////////////////////////////////////////////
    //ui->channelLabelLayout->setContentsMargins(margin*0.75, margin*0.25, margin*2*10, margin);
    //ui->channelLabelLayout->setHorizontalSpacing(4);
    //ui->channelLabelLayout->setVerticalSpacing(1);

    //ui->VMMChanLabel->setFixedSize(channel_square_size*1.22, channel_square_size);
    //ui->VMMChanLabel->setEnabled(false);

    ui->SDLabel_2->setFixedSize(55, channel_square_size*1.15);
    ui->SDLabel2_2->setFixedSize(55, channel_square_size*1.15);

    ui->SCLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SLLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->STLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->STHLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SMLabel->setFixedSize(channel_square_size, channel_square_size);
    ui->SMXLabel->setFixedSize(31, channel_square_size);
    ui->SZ010bLabel_2->setFixedSize(55, channel_square_size);
    ui->SZ08bLabel_2->setFixedSize(55, channel_square_size);
    ui->SZ06bLabel_2->setFixedSize(55, channel_square_size);
    
    //ui->VMMChanLabel2->setFixedSize(channel_square_size*1.22, channel_square_size);
    //ui->VMMChanLabel2->setEnabled(false);

    ui->SCLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SLLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->STLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->STHLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SMLabel2->setFixedSize(channel_square_size, channel_square_size);
    ui->SMXLabel2->setFixedSize(31, channel_square_size);
    ui->SZ010bLabel2_2->setFixedSize(55, channel_square_size);
    ui->SZ08bLabel2_2->setFixedSize(55, channel_square_size);
    ui->SZ06bLabel2_2->setFixedSize(55, channel_square_size);
    

    //////////////////////////////////////////////////////
    // add the channel stuff
    //////////////////////////////////////////////////////

    channel_square_size = 15;
    //void  setContentsMargins(int left, int top, int right, int bottom)
    channelGridLayout->setContentsMargins(margin*0.75, margin*0.25, margin*2*10, margin);
    channelGridLayout->setHorizontalSpacing(4);
    channelGridLayout->setVerticalSpacing(1);

    QString initialValueRadio = "";
    QString counter;

    // add "global" channel trim labels
    for(int i = 0; i < 32; i++) {
        ui->SDLabel_2->addItem(counter.setNum(i)+" mV");
        ui->SDLabel2_2->addItem(counter.setNum(i)+" mV");
    }
    // add "global" 10-bit zero values
    for(int i = 0; i < 32; i++) {
        ui->SZ010bLabel_2->addItem(counter.setNum(i)+"");
        ui->SZ010bLabel2_2->addItem(counter.setNum(i)+"");
    }
    // add "global" 8-bit zero values
    for(int i = 0; i < 16; i++) {
        ui->SZ08bLabel_2->addItem(counter.setNum(i)+"");
        ui->SZ08bLabel2_2->addItem(counter.setNum(i)+"");
    }
    // add "global" 6-bit zero values
    for(int i = 0; i < 8; i++) {
        ui->SZ06bLabel_2->addItem(counter.setNum(i)+"");
        ui->SZ06bLabel2_2->addItem(counter.setNum(i)+"");
    }


    // set initial values for "global" channel bools
    VMMSPBoolAll=0;
    VMMSCBoolAll=0;
    VMMSLBoolAll=0;
    VMMSTBoolAll=0;
    VMMSTHBoolAll=0;
    VMMSMBoolAll=0;
    VMMSMXBoolAll=0;
    VMMSZ010bBoolAll=0;
    VMMSZ08bBoolAll=0;
    VMMSZ06bBoolAll=0;
    
    VMMSPBoolAll2=0;
    VMMSCBoolAll2=0;
    VMMSLBoolAll2=0;
    VMMSTBoolAll2=0;
    VMMSTHBoolAll2=0;
    VMMSMBoolAll2=0;
    VMMSMXBoolAll2=0;
    VMMSZ010bBoolAll2=0;
    VMMSZ08bBoolAll2=0;
    VMMSZ06bBoolAll2=0;

    // prevent scrolling (mouse wheel) from changing these widgets
    ui->SDLabel_2     ->installEventFilter(this);
    ui->SDLabel2_2    ->installEventFilter(this);
    ui->SZ010bLabel_2 ->installEventFilter(this);
    ui->SZ08bLabel_2  ->installEventFilter(this);
    ui->SZ06bLabel_2  ->installEventFilter(this);

    ui->SZ010bLabel2_2 ->installEventFilter(this);
    ui->SZ08bLabel2_2  ->installEventFilter(this);
    ui->SZ06bLabel2_2  ->installEventFilter(this);

    // left hand column
    std::stringstream num_string;
    for(int i = 0; i < 32; i++) {

        if(i==0) {
            //ui->sd_list_widget->addItem("");
            //ui->sd_list_widget->setItemWidget(ui->sd_list_widget->item(i), ui->SDLabel_2);
            //ui->chan_trim_layout->addWidget(ui->SDLabel_2, i, 0, Qt::AlignCenter);
        }

        // this is the fixed channel label for each row on the LHS
        num_string.str("");
        num_string << i;
        ui->chan_label_list_widget->addItem(QString::fromStdString(num_string.str()));
        ui->chan_label_list_widget->item(i)->setTextAlignment(Qt::AlignHCenter);

        //////////////////////////////////////////////////////////
        // VMMSD
        //////////////////////////////////////////////////////////
        VMMSDVoltage[i] = new QComboBox(ui->tab_8);
        VMMSDVoltage[i]->setFixedSize(55, 15);
        //VMMSDVoltage[i]->setFixedSize(55, channel_square_size*0.95);
        VMMSDVoltage[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSDVoltage[i]->addItem(counter.setNum(j)+" mV");
        ui->sd_list_widget->addItem("");
        ui->sd_list_widget->setItemWidget(ui->sd_list_widget->item(i), VMMSDVoltage[i]);

        //////////////////////////////////////////////////////////
        // ADC zeros
        //////////////////////////////////////////////////////////
        VMMSZ010bCBox[i] = new QComboBox(ui->tab_8);
        VMMSZ08bCBox[i]  = new QComboBox(ui->tab_8);
        VMMSZ06bCBox[i]  = new QComboBox(ui->tab_8);

        VMMSZ010bCBox[i]->setFixedSize(55, 15); 
        VMMSZ010bCBox[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSZ010bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ010bCBox[i]->setCurrentIndex(0);
        VMMSZ010bValue[i] = 0;
        ui->sc10_list_widget->addItem("");
        ui->sc10_list_widget->setItemWidget(ui->sc10_list_widget->item(i), VMMSZ010bCBox[i]);

        VMMSZ08bCBox[i]->setFixedSize(55, 15); 
        VMMSZ08bCBox[i]->setFont(Font);
        for(int j = 0; j < 16; j++) VMMSZ08bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ08bCBox[i]->setCurrentIndex(0);
        VMMSZ08bValue[i] = 0;
        ui->sc8_list_widget->addItem("");
        ui->sc8_list_widget->setItemWidget(ui->sc8_list_widget->item(i), VMMSZ08bCBox[i]);

        VMMSZ06bCBox[i]->setFixedSize(55, 15); 
        VMMSZ06bCBox[i]->setFont(Font);
        for(int j = 0; j < 8; j++) VMMSZ06bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ06bCBox[i]->setCurrentIndex(0);
        VMMSZ06bValue[i] = 0;
        ui->sc6_list_widget->addItem("");
        ui->sc6_list_widget->setItemWidget(ui->sc6_list_widget->item(i), VMMSZ06bCBox[i]);


        //VMMChannel[i] = new QLineEdit(counter.setNum(i), ui->groupBox_22);
        //VMMChannel[i]->setAlignment(Qt::AlignHCenter);
        //VMMChannel[i]->setEnabled(false);
        //VMMChannel[i]->setFixedSize(channel_square_size*1.00, channel_square_size);
        //VMMChannel[i]->setFixedSize(channel_square_size*1.22, channel_square_size);

        //addWidget(QWidget * widget, int row, int column, Qt::Alignment alignment = 0)
        //int initial_offset = 2;
        //if(i==0) {
        //    //channelGridLayout->addWidget(ui->VMMChanLabel,      i, 1, Qt::AlignCenter);
        //    ui->chan_label_layout->addWidget(ui->VMMChanLabel,      i, 1, Qt::AlignCenter);
        //}
        ////channelGridLayout->addWidget(VMMChannel[i],             i+initial_offset+1, 1, Qt::AlignCenter);
        //ui->chan_label_layout->addWidget(VMMChannel[i], i+1, 1, Qt::AlignCenter);
        
    } // i (LHS column)

    // left hand column
    for(int i = 32; i < 64; i++) {

        // this is the fixed channel label for each row on the LHS
        num_string.str("");
        num_string << i;
        ui->chan_label_list_widget_2->addItem(QString::fromStdString(num_string.str()));
        ui->chan_label_list_widget_2->item(i-32)->setTextAlignment(Qt::AlignHCenter);

        //////////////////////////////////////////////////////////
        // VMMSD
        //////////////////////////////////////////////////////////
        VMMSDVoltage[i] = new QComboBox(ui->tab_8);
        VMMSDVoltage[i]->setFixedSize(55, 15);
        //VMMSDVoltage[i]->setFixedSize(55, channel_square_size*0.95);
        VMMSDVoltage[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSDVoltage[i]->addItem(counter.setNum(j)+" mV");
        ui->sd_list_widget_2->addItem("");
        ui->sd_list_widget_2->setItemWidget(ui->sd_list_widget_2->item(i-32), VMMSDVoltage[i]);

        //////////////////////////////////////////////////////////
        // ADC zeros
        //////////////////////////////////////////////////////////
        VMMSZ010bCBox[i] = new QComboBox(ui->tab_8);
        VMMSZ08bCBox[i]  = new QComboBox(ui->tab_8);
        VMMSZ06bCBox[i]  = new QComboBox(ui->tab_8);

        VMMSZ010bCBox[i]->setFixedSize(55, 15); 
        VMMSZ010bCBox[i]->setFont(Font);
        for(int j = 0; j < 32; j++) VMMSZ010bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ010bCBox[i]->setCurrentIndex(0);
        VMMSZ010bValue[i] = 0;
        ui->sc10_list_widget_2->addItem("");
        ui->sc10_list_widget_2->setItemWidget(ui->sc10_list_widget_2->item(i-32), VMMSZ010bCBox[i]);

        VMMSZ08bCBox[i]->setFixedSize(55, 15); 
        VMMSZ08bCBox[i]->setFont(Font);
        for(int j = 0; j < 16; j++) VMMSZ08bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ08bCBox[i]->setCurrentIndex(0);
        VMMSZ08bValue[i] = 0;
        ui->sc8_list_widget_2->addItem("");
        ui->sc8_list_widget_2->setItemWidget(ui->sc8_list_widget_2->item(i-32), VMMSZ08bCBox[i]);

        VMMSZ06bCBox[i]->setFixedSize(55, 15); 
        VMMSZ06bCBox[i]->setFont(Font);
        for(int j = 0; j < 8; j++) VMMSZ06bCBox[i]->addItem(counter.setNum(j)+"");
        VMMSZ06bCBox[i]->setCurrentIndex(0);
        VMMSZ06bValue[i] = 0;
        ui->sc6_list_widget_2->addItem("");
        ui->sc6_list_widget_2->setItemWidget(ui->sc6_list_widget_2->item(i-32), VMMSZ06bCBox[i]);


        //VMMChannel[i] = new QLineEdit(counter.setNum(i), ui->groupBox_22);
        //VMMChannel[i]->setAlignment(Qt::AlignHCenter);
        //VMMChannel[i]->setEnabled(false);
        //VMMChannel[i]->setFixedSize(channel_square_size*1.00, channel_square_size);
        //VMMChannel[i]->setFixedSize(channel_square_size*1.22, channel_square_size);

        //addWidget(QWidget * widget, int row, int column, Qt::Alignment alignment = 0)
        //int initial_offset = 2;
        //if(i==0) {
        //    //channelGridLayout->addWidget(ui->VMMChanLabel,      i, 1, Qt::AlignCenter);
        //    ui->chan_label_layout->addWidget(ui->VMMChanLabel,      i, 1, Qt::AlignCenter);
        //}
        ////channelGridLayout->addWidget(VMMChannel[i],             i+initial_offset+1, 1, Qt::AlignCenter);
        //ui->chan_label_layout->addWidget(VMMChannel[i], i+1, 1, Qt::AlignCenter);
        
    } // i (LHS column)



    ///////////////////////////////////////////////
    // multi
    ///////////////////////////////////////////////
    vector<QListWidget*> channel_list_widgets;
    channel_list_widgets.push_back(ui->sc_list_widget);
    channel_list_widgets.push_back(ui->sl_list_widget);
    channel_list_widgets.push_back(ui->sth_list_widget);
    channel_list_widgets.push_back(ui->st_list_widget);
    channel_list_widgets.push_back(ui->sm_list_widget);

    channel_list_widgets.push_back(ui->sc_list_widget_2);
    channel_list_widgets.push_back(ui->sl_list_widget_2);
    channel_list_widgets.push_back(ui->sth_list_widget_2);
    channel_list_widgets.push_back(ui->st_list_widget_2);
    channel_list_widgets.push_back(ui->sm_list_widget_2);

    channel_list_widgets.push_back(ui->smx_list_widget);
    channel_list_widgets.push_back(ui->smx_list_widget_2);

    //for(auto list : channel_list_widgets) {
    for(size_t i = 0; i < channel_list_widgets.size(); i++) {
        QListWidget* list = channel_list_widgets.at(i);
        //list->setStyleSheet( "QListWidget::item { border-bottom: 1px solid black; }" );
        if(i < 10) {
            for(uint ichan = 0; ichan < 32; ichan++) {
                list->addItem("");
                //list->item(ichan)->setSizeHint(QSize(list->item(ichan)->sizeHint().width(), list->height()/32));
                list->item(ichan)->setSizeHint(QSize(list->item(ichan)->sizeHint().width(), 14));
            }
        }
        else {
            for(uint ichan = 0; ichan < 32; ichan++) {
                list->addItem("");
                list->item(ichan)->setSizeHint(QSize(list->item(ichan)->sizeHint().width(), 14));
            } // ichan
        }
    }

    ////////////////////////////////////////////////////////////////
    // widget connections for "all channel" buttons at the top
    ////////////////////////////////////////////////////////////////

    // SC
    connect(ui->SCLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->SCLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    // SL
    connect(ui->SLLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->SLLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    // STH
    connect(ui->STHLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->STHLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    // ST
    connect(ui->STLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->STLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    // SM
    connect(ui->SMLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->SMLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    // SMX
    connect(ui->SMXLabel_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));
    connect(ui->SMXLabel2_2,  SIGNAL(pressed()),  this, SLOT(updateChannelState()));

    //connect(ui->sc_list_widget, SIGNAL(itemChanged(QListWidgetItem*)),
    //        this, SLOT(updateIndividualChannelState(QListWidgetItem*)));
    // SC
    connect(ui->sc_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->sc_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    // SL
    connect(ui->sl_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->sl_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    // STH
    connect(ui->sth_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->sth_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    // ST 
    connect(ui->st_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->st_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    // SM 
    connect(ui->sm_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->sm_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    // SMX 
    connect(ui->smx_list_widget, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));
    connect(ui->smx_list_widget_2, SIGNAL(itemSelectionChanged()), this, SLOT(updateIndividualChannelState()));

    // ADC offsets
    connect(ui->SZ010bLabel_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ08bLabel_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ06bLabel_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));
    
    connect(ui->SZ010bLabel2_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ08bLabel2_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));
    connect(ui->SZ06bLabel2_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelADCs(int)));

    // Channel threshold trims
    connect(ui->SDLabel_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelVoltages(int)));
    connect(ui->SDLabel2_2, SIGNAL(currentIndexChanged(int)),
            this, SLOT(updateChannelVoltages(int)));

    for(int i = 0; i < 64; i++) {
        connect(VMMSDVoltage[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelVoltages(int)));
        connect(VMMSZ010bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
        connect(VMMSZ08bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
        connect(VMMSZ06bCBox[i], SIGNAL(currentIndexChanged(int)),
                                    this, SLOT(updateChannelADCs(int)));
    }



}
// ------------------------------------------------------------------------- //
void MainWindow::updateChannelState()
{
    ignore_channel_update = true;

    // ***********************  SC ALL *************************** //
    if(ui->SCLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->sc_list_widget->item(j)->setSelected(!VMMSCBoolAll);
            VMMSCBool[j] = !VMMSCBoolAll;
            //cout << "VMMSCBool[" << j << "] " << VMMSCBool[j] << endl;
        }
        VMMSCBoolAll=!VMMSCBoolAll;
    }
    else if(ui->SCLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->sc_list_widget_2->item(j-32)->setSelected(!VMMSCBoolAll2);
            VMMSCBool[j] = !VMMSCBoolAll2;
            //cout << "VMMSCBool[" << j << "] " << VMMSCBool[j] << endl;
        }
        VMMSCBoolAll2=!VMMSCBoolAll2;
    }

    // ***********************  SL ALL *************************** //
    if(ui->SLLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->sl_list_widget->item(j)->setSelected(!VMMSLBoolAll);
            VMMSLBool[j] = !VMMSLBoolAll;
            //cout << "VMMSLBool[" << j << "] " << VMMSLBool[j] << endl;
            
        }
        VMMSLBoolAll=!VMMSLBoolAll;
    }
    else if(ui->SLLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->sl_list_widget_2->item(j-32)->setSelected(!VMMSLBoolAll2);
            VMMSLBool[j] = !VMMSLBoolAll2;
            //cout << "VMMSLBool[" << j << "] " << VMMSLBool[j] << endl;
        }
        VMMSLBoolAll2=!VMMSLBoolAll2;
    }
    
    // ***********************  STH ALL *************************** //
    if(ui->STHLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->sth_list_widget->item(j)->setSelected(!VMMSTHBoolAll);
            VMMSTHBool[j]=!VMMSTHBoolAll;
            //cout << "VMMSTHBool[" << j << "] " << VMMSTHBool[j] << endl;
        }
        VMMSTHBoolAll=!VMMSTHBoolAll;
    }
    else if(ui->STHLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->sth_list_widget_2->item(j-32)->setSelected(!VMMSTHBoolAll2);
            VMMSTHBool[j]=!VMMSTHBoolAll2;

            //cout << "VMMSTHBool[" << j << "] " << VMMSTHBool[j] << endl;
        }
        VMMSTHBoolAll2=!VMMSTHBoolAll2;
    }

    // ***********************  ST ALL *************************** //
    if(ui->STLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->st_list_widget->item(j)->setSelected(!VMMSTBoolAll);
            VMMSTBool[j]=!VMMSTBoolAll;
            //cout << "VMMSTBool[" << j << "] " << VMMSTBool[j] << endl;
        }
        VMMSTBoolAll=!VMMSTBoolAll;
    } 
    if(ui->STLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->st_list_widget_2->item(j-32)->setSelected(!VMMSTBoolAll2);
            VMMSTBool[j]=!VMMSTBoolAll2;
            //cout << "VMMSTBool[" << j << "] " << VMMSTBool[j] << endl;
        }
        VMMSTBoolAll2=!VMMSTBoolAll2;
    } 
    // ***********************  SM ALL *************************** //
    if(ui->SMLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->sm_list_widget->item(j)->setSelected(!VMMSMBoolAll);
            VMMSMBool[j]=!VMMSMBoolAll;
            //cout << "VMMSMBool[" << j << "] " << VMMSMBool[j] << endl;
        }
        VMMSMBoolAll=!VMMSMBoolAll;
    }
    if(ui->SMLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->sm_list_widget_2->item(j-32)->setSelected(!VMMSMBoolAll2);
            VMMSMBool[j]=!VMMSMBoolAll2;
            //cout << "VMMSMBool[" << j << "] " << VMMSMBool[j] << endl;
        }
        VMMSMBoolAll2=!VMMSMBoolAll2;
    }

    // ***********************  SMX ALL *************************** //
    if(ui->SMXLabel_2 == QObject::sender()) {
        for(int j = 0; j < 32; j++) {
            ui->smx_list_widget->item(j)->setSelected(!VMMSMXBoolAll);
            VMMSMXBool[j]=!VMMSMXBoolAll;
            //cout << "VMMSMXBool[" << j << "] " << VMMSMXBool[j] << endl;
        }
        VMMSMXBoolAll=!VMMSMXBoolAll;
    }
    if(ui->SMXLabel2_2 == QObject::sender()) {
        for(int j = 32; j < 64; j++) {
            ui->smx_list_widget_2->item(j-32)->setSelected(!VMMSMXBoolAll2);
            VMMSMXBool[j]=!VMMSMXBoolAll2;
            //cout << "VMMSMXBool[" << j << "] " << VMMSMXBool[j] << endl;
        }
        VMMSMXBoolAll2=!VMMSMXBoolAll2;
    }
    ignore_channel_update = false;


   // /////////////////////////////////////////////////////////////////
   // // individual channels are acted upon
   // /////////////////////////////////////////////////////////////////
   // for(int i = 0; i < 64; i++) {
   //     if(i<32) {
   //         ////////////////////////////////////////
   //         // SC
   //         ////////////////////////////////////////
   //         if(ui->sc_list_widget->item(i)
   //     } // < 32

   // } // i

/*
    // ***********************  SC  ********************************* //
    if(ui->SCLabel == QObject::sender()){
        if(VMMSCBoolAll==0){
            for(int j=0;j<32;j++){
                VMMSC[j]->setStyleSheet("background-color: green");
                VMMSCBool[j]=true;
            }
            VMMSCBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMSC[j]->setStyleSheet("background-color: lightGray");
                VMMSCBool[j]=0;
            }
            VMMSCBoolAll=0;
        }
    }
    else if(ui->SCLabel2 == QObject::sender()){
        if(VMMSCBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMSC[j]->setStyleSheet("background-color: green");
                VMMSCBool[j]=true;
            }
            VMMSCBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMSC[j]->setStyleSheet("background-color: lightGray");
                VMMSCBool[j]=0;
            }
            VMMSCBoolAll2=0;
        }
    }
    // ***********************  SL  ********************************* //
    if(ui->SLLabel == QObject::sender()){
        if(VMMSLBoolAll==0){
            for(int j=0;j<32;j++){
                VMMSL[j]->setStyleSheet("background-color: green");
                VMMSLBool[j]=true;
            }
            VMMSLBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMSL[j]->setStyleSheet("background-color: lightGray");
                VMMSLBool[j]=0;
            }
            VMMSLBoolAll=0;
        }
    }
    else if(ui->SLLabel2 == QObject::sender()){
        if(VMMSLBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMSL[j]->setStyleSheet("background-color: green");
                VMMSLBool[j]=true;
            }
            VMMSLBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMSL[j]->setStyleSheet("background-color: lightGray");
                VMMSLBool[j]=0;
            }
            VMMSLBoolAll2=0;
        }
    }
    // ***********************  ST  ********************************* //
    if(ui->STLabel == QObject::sender()){
        if(VMMSTBoolAll==0){
            for(int j=0;j<32;j++){
                VMMST[j]->setStyleSheet("background-color: green");
                VMMSTBool[j]=true;
            }
            VMMSTBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMST[j]->setStyleSheet("background-color: lightGray");
                VMMSTBool[j]=0;
            }
            VMMSTBoolAll=0;
        }
    }
    else if(ui->STLabel2 == QObject::sender()){
        if(VMMSTBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMST[j]->setStyleSheet("background-color: green");
                VMMSTBool[j]=true;
            }
            VMMSTBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMST[j]->setStyleSheet("background-color: lightGray");
                VMMSTBool[j]=0;
            }
            VMMSTBoolAll2=0;
        }
    }
    // ***********************  STH  ********************************* //
    if(ui->STHLabel == QObject::sender()){
        if(VMMSTHBoolAll==0){
            for(int j=0;j<32;j++){
                VMMSTH[j]->setStyleSheet("background-color: green");
                VMMSTHBool[j]=true;
            }
            VMMSTHBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMSTH[j]->setStyleSheet("background-color: lightGray");
                VMMSTHBool[j]=0;
            }
            VMMSTHBoolAll=0;
        }
    }
    else if(ui->STHLabel2 == QObject::sender()){
        if(VMMSTHBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMSTH[j]->setStyleSheet("background-color: green");
                VMMSTHBool[j]=true;
            }
            VMMSTHBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMSTH[j]->setStyleSheet("background-color: lightGray");
                VMMSTHBool[j]=0;
            }
            VMMSTHBoolAll2=0;
        }
    }
    // ***********************  SM  ********************************* //
    if(ui->SMLabel == QObject::sender()){
        if(VMMSMBoolAll==0){
            for(int j=0;j<32;j++){
                VMMSM[j]->setStyleSheet("background-color: green");
                VMMSMBool[j]=true;
            }
            VMMSMBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMSM[j]->setStyleSheet("background-color: lightGray");
                VMMSMBool[j]=0;
            }
            VMMSMBoolAll=0;
        }
    }
    else if(ui->SMLabel2 == QObject::sender()){
        if(VMMSMBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMSM[j]->setStyleSheet("background-color: green");
                VMMSMBool[j]=true;
            }
            VMMSMBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMSM[j]->setStyleSheet("background-color: lightGray");
                VMMSMBool[j]=0;
            }
            VMMSMBoolAll2=0;
        }
    }
    // ***********************  SMX  ********************************* //
    if(ui->SMXLabel == QObject::sender()){
        if(VMMSMXBoolAll==0){
            for(int j=0;j<32;j++){
                VMMSMX[j]->setStyleSheet("background-color: green");
                VMMSMXBool[j]=true;
            }
            VMMSMXBoolAll=1;
        }else{
            for(int j=0;j<32;j++){
                VMMSMX[j]->setStyleSheet("background-color: lightGray");
                VMMSMXBool[j]=0;
            }
            VMMSMXBoolAll=0;
        }
    }
    else if(ui->SMXLabel2 == QObject::sender()){
        if(VMMSMXBoolAll2==0){
            for(int j=32;j<64;j++){
                VMMSMX[j]->setStyleSheet("background-color: green");
                VMMSMXBool[j]=true;
            }
            VMMSMXBoolAll2=1;
        }else{
            for(int j=32;j<64;j++){
                VMMSMX[j]->setStyleSheet("background-color: lightGray");
                VMMSMXBool[j]=0;
            }
            VMMSMXBoolAll2=0;
        }
    }
    // *********************  Loop Individually  ********************** //
    for(int i=0;i<64;i++){
        if(VMMSC[i] == QObject::sender()){
            if(VMMSCBool[i]==0){
                VMMSC[i]->setStyleSheet("background-color: green");
                VMMSCBool[i]=true;
            }else if(VMMSCBool[i]==1){
                VMMSC[i]->setStyleSheet("background-color: lightGray");
                VMMSCBool[i]=false;
            }
        }else if(VMMST[i] == QObject::sender()){
            if(VMMSTBool[i]==0){
                VMMST[i]->setStyleSheet("background-color: green");
                VMMSTBool[i]=true;
            }else if(VMMSTBool[i]==1){
                VMMST[i]->setStyleSheet("background-color: lightGray");
                VMMSTBool[i]=false;
            }
        }else if(VMMSTH[i] == QObject::sender()){
            if(VMMSTHBool[i]==0) {
                VMMSTH[i]->setStyleSheet("background-color: green");
                VMMSTHBool[i]=true;
            }else if(VMMSTHBool[i]==1){
                VMMSTH[i]->setStyleSheet("background-color: lightGray");
                VMMSTHBool[i]=false;
            }
        }else if(VMMSL[i] == QObject::sender()){
            if(VMMSLBool[i]==0){
                VMMSL[i]->setStyleSheet("background-color: green");
                VMMSLBool[i]=true;
            }else if(VMMSLBool[i]==1){
                VMMSL[i]->setStyleSheet("background-color: lightGray");
                VMMSLBool[i]=false;
            }
        }else if(VMMSM[i] == QObject::sender()){
            if(VMMSMBool[i]==0){
                VMMSM[i]->setStyleSheet("background-color: green");
                VMMSMBool[i]=true;
            }else if(VMMSMBool[i]==1){
                VMMSM[i]->setStyleSheet("background-color: lightGray");
                VMMSMBool[i]=false;
            }
        }else if(VMMSMX[i] == QObject::sender()){
            if(VMMSMXBool[i]==0){
                VMMSMX[i]->setStyleSheet("background-color: green");
                VMMSMXBool[i]=true;
            }else if(VMMSMXBool[i]==1){
                VMMSMX[i]->setStyleSheet("background-color: lightGray");
                VMMSMXBool[i]=false;
            }
        }
    }
*/
}
// ------------------------------------------------------------------------- //
void MainWindow::updateIndividualChannelState()
{
    if(ignore_channel_update) return;
    vector<QListWidget*> channel_list_widgets;
    channel_list_widgets.push_back(ui->sc_list_widget);
    channel_list_widgets.push_back(ui->sl_list_widget);
    channel_list_widgets.push_back(ui->sth_list_widget);
    channel_list_widgets.push_back(ui->st_list_widget);
    channel_list_widgets.push_back(ui->sm_list_widget);

    channel_list_widgets.push_back(ui->sc_list_widget_2);
    channel_list_widgets.push_back(ui->sl_list_widget_2);
    channel_list_widgets.push_back(ui->sth_list_widget_2);
    channel_list_widgets.push_back(ui->st_list_widget_2);
    channel_list_widgets.push_back(ui->sm_list_widget_2);

    channel_list_widgets.push_back(ui->smx_list_widget);
    channel_list_widgets.push_back(ui->smx_list_widget_2);

    //cout << "individual \n" << endl;
    // ---------------- SC ----------------- //
    if(ui->sc_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSCBool[i] = ui->sc_list_widget->item(i)->isSelected();
            //cout << "VMMSCBool[" << i << "] : " << VMMSCBool[i] << endl;
        }
    }
    else if(ui->sc_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSCBool[i] = ui->sc_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSCBool[" << i << "] : " << VMMSCBool[i] << endl;
        }
    }
    // ---------------- SL ----------------- //
    if(ui->sl_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSLBool[i] = ui->sl_list_widget->item(i)->isSelected();
            //cout << "VMMSLBool[" << i << "] : " << VMMSLBool[i] << endl;
        }
    }
    else if(ui->sl_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSLBool[i] = ui->sl_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSLBool[" << i << "] : " << VMMSLBool[i] << endl;
        }
    }
    // ---------------- STH ----------------- //
    if(ui->sth_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSTHBool[i] = ui->sth_list_widget->item(i)->isSelected();
            //cout << "VMMSTHBool[" << i << "] : " << VMMSTHBool[i] << endl;
        }
    }
    else if(ui->sth_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSTHBool[i] = ui->sth_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSTHBool[" << i << "] : " << VMMSTHBool[i] << endl;
        }
    }
    // ---------------- ST ----------------- //
    if(ui->st_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSTBool[i] = ui->st_list_widget->item(i)->isSelected();
            //cout << "VMMSTBool[" << i << "] : " << VMMSTBool[i] << endl;
        }
    }
    else if(ui->st_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSTBool[i] = ui->st_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSTBool[" << i << "] : " << VMMSTBool[i] << endl;
        }
    }
    // ---------------- SM ----------------- //
    if(ui->sm_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSMBool[i] = ui->sm_list_widget->item(i)->isSelected();
            //cout << "VMMSMBool[" << i << "] : " << VMMSMBool[i] << endl;
        }
    }
    else if(ui->sm_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSMBool[i] = ui->sm_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSMBool[" << i << "] : " << VMMSMBool[i] << endl;
        }
    }
    // ---------------- SMX ----------------- //
    if(ui->smx_list_widget == QObject::sender()) {
        for(int i = 0; i < 32; i++) {
            VMMSMXBool[i] = ui->smx_list_widget->item(i)->isSelected();
            //cout << "VMMSMXBool[" << i << "] : " << VMMSMXBool[i] << endl;
        }
    }
    else if(ui->smx_list_widget_2 == QObject::sender()) {
        for(int i = 32; i < 64; i++) {
            VMMSMXBool[i] = ui->smx_list_widget_2->item(i-32)->isSelected();
            //cout << "VMMSMXBool[" << i << "] : " << VMMSMXBool[i] << endl;
        }
    }


}
// ------------------------------------------------------------------------- //
void MainWindow::updateChannelADCs(int index)
{
    // ***********************  SD  ******************************* //
    for(int j=0;j<32;j++){
        if(ui->SZ010bLabel_2 == QObject::sender()){
            VMMSZ010bCBox[j]->setCurrentIndex(index);
            VMMSZ010bValue[j]=index;
            //cout << "SZ10[" << j << "] : " << VMMSZ010bValue[j] << endl;
        }
        if(ui->SZ08bLabel_2 == QObject::sender()){
            VMMSZ08bCBox[j]->setCurrentIndex(index);
            VMMSZ08bValue[j]=index;
            //cout << "SZ8[" << j << "] : " << VMMSZ08bValue[j] << endl;
        }
        if(ui->SZ06bLabel_2 == QObject::sender()){
            VMMSZ06bCBox[j]->setCurrentIndex(index);
            VMMSZ06bValue[j]=index;
            //cout << "SZ6[" << j << "] : " << VMMSZ06bValue[j] << endl;
        }
    }
    for(int j=32;j<64;j++){
        if(ui->SZ010bLabel2_2 == QObject::sender()){
            VMMSZ010bCBox[j]->setCurrentIndex(index);
            VMMSZ010bValue[j]=index;
            //cout << "SZ10[" << j << "] : " << VMMSZ010bValue[j] << endl;
        }
        if(ui->SZ08bLabel2_2 == QObject::sender()){
            VMMSZ08bCBox[j]->setCurrentIndex(index);
            VMMSZ08bValue[j]=index;
            //cout << "SZ8[" << j << "] : " << VMMSZ08bValue[j] << endl;
        }
        if(ui->SZ06bLabel2_2 == QObject::sender()){
            VMMSZ06bCBox[j]->setCurrentIndex(index);
            VMMSZ06bValue[j]=index;
            //cout << "SZ6[" << j << "] : " << VMMSZ06bValue[j] << endl;
        }
    }

    for(int i=0;i<64;i++){
        if(VMMSZ010bCBox[i] == QObject::sender()){
            VMMSZ010bValue[i]=index;
            //cout << "combo SZ10[" << i << "] : " << VMMSZ010bValue[i] << endl;
        }
        if(VMMSZ08bCBox[i] == QObject::sender()){
            VMMSZ08bValue[i]=index;
            //cout << "combo SZ8[" << i << "] : " << VMMSZ08bValue[i] << endl;
        }
        if(VMMSZ06bCBox[i] == QObject::sender()){
            VMMSZ06bValue[i]=index;
            //cout << "combo SZ6[" << i << "] : " << VMMSZ06bValue[i] << endl;
        }
    }


}
// ------------------------------------------------------------------------- //
void MainWindow::updateChannelVoltages(int index)
{
    // ***********************  SD  ******************************** //
    if(ui->SDLabel_2 == QObject::sender()){
        for(int j=0;j<32;j++){
            VMMSDVoltage[j]->setCurrentIndex(index);
            VMMSDValue[j]=index;
            //cout << "  VMMSDValue[" << j << "] : " << VMMSDValue[j] << endl;
        }
    }
    if(ui->SDLabel2_2 == QObject::sender()){
        for(int j=32;j<64;j++){
            VMMSDVoltage[j]->setCurrentIndex(index);
            VMMSDValue[j]=index;
            //cout << "  VMMSDValue[" << j << "] : " << VMMSDValue[j] << endl;
        }
    }
    for(int i=0;i<64;i++){
        if(VMMSDVoltage[i] == QObject::sender()){
            VMMSDValue[i]=index;
            //cout << "i VMMSDValue[" << i << "] : " << VMMSDValue[i] << endl;
        }
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::updateConfigState()
{
    // ------------------------------------------- //
    //  GlobalSetting
    // ------------------------------------------- //
    GlobalSetting global = configHandle().globalSettings();

    ui->global_sp->setCurrentIndex(global.sp);
    ui->global_sdp->setCurrentIndex(global.sdp);
    ui->global_sbmx->setCurrentIndex(global.sbmx);
    ui->global_sbft->setCurrentIndex(global.sbft);
    ui->global_sbfp->setCurrentIndex(global.sbfp);
    ui->global_sbfm->setCurrentIndex(global.sbfm);
    ui->global_slg->setCurrentIndex(global.slg);
    ui->global_sm5->setCurrentIndex(global.sm5);
    ui->global_scmx->setCurrentIndex(global.scmx);
    ui->global_sfa->setCurrentIndex(global.sfa);
    ui->global_sfam->setCurrentIndex(global.sfam);
    ui->global_st->setCurrentIndex(global.st);
    ui->global_sfm->setCurrentIndex(global.sfm);
    ui->global_sg->setCurrentIndex(global.sg);
    ui->global_sng->setCurrentIndex(global.sng);
    ui->global_stot->setCurrentIndex(global.stot);
    ui->global_stpp->setCurrentIndex(global.stpp);
    ui->global_sttt->setCurrentIndex(global.sttt);
    ui->global_ssh->setCurrentIndex(global.ssh);
    ui->global_stc->setCurrentIndex(global.stc);
    ui->global_sdt_dac->setValue(global.sdt_dac);
    ui->global_sdp_dac->setValue(global.sdp_dac);
    ui->global_sc10b->setCurrentIndex(global.sc10b);
    ui->global_sc8b->setCurrentIndex(global.sc8b);
    ui->global_sc6b->setCurrentIndex(global.sc6b);
    ui->global_s8b->setCurrentIndex(global.s8b);
    ui->global_s6b->setCurrentIndex(global.s6b);
    ui->global_s10b->setCurrentIndex(global.s10b);
    ui->global_sdcks->setCurrentIndex(global.sdcks);
    ui->global_sdcka->setCurrentIndex(global.sdcka);
    ui->global_sdck6b->setCurrentIndex(global.sdck6b);
    ui->global_sdrv->setCurrentIndex(global.sdrv);
    ui->global_slvs->setCurrentIndex(global.slvs);
    ui->global_stcr->setCurrentIndex(global.stcr);
    ui->global_ssart->setCurrentIndex(global.ssart);
    ui->global_s32->setCurrentIndex(global.s32);
    ui->global_stlc->setCurrentIndex(global.stlc);
    ui->global_srec->setCurrentIndex(global.srec);
    ui->global_sbip->setCurrentIndex(global.sbip);
    ui->global_srat->setCurrentIndex(global.srat);
    ui->global_sfrst->setCurrentIndex(global.sfrst);
    ui->global_slvsbc->setCurrentIndex(global.slvsbc);
    ui->global_slvstp->setCurrentIndex(global.slvstp);
    ui->global_slvstk->setCurrentIndex(global.slvstk);
    ui->global_slvsdt->setCurrentIndex(global.slvsdt);
    ui->global_slvsart->setCurrentIndex(global.slvsart);
    ui->global_slvstki->setCurrentIndex(global.slvstki);
    ui->global_slvsena->setCurrentIndex(global.slvsena);
    ui->global_slvs6b->setCurrentIndex(global.slvs6b);
    ui->global_sL0enaV->setCurrentIndex(global.sL0enaV);
    ui->global_slh->setCurrentIndex(global.slh);
    ui->global_slxh->setCurrentIndex(global.slxh);
    ui->global_stgc->setCurrentIndex(global.stgc);
    ui->global_reset->setCurrentIndex(0);
    ui->global_sL0ena->setCurrentIndex(global.sL0ena);
    ui->global_l0offset->setValue(global.l0offset);
    ui->global_offset->setValue(global.offset);
    ui->global_rollover->setValue(global.rollover);
    ui->global_window->setValue(global.window);
    ui->global_truncate->setValue(global.truncate);
    ui->global_nskip->setValue(global.nskip);
    ui->global_sL0cktest->setCurrentIndex(global.sL0cktest);
    ui->global_sL0ckinv->setCurrentIndex(global.sL0ckinv);
    ui->global_sL0dckinv->setCurrentIndex(global.sL0dckinv);
    ui->global_nskipm->setCurrentIndex(global.nskipm);


//// old [below]
/*
    ui->spg->setCurrentIndex(global.polarity);
    ui->slg->setCurrentIndex(global.leakage_current);
    ui->sdrv->setCurrentIndex(global.analog_tristates);
    ui->sfm->setCurrentIndex(global.double_leakage);
    ui->sg->setCurrentIndex(global.gain);
    ui->st->setCurrentIndex(global.peak_time);
    ui->sng->setCurrentIndex(global.neighbor_trigger);
    ui->stc->setCurrentIndex(global.tac_slope);
    ui->sdp->setCurrentIndex(global.disable_at_peak);
    ui->sfa->setCurrentIndex(global.art);
    ui->sfam->setCurrentIndex(global.art_mode);
    ui->sdcka->setCurrentIndex(global.dual_clock_art);
    ui->sbfm->setCurrentIndex(global.out_buffer_mo);
    ui->sbfp->setCurrentIndex(global.out_buffer_pdo);
    ui->sbft->setCurrentIndex(global.out_buffer_tdo);
    ui->sm5_sm0->setCurrentIndex(global.channel_monitor);
    ui->scmx->setCurrentIndex(global.monitoring_control);
    ui->sbmx->setCurrentIndex(global.monitor_pdo_out);
    ui->spdc->setCurrentIndex(global.adcs);
    ui->ssh->setCurrentIndex(global.sub_hysteresis);
    ui->sttt->setCurrentIndex(global.direct_time);
    ui->stpp->setCurrentIndex(global.direct_time_mode0);
    ui->stot->setCurrentIndex(global.direct_time_mode1);
    ui->s8b->setCurrentIndex(global.conv_mode_8bit);
    ui->s6b->setCurrentIndex(global.enable_6bit);
    ui->sc010b->setCurrentIndex(global.adc_10bit);
    ui->sc08b->setCurrentIndex(global.adc_8bit);
    ui->sc06b->setCurrentIndex(global.adc_6bit);
    ui->sdcks->setCurrentIndex(global.dual_clock_data);
    ui->sdck6b->setCurrentIndex(global.dual_clock_6bit);
    ui->sdt->setValue(global.threshold_dac);
    ui->sdp_2->setValue(global.test_pulse_dac);
*/
    // ------------------------------------------- //
    //  VMM Map
    // ------------------------------------------- //
    //vector<QCheckBox*> vmm_select;
    vector<QRadioButton*> vmm_select;
    vmm_select.push_back(ui->vmm_select_1);
    vmm_select.push_back(ui->vmm_select_2);
    vmm_select.push_back(ui->vmm_select_3);
    vmm_select.push_back(ui->vmm_select_4);
    vmm_select.push_back(ui->vmm_select_5);
    vmm_select.push_back(ui->vmm_select_6);
    vmm_select.push_back(ui->vmm_select_7);
    vmm_select.push_back(ui->vmm_select_8);
    
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        tmp = tmp << i;
        if(configHandle().vmmMap().mask & tmp) {
            vmm_select.at(i)->setChecked(true);
        }
        else {
            vmm_select.at(i)->setChecked(false);
        }
    }

/*
    // ------------------------------------------- //
    //  HDMI Channel Map
    // ------------------------------------------- //
    vector<VMMMap> chMap;
    for(int i = 0; i < 8; i++) {
        chMap.push_back(configHandle().hdmiChannelSettings(i));
    }
    int ch = 0;
    ui->hdmi1->setChecked(  chMap[ch].on);
    ui->hdmi1_1->setChecked(chMap[ch].first);
    ui->hdmi1_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi2->setChecked(  chMap[ch].on);
    ui->hdmi2_1->setChecked(chMap[ch].first);
    ui->hdmi2_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi3->setChecked(  chMap[ch].on);
    ui->hdmi3_1->setChecked(chMap[ch].first);
    ui->hdmi3_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi4->setChecked(  chMap[ch].on);
    ui->hdmi4_1->setChecked(chMap[ch].first);
    ui->hdmi4_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi5->setChecked(  chMap[ch].on);
    ui->hdmi5_1->setChecked(chMap[ch].first);
    ui->hdmi5_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi6->setChecked(  chMap[ch].on);
    ui->hdmi6_1->setChecked(chMap[ch].first);
    ui->hdmi6_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi7->setChecked(  chMap[ch].on);
    ui->hdmi7_1->setChecked(chMap[ch].first);
    ui->hdmi7_2->setChecked(chMap[ch].second);
    ch++;
    ui->hdmi8->setChecked(  chMap[ch].on);
    ui->hdmi8_1->setChecked(chMap[ch].first);
    ui->hdmi8_2->setChecked(chMap[ch].second);
*/

    // ------------------------------------------------- //
    //  individual channels
    // ------------------------------------------------- //
    for(int i = 0; i < 64; i++) {
        Channel chan = configHandle().channelSettings(i);
        if(!((int)chan.number == i)) {
            stringstream ss;
            ss << "Channel config from ConfigHandler is out of sync!"
               << " Attempting to access state of channel number " << i
               << " but corresponding index in ConfigHandler channel map is "
               << chan.number << "!";
            msg()(ss, "MainWindow::updateConfigState");
        }

        /*
        // for some reason in the original code
        // booleans of 0 are 'true' and booleans
        // of 1 are 'false'. go figure?

        VMMSCBool[i] = !chan.sc;
        VMMSLBool[i] = !chan.sl;
        VMMSTHBool[i] = !chan.sth;
        VMMSTBool[i] = !chan.st;
        VMMSMBool[i] = !chan.sm;
        VMMSDValue[i] = chan.sd;
        VMMSMXBool[i] = !chan.smx;
        VMMSZ010bValue[i] = chan.sz10b;
        VMMSZ08bValue[i] = chan.sz8b;
        VMMSZ06bValue[i] = chan.sz6b;
        */

        VMMSCBool[i]        = chan.sc;
        VMMSLBool[i]        = chan.sl;
        VMMSTHBool[i]       = chan.sth;
        VMMSTBool[i]        = chan.st;
        VMMSMBool[i]        = chan.sm;
        VMMSDValue[i]       = chan.sd;
        VMMSMXBool[i]       = chan.smx;
        VMMSZ010bValue[i]   = chan.sz10b;
        VMMSZ08bValue[i]    = chan.sz8b;
        VMMSZ06bValue[i]    = chan.sz6b;

        // make the gui gui
        //VMMSMX[i]->click();
        //VMMSM[i]->click();
        //VMMSL[i]->click();
        //VMMSTH[i]->click();
        //VMMST[i]->click();
        //VMMSC[i]->click();
        if(i<32) {
            // sc
            ui->sc_list_widget->item(i)->setSelected(VMMSCBool[i]);
            // sl
            ui->sl_list_widget->item(i)->setSelected(VMMSLBool[i]);
            // sth
            ui->sth_list_widget->item(i)->setSelected(VMMSTHBool[i]);
            // st
            ui->st_list_widget->item(i)->setSelected(VMMSTBool[i]);
            // sm
            ui->sm_list_widget->item(i)->setSelected(VMMSMBool[i]);
            // smx
            ui->smx_list_widget->item(i)->setSelected(VMMSMXBool[i]);
        }
        else {
            // sc
            ui->sc_list_widget_2->item(i-32)->setSelected(VMMSCBool[i]);
            // sl
            ui->sl_list_widget_2->item(i-32)->setSelected(VMMSLBool[i]);
            // sth
            ui->sth_list_widget_2->item(i-32)->setSelected(VMMSTHBool[i]);
            // st
            ui->st_list_widget_2->item(i-32)->setSelected(VMMSTBool[i]);
            // sm
            ui->sm_list_widget_2->item(i-32)->setSelected(VMMSMBool[i]);
            // smx
            ui->smx_list_widget_2->item(i-32)->setSelected(VMMSMXBool[i]);
        }

        VMMSDVoltage[i]->setCurrentIndex(VMMSDValue[i]);
        VMMSZ010bCBox[i]->setCurrentIndex(VMMSZ010bValue[i]);
        VMMSZ08bCBox[i]->setCurrentIndex(VMMSZ08bValue[i]);
        VMMSZ06bCBox[i]->setCurrentIndex(VMMSZ06bValue[i]);
    }

    // ------------------------------------------------- //
    //  T/DAQ settings
    // ------------------------------------------------- //
    TriggerDAQ tdaq = configHandle().daqSettings();
    ui->trigExternalBCLatency->setValue(tdaq.trigger_latency);

    // ------------------------------------------------- //
    //  Clocks Settings
    // ------------------------------------------------- //
    FPGAClocks clocks = configHandle().clocksSettings();
    ui->cktk_max_number->setValue(clocks.cktk_max_number);
    ui->ckbc_frequency->setCurrentIndex(clocks.ckbc_frequency);
    ui->cktp_max_number->setValue(clocks.cktp_max_number);
    ui->cktp_skew_steps->setValue(clocks.cktp_skew_steps);
    ui->cktp_period->setValue(clocks.cktp_period);
    ui->cktp_width->setValue(clocks.cktp_width);

}
// ------------------------------------------------------------------------- //
void MainWindow::writeConfigurationToFile()
{

    // open a dialog to select the output file name
    QString filename = QFileDialog::getSaveFileName(this,
        tr("Save XML Configuration File"), "../readout_configuration/configuration/",
        tr("XML Files (*.xml)"));

    if(filename.isNull()) return;

    ///////////////////////////////////////////////////////
    // grab the global register values from the ui
    ///////////////////////////////////////////////////////
    GlobalSetting global;

    global.sp               = ui->global_sp->currentIndex();
    global.sdp              = ui->global_sdp->currentIndex();
    global.sbmx             = ui->global_sbmx->currentIndex();
    global.sbft             = ui->global_sbft->currentIndex();
    global.sbfp             = ui->global_sbfp->currentIndex();
    global.sbfm             = ui->global_sbfm->currentIndex();
    global.slg              = ui->global_slg->currentIndex();
    global.sm5              = ui->global_sm5->currentIndex();
    global.scmx             = ui->global_scmx->currentIndex();
    global.sfa              = ui->global_sfa->currentIndex();
    global.sfam             = ui->global_sfam->currentIndex();
    global.st               = ui->global_st->currentIndex();
    global.sfm              = ui->global_sfm->currentIndex();
    global.sg               = ui->global_sg->currentIndex();
    global.sng              = ui->global_sng->currentIndex();
    global.stot             = ui->global_stot->currentIndex();
    global.stpp             = ui->global_stpp->currentIndex();
    global.sttt             = ui->global_sttt->currentIndex();
    global.ssh              = ui->global_ssh->currentIndex();
    global.stc              = ui->global_stc->currentIndex();
    global.sdt_dac          = ui->global_sdt_dac->value();
    global.sdp_dac          = ui->global_sdp_dac->value();
    global.sc10b            = ui->global_sc10b->currentIndex();
    global.sc8b             = ui->global_sc8b->currentIndex();
    global.sc6b             = ui->global_sc6b->currentIndex();
    global.s8b              = ui->global_s8b->currentIndex();
    global.s6b              = ui->global_s6b->currentIndex();
    global.s10b             = ui->global_s10b->currentIndex();
    global.sdcks            = ui->global_sdcks->currentIndex();
    global.sdcka            = ui->global_sdcka->currentIndex();
    global.sdck6b           = ui->global_sdck6b->currentIndex();
    global.sdrv             = ui->global_sdrv->currentIndex();
    global.slvs             = ui->global_slvs->currentIndex();
    global.stcr             = ui->global_stcr->currentIndex();
    global.ssart            = ui->global_ssart->currentIndex();
    global.s32              = ui->global_s32->currentIndex();
    global.stlc             = ui->global_stlc->currentIndex();
    global.srec             = ui->global_srec->currentIndex();
    global.sbip             = ui->global_sbip->currentIndex();
    global.srat             = ui->global_srat->currentIndex();
    global.sfrst            = ui->global_sfrst->currentIndex();
    global.slvsbc           = ui->global_slvsbc->currentIndex();
    global.slvstp           = ui->global_slvstp->currentIndex();
    global.slvstk           = ui->global_slvstk->currentIndex();
    global.slvsdt           = ui->global_slvsdt->currentIndex();
    global.slvsart          = ui->global_slvsart->currentIndex();
    global.slvstki          = ui->global_slvstki->currentIndex();
    global.slvsena          = ui->global_slvsena->currentIndex();
    global.slvs6b           = ui->global_slvs6b->currentIndex();
    global.sL0enaV          = ui->global_sL0enaV->currentIndex();
    global.slh              = ui->global_slh->currentIndex();
    global.slxh             = ui->global_slxh->currentIndex();
    global.stgc             = ui->global_stgc->currentIndex();
    global.reset            = ui->global_reset->currentIndex();
    global.sL0ena           = ui->global_sL0ena->currentIndex();
    global.l0offset         = ui->global_l0offset->value();
    global.offset           = ui->global_offset->value();
    global.rollover         = ui->global_rollover->value();
    global.window           = ui->global_window->value();
    global.truncate         = ui->global_truncate->value();
    global.nskip            = ui->global_nskip->value();
    global.sL0cktest        = ui->global_sL0cktest->currentIndex();
    global.sL0ckinv         = ui->global_sL0ckinv->currentIndex();
    global.sL0dckinv        = ui->global_sL0dckinv->currentIndex();
    global.nskipm           = ui->global_nskipm->currentIndex();



    ///////////////////////////////////////////////////////////
    // VMM mask
    ///////////////////////////////////////////////////////////
    VMMMap vmmMap;
    vmmMap.mask = 0;
    std::vector<QRadioButton*> vmm_masks;
    vmm_masks.push_back(ui->vmm_select_1);
    vmm_masks.push_back(ui->vmm_select_2);
    vmm_masks.push_back(ui->vmm_select_3);
    vmm_masks.push_back(ui->vmm_select_4);
    vmm_masks.push_back(ui->vmm_select_5);
    vmm_masks.push_back(ui->vmm_select_6);
    vmm_masks.push_back(ui->vmm_select_7);
    vmm_masks.push_back(ui->vmm_select_8);
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        if(vmm_masks.at(i)->isChecked()) {
            tmp = (tmp << i);
            vmmMap.mask |= tmp;
        }
    }

    ///////////////////////////////////////////////////////////
    // grab the channel registers
    ///////////////////////////////////////////////////////////
    vector<Channel> channels;
    for(int i = 0; i < 64; i++) {
        Channel ch;
        ch.number       = i;
        ch.sc           = VMMSCBool[i];
        ch.sl           = VMMSLBool[i];
        ch.st           = VMMSTBool[i];
        ch.sth          = VMMSTHBool[i];
        ch.sm           = VMMSMBool[i];
        ch.sd           = VMMSDValue[i];
        ch.smx          = VMMSMXBool[i];
        ch.sz10b        = VMMSZ010bValue[i];
        ch.sz8b         = VMMSZ08bValue[i];
        ch.sz6b         = VMMSZ06bValue[i];
        channels.push_back(ch);
    } // i

    ///////////////////////////////////////////////////////////
    // grab the acqusition configuration
    ///////////////////////////////////////////////////////////
    TriggerDAQ daq;

    daq.trigger_latency  = ui->trigExternalBCLatency->value();
    daq.run_mode = "pulser";
    daq.mapping_file = ui->daqSetupFilenameField->text().toStdString();
    daq.output_path     = ui->runDirectoryField->text().toStdString();
    
    ///////////////////////////////////////////////////////////
    // clocks
    ///////////////////////////////////////////////////////////
    FPGAClocks clocks_current;
    clocks_current.cktk_max_number      = ui->cktk_max_number->value();
    clocks_current.ckbc_frequency       = ui->ckbc_frequency->currentIndex();
    clocks_current.cktp_max_number      = ui->cktp_max_number->value();
    clocks_current.cktp_skew_steps      = ui->cktp_skew_steps->value();
    clocks_current.cktp_period          = ui->cktp_period->value();
    clocks_current.cktp_width           = ui->cktp_width->value();


    // load them into the config_handler
    configHandle().LoadBoardConfiguration(global, vmmMap, channels);
    configHandle().LoadFPGAClocksConfiguration(clocks_current);
    configHandle().LoadTDAQConfiguration(daq);

    // make the output XML
    configHandle().WriteConfig(filename);

}
// ------------------------------------------------------------------------- //
//void MainWindow::setAndSendEventHeaders()
//{
//    runModule().setEventHeaders(ui->evbld_infodata->currentIndex(),
//                                ui->evbld_mode->currentIndex(),
//                                ui->timeStampResCheckBox->isChecked());
//}
// ------------------------------------------------------------------------- //
void MainWindow::performHardReset()
{
    if(vmm_type()==2) {
        performHardReset_VMM2();
        return;
    }

    if(m_in_hard_reset) return;

    VMMMap vmmMap;
    vmmMap.mask = 0;
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        if(m_vmm_mask.at(i)->isChecked()) {
            tmp = (tmp << i);
            vmmMap.mask |= tmp;
        }
    }

    if(vmmMap.mask != 0) {

        m_in_hard_reset = true;

        msg()("Resetting VMM(s)");
        int initial_index = ui->global_reset->currentIndex();

        // set hard reset register high
        ui->global_reset->setCurrentIndex(1);

        // send configuration register
        prepareAndSendBoardConfig();
        //delay();
        //delay();
        prepareAndSendBoardConfig();
        //delay();
        //delay();

        // set reset register back to what it was to begin with
        ui->global_reset->setCurrentIndex(initial_index);

        m_in_hard_reset = false;
    }
    else {
        msg()("No VMM selected! Cannot perform reset.");
        m_in_hard_reset = false;
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::performHardReset_VMM2()
{
    runModule().resetASICs();
}
//// ------------------------------------------------------------------------- //
//void MainWindow::resetFEC()
//{
//    bool do_reset = (ui->fec_reset == QObject::sender() ? true : false);
//    runModule().resetFEC(do_reset);
//
//    ui->trgExternal->setChecked(false);
//    ui->trgPulser->setChecked(false);
//    ui->trgExternalBC->setChecked(false);
//    ui->setMask->setChecked(false);
//    ui->onACQ->setChecked(false);
//    ui->offACQ->setChecked(false);
//
//    SetInitialState();
//    m_commOK = true;
//    m_configOK = false;
//    m_tdaqOK = false;
//    m_runModeOK = false;
//    m_acqMode = "";
//    emit checkFSM();
//}
// ------------------------------------------------------------------------- //
/*
void MainWindow::setHDMIMask()
{
    runModule().setMask();

    if(m_hdmiMaskON) {
        ui->setMask->setDown(true);
    }
    ui->setMask->setDown(true);
    m_hdmiMaskON = true;

}
*/
// ------------------------------------------------------------------------- //
//void MainWindow::setART()
//{
//    bool enabling = true;
//    bool enableHoldOff = ui->enabled_holdoff_button->isChecked();
//    if(!(ui->enableART->isChecked())) enabling = false;
//    runModule().enableART(enabling, enableHoldOff);
//}
//// ------------------------------------------------------------------------- //
//void MainWindow::checkLinkStatus()
//{
//    runModule().checkLinkStatus();
//}
// ------------------------------------------------------------------------- //

//void MainWindow::writeFECStatus()
//{
//
//    QByteArray buff;
//    buff.clear();
//    buff.resize(socketHandle().fecSocket().pendingDatagramSize());
//    socketHandle().fecSocket().readDatagram(buff.data(), buff.size());
//    if(buff.size()==0) return;
//
//    bool ok;
//    QString sizeOfPackageReceived, datagramCheck;
//    datagramCheck = buff.mid(0,4).toHex();
//    quint32 check = datagramCheck.toUInt(&ok,16);
//
//    sizeOfPackageReceived = sizeOfPackageReceived.number(buff.size(),10);
//
//    // why do we check if it is <1000000 ?
//    if(check<1000000) {
//        stringstream ss;
//        ss << " ****** NEW PACKET RECEIVED ****** " << endl;
//        ss << " Data received size: " << sizeOfPackageReceived.toStdString()
//           << " bytes" << endl;
//        QString bin, hex;
//        for(int i = 0; i < buff.size()/4; i++) {
//            hex = buff.mid(i*4, 4).toHex();
//            quint32 tmp32 = hex.toUInt(&ok,16);
//            if(i==0) ss << " Rec'd ID: " << bin.number(tmp32,10).toStdString() << endl;
//            else {
//                ss << " Data, " << i << ": " << bin.number(tmp32,16).toStdString() << endl;
//            }
//        } // i
//        ui->debugScreen->append(QString::fromStdString(ss.str()));
//        ui->debugScreen->moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
//    }
//}
//// ------------------------------------------------------------------------- //
//void MainWindow::resetLinks()
//{
//    runModule().resetLinks();
//}
// ------------------------------------------------------------------------- //
void MainWindow::eventCountReached()
{
    ui->stopTriggerCnt->click();
}
// ------------------------------------------------------------------------- //
void MainWindow::effCountReached()
{
    return;
}
// ------------------------------------------------------------------------- //
void MainWindow::updateRunNumber(int run_number)
{
    ui->runNumber->setValue(run_number);
}
// ------------------------------------------------------------------------- //
bool MainWindow::is_calibration_setup()
{
    // calibration type toggles
    bool calibration_type_setup = (ui->calibration_type_threshold->isChecked() || ui->calibration_type_charge->isChecked());
    stringstream sx;

    // specific type toggles
    bool calibration_specific_setup = false;
    if(calibration_type_setup) {
        if(ui->calibration_type_threshold->isChecked()) {
            bool global = ui->xadccalibration_global_threshold_checkbox->isChecked();
            bool trim = ui->xadccalibration_channel_trims_checkbox->isChecked();
            bool tp_dac = ui->xadccalibration_global_testpulse_checkbox->isChecked();
            bool x_baseline = ui->xadccalibration_channel_baselines_checkbox->isChecked();
            calibration_specific_setup = (global || trim || tp_dac || x_baseline);
        }
        else if(ui->calibration_type_charge->isChecked()) {
            bool custom = ui->calibration_custom_loop_checkbox->isChecked();
            bool extrap = ui->calibration_baselines_extrapolation_checkbox->isChecked();
            bool neighbor = ui->calibration_baselines_neighbor_checkbox->isChecked();
            bool time = ui->calibration_time_checkbox->isChecked();
            bool eff_th = ui->calibration_efficiency_threshold_checkbox->isChecked();
            bool eff_ch = ui->calibration_efficiency_amplitude_checkbox->isChecked();
            bool analog_pulse = ui->calibration_scan_do_analog->isChecked();
            calibration_specific_setup = (custom || extrap || neighbor || time || eff_th || eff_ch || analog_pulse);
        }
    }
    if(!calibration_specific_setup) {
        sx.str(""); sx << "ERROR You have not selected a specific CalibrationType in the Calibration panel!";
        msg()(sx);
    }

    bool type_ok = true;
    if(ui->do_calibration_button->isChecked() && (get_calibration_type() == CalibrationType::Invalid)) {
        type_ok = false;
        sx.str(""); sx << "ERROR CalibrationType is Invalid. Calibration is not setup correctly.";
        msg()(sx);
    }
    bool has_vmm_selected = false;
    for(int i = 0; i < 8; i++) {
        if(ui->calibration_vmm_select->item(0,i)->isSelected()) { has_vmm_selected = true; break; }
    }
    if(!has_vmm_selected) {
        sx.str(""); sx << "ERROR You have not selected a VMM to calibrate in the Calibration panel!";
        msg()(sx);
    }
    calibration_specific_setup = has_vmm_selected; 

    return (calibration_type_setup && calibration_specific_setup && type_ok);
}
// ------------------------------------------------------------------------- //
bool MainWindow::load_pulser_calib()
{

    // connect to load the IPList
    ui->openConnection->click();

    //vmmDataHandler->LoadModulesForCalibration(configModule(), runModule());//, socketHandle());

    // this will be the PulserCalibDef for PulserCalib
    PulserCalibDef def;

    def.n_samples_per_channel = ui->calibration_n_samples_select->value();
    def.n_expected = ui->cktp_max_number->value();

    // get the board #'s that we expect pulser data to come from
    // during the calibration run
    int board_number_low = ui->ip4->text().toInt();
    int board_number_high = (ui->ip4->text().toInt() + (ui->numberOfFecs->text().toInt() - 1));
    stringstream bs;

    stringstream iplist;
    iplist << ui->ip1->text().toInt() << "." << ui->ip2->text().toInt() << "." << ui->ip3->text().toInt();
    
    for(int ib = board_number_low; ib <= board_number_high; ib++) {
        bs << ib << ",";
        
    } 

    string board_sel_string = ui->calibration_board_select->currentText().toStdString();
    int selection = -1;
    if(board_sel_string != "All") {
        try {
            selection = std::stoi(board_sel_string);
        }
        catch(std::exception& e) {
            cout << "MainWindow::load_pulser_calib    Could not determine board selection ID, " << board_sel_string << endl;
            selection = -1;
        }
    } 
    if(selection < 0) {
        ui->setVMMs->setCurrentIndex(ui->setVMMs->count() - 1);
    }
    else {
        ui->setVMMs->setCurrentIndex(selection-1);
    }

    string bss = bs.str();
    vector<int> test_boards;
    size_t pos = 0;
    std::string token;
    std::string delimiter(",");
    while((pos = bss.find(delimiter)) != std::string::npos) {
        token = bss.substr(0,pos);
        if(token != "") {
            int bn = 0;
            try {
                bn = std::stoi(token);
            }
            catch(std::exception& e) {
                cout << "MainWindow::load_pulser_calib    Could not get board ID from loaded IP, " << bss << endl; 
            }
            test_boards.push_back(bn);
        }
        bss.erase(0,pos+delimiter.length());
    }
    def.board_numbers.clear();
    if(selection < 0) {
        for(auto x : test_boards) def.board_numbers.push_back(x);
    }
    else {
        def.board_numbers.push_back(test_boards.at(selection-1));
    }

    bool ok;
    if(!(selection < 0)) {
        def.board_selection = (selection - 1); 
    }
    else {
        def.board_selection = -1; 
    }

    // get the VMM mask for the VMMs we want to calibrate
    int mask = 0;
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
            tmp = (tmp << i);
            mask |= tmp;
        }
    }
    def.vmm_mask = mask; 

    def.channel_low = ui->calibration_channel_start->value();
    def.channel_high = ui->calibration_channel_end->value();

    def.gain_idx_start = ui->calibration_gain_start->currentIndex();
    def.gain_idx_stop = ui->calibration_gain_end->currentIndex();

    def.peak_time_idx_start = ui->calibration_peak_time_start->currentIndex();
    def.peak_time_idx_stop = ui->calibration_peak_time_end->currentIndex();

    def.tac_idx_start = ui->calibration_tac_slope_start->currentIndex();
    def.tac_idx_stop = ui->calibration_tac_slope_end->currentIndex();

    int freq = ui->ckbc_frequency->currentText().toInt(&ok,10);
    def.tp_time_per_step = ( (freq==40) ? 1 : 6.25 ); // ns 
    def.tp_skew_start = ui->calibration_cktp_skew_start->value();
    def.tp_skew_stop = ui->calibration_cktp_skew_end->value();
    def.tp_skew_step = ui->calibration_cktp_skew_step->value(); 

    def.threshold_dac_val_start = ui->calibration_global_threshold_start->value();
    def.threshold_dac_val_stop = ui->calibration_global_threshold_end->value(); 
    def.threshold_dac_val_step = ui->calibration_global_threshold_step->value(); 

    def.pulser_dac_val_start = ui->calibration_global_testpulse_start->value();
    def.pulser_dac_val_stop = ui->calibration_global_testpulse_end->value();
    def.pulser_dac_val_step = ui->calibration_global_testpulse_step->value();

    def.analog_dac_val_start = ui->calibration_scan_analog_start->value();
    def.analog_dac_val_stop = ui->calibration_scan_analog_end->value();
    def.analog_dac_val_step = ui->calibration_scan_analog_step->value();

    // propagate the calibration type to CalibModule so that the decoding is setup properly
    // (the first 3 arguments are dummy and not used when using PulserCalib)
    //CalibrationType type = get_calibration_type();
    //dataHandle().initializeCalibration(1, 5, 0, type, false, do_l0_decoding());
    CalibrationType cal_type = get_calibration_type();
    def.type = cal_type; 

    return vmmDataHandler->load_pulser_calib(def);
}
// ------------------------------------------------------------------------- //
bool MainWindow::load_pulser_calib_properties()
{
    PulserCalibRunProperties properties;

    properties.run_number = ui->runNumber->value();
    properties.output_dir = ui->runDirectoryField->text().toStdString();
    properties.run_comment = ui->userComments->text().toStdString();
    properties.l0_decoding = do_l0_decoding();
    properties.subhysteresis = (ui->global_ssh->currentIndex() == 1);
    properties.neighbor_enable = (ui->global_sng->currentIndex() == 1);
    properties.channel_trims = getChannelTrims();

    bool ok;
    properties.cktk_max = ui->cktk_max_number->value();
    properties.ckbc_frequency = ui->ckbc_frequency->currentText().toInt(&ok,10);
    properties.cktp_period = ui->cktp_period->value();
    properties.cktp_width = ui->cktp_width->value();

    return vmmDataHandler->load_pulser_calib_run_properties(properties);
} 
// ------------------------------------------------------------------------- //
void MainWindow::stop_pulser_calib_run()
{
    ui->offACQ->click();
    ui->stopTriggerCnt->click();


}
// ------------------------------------------------------------------------- //
void MainWindow::send_initial_config_to_pulser_calib()
{
    // provide the PulserCalib tool with the configuration params at the
    // start of the calibration run -- PulserCalib will then only adjust
    // those parameters defined in the calibration loop

    GlobalSetting global;

    global.sp               = ui->global_sp->currentIndex();
    global.sdp              = ui->global_sdp->currentIndex();
    global.sbmx             = ui->global_sbmx->currentIndex();
    global.sbft             = ui->global_sbft->currentIndex();
    global.sbfp             = ui->global_sbfp->currentIndex();
    global.sbfm             = ui->global_sbfm->currentIndex();
    global.slg              = ui->global_slg->currentIndex();
    global.sm5              = ui->global_sm5->currentIndex();
    global.scmx             = ui->global_scmx->currentIndex();
    global.sfa              = ui->global_sfa->currentIndex();
    global.sfam             = ui->global_sfam->currentIndex();
    global.st               = ui->global_st->currentIndex();
    global.sfm              = ui->global_sfm->currentIndex();
    global.sg               = ui->global_sg->currentIndex();
    global.sng              = ui->global_sng->currentIndex();
    global.stot             = ui->global_stot->currentIndex();
    global.stpp             = ui->global_stpp->currentIndex();
    global.sttt             = ui->global_sttt->currentIndex();
    global.ssh              = ui->global_ssh->currentIndex();
    global.stc              = ui->global_stc->currentIndex();
    global.sdt_dac          = ui->global_sdt_dac->value();
    global.sdp_dac          = ui->global_sdp_dac->value();
    global.sc10b            = ui->global_sc10b->currentIndex();
    global.sc8b             = ui->global_sc8b->currentIndex();
    global.sc6b             = ui->global_sc6b->currentIndex();
    global.s8b              = ui->global_s8b->currentIndex();
    global.s6b              = ui->global_s6b->currentIndex();
    global.s10b             = ui->global_s10b->currentIndex();
    global.sdcks            = ui->global_sdcks->currentIndex();
    global.sdcka            = ui->global_sdcka->currentIndex();
    global.sdck6b           = ui->global_sdck6b->currentIndex();
    global.sdrv             = ui->global_sdrv->currentIndex();
    global.slvs             = ui->global_slvs->currentIndex();
    global.stcr             = ui->global_stcr->currentIndex();
    global.ssart            = ui->global_ssart->currentIndex();
    global.s32              = ui->global_s32->currentIndex();
    global.stlc             = ui->global_stlc->currentIndex();
    global.srec             = ui->global_srec->currentIndex();
    global.sbip             = ui->global_sbip->currentIndex();
    global.srat             = ui->global_srat->currentIndex();
    global.sfrst            = ui->global_sfrst->currentIndex();
    global.slvsbc           = ui->global_slvsbc->currentIndex();
    global.slvstp           = ui->global_slvstp->currentIndex();
    global.slvstk           = ui->global_slvstk->currentIndex();
    global.slvsdt           = ui->global_slvsdt->currentIndex();
    global.slvsart          = ui->global_slvsart->currentIndex();
    global.slvstki          = ui->global_slvstki->currentIndex();
    global.slvsena          = ui->global_slvsena->currentIndex();
    global.slvs6b           = ui->global_slvs6b->currentIndex();
    global.sL0enaV          = ui->global_sL0enaV->currentIndex();
    global.slh              = ui->global_slh->currentIndex();
    global.slxh             = ui->global_slxh->currentIndex();
    global.stgc             = ui->global_stgc->currentIndex();
    global.reset            = ui->global_reset->currentIndex();
    global.sL0ena           = ui->global_sL0ena->currentIndex();
    global.l0offset         = ui->global_l0offset->value();
    global.offset           = ui->global_offset->value();
    global.rollover         = ui->global_rollover->value();
    global.window           = ui->global_window->value();
    global.truncate         = ui->global_truncate->value();
    global.nskip            = ui->global_nskip->value();
    global.sL0cktest        = ui->global_sL0cktest->currentIndex();
    global.sL0ckinv         = ui->global_sL0ckinv->currentIndex();
    global.sL0dckinv        = ui->global_sL0dckinv->currentIndex();
    global.nskipm           = ui->global_nskipm->currentIndex();


    ///////////////////////////////////////////////////////////
    // VMM mask
    ///////////////////////////////////////////////////////////
    VMMMap vmmMap;
    vmmMap.mask = 0;
    std::vector<QRadioButton*> vmm_masks;
    vmm_masks.push_back(ui->vmm_select_1);
    vmm_masks.push_back(ui->vmm_select_2);
    vmm_masks.push_back(ui->vmm_select_3);
    vmm_masks.push_back(ui->vmm_select_4);
    vmm_masks.push_back(ui->vmm_select_5);
    vmm_masks.push_back(ui->vmm_select_6);
    vmm_masks.push_back(ui->vmm_select_7);
    vmm_masks.push_back(ui->vmm_select_8);
    for(int i = 0; i < 8; i++) {
        int tmp = 1;
        if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
            tmp = (tmp << i);
            vmmMap.mask |= tmp;
        }
    }

    ///////////////////////////////////////////////////////////
    // grab the channel registers
    ///////////////////////////////////////////////////////////
    vector<Channel> channels;
    for(int i = 0; i < 64; i++) {
        Channel ch;
        ch.number       = i;
        ch.sc           = VMMSCBool[i];
        ch.sl           = VMMSLBool[i];
        ch.st           = VMMSTBool[i];
        ch.sth          = VMMSTHBool[i];
        ch.sm           = VMMSMBool[i];
        ch.sd           = VMMSDValue[i];
        ch.smx          = VMMSMXBool[i];
        ch.sz10b        = VMMSZ010bValue[i];
        ch.sz8b         = VMMSZ08bValue[i];
        ch.sz6b         = VMMSZ06bValue[i];
        channels.push_back(ch);
    } // i

    ///////////////////////////////////////////////////////////
    // grab the acqusition configuration
    ///////////////////////////////////////////////////////////
    TriggerDAQ daq;

    daq.trigger_latency  = ui->trigExternalBCLatency->value();
    daq.run_mode = "pulser";
    daq.mapping_file = ui->daqSetupFilenameField->text().toStdString();
    daq.output_path     = ui->runDirectoryField->text().toStdString();
    
    ///////////////////////////////////////////////////////////
    // clocks
    ///////////////////////////////////////////////////////////
    FPGAClocks clocks_current;
    clocks_current.cktk_max_number      = ui->cktk_max_number->value();
    clocks_current.ckbc_frequency       = ui->ckbc_frequency->currentIndex();
    clocks_current.cktp_max_number      = ui->cktp_max_number->value();
    clocks_current.cktp_skew_steps      = ui->cktp_skew_steps->value();
    clocks_current.cktp_period          = ui->cktp_period->value();
    clocks_current.cktp_width           = ui->cktp_width->value();

    vmmDataHandler->send_initial_config_to_pulser_calib(global, vmmMap, channels,
                                                                clocks_current, daq);

}
// ------------------------------------------------------------------------- //
int MainWindow::get_first_selected_vmm()
{
    int first_selected = -1;
    for(int i = 0; i < 8; i++) {
        //if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
        if(m_vmm_mask.at(i)->isChecked()) {
            first_selected = i;
            break;
        }
    } 
    return first_selected;
}
// ------------------------------------------------------------------------- //
vector<int> MainWindow::get_selected_vmms_for_calibration()
{
    vector<int> vmms;
    for(int i = 0; i < 8; i++) {
        if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
            vmms.push_back(i);
        }
    } 

    return vmms;
}
// ------------------------------------------------------------------------- //
CalibrationType MainWindow::get_calibration_type()
{
    CalibrationType type = CalibrationType::Invalid;

    if(ui->do_calibration_button->isChecked()) {

        // threshold
        if(ui->calibration_type_threshold->isChecked()) {
            if(ui->xadccalibration_global_threshold_checkbox->isChecked()) {
                type = CalibrationType::GlobalThresholdDAC;
            }
            else if(ui->xadccalibration_channel_trims_checkbox->isChecked()) {
                type = CalibrationType::ChannelTrim;
            }
            else if(ui->xadccalibration_global_testpulse_checkbox->isChecked()) {
                type = CalibrationType::GlobalTestPulseDAC;
            }
            else if(ui->xadccalibration_channel_baselines_checkbox->isChecked()) {
                type = CalibrationType::ChannelBaseline;
            }
        }

        // charge
        if(ui->calibration_type_charge->isChecked()) {
            if(ui->calibration_custom_loop_checkbox->isChecked()) {
                type = CalibrationType::Custom;
            }
            else if(ui->calibration_baselines_extrapolation_checkbox->isChecked()) {
                type = CalibrationType::BaselineExtrapolation;
            }
            else if(ui->calibration_baselines_neighbor_checkbox->isChecked()) {
                type = CalibrationType::BaselineNeighbor;
            }
            else if(ui->calibration_time_checkbox->isChecked()) {
                type = CalibrationType::Time;
           }
            else if(ui->calibration_efficiency_threshold_checkbox->isChecked()
            || ui->calibration_efficiency_amplitude_checkbox->isChecked()) {
                type = CalibrationType::Efficiency;
            }
            else if(ui->calibration_scan_do_analog->isChecked()) {
                type = CalibrationType::AnalogPulse;
            }
        }
    }

    return type;
}
// ------------------------------------------------------------------------- //
void MainWindow::activateRC()
{
    ui->stopTriggerCnt->setEnabled(true);
}
// ------------------------------------------------------------------------- //
bool MainWindow::do_l0_decoding()
{
    return (ui->global_sL0ena->currentIndex() == 1);
}
// ------------------------------------------------------------------------- //
void MainWindow::startRun()
{
    if(ui->monitoring_enable_button->isChecked() && m_mapping_loaded_ok) {
        // send the mon configuration
        // this button is off-screen
        //ui->monitoring_send_config_button->click();
        //emit send_monitor_config();
        dataHandle().sendMonitoringConfiguration();
        boost::this_thread::sleep(boost::posix_time::milliseconds(100));
    }

    // first check that things make sense
    bool do_calibration_run = ui->do_calibration_button->isChecked();
    if(do_calibration_run) {
        bool calibration_setup = is_calibration_setup();
        if(!calibration_setup) {
            stringstream sx;
            sx << "ERROR Run will not be started. You have asked for a calibration run"
                << " but have not correctly set the calibration in the Calibraton panel.";

            msg()(sx);
            return;
        }
    }

    if(do_calibration_run && ui->calibration_type_charge->isChecked()) {
        // let the PulserCalib start the acq itself
        ui->offACQ->click();
	if(ui->calibration_scan_do_analog->isChecked()) {
		ui->trgExternal->click();
		ui->send_tdaq_settings_button->click();
	}
        if(!load_pulser_calib()) {
            stringstream sx;
            sx << "ERROR Pulser-based calibration loop is not configured properly, check the Calibration panel";
            msg()(sx);
            return;
        }
        send_initial_config_to_pulser_calib();
        delay();
        m_have_loaded_pulser = true;
    }

    int count_to_stop_at = ui->eventCountStop->text().toInt();
    //bool is_L0 = (ui->global_sL0ena->currentIndex() == 1);
    bool is_L0 = do_l0_decoding();
    bool is_xadc = (do_calibration_run && ui->calibration_type_threshold->isChecked());
    bool ok_to_start = dataHandle().initializeRun(ui->writeRaw->isChecked(), ui->writeData->isChecked(),
                            ui->runDirectoryField->text().toStdString(),
                            ui->runNumber->value(),
                            count_to_stop_at,
                            do_calibration_run,
                            is_L0,
                            is_xadc);
    if(do_calibration_run && ui->calibration_type_charge->isChecked()) {
        if(!load_pulser_calib_properties()) {
            stringstream sx;
            sx << "ERROR Pulser-based calibration loop is not configured properly, check the Calibration panel";
            msg()(sx);
            return;
        }
    }

    int cktp_limit = ui->cktp_max_number->text().toInt();
    // set the initial CKTP max for Efficiency runs here... could be checked otherwise but
    // works.
    if(do_calibration_run && get_calibration_type()==CalibrationType::Efficiency) {
        cktp_limit = ui->calibration_n_samples_select->value()+1;
        ui->cktp_max_number->setValue(cktp_limit);
    }
    dataHandle().set_cktp_limit(cktp_limit);

    if(ok_to_start && do_calibration_run && !ui->calibration_type_charge->isChecked()) {
        updateCalibrationState(true);
    }

    if(!ok_to_start) {
        msg()("Unable to start the run");
        return;
    }

    //// right now, don't allow user to turn on/off monitoring in the middle of a run
    //// simply have not tested that this is OK to do
    //ui->monitoring_enable_button->setEnabled(false);


    // don't allow user to try to disable mapping in the middle of a run (this will avoid confusion down the line)
    setDaqConfigurationEnabled(false);


    //if(ui->doCalib->isChecked() && (ui->doPDOCalib->isChecked() || ui->doTDOCalib->isChecked()))
    //    dataHandle().setCalibrationRun(true);
    //else { dataHandle().setCalibrationRun(false); }

    //////////////////////////////////////////////////////
    // fill the run properties
    //////////////////////////////////////////////////////
    //double gain = GlobalSetting::all_gains[configHandle().globalSettings().sg].toDouble();
    int gain = configHandle().globalSettings().sg; // store index of gain!
    int tac_slope = configHandle().globalSettings().stc;
    int peak_time = configHandle().globalSettings().st; 
    int dac_threshold = configHandle().globalSettings().sdt_dac;
    int dac_amplitude = configHandle().globalSettings().sdp_dac;
    int angle = ui->angle->value();
    int tp_skew_steps = ui->cktp_skew_steps->value();
    int ckbc_freq = 40;
    string ckbc_freq_str = ui->ckbc_frequency->currentText().toStdString();
    try {
        ckbc_freq = std::stoi(ckbc_freq_str);
    }
    catch(std::exception& e) {
        cout << "MainWindow    Unable to convert CKBC frequency to int for filling run properties "
            << "will store as -1" << endl;
        ckbc_freq = -1;
    }
    //double tp_skew = 0.0;
    //if(ui->ckbc_frequency->currentIndex()==0) {
    //    tp_skew = tp_skew_steps * 2; // ns
    //}
    //else {
    //    tp_skew = tp_skew_steps * 6.25; // ns
    //}

    if(!do_calibration_run)
        dataHandle().fillRunProperties(gain, tac_slope, peak_time, dac_threshold,
                            dac_amplitude, angle, tp_skew_steps, ckbc_freq);

    //////////////////////////////////////////////////////
    // fill the user comments for this run
    //////////////////////////////////////////////////////
    string text_for_info;
    if( (ui->userComments->text()!="" || do_calibration_run) && ui->userComments->isEnabled()) {

        // fill the output text
        string user_comments = ui->userComments->text().toStdString();
        text_for_info = (user_comments == "") ? "" : user_comments;
        if(do_calibration_run) {
            bool do_channel_threshold_run = ui->xadccalibration_channel_trims_checkbox->isChecked();
            bool do_global_threshold_run = ui->xadccalibration_global_threshold_checkbox->isChecked();
            string text_tmp = "Calibration run ";
                size_t has_string = text_for_info.find(text_tmp);
            if(has_string == string::npos) {
                size_t has_channel = text_for_info.find("channel threshold");
                size_t has_global = text_for_info.find("global threshold");
                if(do_channel_threshold_run && (has_channel==string::npos))
                        text_tmp = text_tmp + "(channel threshold scan)";
                else if(do_global_threshold_run && (has_global==string::npos))
                        text_tmp = text_tmp + "(global threshold scan)";
                if(!(text_for_info=="")) text_tmp = text_tmp + " : ";
                text_tmp = text_tmp + text_for_info;
                text_for_info = text_tmp;
            }
        }
        dataHandle().fillRunComment(text_for_info);

        // add run specification in GUI
        //QString new_text = ui->userComments->text();
        QString new_text = QString::fromStdString(text_for_info);
        QString run_number = QString::number(ui->runNumber->value());
        new_text = "Run " + run_number + " comment: " + new_text;
        ui->userComments->setText(new_text);

        // disable to prevent (by default) same comment used in multiple runs
        ui->userComments->setEnabled(false);
    }
    else {
        dataHandle().fillRunComment("None");
        text_for_info = "None";
    }

    if(ui->writeData->isChecked()) {
        stringstream sx;
        sx << " ------------------------------------\n";
        sx << " *** Starting "
            << (do_calibration_run ? "CALIBRATION run " : "DAQ run " )<< ui->runNumber->value() << " *** \n";
        if(do_calibration_run)
            sx << "   Calibration type: " << CalibrationType2Str(get_calibration_type()) << "\n"; 
        sx << " ------------------------------------\n";
        msg()(sx); sx.str("");
        sx << "  User comments: " << text_for_info << "\n";
        sx << " - - - - - - - - - - - - - - - - - - - -";
        msg()(sx); sx.str("");
    }


    // reset the event before starting
    dataHandle().resetCounts();
    ui->triggerCntLabel->setText(QString::number(dataHandle().getCounts(), 10));

    m_daqInProgress = true;
    dataHandle().startGathering();
    if(do_calibration_run && ui->calibration_type_charge->isChecked()) {
        //cout << "MainWindow   begin pulser calibration called from thread " << boost::this_thread::get_id() << endl;
        if(!dataHandle().begin_pulser_calibration()) {
            stringstream sx;
            sx << "ERROR Unable to start pulser-based calibration, run will not begin";
            msg()(sx); sx.str("");
            m_daqInProgress = false;
            return; 
        }
    }    

    // don't allow switching between VMM types in the middle of running
    ui->set_vmm2_button->setEnabled(false);
    ui->set_vmm3_button->setEnabled(false);
    ui->set_l0_button->setEnabled(false);

    m_start_run_time = QTime::currentTime();
    if(ui->dataflow_enable_button->isChecked()) {
        //m_dataflow_window->close();
        //ui->dataflow_enable_button->click();
        //ui->dataflow_enable_button->click();
        m_dataflow_window->start_dataflow_monitor(ui->runNumber->value(), m_start_run_time);
        vmmDataHandler->toggle_occupancy_check(true);
        //m_dataflow_window->start_dataflow_monitor(ui->runNumber->value(), QTime(QTime::currentTime()));
        m_dataflow_window->initialize_occupancy_plots(ui->ip4->text().toInt(), ui->numberOfFecs->value());
        m_dataflow_window->initialize_trig_delta_plots(ui->ip4->text().toInt(), ui->numberOfFecs->value());
    }

    if(ui->writeData->isChecked()) {
        ui->runStatusField->setText("Run:"+ui->runNumber->text()+" ongoing");
    }
    else {
        ui->runStatusField->setText("Data Incoming");
    }
    ui->runStatusField->setStyleSheet("background-color:green");
    ui->checkTriggers->setEnabled(false);
    ui->stopTriggerCnt->setEnabled(true);

    //if(count_to_stop_at > 10000)
    //    m_timer->setInterval(200);
    //else if(count_to_stop_at < 2000) {
    //    ui->stopTriggerCnt->setEnabled(true);
    //} 
    //else
    //    m_timer->setInterval(1);
    //m_timer->start();

    ui->runNumber->setEnabled(false);

}
// ------------------------------------------------------------------------- //
void MainWindow::stopRun()
{
    stringstream sx;

    if(!m_daqInProgress) return;
    ui->clearTriggerCnt->setEnabled(true);
    if(ui->writeData->isChecked()) {
        sx << " ---------------------------- \n";
        sx << " *** Ending run " << ui->runNumber->value() << " ***\n";
        sx << " ---------------------------- ";
        msg()(sx); sx.str("");
    }
    ui->eventCountStop->setEnabled(true);

    // stop the worker threads and stop the io_service
    dataHandle().endRun(true);

    //dataflow
    if(ui->dataflow_enable_button->isChecked()) {
        vmmDataHandler->toggle_occupancy_check(false);
        m_dataflow_window->stop_dataflow_monitor();
    }

    m_daqInProgress = false;

    if(ui->writeData->isChecked()) {
        ui->runStatusField->setText("Run:"+ui->runNumber->text()+" finished");
    }
    else {
        ui->runStatusField->setText("All Quiet");
    }
    ui->runStatusField->setStyleSheet("background-color: lightGray");

    if(ui->writeData->isChecked())
        ui->runNumber->setValue(ui->runNumber->value()+1);
    ui->checkTriggers->setEnabled(true);
    ui->stopTriggerCnt->setEnabled(false);
    ui->runNumber->setEnabled(true);

    if(m_mapping_loaded_ok)
        ui->monitoring_enable_button->setEnabled(true);

    // run over, allow user to load/disable mapping at their will
    setDaqConfigurationEnabled(true);

    // open up VMM type switching
    ui->set_vmm2_button->setEnabled(true);
    ui->set_vmm3_button->setEnabled(true);
    ui->set_l0_button->setEnabled(true);
}

// ------------------------------------------------------------------------- //
void MainWindow::setDaqConfigurationEnabled(bool enable_it)
{
    // don't edit the name/path of the config file in the middle of a run -- that is confusion
    ui->daqSetupFilenameField->setReadOnly(!enable_it);

    // don't allow browsing for new file in the middle of a run -- this will attempt to over-write the text edit field
    ui->selectDir_daqSetup->setEnabled(enable_it);

    // don't allow loading of new file (re-building the mapping) in the middle of a run
    ui->load_daqSetup->setEnabled(enable_it);

    // don't allow disabling of mapping in the middle of the run (for simplicity/confusion avoidance)
    ui->disable_mapping->setEnabled(enable_it);
}
// ------------------------------------------------------------------------- //
void MainWindow::triggerHandler()

{
    if(QObject::sender() == ui->checkTriggers)
        startRun();

    if(QObject::sender() == ui->clearTriggerCnt) {
        dataHandle().resetCounts();
        delay();
        delay();
        ui->triggerCntLabel->setText(QString::number(dataHandle().getCounts(), 10));
    }

    if(QObject::sender() == ui->stopTriggerCnt)
        stopRun();
}
// ------------------------------------------------------------------------- //
// ------------------------------------------------------------------------- //
void MainWindow::calibrationLoopState(bool calib_on)
{
    m_inCalibrationLoop = calib_on;
}
// ------------------------------------------------------------------------- //
void MainWindow::updateTriggerCount(int count, int trigger_count,
        unsigned int input_packet_count, unsigned int input_empty_packet_count, bool fine)
{
    //QString cnt_;
    //int count = dataHandle().getCounts();

    // as the DAQ is on a different thread, the 
    // call here can be behind the DAQ count at the
    // time at which the counter was incremented
    // just round it down to the nearest mult. of 100
    // (be sure to adjust this if you begin changing
    // the increment of the counter update)

    if(fine) {
        ui->triggerCntLabel->setText(QString::number(count,10));
        ui->triggerCntLabel_fpga->setText(QString::number(trigger_count, 10));
    }
    else if ( ui->do_calibration_button->isChecked() ) {
        count = (count + 500) / 1000 * 1000;
        ui->triggerCntLabel->setText(QString::number(count,10));
        ui->triggerCntLabel_fpga->setText(QString::number(trigger_count, 10));
    }
    else {
        count = (count + 50) / 100 * 100;
        ui->triggerCntLabel->setText(QString::number(count,10));
        ui->triggerCntLabel_fpga->setText(QString::number(trigger_count, 10));
    }

    if(ui->dataflow_enable_button->isChecked()) {
        m_dataflow_window->update_counters(count, trigger_count,
                                input_packet_count, input_empty_packet_count);
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::set_calibration_loop_params()
{

    ///////////////////////////////////////////////////////////
    // xADC calibration
    ///////////////////////////////////////////////////////////
    if(QObject::sender() == ui->xadccalibration_global_threshold_checkbox) {
        ui->xadccalibration_channel_trim_end->setEnabled(false);
        ui->xadccalibration_global_threshold_end->setEnabled(true);
        ui->xadccalibration_global_testpulse_end->setEnabled(false);

        // set gui testpulse
        ui->xadccalibration_global_testpulse_start->setValue(ui->global_sdp_dac->value());
        ui->xadccalibration_global_testpulse_end->setValue(ui->global_sdp_dac->value());
    }
    else if(QObject::sender() == ui->xadccalibration_global_testpulse_checkbox) {
        ui->xadccalibration_channel_trim_end->setEnabled(false);
        ui->xadccalibration_global_threshold_end->setEnabled(false);
        ui->xadccalibration_global_testpulse_end->setEnabled(true);

        // set gui threshold
        ui->xadccalibration_global_threshold_start->setValue(ui->global_sdt_dac->value());
        ui->xadccalibration_global_threshold_end->setValue(ui->global_sdt_dac->value());
    }
    else if(QObject::sender() == ui->xadccalibration_channel_trims_checkbox) {
        ui->xadccalibration_channel_trim_end->setEnabled(true);
        ui->xadccalibration_global_threshold_end->setEnabled(false);
        ui->xadccalibration_global_testpulse_end->setEnabled(false);

        // set gui threshold and testpulse
        ui->xadccalibration_global_testpulse_start->setValue(ui->global_sdp_dac->value());
        ui->xadccalibration_global_testpulse_end->setValue(ui->global_sdp_dac->value());
        ui->xadccalibration_global_threshold_start->setValue(ui->global_sdt_dac->value());
        ui->xadccalibration_global_threshold_end->setValue(ui->global_sdt_dac->value());

    }
    ///////////////////////////////////////////////////////////
    // non-xADC calibration
    ///////////////////////////////////////////////////////////
    else if(QObject::sender() == ui->calibration_custom_loop_checkbox) {
        ui->calibration_gain_end->setEnabled(true);
        ui->calibration_peak_time_end->setEnabled(true); 
        ui->calibration_tac_slope_end->setEnabled(true);
        ui->calibration_global_threshold_end->setEnabled(true);
        ui->calibration_global_testpulse_end->setEnabled(true);

        ui->calibration_cktp_skew_start->setEnabled(true);
        ui->calibration_cktp_skew_end->setEnabled(true);

        // set for infinite # for data collecting in loop
        ui->cktp_max_number->setValue(-1);

        ui->calibration_cktp_skew_start->setValue(5);
        ui->calibration_cktp_skew_end->setValue(5);

    }
    else if(QObject::sender() == ui->calibration_baselines_extrapolation_checkbox) {
        ui->calibration_gain_end->setEnabled(false);
        ui->calibration_peak_time_end->setEnabled(false); 
        ui->calibration_tac_slope_end->setEnabled(false);
        ui->calibration_global_threshold_end->setEnabled(false);
        ui->calibration_global_testpulse_end->setEnabled(true);

        // don't allow time loop for this
        ui->calibration_cktp_skew_start->setValue(ui->cktp_skew_steps->value());
        ui->calibration_cktp_skew_start->setEnabled(false);
        ui->calibration_cktp_skew_end->setValue(ui->cktp_skew_steps->value());
        ui->calibration_cktp_skew_end->setEnabled(false);

        // set for infinite # for data collecting in loop
        ui->cktp_max_number->setValue(-1);

        ui->calibration_cktp_skew_start->setValue(5);
        ui->calibration_cktp_skew_end->setValue(5);

        // set gui gain
        ui->calibration_gain_start->setCurrentIndex(ui->global_sg->currentIndex());
        ui->calibration_gain_end->setCurrentIndex(ui->global_sg->currentIndex());
        // set gui peak time
        ui->calibration_peak_time_start->setCurrentIndex(ui->global_st->currentIndex());
        ui->calibration_peak_time_end->setCurrentIndex(ui->global_st->currentIndex());
        // set gui TAC slope
        ui->calibration_tac_slope_start->setCurrentIndex(ui->global_stc->currentIndex());
        ui->calibration_tac_slope_end->setCurrentIndex(ui->global_stc->currentIndex());
        // set gui threshold
        ui->calibration_global_threshold_start->setValue(ui->global_sdt_dac->value());
        ui->calibration_global_threshold_end->setValue(ui->global_sdt_dac->value());
    }
    else if(QObject::sender() == ui->calibration_baselines_neighbor_checkbox) {
        ui->calibration_gain_end->setEnabled(false);
        ui->calibration_peak_time_end->setEnabled(false); 
        ui->calibration_tac_slope_end->setEnabled(false);
        ui->calibration_global_threshold_end->setEnabled(false);
        ui->calibration_global_testpulse_end->setEnabled(false);

        // don't allow time loop for this
        ui->calibration_cktp_skew_start->setValue(ui->cktp_skew_steps->value());
        ui->calibration_cktp_skew_start->setEnabled(false);
        ui->calibration_cktp_skew_end->setValue(ui->cktp_skew_steps->value());
        ui->calibration_cktp_skew_end->setEnabled(false);
        
        // set for infinite # for data collecting in loop
        ui->cktp_max_number->setValue(-1);

        ui->calibration_cktp_skew_start->setValue(5);
        ui->calibration_cktp_skew_end->setValue(5);

        // set gui gain
        ui->calibration_gain_start->setCurrentIndex(ui->global_sg->currentIndex());
        ui->calibration_gain_end->setCurrentIndex(ui->global_sg->currentIndex());
        // set gui peak time
        ui->calibration_peak_time_start->setCurrentIndex(ui->global_st->currentIndex());
        ui->calibration_peak_time_end->setCurrentIndex(ui->global_st->currentIndex());
        // set gui TAC slope
        ui->calibration_tac_slope_start->setCurrentIndex(ui->global_stc->currentIndex());
        ui->calibration_tac_slope_end->setCurrentIndex(ui->global_stc->currentIndex());
        // set gui threshold
        ui->calibration_global_threshold_start->setValue(ui->global_sdt_dac->value());
        ui->calibration_global_threshold_end->setValue(ui->global_sdt_dac->value());
        // set gui testpulse
        ui->calibration_global_testpulse_start->setValue(ui->global_sdp_dac->value());
        ui->calibration_global_testpulse_end->setValue(ui->global_sdp_dac->value());
    }
    else if(QObject::sender() == ui->calibration_time_checkbox) {
        ui->calibration_gain_end->setEnabled(false);
        ui->calibration_peak_time_end->setEnabled(false); 
        ui->calibration_tac_slope_end->setEnabled(false);
        ui->calibration_global_threshold_end->setEnabled(false);
        ui->calibration_global_testpulse_end->setEnabled(false);

        ui->calibration_cktp_skew_start->setEnabled(true);
        ui->calibration_cktp_skew_end->setEnabled(true);
        ui->calibration_cktp_skew_end->setValue(10);

        // set for infinite # for data collecting in loop
        ui->cktp_max_number->setValue(-1);
        
        // set gui gain
        ui->calibration_gain_start->setCurrentIndex(ui->global_sg->currentIndex());
        ui->calibration_gain_end->setCurrentIndex(ui->global_sg->currentIndex());
        // set gui peak time
        ui->calibration_peak_time_start->setCurrentIndex(ui->global_st->currentIndex());
        ui->calibration_peak_time_end->setCurrentIndex(ui->global_st->currentIndex());
        // set gui TAC slope
        ui->calibration_tac_slope_start->setCurrentIndex(ui->global_stc->currentIndex());
        ui->calibration_tac_slope_end->setCurrentIndex(ui->global_stc->currentIndex());
        // set gui threshold
        ui->calibration_global_threshold_start->setValue(ui->global_sdt_dac->value());
        ui->calibration_global_threshold_end->setValue(ui->global_sdt_dac->value());
        // set gui testpulse
        ui->calibration_global_testpulse_start->setValue(ui->global_sdp_dac->value());
        ui->calibration_global_testpulse_end->setValue(ui->global_sdp_dac->value());
    }
    else if(QObject::sender() == ui->calibration_efficiency_threshold_checkbox
            || QObject::sender() == ui->calibration_efficiency_amplitude_checkbox) {

        bool is_thresh = (ui->calibration_efficiency_threshold_checkbox == QObject::sender());

        ui->calibration_cktp_skew_start->setValue(5);
        ui->calibration_cktp_skew_end->setValue(5);

        ui->calibration_gain_end->setCurrentIndex(ui->calibration_gain_start->currentIndex());
        ui->calibration_peak_time_end->setCurrentIndex(ui->calibration_peak_time_start->currentIndex());
        ui->calibration_tac_slope_end->setCurrentIndex(ui->calibration_tac_slope_start->currentIndex());
        ui->calibration_cktp_skew_end->setValue(ui->calibration_cktp_skew_start->value());
        
        ui->calibration_gain_end->setEnabled(false);
        ui->calibration_peak_time_end->setEnabled(false);
        ui->calibration_tac_slope_end->setEnabled(false);
        ui->calibration_cktp_skew_end->setEnabled(false);

        if(is_thresh) {
            ui->calibration_global_testpulse_end->setValue(ui->calibration_global_testpulse_start->value());
            ui->calibration_global_testpulse_end->setEnabled(false);
            ui->calibration_global_threshold_end->setEnabled(true);
            ui->calibration_global_threshold_start->setEnabled(true);
        } 
        else {
            ui->calibration_global_threshold_end->setValue(ui->calibration_global_threshold_start->value());
            ui->calibration_global_threshold_end->setEnabled(false);
            ui->calibration_global_testpulse_end->setEnabled(true);
            ui->calibration_global_testpulse_start->setEnabled(true);
        }
    }


}
// ------------------------------------------------------------------------- //
void MainWindow::set_calibration()
{
    if(QObject::sender() == ui->do_calibration_button) {
        if(ui->do_calibration_button->isChecked()) {
            ui->do_calibration_button->setStyleSheet("QPushButton {background-color: rgb(15,147,255);}");
        }
        else if(!ui->do_calibration_button->isChecked()) {
            ui->do_calibration_button->setStyleSheet("QPushButton {background-color: white;}");
        }
    }
    else if(QObject::sender() == ui->calibration_type_threshold) {
        ui->calibration_type_charge->setChecked(false);
        ui->calibration_charge_box->setEnabled(false);

        if(ui->calibration_type_threshold->isChecked()) {

            ui->calibration_charge_box->setEnabled(false);

            ui->xadccalibration_global_threshold_checkbox->setChecked(false);
            ui->xadccalibration_channel_trims_checkbox->setChecked(false);

            //ui->xadccalibration_global_threshold_checkbox->setAutoExclusive(true);
            //ui->xadccalibration_channel_trims_checkbox->setAutoExclusive(true);

            ui->calibration_threshold_box->setEnabled(true);


            ui->calibration_n_samples_select->setMaximum(2000);
        }
        else if(!ui->calibration_type_threshold->isChecked()) {
            ui->calibration_threshold_box->setEnabled(false);
            ui->calibration_charge_box->setEnabled(false);

            //ui->xadccalibration_global_threshold_checkbox->setAutoExclusive(false);
            //ui->xadccalibration_channel_trims_checkbox->setAutoExclusive(false);

            ui->xadccalibration_global_threshold_checkbox->setChecked(false);
            ui->xadccalibration_channel_trims_checkbox->setChecked(false);
            ui->calibration_n_samples_select->setMaximum(1000000);
        }
    }
    else if(QObject::sender() == ui->calibration_type_charge) {
        ui->calibration_type_threshold->setChecked(false);
        ui->calibration_threshold_box->setEnabled(false);

        //ui->xadccalibration_global_threshold_checkbox->setAutoExclusive(false);
        //ui->xadccalibration_channel_trims_checkbox->setAutoExclusive(false);

        //ui->xadccalibration_global_threshold_checkbox->setChecked(false);
        //ui->xadccalibration_channel_trims_checkbox->setChecked(false);

        if(ui->calibration_type_charge->isChecked()) {
            ui->calibration_threshold_box->setEnabled(false);
            ui->calibration_charge_box->setEnabled(true);
            //ui->calibration_vmm_select->setEnabled(true);
            ui->calibration_n_samples_select->setMaximum(1000000);
        }
        else if(!ui->calibration_type_charge->isChecked()) {
            ui->calibration_threshold_box->setEnabled(false);
            ui->calibration_charge_box->setEnabled(false);
            //ui->calibration_vmm_select->setEnabled(false);
            ui->calibration_n_samples_select->setMaximum(1000000);
        }
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::updateCalibrationState(bool is_start_of_run)
{

    //dataHandle().resetCounts();
    //boost::this_thread::sleep(boost::posix_time::milliseconds(1000));
    //if(is_start_of_run) {
    //    m_calib_time = QTime::currentTime();
    //    m_last_calib_time_check = 0.0;
    //}
    //else {
    //    double elapsed_time = m_calib_time.elapsed()/1000.; // in seconds
    //    double since_last = (elapsed_time - m_last_calib_time_check);
    //    cout << "elapsed_time = " << elapsed_time <<  "   since last: " << since_last << endl;;
    //    if( (elapsed_time - m_last_calib_time_check) < 0.2 ) {
    //        updateCalibrationState(false);
    //    }
    //    else {
    //        m_last_calib_time_check = elapsed_time;
    //    }
    //}

    CalibrationType type = get_calibration_type();

    vector<int> selected_vmms = get_selected_vmms_for_calibration();
    map<int, int> vmm_idx_map;
    for(int idx = 0; idx < (int)selected_vmms.size(); idx++) {
        vmm_idx_map[idx] = selected_vmms.at(idx);
    }
    //cout << "MainWindow::updateCalibrationState    "
    //        << "Selected VMMs: ";
    //for(auto idx : vmm_idx_map) {
    //    cout << " " << idx.first << ":" << idx.second;
    //}
    //cout << endl;

    if(type==CalibrationType::GlobalThresholdDAC || type==CalibrationType::GlobalTestPulseDAC) {

        ui->cktp_max_number->setValue(-1);

        bool is_threshold = (type==CalibrationType::GlobalThresholdDAC);
        dataHandle().resetCounts();

        ui->global_sbmx->setCurrentIndex(1);
        ui->global_scmx->setCurrentIndex(0); // disable channel multiplexing

        if(is_threshold)
            ui->global_sm5->setCurrentIndex(2);  // set to channel global DAC threshold
        else
            ui->global_sm5->setCurrentIndex(1);  // set to channel global DAC pulser

        bool ok;

        // board
        string board_sel_string = ui->calibration_board_select->currentText().toStdString();
        QString all_text = "All";
        int all_boards_idx = (ui->calibration_board_select->findText(all_text));
        int board_to_calibrate_idx = ui->calibration_board_select->currentIndex();

        int board_start = -1;
        int board_end = -1;

        if(board_sel_string == "All") {
            board_start = 0;
            board_end = (all_boards_idx - 1);
        }
        else {
            board_start = board_to_calibrate_idx;
            board_end = board_to_calibrate_idx;
        }

        int board_at_end = (board_start == board_end);
        int board_current = ui->setVMMs->currentIndex();
        if(board_sel_string == "All" && board_current==(all_boards_idx - 1)) {
            board_at_end = true;
        }
        else if(board_current == board_end) {
            board_at_end = true;
        }
        int board_for_calibration = -1;

        // VMM
        int vmm_idx_start = 0;
        int vmm_idx_end = (selected_vmms.size() - 1);
        int vmm_current = get_first_selected_vmm();
        int vmm_idx_current = -1;

        for(auto x : vmm_idx_map) {
            if(x.second == vmm_current)
                vmm_idx_current = x.first;
        } 

        int vmm_idx_to_calibrate = -1;
        bool vmm_idx_at_end = (vmm_idx_current == vmm_idx_end);
        if(selected_vmms.size()==8) vmm_idx_at_end = true; // all simultaneous, avoid loop

        // threshold/pulser
        int dac_start = ui->xadccalibration_global_threshold_start->value();
        int dac_end   = ui->xadccalibration_global_threshold_end->value();
        int dac_step  = ui->xadccalibration_threshold_step->value();

        int dac_current = ui->global_sdt_dac->value();
        int dac_to_calibrate = -1;
        bool dac_at_end = (dac_current == dac_end);

        if(!is_threshold) {
            dac_start = ui->xadccalibration_global_testpulse_start->value();
            dac_end   = ui->xadccalibration_global_testpulse_end->value();
            dac_step  = ui->xadccalibration_global_test_pulse_step->value();

            dac_current = ui->global_sdp_dac->value();
            dac_to_calibrate = -1;
            dac_at_end = (dac_current == dac_end);
        }

        if (is_start_of_run) {
            dac_to_calibrate = dac_start;
            vmm_idx_to_calibrate = vmm_idx_start;
            //if(vmm_idx_current < 0) vmm_idx_current = vmm_idx_start;
            board_for_calibration = board_start;
        } // is start of run
        if (!is_start_of_run) {
            if(!dac_at_end) {
                dac_to_calibrate = (dac_current + dac_step);
                vmm_idx_to_calibrate = vmm_idx_current;
                board_for_calibration = board_current;
            } // not at end of threshold loop
            else if(dac_at_end && !vmm_idx_at_end) {
                dac_to_calibrate = dac_start;
                vmm_idx_to_calibrate = (vmm_idx_current + 1);
                board_for_calibration = board_current;
            }
            else if(dac_at_end && vmm_idx_at_end && !board_at_end) {
                dac_to_calibrate = dac_start;
                vmm_idx_to_calibrate = vmm_idx_start;
                board_for_calibration = (board_current + 1);
            }
            else if(dac_at_end && vmm_idx_at_end && board_at_end) {
                ui->stopTriggerCnt->click();
                return;
            }
        } // is NOT start of run

        // set the threshold on the gui
        if(is_threshold)
            ui->global_sdt_dac->setValue(dac_to_calibrate);
        else
            ui->global_sdp_dac->setValue(dac_to_calibrate);

        // set the board to calibrate
        ui->setVMMs->setCurrentIndex(board_for_calibration);

        // xadc stuff
        int vmm_to_sample = 1;
        int vmm_to_test = -1;
        if(!(selected_vmms.size()==8)) {
            vmm_to_test = vmm_idx_map[vmm_idx_to_calibrate];
        }
        int n_vmms = setVMMsForCalib(type, vmm_to_test); 
        if(n_vmms<0) {
            stringstream err;
            err << "ERROR Number of VMMs set for calibration scan is <0!";
            msg()(err);
            ui->stopTriggerCnt->click();
            return;
        }
        if(n_vmms==8) vmm_to_sample = 0; // xADC FW sees '0' as 'All VMMs'
        else {
            vmm_to_sample = (vmm_idx_map[vmm_idx_to_calibrate] + 1);
            //for(int i = 0; i < 8; i++){ 
            //    if(m_vmm_mask.at(i)->isChecked()) {
            //        vmm_to_sample = (i+1);
            //        break;
            //    }
            //} // i
        }

        //cout << "vmm idx to calibrate = " << vmm_idx_to_calibrate << " (" << vmm_idx_map[vmm_idx_to_calibrate] << ")  current = " << vmm_idx_current << "  vmm_to_sample = " << vmm_to_sample << endl;

        int n_samples = ui->calibration_n_samples_select->text().toInt(&ok, 10);
        int sample_period = ui->xadccalibration_sample_period->value();

        // clear all samples
        ui->STLabel_2->click();
        ui->STLabel_2->click();
        ui->STLabel2_2->click();
        ui->STLabel2_2->click();

        ui->SMXLabel_2->click();
        ui->SMXLabel2_2->click();
        ui->SMXLabel_2->click();
        ui->SMXLabel2_2->click();
        

        // ok send the configuration
        // delay
        //delay();
        //delay();
        ui->SendConfiguration->click();

        // load the calibration state into the calibration module
        CalibrationState state;
        int channel_set = ui->global_sm5->currentIndex();
        int run_number_set = ui->runNumber->value();
        double gain_set = ui->global_sg->currentText().remove(" mV/fC").toDouble();
        int tac_slope_set = ui->global_stc->currentText().remove("ns").toInt();
        int neighbor_trigger_set = ui->global_sng->currentIndex();
        int subhyst_set = ui->global_ssh->currentIndex();
        int peak_time_set = ui->global_st->currentText().remove("ns").toInt();
        int dac_counts_set = ui->global_sdt_dac->value();
        int test_pulse_set = ui->global_sdp_dac->value();

        int skew_idx = ui->cktp_skew_steps->value();
        int ckbc_idx = ui->ckbc_frequency->currentIndex();
        double tp_skew_value = 0.0;
        if(ckbc_idx==0) { // 40 MHz CKBC
            tp_skew_value = skew_idx * 1; // ns
        }
        else if(ckbc_idx==1 || ckbc_idx ==2) { // 20 MHz or 10 MHz CKBC
            tp_skew_value = skew_idx * 6.25; // ns
        }

        state.channel = channel_set;
        state.run_number = run_number_set;
        state.gain = gain_set;
        state.tac_slope = tac_slope_set;
        state.neighbor_trigger = neighbor_trigger_set;
        state.subhysteresis = subhyst_set;
        state.peak_time = peak_time_set;
        state.test_pulse = test_pulse_set;
        state.dac_counts = dac_counts_set;
        state.channel_trims = getChannelTrims();
        state.tp_skew = tp_skew_value;
        int n_expected_pulses = ui->cktp_max_number->value();
        state.n_expected_pulses = n_expected_pulses;

        dataHandle().loadCalibrationState(state.channel, state.gain, state.run_number,
                state.tac_slope, state.neighbor_trigger, state.subhysteresis,
                state.peak_time, state.test_pulse, state.dac_counts, state.channel_trims,
                state.tp_skew, state.n_expected_pulses);

        CalibrationType type = get_calibration_type();
        dataHandle().initializeCalibration(vmm_to_sample, n_samples, sample_period, type, false, do_l0_decoding()); 


        //cout << "BLAH board ip : " << ips[board_for_calibration].toStdString() << "  vmms: " << vmm_to_sample << "  board_for_calibration: " << board_for_calibration << endl; 

        dataHandle().start_xadc_sampling(ips[board_for_calibration].toStdString());
        //dataHandle().start_xadc_sampling(ips[0].toStdString());

    }
    /////////////////////////////////////////////////////////////////
    // Channel Trim
    /////////////////////////////////////////////////////////////////
    else if(type==CalibrationType::ChannelTrim || type==CalibrationType::ChannelBaseline) {

        ui->cktp_max_number->setValue(-1);

        dataHandle().resetCounts();

        // set the necessary configurations
        ui->global_scmx->setCurrentIndex(1); // route analog output to MO
        ui->global_sbmx->setCurrentIndex(1); // enable channel multiplexing

        bool ok;


        // threshold
        int threshstart = ui->xadccalibration_global_threshold_start->value();
        int thresh_step = ui->xadccalibration_threshold_step->value();
        int threshend = ui->xadccalibration_global_threshold_end->value();
        int currentthresh = ui->global_sdt_dac->value();

        // board
        string board_sel_string = ui->calibration_board_select->currentText().toStdString();
        QString all_text = "All";
        int all_boards_idx = (ui->calibration_board_select->findText(all_text));
        int board_to_calibrate_idx = ui->calibration_board_select->currentIndex();

        int board_start = -1;
        int board_end = -1;
        if(board_to_calibrate_idx == all_boards_idx) {
            board_start = 0; // index 0 is start
            board_end = (all_boards_idx - 1);
        }
        else {
            board_start = board_to_calibrate_idx;
            board_end = board_to_calibrate_idx;
        }

        if(board_sel_string == "All") {
            board_start = 0;
            board_end = (all_boards_idx - 1);
        }
        else {
            board_start = board_to_calibrate_idx;
            board_end = board_to_calibrate_idx;
        }

        bool board_at_end = (board_start == board_end);
        int board_current = ui->setVMMs->currentIndex();
        if(board_sel_string == "All" && board_current==(all_boards_idx - 1)) {
            board_at_end = true;
        }
        int board_for_calibration = -1;

        // VMM loop
        int vmm_idx_start = 0;
        int vmm_idx_end = (selected_vmms.size() - 1);
        int vmm_current = get_first_selected_vmm();

        int vmm_idx_current = -1;
        for(auto x : vmm_idx_map) {
            if(x.second == vmm_current)
                vmm_idx_current = x.first;
        }
        int vmm_idx_to_calibrate = -1;
        bool vmm_idx_at_end = (vmm_idx_current == vmm_idx_end);
        if(selected_vmms.size()==8) vmm_idx_at_end = true; // all simultaneous, avoid loop

        // channel
        int channelstart = ui->calibration_channel_start->value();
        int channelend = ui->calibration_channel_end->value();
        int channel = ui->global_sm5->currentIndex();
        int channel_to_calibrate = -1;


        // global threshold
        int global_to_calibrate = -1;
        bool global_at_end = (currentthresh == threshend);
        bool channel_at_end = ( channel == channelend );

        // channel trim
        int trim_start = ui->xadccalibration_channel_trim_start->value();
        int trim_end   = ui->xadccalibration_channel_trim_end->value();
        int trim_stepsize = ui->xadccalibration_trim_step->currentText().toInt(&ok,10);
        int current_trim = ui->SDLabel_2->currentIndex();
        bool trim_at_end = (current_trim == trim_end);
        int trim_to_set = -1;
        if(vmm_type()==2) {
            if((current_trim + trim_stepsize) > 15) trim_at_end = true;
        }
        else if(vmm_type()==3) {
            if((current_trim + trim_stepsize) > 31) trim_at_end = true;
        }



        // for channel baselines with xADC, do not loop over trim or global threshold
        if(type==CalibrationType::ChannelBaseline) {
            global_at_end = true;
            trim_at_end = true;
            // set MO off to remove any bias level
            ui->global_sbfm->setCurrentIndex(0);
        }

        if (is_start_of_run)
        {
            board_for_calibration = board_start;
            vmm_idx_to_calibrate = vmm_idx_start;
            channel_to_calibrate = channelstart;
            global_to_calibrate = threshstart;
            trim_to_set = trim_start;
        }
    
        if (!is_start_of_run)
        {
            if (!channel_at_end)
            {
                channel_to_calibrate = channel + 1; // move forward in the channel loop
                global_to_calibrate = currentthresh;
                trim_to_set = current_trim;
                vmm_idx_to_calibrate = vmm_idx_current;
                board_for_calibration = board_current;
            }
            else if (channel_at_end && !trim_at_end) {
                channel_to_calibrate = channelstart;
                global_to_calibrate = currentthresh;
                trim_to_set = (current_trim + trim_stepsize);
                vmm_idx_to_calibrate = vmm_idx_current;
                board_for_calibration = board_current;
            } 
            else if (channel_at_end && trim_at_end && !global_at_end)
            {
                channel_to_calibrate = channelstart;
                global_to_calibrate = currentthresh + thresh_step;
                trim_to_set = trim_start;
                vmm_idx_to_calibrate = vmm_idx_current;
                board_for_calibration = board_current;
            }
            else if (channel_at_end && trim_at_end && global_at_end && !vmm_idx_at_end) {
                channel_to_calibrate = channelstart;
                global_to_calibrate = threshstart;
                trim_to_set = trim_start;
                vmm_idx_to_calibrate = (vmm_idx_current + 1);
                board_for_calibration = board_current;
            }
            else if (channel_at_end && trim_at_end && global_at_end && vmm_idx_at_end && !board_at_end) {
                channel_to_calibrate = channelstart;
                global_to_calibrate = threshstart;
                trim_to_set = trim_start;
                vmm_idx_to_calibrate = vmm_idx_start;
                board_for_calibration = (board_current + 1);
            }
            else if (channel_at_end && trim_at_end && global_at_end && vmm_idx_at_end && board_at_end)
            {
                ui->stopTriggerCnt->click();
                return;
            }
    
        }
    
        ui->global_sdt_dac->setValue(global_to_calibrate);
        ui->global_sm5->setCurrentIndex(channel_to_calibrate);
     
   
        // xadc stuff
        int vmm_to_sample = 1;
        int vmm_to_test = -1;
        if(!(selected_vmms.size()==8)) {
            vmm_to_test = vmm_idx_map[vmm_idx_to_calibrate];
        }
        int n_vmms = setVMMsForCalib(type, vmm_to_test);
        if(n_vmms<0) {
            stringstream err;
            err << "ERROR Number of VMMs set for calibration scan is <0!";
            msg()(err);
            ui->stopTriggerCnt->click();
            return;
        }
        if(n_vmms==8) vmm_to_sample = 0;
        else {
            vmm_to_sample = (vmm_idx_map[vmm_idx_to_calibrate] + 1);
        }

        // set the board to calibrate
        ui->setVMMs->setCurrentIndex(board_for_calibration);

        int n_samples = ui->calibration_n_samples_select->text().toInt(&ok, 10);
        int sample_period = ui->xadccalibration_sample_period->value();
        ui->SDLabel_2->setCurrentIndex(trim_to_set);
        ui->SDLabel2_2->setCurrentIndex(trim_to_set);

        ui->SMXLabel_2->click();
        ui->SMXLabel_2->click();
        ui->SMXLabel2_2->click();
        ui->SMXLabel2_2->click();

        setChannelsForCalib(type, channel_to_calibrate);

        //VMMSMX[channel_to_calibrate-1]->click();
        ui->SendConfiguration->click();
        //boost::this_thread::sleep(boost::posix_time::milliseconds(1000));

        // load the calibration state into the calibration module
        CalibrationState state;
        int channel_set = ui->global_sm5->currentIndex(); // count 0-63!
        int run_number_set = ui->runNumber->value();
        double gain_set = ui->global_sg->currentText().remove(" mV/fC").toDouble();
        int tac_slope_set = ui->global_stc->currentText().remove("ns").toInt();
        int neighbor_trigger_set = ui->global_sng->currentIndex();
        int subhysteresis_set = ui->global_ssh->currentIndex();
        int peak_time_set = ui->global_st->currentText().remove("ns").toInt();
        int dac_counts_set = ui->global_sdt_dac->value();
        int test_pulse_set = ui->global_sdp_dac->value();

        int skew_idx = ui->cktp_skew_steps->value();
        int ckbc_idx = ui->ckbc_frequency->currentIndex();
        double tp_skew_value = 0.0;
        if(ckbc_idx==0) { // 40 MHz CKBC
            tp_skew_value = skew_idx * 1; // ns
        }
        else if(ckbc_idx==1 || ckbc_idx ==2) { // 20 MHz or 10 MHz CKBC
            tp_skew_value = skew_idx * 6.25; // ns
        }
     
        state.channel = channel_set;
        state.run_number = run_number_set;
        state.gain = gain_set;
        state.tac_slope = tac_slope_set;
        state.neighbor_trigger = neighbor_trigger_set;
        state.subhysteresis = subhysteresis_set;
        state.peak_time = peak_time_set;
        state.test_pulse = test_pulse_set;
        state.dac_counts = dac_counts_set;
        state.channel_trims = getChannelTrims();
        state.tp_skew = tp_skew_value;
        int n_expected_pulses = ui->cktp_max_number->value();
        state.n_expected_pulses = n_expected_pulses;

        //dataHandle().loadCalibrationState(state);
        dataHandle().loadCalibrationState(state.channel, state.gain, state.run_number,
                state.tac_slope, state.neighbor_trigger, state.subhysteresis,
                state.peak_time, state.test_pulse, state.dac_counts, state.channel_trims,
                state.tp_skew, state.n_expected_pulses);

        // now send the xADC sampling command
        //cout << " StartRun configuration sent" << channel << endl;
        CalibrationType type = get_calibration_type();
        dataHandle().initializeCalibration(vmm_to_sample, n_samples, sample_period, type, false, do_l0_decoding());
        //delay();
        //delay();

        //cout << "BLAH board ip : " << ips[board_for_calibration].toStdString() << "  vmms: " << vmm_to_sample << endl; 
        dataHandle().start_xadc_sampling(ips[board_for_calibration].toStdString());
        //dataHandle().start_xadc_sampling(ips[0].toStdString());
    } 

    /////////////////////////////////////////////////////////////////////
    // non-xADC calibration loops
    /////////////////////////////////////////////////////////////////////
    else if(type==CalibrationType::BaselineNeighbor || type==CalibrationType::BaselineExtrapolation
        || type==CalibrationType::Custom || type==CalibrationType::Time
        || type==CalibrationType::Efficiency) {


        return;
        cout << "MainWindow::updateCalibrationState    is_start_of_run = " << is_start_of_run << endl;
        ui->offACQ->click();
        ui->offACQ->click();
        ui->offACQ->click();


        ui->cktp_max_number->setValue(-1);


        stringstream sx;

        /*-------------------------------------------------------*/
        // display progress
        /*-------------------------------------------------------*/
        //if(!m_calculate_total_steps) {
        //    n_current_calibration_step++;
        //    if( ((n_current_calibration_step % n_calib_update_progress) == 0)
        //            || (n_current_calibration_step == n_total_steps_in_loop) ) {
        //        int per = 10*(n_times_checked_progress+1);
        //        sx << "Calibration progress update: " << std::setw(3) << per << "%";
        //        msg()(sx); sx.str("");
        //        n_times_checked_progress++;
        //    }
        //}

        /*-------------------------------------------------------*/
        // for neighbor calib, set global configuration parameter
        // and can only do channel loop here
        /*-------------------------------------------------------*/
        if(type==CalibrationType::BaselineNeighbor) {
            ui->global_sng->setCurrentIndex(1);
            ui->calibration_channel_loop_enable->setChecked(true);
        }


        // set acquisition off for configuration setting
        //ui->offACQ->click();
        //performHardReset();
        //delay();
        //delay();

        //ui->vmm3_hard_reset_button->click();
        //delay();
        //delay();


        dataHandle().resetCounts();
        delay();
        dataHandle().resetCounts();
        delay();
        //dataHandle().resetCounts();
        //delay();
        //dataHandle().resetCounts();
        //delay();
        //dataHandle().resetCounts();
        //delay();
        //if(!is_start_of_run) {
        //    if(dataHandle().getCounts() != 0)
        //        cout << "DH getCounts is nonzero!" << endl;
        //        dataHandle().resetCounts();
        //        dataHandle().resetCounts();
        //        dataHandle().resetCounts();
        //}
        //if(!is_start_of_run) {
        //    while(ui->eventCountStop->text().toInt() != 0) {
        //        updateCalibrationState(is_start_of_run);
        //    }
        //}

        
        /*-------------------------------------------------------*/
        // get the loop definition
        /*-------------------------------------------------------*/

        bool ok;
        // board
        string board_sel_string = ui->calibration_board_select->currentText().toStdString();
        QString all_text = "All";
        int all_boards_idx = (ui->calibration_board_select->findText(all_text));
        int board_to_calibrate_idx = ui->calibration_board_select->currentIndex();

        int board_start = -1;
        int board_end = -1;
        if(board_to_calibrate_idx == all_boards_idx) {
            board_start = 0; // index 0 is start
            board_end = (all_boards_idx - 1);
        }
        else {
            board_start = board_to_calibrate_idx;
            board_end = board_to_calibrate_idx;
        }

        if(board_sel_string == "All") {
            board_start = 0;
            board_end = (all_boards_idx - 1);
        }
        else {
            board_start = board_to_calibrate_idx;
            board_end = board_to_calibrate_idx;
        }

        bool board_at_end = (board_start == board_end);
        int board_current = ui->setVMMs->currentIndex();
        if(board_sel_string == "All" && board_current==(all_boards_idx-1)) {
            board_at_end = true; 
        }
        int board_for_calibration = -1;

        // gain 
        int gain_start = ui->calibration_gain_start->currentIndex();
        int gain_end   = ui->calibration_gain_end->currentIndex();

        int gain_current = ui->global_sg->currentIndex();
        int gain_for_calibration = -1;
        bool gain_at_end = (gain_current == gain_end);

        // peak time
        int peak_time_start = ui->calibration_peak_time_start->currentIndex();
        int peak_time_end   = ui->calibration_peak_time_end->currentIndex();

        int peak_time_current = ui->global_st->currentIndex();
        int peak_time_for_calibration = -1;
        bool peak_time_at_end = (peak_time_current == peak_time_end);

        // tac slope
        int tac_start = ui->calibration_tac_slope_start->currentIndex();
        int tac_end   = ui->calibration_tac_slope_end->currentIndex();

        int tac_current = ui->global_stc->currentIndex();
        int tac_for_calibration = -1;
        bool tac_at_end = (tac_current == tac_end);

        // time
        int time_start = ui->calibration_cktp_skew_start->value();
        int time_end   = ui->calibration_cktp_skew_end->value();
        int time_step  = ui->calibration_cktp_skew_step->value();

        int time_current = ui->cktp_skew_steps->value();
        int time_for_calibration = -1;
        bool time_at_end = (time_current == time_end);

        // threshold
        int threshold_start = ui->calibration_global_threshold_start->value();
        int threshold_end   = ui->calibration_global_threshold_end->value();
        int threshold_step  = ui->calibration_global_threshold_step->value();

        int threshold_current = ui->global_sdt_dac->value();
        int threshold_for_calibration = -1;
        bool threshold_at_end = (threshold_current == threshold_end);

        // dac amplitude
        int amplitude_start = ui->calibration_global_testpulse_start->value();
        int amplitude_end   = ui->calibration_global_testpulse_end->value();
        int amplitude_step  = ui->calibration_global_testpulse_step->value();

        int amplitude_current = ui->global_sdp_dac->value();
        int amplitude_for_calibration = -1;
        bool amplitude_at_end = (amplitude_current == amplitude_end);
        //cout << " [1] amplitude_current: " << amplitude_current << "  for calib: " << amplitude_for_calibration <<"  end: " << amplitude_end << "  at end? " << amplitude_at_end << endl;
    
        // channel
        int channel_start   = ui->calibration_channel_start->value();
        int channel_end     = ui->calibration_channel_end->value();
        int channel_current = ui->global_sm5->currentIndex();
        int channel_to_calibrate = -1;
        bool channel_at_end = (channel_current == channel_end);

        bool do_channel_loop = ui->calibration_channel_loop_enable->isChecked();

        ///////////////////////////////////////////////////////
        // depending on calibration type, disable loops
        ///////////////////////////////////////////////////////
        if(type==CalibrationType::BaselineNeighbor || type==CalibrationType::BaselineExtrapolation || type==CalibrationType::Time) {
            gain_at_end = true;
            peak_time_at_end = true;
            tac_at_end = true;
            threshold_at_end = true;
            if(type!=CalibrationType::Time) time_at_end = true;
            //amplitude_at_end = (type==CalibrationType::BaselineNeighbor);
            if(type==CalibrationType::BaselineNeighbor) amplitude_at_end = true;
        }
        //cout << " [2] amplitude_current: " << amplitude_current << "  for calib: " << amplitude_for_calibration <<"  end: " << amplitude_end << "  at end? " << amplitude_at_end << endl;

        if(type==CalibrationType::Efficiency) {
            gain_at_end = true;
            peak_time_at_end = true;
            tac_at_end = true;
            time_at_end = true;

            bool is_thresh = ui->calibration_efficiency_threshold_checkbox->isChecked();
            if(is_thresh) { amplitude_at_end = true; }
            else { threshold_at_end = true; }
        }


        /*
        ///////////////////////////////////////////////////////////
        // get the total number of steps in the loop
        ///////////////////////////////////////////////////////////
        if(m_calculate_total_steps) {
            int total_board_steps = 0;
            for(int i = board_end; i >= board_start; i--)
                total_board_steps++;
            int total_gain_steps = 0;
            for(int i = gain_end; i >= gain_start; i--)
                total_gain_steps++;
            int total_peak_time_steps = 0;
            for(int i = peak_time_end; i >= peak_time_start; i--)
                total_peak_time_steps++;
            int total_tac_steps = 0;
            for(int i = tac_end; i >= tac_start; i--)
                total_tac_steps++;
            int total_threshold_steps = 0;
            for(int i = threshold_end; i >= threshold_start; i-=threshold_step)
                total_threshold_steps++;
            int total_amplitude_steps = 0;
            for(int i = amplitude_start; i >= amplitude_end; i-=amplitude_step)
                total_amplitude_steps++;
            int total_channel_steps = 0;
            for(int i = channel_end; i >= channel_start; i--)
                total_channel_steps++;
            int total_time_steps = 0;
            for(int i = time_end; i >= time_start; i-=time_step)
                total_time_steps++;

            if(board_at_end) total_board_steps = 1;
            if(gain_at_end) total_gain_steps = 1;
            if(peak_time_at_end) total_peak_time_steps = 1;
            if(tac_at_end) total_tac_steps = 1;
            if(threshold_at_end) total_threshold_steps = 1;
            if(amplitude_at_end) total_amplitude_steps = 1;
            if(channel_at_end) total_channel_steps = 1;
            if(time_at_end) total_time_steps = 1;

            int total_number_of_steps_in_loop = total_board_steps * total_gain_steps;
            total_number_of_steps_in_loop *= total_peak_time_steps * total_tac_steps;
            total_number_of_steps_in_loop *= total_threshold_steps * total_amplitude_steps;
            if(do_channel_loop)
                total_number_of_steps_in_loop *= total_channel_steps;

            n_total_steps_in_loop = total_number_of_steps_in_loop;
            n_calib_update_progress = (int)(((n_total_steps_in_loop * 1.00) / 100.0) * 10);
            m_calculate_total_steps = false;
        }
        */

        ///////////////////////////////////////////////////////
        // looping over the channels
        ///////////////////////////////////////////////////////
        if(do_channel_loop) {
            if (is_start_of_run) {
                board_for_calibration = board_start;
                gain_for_calibration = gain_start;
                peak_time_for_calibration = peak_time_start;
                tac_for_calibration = tac_start;
                threshold_for_calibration = threshold_start;
                time_for_calibration = time_start;
                amplitude_for_calibration = amplitude_start;
                channel_to_calibrate = channel_start;

                sx << "Begin calibration scan of board #" << board_for_calibration << " (IP: " << ips[board_for_calibration].toStdString() << ")";
                msg()(sx); sx.str("");

                n_current_calibration_step = 1;
                n_times_checked_progress = 0;
            } // start of run

            if (!is_start_of_run) {
                if(!channel_at_end) {
                    channel_to_calibrate = (channel_current + 1);
                    peak_time_for_calibration = peak_time_current;
                    tac_for_calibration = tac_current;
                    amplitude_for_calibration = amplitude_current;
                    threshold_for_calibration = threshold_current;
                    time_for_calibration = time_current;
                    gain_for_calibration = gain_current;
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && !amplitude_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = (amplitude_current + amplitude_step);
                    threshold_for_calibration = threshold_current;
                    tac_for_calibration = tac_current;
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
        //cout << " [3] amplitude_current: " << amplitude_current << "  for calib: " << amplitude_for_calibration <<"  end: " << amplitude_end << "  at end? " << amplitude_at_end << endl;
                }
                else if(channel_at_end && amplitude_at_end && !threshold_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = (threshold_current + threshold_step);
                    tac_for_calibration = tac_current;
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && !tac_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = (tac_current + 1);
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && tac_at_end && !peak_time_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = (peak_time_current + 1);
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && !gain_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = (gain_current + 1);
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && !time_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = gain_start;
                    time_for_calibration = (time_current + time_step);
                    board_for_calibration = board_current;
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && time_at_end && !board_at_end) {
                    channel_to_calibrate = channel_start;
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = gain_start;
                    time_for_calibration = time_start;
                    board_for_calibration = (board_current + 1);

                    sx << "Begin calibration scan of board #" << board_for_calibration << " (IP: " << ips[board_for_calibration].toStdString() << ")";
                    msg()(sx); sx.str("");
                }
                else if(channel_at_end && amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && time_at_end && board_at_end) {
                    ui->stopTriggerCnt->click();
                    ui->offACQ->click();
                    m_calculate_total_steps = true;
                    return;
                }
            } // not start of run
        }
        ///////////////////////////////////////////////////////
        // pulsing all channels at once (not looping over them)
        ///////////////////////////////////////////////////////
        else {
            if (is_start_of_run) {
                board_for_calibration = board_start;
                gain_for_calibration = gain_start;
                peak_time_for_calibration = peak_time_start;
                tac_for_calibration = tac_start;
                threshold_for_calibration = threshold_start;
                time_for_calibration = time_start;
                amplitude_for_calibration = amplitude_start;

                sx << "Begin calibration scan of board #" << board_for_calibration << " (IP: " << ips[board_for_calibration].toStdString() << ")";
                msg()(sx); sx.str("");

                n_current_calibration_step = 1;
                n_times_checked_progress = 0;
            } // start of run

            if (!is_start_of_run) {
                if(!amplitude_at_end) {
                    amplitude_for_calibration = (amplitude_current + amplitude_step);
                    threshold_for_calibration = threshold_current;
                    tac_for_calibration = tac_current;
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current; 
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && !threshold_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = (threshold_current + threshold_step);
                    tac_for_calibration = tac_current;
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && threshold_at_end && !tac_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = (tac_current + 1);
                    peak_time_for_calibration = peak_time_current;
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && threshold_at_end && tac_at_end && !peak_time_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = (peak_time_current + 1);
                    gain_for_calibration = gain_current;
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && !gain_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = (gain_current + 1);
                    time_for_calibration = time_current;
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && !time_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = gain_start;
                    time_for_calibration = (time_current + time_step);
                    board_for_calibration = board_current;
                }
                else if(amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && time_at_end && !board_at_end) {
                    amplitude_for_calibration = amplitude_start;
                    threshold_for_calibration = threshold_start;
                    tac_for_calibration = tac_start;
                    peak_time_for_calibration = peak_time_start;
                    gain_for_calibration = gain_start;
                    board_for_calibration = (board_current + 1);

                    sx << "Begin calibration scan of board #" << board_for_calibration << " (IP: " << ips[board_for_calibration].toStdString() << ")";
                    msg()(sx); sx.str("");
                }
                else if(amplitude_at_end && threshold_at_end && tac_at_end && peak_time_at_end && gain_at_end && time_at_end && board_at_end) {
                    ui->stopTriggerCnt->click();
                    ui->offACQ->click();
                    m_calculate_total_steps = true;
                    return;
                }
            } // not start of run
        } // not looping
    
        // set the parameters on the GUI to be configured
        // board
        ui->setVMMs->setCurrentIndex(board_for_calibration);
        // channel
        ui->global_sm5->setCurrentIndex((do_channel_loop ? (channel_to_calibrate) : 0));
        // amplitude
        ui->global_sdp_dac->setValue(amplitude_for_calibration);
        // threshold
        ui->global_sdt_dac->setValue(threshold_for_calibration);
        // gain
        ui->global_sg->setCurrentIndex(gain_for_calibration);
        // peak time
        ui->global_st->setCurrentIndex(peak_time_for_calibration);
        // tac slope
        ui->global_stc->setCurrentIndex(tac_for_calibration);

        // time
        ui->cktp_max_number->setValue(-1);
        if(type==CalibrationType::Time || type==CalibrationType::Custom) {
            ui->cktp_skew_steps->setValue(time_for_calibration);
            ui->clocks_send_settings_button->click();
            //delay();
            //delay();
        }
        
        ui->global_scmx->setCurrentIndex(1);
        ui->global_sbmx->setCurrentIndex(0);
        // set MO
        ui->global_sbfm->setCurrentIndex(1);

        //int vmm_to_sample = ui->calibration_xadc_vmm_select->value();
        int vmm_to_sample = 1; // this paramter is not used for these calibrations
        int n_vmms = setVMMsForCalib(type);
        if(n_vmms<0) {
            stringstream err;
            err << "ERROR Number of VMMs set for calibration scan is <0!";
            msg()(err);
            ui->stopTriggerCnt->click();
            return;
        }
        if(n_vmms<0) n_vmms = 1;

        //int n_samples = ui->calibration_n_samples_select->text().toInt(&ok, 10);
        int n_samples_per_channel = ui->calibration_n_samples_select->text().toInt(&ok, 10);
        int sample_period = ui->xadccalibration_sample_period->value();

        // for efficiency runs set the n_samples to the number of cktps to send from the front end
        int n_pulses = -1;
        if(type==CalibrationType::Efficiency) {
            n_pulses = n_samples_per_channel;
            ui->cktp_skew_steps->setValue(time_for_calibration);
            //if(!do_channel_loop) {
            //    int chan_beg = ui->calibration_channel_start->value();
            //    int chan_end = ui->calibration_channel_end->value();
            //    int multiplier = (int)(chan_end - chan_beg);
            //    if(multiplier > 0)
            //        n_pulses = (int)(n_pulses / multiplier);
            //}
            
            ui->cktp_max_number->setValue(n_pulses); 
            ui->clocks_send_settings_button->click();
        }

        

        // set the charge injection for the channel
        setChannelsForCalib(type, channel_to_calibrate, do_channel_loop);
        //delay();
        //delay();
        //setChannelsForCalib(type, channel_to_calibrate-1, do_channel_loop);
        //delay();
        //delay();
        //delay();

        
        ui->SendConfiguration->click();
        delay();
        ui->SendConfiguration->click();
        delay();
        //boost::this_thread::sleep(boost::posix_time::milliseconds(150));
        //ui->SendConfiguration->click();
        //boost::this_thread::sleep(boost::posix_time::milliseconds(150));
        //delay();
        //delay();
        //ui->SendConfiguration->click();
        ////ui->SendConfiguration->click();
        //delay();
        //delay();
        //delay();

        // load the calibration state into the calibration module
        CalibrationState state;
        int channel_set = ui->global_sm5->currentIndex(); // count 0-63!
        int run_number_set = ui->runNumber->value();
        double gain_set = ui->global_sg->currentText().remove(" mV/fC").toDouble();
        int tac_slope_set = ui->global_stc->currentText().remove("ns").toInt();
        int neighbor_trigger_set = ui->global_sng->currentIndex();
        int subhysteresis_set = ui->global_ssh->currentIndex();
        int peak_time_set = ui->global_st->currentText().remove("ns").toInt();
        int dac_counts_set = ui->global_sdt_dac->value();
        int test_pulse_set = ui->global_sdp_dac->value();

        int skew_idx = ui->cktp_skew_steps->value();
        int ckbc_idx = ui->ckbc_frequency->currentIndex();
        double tp_skew_value = 0.0;
        if(ckbc_idx==0) { // 40 MHz CKBC
            tp_skew_value = skew_idx * 1; // ns
        }
        else if(ckbc_idx==1 || ckbc_idx ==2) { // 20 MHz or 10 MHz CKBC
            tp_skew_value = skew_idx * 6.25; // ns
        }
     
        state.channel = channel_set;
        state.run_number = run_number_set;
        state.gain = gain_set;
        state.tac_slope = tac_slope_set;
        state.neighbor_trigger = neighbor_trigger_set;
        state.subhysteresis = subhysteresis_set;
        state.peak_time = peak_time_set;
        state.test_pulse = test_pulse_set;
        state.dac_counts = dac_counts_set;
        state.channel_trims = getChannelTrims();
        state.tp_skew = tp_skew_value;
        int n_expected_pulses = ui->cktp_max_number->value();
        state.n_expected_pulses = (n_expected_pulses);

        //dataHandle().loadCalibrationState(state);
        dataHandle().loadCalibrationState(state.channel, state.gain, state.run_number,
                state.tac_slope, state.neighbor_trigger, state.subhysteresis,
                state.peak_time, state.test_pulse, state.dac_counts, state.channel_trims,
                state.tp_skew, state.n_expected_pulses);
        //delay();
        //delay();

        //CalibrationType type = get_calibration_type();

        // note: for threshold calibration using xADC the samples value
        //       should NOT be multiplied by number of VMMs
        // note2: for PDO calibration, 'vmm_to_sample' parameter is not used
        // note3: for PDO calibration, 'sample_period' parameter is not used
        bool skip_dead_channels = true;
        //if(!do_channel_loop) {
        //    skip_dead_channels = false;
        //    int multiplier = (ui->calibration_channel_end->value() - ui->calibration_channel_start->value()) + 1;
        //    n_samples_per_channel = (n_samples_per_channel * multiplier);
        //}
        dataHandle().initializeCalibration(vmm_to_sample, n_vmms * n_samples_per_channel,
                                                            sample_period, type, skip_dead_channels, do_l0_decoding());
        //dataHandle().initializeCalibration(vmm_to_sample, n_samples, sample_period, type);
       // delay();
        //delay();
        //delay();

        // now start the acquisition
        //ui->trgPulser->click();
        //delay();

        //if(type==CalibrationType::Time) {
        //    for(int i = 0; i < 5; i++) delay();
        //}
        //if(type==CalibrationType::Time || type==CalibrationType::Custom) {
        //    ui->clocks_send_settings_button->click();
        //    delay();
        //}
        
        ui->onACQ->click();
        //ui->offACQ->click();
        //delay();
        //ui->onACQ->click();

    }
    
}
// ------------------------------------------------------------------------- //
void MainWindow::setPDOCalibrationState(int gain, int threshold_daq, int pulser_daq)
{
    ui->sg->setCurrentIndex(gain);
    ui->sdt->setValue(threshold_daq);
    ui->sdp_2->setValue(pulser_daq);

    // send overall calibration state
    emit updateCalibrationState(ui->sg->currentIndex(),
                                ui->sdt->value(),
                                ui->sdp_2->value(),
                                ui->tpSkew->currentIndex(),
                                ui->st->currentIndex());
}
// ------------------------------------------------------------------------- //
void MainWindow::changeDelayLabels(int delayIdx)
{

    QString tmp;
    //#warning current TDO loop always starts at zero delay

    ui->tdoCalib_skewLabLow->setText(tmp.number(0.000) + " ns");
    ui->tdoCalib_skewLabHigh->setText(tmp.number((3.125*delayIdx), 'f', 3) + " ns");
    //ui->tdoCalib_skewLabHigh->setText(tmp.number((3.125*ui->delaySteps->currentIndex()), 'f', 2) + " ns");
}
// ------------------------------------------------------------------------- //
void MainWindow::setTDOCalibrationState(int delay, int gain, int amplitude, int threshold)
{
    ui->sg->setCurrentIndex(gain);
    ui->sdt->setValue(threshold);
    ui->sdp_2->setValue(amplitude);

    ui->tpSkew->setCurrentIndex(delay);

    emit ui->setTp_s6->click();

    emit updateCalibrationState(ui->sg->currentIndex(),
                                ui->sdt->value(),
                                ui->sdp_2->value(),
                                ui->tpSkew->currentIndex(),
                                ui->st->currentIndex());

}
// ------------------------------------------------------------------------- //
int MainWindow::setVMMsForCalib(CalibrationType type, int vmm_to_calib)
{
    int n_vmms = -1;

    if(type==CalibrationType::BaselineNeighbor || type==CalibrationType::BaselineExtrapolation
        || type==CalibrationType::Custom || type==CalibrationType::Time || type==CalibrationType::Efficiency) {
        n_vmms = 0;

        // clear the current VMM selection
        for(int i = 0; i < 8; i++) {
            m_vmm_mask.at(i)->setChecked(false);
        }

        for(int i = 0; i < 8; i++) {
            if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
                m_vmm_mask.at(i)->setChecked(true);
                n_vmms++;
            }
        }
    }
    else {
        for(int i = 0; i < 8; i++) {
            m_vmm_mask.at(i)->setChecked(false);
        }
        n_vmms = 0;

        if(!(vmm_to_calib < 0)) {
            m_vmm_mask.at(vmm_to_calib)->setChecked(true);
            n_vmms++;
        }
        else {
            for(int i = 0; i < 8; i++) {
                m_vmm_mask.at(i)->setChecked(true);
                n_vmms++;
            }
        }


        /*
        bool has_selected = false;
        for(int i = 0; i < 8; i++) {
            if(ui->calibration_vmm_select->item(0,i)->isSelected()) {
                if(!has_selected) {
                    m_vmm_mask.at(i)->setChecked(true);
                    has_selected = true;
                }
                n_vmms++;
            }
        } // i
        // if all 8 are selected then this is OK, downstream we will
        // send the xADC command that samples from all 8 so we should
        // set the VMM mask to 8 so that we configure them all
        // properly
        if(n_vmms==8) {
            for(int i = 0; i < 8; i++) {
                m_vmm_mask.at(i)->setChecked(true);
            } // i
        }
        */
    }
    return n_vmms;

}
// ------------------------------------------------------------------------- //
/*
void MainWindow::setChannelsForCalib(int calib_channel, bool do_channel_loop)
{
    QString style_off = "background-color: lightGray";

    bool do_masking = ui->calibration_mask_charge->isChecked();

    if(do_channel_loop) {
        for(int ch = 0; ch < 64; ch++) {
            if(ch<32) {
                #warning not masking channels for calibration for MDT board
                //ui->sm_list_widget->item(ch)->setSelected(false);
                //VMMSMBool[ch] = 0;
                //VMMSMBoolAll = 0;
                //VMMSMBoolAll2 = 0;

                ui->st_list_widget->item(ch)->setSelected(false);
                VMMSTBool[ch] = 0;
                VMMSTBoolAll = 0;
                VMMSTBoolAll2 = 0;
            }
            else {
                //ui->sm_list_widget_2->item(ch-32)->setSelected(false);
                //VMMSMBool[ch] = 0;
                //VMMSMBoolAll = 0;
                //VMMSMBoolAll2 = 0;

                ui->st_list_widget_2->item(ch-32)->setSelected(false);
                VMMSTBool[ch] = 0;
                VMMSTBoolAll = 0;
                VMMSTBoolAll2 = 0;
            }
            //VMMSM[ch]->setStyleSheet(style_off);
            //VMMSMBool[ch] = 0;
            //VMMSMBoolAll = 0;
            //VMMST[ch]->setStyleSheet(style_off);
            //VMMSTBool[ch] = 0;
            //VMMSTBoolAll = 0;
        } // ch

        if(do_masking) {
            #warning calibration masking is fixed for only MDT type board
            for(int i = 0; i < 24; i++)
                ui->sm_list_widget->item(i)->setSelected(true);
            ui->sm_list_widget->item(calib_channel)->setSelected(false);

            if(ui->global_sng->currentIndex()==1) {
                if(calib_channel>0)
                    ui->sm_list_widget->item(calib_channel-1)->setSelected(false);
                if(calib_channel<23)
                    ui->sm_list_widget->item(calib_channel+1)->setSelected(false);
            }
        }

        //VMMST[calib_channel]->click();
        if(calib_channel<32)
            ui->st_list_widget->item(calib_channel)->setSelected(true);
        else if(calib_channel>=32)
            ui->st_list_widget_2->item(calib_channel-32)->setSelected(true);
    } // do channel loop
    else {
        for(int ch = 0; ch < 64; ch++) {
            if(ch < 32) {
                ui->st_list_widget->item(ch)->setSelected(false);
                VMMSTBool[ch] = 0;
                VMMSTBoolAll = 0;
                VMMSTBoolAll2 = 0;
            }
            if(ch >= 32) {
                ui->st_list_widget_2->item(ch-32)->setSelected(false);
                VMMSTBool[ch] = 0;
                VMMSTBoolAll = 0;
                VMMSTBoolAll2 = 0;
            }
        } // ch
        int channel_start = ui->calibration_channel_start->value()-1;
        int channel_end   = ui->calibration_channel_end->value()-1;
        for(int i = channel_start; i <= channel_end; i++) {
            if(i<32) {
                ui->st_list_widget->item(i)->setSelected(true);
                VMMSTBool[i] = 1;
            }
            else if(i>=32) {
                ui->st_list_widget_2->item(i-32)->setSelected(true);
                VMMSTBool[i] = 1;
            }
        } // i
    } // not doing channel loop
}
*/
// ------------------------------------------------------------------------- //
void MainWindow::setChannelsForCalib(CalibrationType caltype, int calib_channel, bool do_channel_loop)
{
    ////////////////////////////////////////////////////////
    // PDO
    ////////////////////////////////////////////////////////
    if(caltype==CalibrationType::BaselineNeighbor || caltype==CalibrationType::BaselineExtrapolation
        || caltype==CalibrationType::Custom || caltype==CalibrationType::Time
        || caltype==CalibrationType::Efficiency) {
        bool do_masking = ui->calibration_mask_charge->isChecked();

        if(do_channel_loop) {
            for(int ch = 0; ch < 64; ch++) {
                if(ch<32) {
                    //#warning not masking channels for calibration for MDT board
                    //ui->sm_list_widget->item(ch)->setSelected(false);
                    //VMMSMBool[ch] = 0;
                    //VMMSMBoolAll = 0;
                    //VMMSMBoolAll2 = 0;

                    ui->st_list_widget->item(ch)->setSelected(false);
                    VMMSTBool[ch] = 0;
                    VMMSTBoolAll = 0;
                    VMMSTBoolAll2 = 0;
                }
                else {
                    //ui->sm_list_widget_2->item(ch-32)->setSelected(false);
                    //VMMSMBool[ch] = 0;
                    //VMMSMBoolAll = 0;
                    //VMMSMBoolAll2 = 0;

                    ui->st_list_widget_2->item(ch-32)->setSelected(false);
                    VMMSTBool[ch] = 0;
                    VMMSTBoolAll = 0;
                    VMMSTBoolAll2 = 0;
                }
                //VMMSM[ch]->setStyleSheet(style_off);
                //VMMSMBool[ch] = 0;
                //VMMSMBoolAll = 0;
                //VMMST[ch]->setStyleSheet(style_off);
                //VMMSTBool[ch] = 0;
                //VMMSTBoolAll = 0;
            } // ch

            if(do_masking) {
           //     #warning calibration masking is fixed for only MDT type board
                for(int i = 0; i < 64; i++) {
                //for(int i = 0; i < 32; i++)
                    if(i<32) {
                        ui->sm_list_widget->item(i)->setSelected(true);
                        VMMSMBool[i] = 1;
                        if(calib_channel==i) {
                            ui->sm_list_widget->item(i)->setSelected(false);
                            VMMSMBool[i] = 0;
                        }
                    }
                    else {
                        ui->sm_list_widget_2->item(i-32)->setSelected(true);
                        VMMSMBool[i] = 1;
                        if(calib_channel==i) {
                            ui->sm_list_widget_2->item(i-32)->setSelected(false);
                            VMMSMBool[i] = 0;
                        }
                    }
                }
                //ui->sm_list_widget->item(calib_channel)->setSelected(false);

                if(ui->global_sng->currentIndex()==1) {
                    if(calib_channel>0 && calib_channel<32) {
                        if(calib_channel==31) {
                            ui->sm_list_widget->item(30)->setSelected(false);
                            ui->sm_list_widget_2->item(0)->setSelected(false);
                            VMMSMBool[30] = 0;
                            VMMSMBool[32] = 0;
                        }
                        else {
                            VMMSMBool[calib_channel-1] = 0;
                            VMMSMBool[calib_channel+1] = 0;
                            ui->sm_list_widget->item(calib_channel-1)->setSelected(false);
                            ui->sm_list_widget->item(calib_channel+1)->setSelected(false);
                        }
                    }
                    else if(calib_channel>=32 && calib_channel < 64) {
                        if(calib_channel==32) {
                            VMMSMBool[31] = 0;
                            VMMSMBool[33] = 0;
                            ui->sm_list_widget->item(31)->setSelected(false);
                            ui->sm_list_widget_2->item(1)->setSelected(false);
                        }
                        else {
                            VMMSMBool[calib_channel] = 0;
                            VMMSMBool[calib_channel] = 0;
                            ui->sm_list_widget_2->item((calib_channel-1)-32)->setSelected(false);
                            ui->sm_list_widget_2->item((calib_channel+1)-32)->setSelected(false);
                        }
                    }

                    //if(calib_channel>0)
                    //    ui->sm_list_widget->item(calib_channel-1)->setSelected(false);
                    //if(calib_channel<23)
                    //    ui->sm_list_widget->item(calib_channel+1)->setSelected(false);
                }
            }

            //VMMST[calib_channel]->click();
            if(calib_channel<32) {
                ui->st_list_widget->item(calib_channel)->setSelected(true);
                VMMSTBool[calib_channel] = 1;
            }
            else if(calib_channel>=32) {
                ui->st_list_widget_2->item(calib_channel-32)->setSelected(true);
                VMMSTBool[calib_channel] = 1;
            }
        } // do channel loop
        else {
            for(int ch = 0; ch < 64; ch++) {
                if(ch < 32) {
                    ui->sm_list_widget->item(ch)->setSelected(false);
                    ui->st_list_widget->item(ch)->setSelected(false);
                    VMMSTBool[ch] = 0;
                    VMMSTBoolAll = 0;
                    VMMSTBoolAll2 = 0;
                    VMMSMBool[ch] = 0;
                    VMMSMBoolAll = 0;
                    VMMSMBoolAll2 = 0;
                }
                if(ch >= 32) {
                    ui->sm_list_widget_2->item(ch-32)->setSelected(false);
                    ui->st_list_widget_2->item(ch-32)->setSelected(false);
                    VMMSTBool[ch] = 0;
                    VMMSTBoolAll = 0;
                    VMMSTBoolAll2 = 0;

                    VMMSMBool[ch] = 0;
                    VMMSMBoolAll = 0;
                    VMMSMBoolAll2 = 0;
                }
            } // ch
            int channel_start = ui->calibration_channel_start->value();
            int channel_end   = ui->calibration_channel_end->value();
            for(int i = channel_start; i <= channel_end; i++) {
                if(i<32) {
                    ui->st_list_widget->item(i)->setSelected(true);
                    VMMSTBool[i] = 1;
                }
                else if(i>=32) {
                    ui->st_list_widget_2->item(i-32)->setSelected(true);
                    VMMSTBool[i] = 1;
                }
            } // i

            for(int i = 0; i < 64; i++) {
                if(i <= channel_end || i >= channel_start) continue;
                if(i<32) {
                    ui->sm_list_widget->item(i)->setSelected(true);
                    VMMSMBool[i] = 1;
                }
                else if(i>=32) {
                    ui->sm_list_widget_2->item(i-32)->setSelected(true);
                    VMMSMBool[i] = 1;
                }
            }
        } // not doing channel loop
    } // PDO
    ////////////////////////////////////////////////////////
    // Channel Trims
    ////////////////////////////////////////////////////////
    else if(caltype==CalibrationType::ChannelTrim) {
        for(int ch = 0; ch < 64; ch++) {
            if(ch!=calib_channel) {
                if(ch<32) {
                    ui->smx_list_widget->item(ch)->setSelected(false);
                    VMMSMXBool[ch] = false;
                    ui->st_list_widget->item(ch)->setSelected(false);
                    VMMSTBool[ch] = false;
                }
                else {
                    ui->smx_list_widget_2->item(ch-32)->setSelected(false);
                    VMMSMXBool[ch] = false;
                    ui->st_list_widget_2->item(ch-32)->setSelected(false);
                    VMMSTBool[ch] = false;
                }
            }
            else {
                if(ch<32) {
                    ui->smx_list_widget->item(ch)->setSelected(true);
                    VMMSMXBool[ch] = true;
                    ui->st_list_widget->item(ch)->setSelected(false);
                    VMMSTBool[ch] = false;
                }
                else {
                    ui->smx_list_widget_2->item(ch-32)->setSelected(true);
                    VMMSMXBool[ch] = true;
                    ui->st_list_widget_2->item(ch-32)->setSelected(false);
                    VMMSTBool[ch] = false;
                }
            }
        } // ch
    } // ChannelThreshold
    ////////////////////////////////////////////////////////
    // ChannelBaseline
    ////////////////////////////////////////////////////////
    else if(caltype==CalibrationType::ChannelBaseline) {
        for(int ch = 0; ch < 64; ch++) {
            if(ch<32) {
                ui->smx_list_widget->item(ch)->setSelected(false);
                VMMSMXBool[ch] = false;
                ui->st_list_widget->item(ch)->setSelected(false);
                VMMSTBool[ch] = false;
            }
            else {
                ui->smx_list_widget_2->item(ch-32)->setSelected(false);
                VMMSMXBool[ch] = false;
                ui->st_list_widget_2->item(ch-32)->setSelected(false);
                VMMSTBool[ch] = false;
            }
        } // ch
    }

    return;
}
// ------------------------------------------------------------------------- //
std::vector<int> MainWindow::getChannelTrims()
{
    std::vector<int> trims;
    for(int i = 0; i < 64; i++) {
        trims.push_back(VMMSDVoltage[i]->currentIndex());
    } 
    return trims;
}
// ------------------------------------------------------------------------- //

/*
//need to update startCalibration to do xADC stuff...?
void MainWindow::startCalibration()
{
    stringstream sx;

    delay();
    bool ok;
    int x = 0;
    int eventsMultiplier = 0;

    sx << " ---------------------------------- \n"
       << "  Starting calibration \n"
       << " ---------------------------------- ";
    msg()(sx); sx.str("");

    for(int g = ui->sg_cal_min->currentIndex(); g <= ui->sg_cal_max->currentIndex(); g++) {
        ui->sg->setCurrentIndex(g);
        for(int thr = ui->sdt_cal_min->value(); thr <= ui->sdt_cal_max->value();
                                                            thr += ui->sdt_cal_step->value()) {
            ui->sdt->setValue(thr);
            for(int p = ui->sdp_cal_min->value(); p <= ui->sdp_cal_max->value();
                                                            p += ui->sdp_cal_step->value()) {
                ui->sdp_2->setValue(p);

                sx << " ---------------------------------- \n"
                   << "  Calibration Loop \n"
                   << "   > Gain            : " << ui->sg->currentText().toStdString() << "\n"
                   << "   > Threshold       : " << ui->sdt->value() << "\n"
                   << "   > Pulser DAC      : " << ui->sdp_2->value() << "\n"
                   << " ----------------------------------";
                msg()(sx); sx.str("");


                // send the updated info to run_module here

                for(int ch = ui->chRange_calib_min->currentIndex();
                        ch <= ui->chRange_calib_max->currentIndex(); ch++) {
                    delay();
                    for(int chT = 0; chT < 64; chT++) {
                        VMMSM[chT]->setStyleSheet("background-color: lightGray");
                        VMMSMBool[chT]=0;
                        VMMSMBoolAll=0;
                        VMMSMBoolAll2=0;
                        VMMST[chT]->setStyleSheet("background-color: lightGray");
                        VMMSTBool[chT]=0;
                        VMMSTBoolAll=0;
                        VMMSTBoolAll2=0;
                    } //chT
                    if(ui->masking_calib->isChecked()) {
                        emit ui->SMLabel->click();
                        //emit ui->SMLabel2->click();
                        VMMSM[ch]->click();
                        if(ch>0)
                            VMMSM[ch-1]->click();
                        if(ch<63)
                            VMMSM[ch+1]->click();
                    } //masking

                    sx << " > Calibrating channel: " << ch;
                    msg()(sx);sx.str("");

                    VMMST[ch]->click();
                    sx << " --> Set mask..."; msg()(sx); sx.str("");
                    emit ui->setMask->click();
                    delay();
                    sx << " --> Set headers..."; msg()(sx); sx.str("");
                    emit ui->setEvbld->click();
                    delay();

                    sx << " --> Initialize electronics..."; msg()(sx); sx.str("");
                    emit ui->SendConfiguration->click();
                    delay();
                    emit ui->SendConfiguration->click();
                    delay();

                    sx << " --> Set trigger constants..."; msg()(sx); sx.str("");
                    emit ui->setTrgAcqConst->click();
                    delay();
                    sx << " --> Setting pulser DAC..."; msg()(sx); sx.str("");
                    emit ui->trgPulser->click();

                    int initialEvents = ui->triggerCntLabel->text().toUInt(&ok,10);

                    sx << " --> Enabling DAQ..."; msg()(sx); sx.str("");
                    emit ui->onACQ->click();

                    eventsMultiplier++;
                    unsigned int eventsForCalibration = ui->evCalib->text().toUInt(&ok,10);

                    sx << " --> Starting pulser with number of events: " << eventsForCalibration;
                    msg()(sx); sx.str("");
                    delay();

                    // I am not sure how this case can happen
                    if(QObject::sender() == ui->stopTriggerCnt)
                        break;

                    while(ui->triggerCntLabel->text().toUInt(&ok,10) - initialEvents <= eventsForCalibration){
                        sx << " * Waiting for event collecting... * "; msg()(sx); sx.str("");
                        delay();
                        delay();
                        delay();
                        delay();
                        delay();
                        if(x>10) { x = 0; break; }
                        x++;
                        if(!ui->stopTriggerCnt->isEnabled())
                            break;
                    }

                    emit ui->offACQ->click();
                    delay();
                    delay();
                    delay();

                    if(QObject::sender() == ui->stopTriggerCnt)
                        break;

                    sx << " ----------------------------------"; msg()(sx); sx.str("");

                } //ch
            } //p
        } //thr
    } //g
    sx << " ---------------------------------- \n"
       << "  Calibration finished \n"
       << " ---------------------------------- ";
    msg()(sx); sx.str("");
    emit ui->stopTriggerCnt->click();

}
*/
// ------------------------------------------------------------------------- //
void MainWindow::setRunDirOK(bool dirStatus)
{
    m_runDirectoryOK = dirStatus;
}
// ------------------------------------------------------------------------- //
void MainWindow::badRunDirectory()
{
    m_runDirectoryOK = false;

    ui->runDirectoryField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); selection-background-color: rgb(233, 99, 0); }");
    delay();
    delay();
    ui->runDirectoryField->setStyleSheet("QLineEdit { background: white; selection-background-color: rgb(233, 99, 0); }");
    delay();
    delay();
    ui->runDirectoryField->setStyleSheet("QLineEdit { background: rgb(220, 0, 0); selection-background-color: rgb(233, 99, 0); }");
    delay();
    delay();
    ui->runDirectoryField->setStyleSheet("QLineEdit { background: white; selection-background-color: rgb(233, 99, 0); }");
}
// ------------------------------------------------------------------------- //
void MainWindow::setIPConfigurationState()
{
    if(ui->ipset_begin_button == QObject::sender()) {
        if(okToChangeBoardIP()) {
            ui->ipset_groupbox->setEnabled(true);
            // don't allow any communication when configuring an ip
            toggleAllCommunication(false);
            return;
        }
        else {
            ui->ipset_groupbox->setEnabled(false);
            toggleAllCommunication(true);
            return;
        }
    }
    else if(ui->ipset_done_button == QObject::sender()) {
        ui->ipset_groupbox->setEnabled(false);
        toggleAllCommunication(true);
    }
    else if(ui->ipset_default_daqIP_button == QObject::sender()) {
        ui->ipset_daqip_0->setText("192");
        ui->ipset_daqip_1->setText("168");
        ui->ipset_daqip_2->setText("0");
        ui->ipset_daqip_3->setText("16");
    }
}
// ------------------------------------------------------------------------- //
bool MainWindow::okToChangeBoardIP()
{
    bool is_ok = true;
    QMessageBox info_box;
    info_box.setWindowTitle("Set IP Error");

    // can't configure ip if in the middle of a run
    if(m_daqInProgress) {
        is_ok = false;
        info_box.setText("You may not configure an MMFE8 IP while\na run is on going.");
    }
    // can't configure ip of board if board is in acqusition mode
    else if(ui->onACQ->isChecked()) {
        is_ok = false;
        info_box.setText("You may not configure an MMFE8 IP while\nin acquisition ON mode.");
    }
    // can only configure if we are communicating with ONE MMFE8
    else if(ui->numberOfFecs->value()>1) {
        is_ok = false;
        stringstream text;
        text << "You may only configure a single MMFE8 at a time.\nYou are set to configure"
                << " " << ui->numberOfFecs->value() << " boards.";
        info_box.setText(QString::fromStdString(text.str()));
        //info_box.setText("You may only configure a single\nMMFE8 at a time.")
    }
    // can't configure ip of board if we have not established communication with it yet
    else if(!m_commOK) {
        // once we have the ability to ping the MMFE8 we need to use that method here
        is_ok = false;
        info_box.setText("You must establish a connection with the MMFE8\nbefore configuring its IP.");
    }

    if(!is_ok) {
        info_box.exec();
    }

    return is_ok;

}
// ------------------------------------------------------------------------- //
void MainWindow::toggleAllCommunication(bool enable)
{
    ui->groupBox->setEnabled(enable);
    ui->groupBox_13->setEnabled(enable);
    ui->groupBox_8->setEnabled(enable);
    ui->groupBox_7->setEnabled(enable);
    ui->groupBox_4->setEnabled(enable);
    ui->groupBox_21->setEnabled(enable);
    ui->runGroupbox->setEnabled(enable);
}
// ------------------------------------------------------------------------- //
void MainWindow::sendIPConfiguration()
{
    std::vector<std::string> current_ip_values;
    std::vector<std::string> new_ip_values;
    std::vector<std::string> daq_ip_values;
    
    current_ip_values.push_back(ui->ipset_currentip_0->text().toStdString());
    current_ip_values.push_back(ui->ipset_currentip_1->text().toStdString());
    current_ip_values.push_back(ui->ipset_currentip_2->text().toStdString());
    current_ip_values.push_back(ui->ipset_currentip_3->text().toStdString());
    
    new_ip_values.push_back(ui->ipset_newip_0->text().toStdString());
    new_ip_values.push_back(ui->ipset_newip_1->text().toStdString());
    new_ip_values.push_back(ui->ipset_newip_2->text().toStdString());
    new_ip_values.push_back(ui->ipset_newip_3->text().toStdString());

    daq_ip_values.push_back(ui->ipset_daqip_0->text().toStdString());
    daq_ip_values.push_back(ui->ipset_daqip_1->text().toStdString());
    daq_ip_values.push_back(ui->ipset_daqip_2->text().toStdString());
    daq_ip_values.push_back(ui->ipset_daqip_3->text().toStdString());
    
    std::stringstream current_ip;
    std::stringstream new_ip;
    std::stringstream daq_ip;
    
    current_ip << current_ip_values.at(0) << "." << current_ip_values.at(1) << "."
               << current_ip_values.at(2) << "." << current_ip_values.at(3);
    new_ip << new_ip_values.at(0) << "." << new_ip_values.at(1) << "."
           << new_ip_values.at(2) << "." << new_ip_values.at(3);
    daq_ip << daq_ip_values.at(0) << "." << daq_ip_values.at(1) << "."
           << daq_ip_values.at(2) << "." << daq_ip_values.at(3);
    
    // tell the configuration module to send the command
    std::string send_status = configModule().configureBoardIP(current_ip.str(), new_ip.str(), daq_ip.str());
    
    if(send_status!="") {

        std::stringstream out_msg;
        out_msg << "ERROR " << send_status;
        std::cout << out_msg.str() << std::endl;

        ui->ipset_status_textbox->append(QString::fromStdString(out_msg.str()));
        ui->ipset_status_textbox->moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);

        ui->ipset_send_button->setStyleSheet("QPushButton { background: rgb(220, 0, 0); }");
        delay();
        delay();
        ui->ipset_send_button->setStyleSheet("QPushButton { background: white; }");
        delay();
        delay();
        ui->ipset_send_button->setStyleSheet("QPushButton { background: rgb(220, 0, 0); }");
        delay();
        delay();
        ui->ipset_send_button->setStyleSheet("QPushButton { background: white; }");
    }
    else {
        std::stringstream out_msg;
        out_msg << "-------------------------------------------------------------\n"
                << "IP configuration sent!\n       > Old IP: " << current_ip.str() << "\n"
                << "       > New IP: " << new_ip.str() << "\n"
                << "       > DAQ IP: " << daq_ip.str() << "\n"
                << "-------------------------------------------------------------";
        ui->ipset_status_textbox->append(QString::fromStdString(out_msg.str()));
        ui->ipset_status_textbox->moveCursor(QTextCursor::End, QTextCursor::MoveAnchor);
        std::cout << out_msg.str() << endl;
    }

}
// ------------------------------------------------------------------------- //
void MainWindow::send_clock_configuration()
{
    /////////////////////////////////
    // CKTK & CKBC stuff
    /////////////////////////////////
    bool ok;
    int cktk_max = ui->cktk_max_number->value();
    int ckbc_frequency = ui->ckbc_frequency->currentText().toUInt(&ok, 10);


    /////////////////////////////////
    // CKTP stuff
    /////////////////////////////////
    int n_pulses_to_send = ui->cktp_max_number->value();
    int cktp_skew_steps = ui->cktp_skew_steps->value();
    int cktp_period = ui->cktp_period->value();
    int cktp_width = ui->cktp_width->value();

    ////////////////////////////////////////
    // send the clock configuration packet
    ////////////////////////////////////////
    runModule().configure_clocks(cktk_max, ckbc_frequency, n_pulses_to_send,
                    cktp_skew_steps, cktp_period, cktp_width);

    string board_sel = ui->setVMMs->currentText().toStdString();
    if(board_sel == "All") {
        for(int i = 0; i < (int)m_clocks_set.size(); i++) {
            m_clocks_set.at(i) = 1;
        }
    } 
    else if(board_sel == "None") {
        m_clocks_set.clear();
    }
    else {
        int board_no = -1;
        try {
            board_no = std::stoi(board_sel);
        }
        catch(std::exception &e) {
            cout << "MainWindow::send_clock_configuration    Exception caught: " << e.what() << endl;
            return;
        }
        m_clocks_set.at(board_no-1) = 1;
    }
}
// ------------------------------------------------------------------------- //
void MainWindow::update_tp_skew_label(int new_idx)
{
    QString label_40MHz = "x 1ns";
    QString label_20MHz = "x 6.25ns";
    QString label_10MHz = "x 6.25ns";
    vector<QString> labels;
    labels.push_back(label_40MHz);
    labels.push_back(label_20MHz);
    labels.push_back(label_10MHz);

    ui->cktp_skew_label->setText(labels.at(new_idx)); 
    ui->calibration_cktp_skew_label->setText(labels.at(new_idx));
}
// ------------------------------------------------------------------------- //
void MainWindow::set_clock_defaults()
{
    ////////////////////////////////////////////////////////////////
    // set CKTK and CKBC defaults
    ////////////////////////////////////////////////////////////////
    if(QObject::sender() == ui->cktk_ckbt_set_defaults_button) {
        ui->cktk_max_number->setValue(7); // set max CKTK ticks to 7 (this is max CKTKs of 'trying' before sending)
        ui->ckbc_frequency->setCurrentIndex(0); // set CKBC to 40 MHz
    }

    ////////////////////////////////////////////////////////////////
    // set CKTP defaults
    ////////////////////////////////////////////////////////////////
    else if(QObject::sender() == ui->cktp_set_defaults_button) {
        ui->cktp_max_number->setValue(-1);
        ui->cktp_skew_steps->setValue(0);
        ui->cktp_period->setValue(5000);
        ui->cktp_width->setValue(4);
    }
    ////////////////////////////////////////////////////////////////
    // set defaults for all
    ////////////////////////////////////////////////////////////////
    else if(QObject::sender() == ui->clocks_set_defaults_button) {
        ui->cktk_ckbt_set_defaults_button->click();
        ui->cktp_set_defaults_button->click();
    }

}
// ------------------------------------------------------------------------- //
//void MainWindow::configureTP()
//{
//    runModule().configTP(ui->tpSkew->currentIndex(),
//                         ui->tpWidth->currentIndex(),
//                         ui->tpPolarity->currentIndex());
//}
// ------------------------------------------------------------------------- //
void MainWindow::delay()
{
    QTime dieTime = QTime::currentTime().addMSecs(200);
    while(QTime::currentTime() < dieTime)
        QCoreApplication::processEvents(QEventLoop::AllEvents,100);
}
// ------------------------------------------------------------------------- //
void MainWindow::delayMs()
{
    QTime dieTime = QTime::currentTime().addMSecs(10);
    while( QTime::currentTime() < dieTime )
        QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
}

bool MainWindow::eventFilter(QObject* /*obj*/, QEvent *event)
{
    if(event->type() == QEvent::Wheel)
    {
        //if(obj->hasFocus()) {
//            ui->scrollArea->setFocus();
        event->ignore();
        //}
    //    qDebug() << "Wheel event blocked";
        return true;
    }
    return false;
}
