// vmm
#include "calibration_module.h"
#include "bit_manip.h"

// std/stl
#include <iostream>
#include <bitset>
using namespace std;

// qt
#include <QEventLoop>
#include <QDataStream>

 //boost
 using boost::asio::ip::udp;
 namespace ip = boost::asio::ip;
 #include <boost/system/error_code.hpp>
 
 #include <QByteArray>
 #include <QUdpSocket>
 
 
#include "map_handler.h"

#include "TFile.h"
#include "TTree.h"
#include "TBranch.h"

//boost
using boost::asio::ip::udp;
namespace ip = boost::asio::ip;
#include <boost/system/error_code.hpp>

#include <QByteArray>
#include <QUdpSocket>


///////////////////////////////////////////////////////////////////////////////
// CalibModule
///////////////////////////////////////////////////////////////////////////////
CalibModule::CalibModule(QObject* parent) :
    QObject(parent),
    m_dbg(false),
    m_vmm_type(3),
    m_do_l0(false),
    m_msg(nullptr),
    m_pulser_calib(nullptr),
    n_push_back(0),
    n_empty_events_received(0),
    n_warnings(0),
    n_send_period(0),
    m_skip_dead_channels(false),
    m_vmm_id(0),
    n_samples(5000),
    m_sample_period(139810),
    m_bind_port(2227),
    m_xadc_port(6608),
    m_socket(NULL),
    m_mapHandler(NULL),
    m_mapping_setup(false),
    m_writeNtuple(true),
    m_run_number(0),
    m_calibRootFile(NULL),
    m_trees_setup(false),
    m_calib_tree(NULL),
    m_last_trigger_read(0xffffffff)
{
    m_trigger_events.clear();
    m_pulser_calib = new PulserCalib(this);
    connect(m_pulser_calib, SIGNAL(pulser_calib_complete()), this, SLOT(pulser_calib_complete()), Qt::DirectConnection);
}

void CalibModule::pulser_calib_complete()
{
    emit pulser_calib_complete_signal();
}

void CalibModule::set_vmm_type(int type)
{
    m_vmm_type = type;
    if(type==2) m_do_l0 = false;
    m_pulser_calib->set_vmm_type(type);
}

void CalibModule::LoadModulesForCalibration(Configuration& config, RunModule& run) //, SocketHandler& socket)
{
    m_pulser_calib->LoadModulesForCalibration(config, run); //, socket);
}

void CalibModule::load_pulser_calib_flags(bool* flags)
{
    m_pulser_calib->load_flags(flags);
}
bool CalibModule::load_pulser_calib(PulserCalibDef def)
{
    return m_pulser_calib->load_calibration_loop(def);
}
bool CalibModule::load_pulser_calib_run_properties(PulserCalibRunProperties props)
{
    return m_pulser_calib->load_run_properties(props);
}

void CalibModule::begin_pulser_calibration()
{
    m_calibration_type = CalibrationType::Custom;
    setup_decoding(false, false);
    bool status = m_pulser_calib->start();
    if(!status) {
        cout << "CalibModule::begin_pulser_calibration    ERROR Did not start pulser-based calibration correctly" << endl;
    }
    //return m_pulser_calib->start();
}

void CalibModule::send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& vmmMap,
        std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq)
{
    m_pulser_calib->load_initial_config(global, vmmMap, channels, clocks, daq);
}

void CalibModule::LoadMessageHandler(MessageHandler& msg)
{
    m_msg = &msg;
}
bool CalibModule::initializeSocket(boost::shared_ptr<boost::asio::io_service> service)
{
    bool init_ok = true;
    if(m_socket) {
        if(m_socket->is_open()) {
            m_socket->close();
            boost::system::error_code ec;
            m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
        }
    }
    m_io_service = service;
    m_socket = boost::shared_ptr<ip::udp::socket>(new ip::udp::socket(*m_io_service,
            ip::udp::endpoint(ip::udp::v4(), m_bind_port)));

    string ip_of_board = "192.168.0.2";
    //string ip_of_board = "127.0.0.1";
    m_remote_endpoint.address(ip::address::from_string(ip_of_board));
    //int test_port = 2224;
    m_remote_endpoint.port(m_xadc_port);
    //remote_endpoint.port(test_port);

    if(!m_socket->is_open()) {
        if(dbg()) {
            cout << "CalibModule::initializeSocket    ERROR Calibration UDP socket unable to be setup properly!" << endl;
        }
        init_ok = false;
    }
    return init_ok;
}
void CalibModule::toggleSocket()
{
    
    std::cout << "CalibModule::toggleSocket()" << std::endl;
    initializeSocket(m_io_service);
    //if(m_socket) {
    //    if(m_socket->is_open()) {
    //        m_socket->close();
    //        boost::system::error_code ec;
    //        m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
    //    }
    //}
    //m_socket = boost::shared_ptr<ip::udp::socket>(new ip::udp::socket(*m_io_service,
    //        ip::udp::endpoint(ip::udp::v4(), m_bind_port)));
}

void CalibModule::closeCalibrationSocket()
{
    //cout << "CalibModule::closeCalibrationSocket" << endl;
    if(m_socket && m_socket->is_open()) {
        m_socket->close();
        boost::system::error_code ec;
        m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
    }
}

void CalibModule::reset_counts()
{
    n_push_back = 0;
    n_empty_events_received = 0;
    n_warnings = 0;
}


void CalibModule::initialize_calibration(int vmm_to_calibrate,
            int xadc_samples, int sample_period, int type, bool skip_dead_channels, bool do_L0){

    reset_counts();

    m_vmm_id = vmm_to_calibrate;
    n_samples = xadc_samples;
    m_sample_period = sample_period;
    m_calibration_type = CalibrationTypeFromInt(type);
    m_skip_dead_channels = skip_dead_channels;

    if(vmm_type()==2) do_L0 = false;
    if(!m_trees_setup && do_xadc())
        setupOutputTrees(do_L0);
    
};

bool CalibModule::setOutputFile(std::string filename, int run_number, bool is_xadc)
{
    // for pulser-based calibrations handle the file output in PulserCalib
    if(!is_xadc) return true;

    stringstream sx;

    cout << "CalibModule::setOutputFile" << endl;
    m_calibRootFile = new TFile(filename.c_str(), "UPDATE");
    if(m_calibRootFile->IsZombie()) {
        sx << "ERROR: Calibration ROOT file unable to be created"; 
        //msg()(sx,"CalibModule::setOutputFile");
        cout << sx.str() << endl;
        delete m_calibRootFile;
        return false;
    }
    m_calibRootFile->cd();
    m_run_number = run_number;
    m_output_rootfilename = filename;
    m_trees_setup = false;
    //setupOutputTrees();
    return true;
}



void CalibModule::start_xadc_sampling(QString ip_address)
{

    if(vmm_type()==2) {
        start_xadc_sampling_VMM2(ip_address);
        return;
    }
    //cout <<"CalibModule::Start_xADC_Sampling starting xADC sampling"<<endl;

    // set the IP address to send the command to
    m_remote_endpoint.address(ip::address::from_string(ip_address.toStdString()));
    //remote_endpoint.port(m_xadc_port);

    if(!m_io_service) {
        msg()("IO service is null, cannot send xADC sampling command","CalibModule::Start_xADC_Sampling");
        return;
    }
    if(m_io_service->stopped()) {
        msg()("IO service has stopped, cannot send xADC sampling command","CalibModule::Start_xADC_Sampling");
        return;
    }
    if(!m_socket->is_open()) {
        if(dbg()) msg()("Initialization UDP socket for data taking","CalibModule::Start_xADC_Sampling");
        initializeSocket(m_io_service);
    }
    bool ok = true;

    uint32_t first_32  = 0xAAFF; // for xADC the serial number is not important
    uint32_t second_32 = 0xAAFF; // current FW does not use the second 32 bit word

    first_32 = bits::reverse_octet_order()(first_32);
    second_32 = bits::reverse_octet_order()(second_32);

    ///////////////////////////////////////////////////////
    // addresses
    ///////////////////////////////////////////////////////
    uint32_t address_vmmid = 0xA1;
    uint32_t address_sample_size = 0xA2;
    uint32_t address_period = 0xA3;

    address_vmmid = bits::reverse_octet_order()(address_vmmid);
    address_sample_size = bits::reverse_octet_order()(address_sample_size);
    address_period = bits::reverse_octet_order()(address_period);
    

    ///////////////////////////////////////////////////////
    // get the xADC parameters
    ///////////////////////////////////////////////////////
    vector<uint32_t> configuration_packet;
    uint32_t vmm_id = 0;
    vmm_id |= (m_vmm_id);
    //vmm_id = bits::reverse_32()(vmm_id);
    uint32_t samples = 0;
    samples |= n_samples;
    //samples = bits::reverse_32()(samples);
    uint32_t period = 0;
    period |= m_sample_period;
    //period = bits::reverse_32()(period);

    vmm_id = bits::reverse_octet_order()(vmm_id);
    samples = bits::reverse_octet_order()(samples);
    period = bits::reverse_octet_order()(period);

    ///////////////////////////////////////////////////////
    // build up the packet
    ///////////////////////////////////////////////////////

    // header
    configuration_packet.push_back(first_32);
    configuration_packet.push_back(second_32);

    // vmm id
    configuration_packet.push_back(address_vmmid);
    configuration_packet.push_back(vmm_id);

    // sample size
    configuration_packet.push_back(address_sample_size);
    configuration_packet.push_back(samples);

    // period
    configuration_packet.push_back(address_period);
    configuration_packet.push_back(period);

    try {
        size_t n_bytes_sent;
        boost::this_thread::sleep(boost::posix_time::milliseconds(100));
        int current = n_push_back;
        for(int i = 0; i < 1; i++) {
            n_bytes_sent = m_socket->send_to(boost::asio::buffer(configuration_packet), m_remote_endpoint);
        }
    }
    catch(boost::system::system_error err) {
        cout << "CalibModule::Start_xADC_Sampling    Exception caught: " << err.what() << endl;
        ok = false;
    }
    return;
};

void CalibModule::start_xadc_sampling_VMM2(QString ip_address)
{
    m_remote_endpoint.address(ip::address::from_string(ip_address.toStdString()));

    if(!m_io_service) {
        msg()("IO service is null, cannot send xADC sampling command","CalibModule::start_xadc_sampling_VMM2");
        return;
    }
    if(m_io_service->stopped()) {
        msg()("IO service has stopped, cannot send xADC sampling command","CalibModule::start_xadc_sampling_VMM2");
        return;
    }
    if(!m_socket->is_open()) {
        if(dbg()) msg()("Initialization UDP socket for data taking","CalibModule::start_xadc_sampling_VMM2");
        initializeSocket(m_io_service);
    }

    bool ok = true;

    vector<uint32_t> configuration_packet;
    uint32_t vmm_id = 0x0;
    vmm_id |= (m_vmm_id);
    uint32_t samples = 0x0;
    samples |= n_samples;
    uint32_t period = 0x0;
    period |= m_sample_period;

    vmm_id = bits::reverse_octet_order()(vmm_id);
    samples = bits::reverse_octet_order()(samples);
    period = bits::reverse_octet_order()(period);

    configuration_packet.push_back(vmm_id);
    configuration_packet.push_back(samples);
    configuration_packet.push_back(period);

    try {
        size_t n_bytes_sent;
        boost::this_thread::sleep(boost::posix_time::milliseconds(100));
        n_bytes_sent = m_socket->send_to(boost::asio::buffer(configuration_packet), m_remote_endpoint);
        //n_bytes_sent = m_socket->send_to(boost::asio::buffer(configuration_packet), m_remote_endpoint);
    } // try
    catch(boost::system::system_error err) {
        cout << "CalibModule::start_xadc_sampling_VMM2    Exception caught: " << err.what() << endl;
        ok = false;
    }

    return;
}

bool CalibModule::do_xadc()
{
    bool do_xadc_decoding = (calibType()==CalibrationType::ChannelTrim
                            || calibType()==CalibrationType::GlobalThresholdDAC
                            || calibType()==CalibrationType::GlobalTestPulseDAC
                            || calibType()==CalibrationType::ChannelBaseline);
    return do_xadc_decoding;
}

void CalibModule::decode_data(std::string& ip_string, std::vector<uint32_t>& datagram, uint32_t length, int& counter)
{
    bool do_xadc_decoding = (calibType()==CalibrationType::ChannelTrim
                            || calibType()==CalibrationType::GlobalThresholdDAC
                            || calibType()==CalibrationType::GlobalTestPulseDAC
                            || calibType()==CalibrationType::ChannelBaseline);

    bool do_vmm_decoding = (calibType()==CalibrationType::BaselineNeighbor
                            || calibType()==CalibrationType::BaselineExtrapolation
                            || calibType()==CalibrationType::Custom
                            || calibType()==CalibrationType::Time
                            || calibType()==CalibrationType::Efficiency);
    //if(calibType()==CalibrationType::ChannelTrim || calibType()==CalibrationType::GlobalThresholdDAC
    //                            || calibType()==CalibrationType::GlobalTestPulseDAC)

    if(do_xadc_decoding && !do_vmm_decoding)
        decode_xadc_sampling(ip_string, datagram, counter);

    //else if(calibType()==CalibrationType::BaselineNeighbor || calibType()==CalibrationType::BaselineExtrapolation
    //                                                       || calibType()==CalibrationType::Custom
    //                                                       || calibType()==CalibrationType::Time)
    else if(!do_xadc_decoding && do_vmm_decoding) {
        //decode_calib_data(ip_string, datagram, counter);
        //decode_calib_data_trig(ip_string, datagram, counter);
        m_pulser_calib->decode_pulser_data(ip_string, datagram, counter);

    }
    else {
        m_sx.str("");
        m_sx << "ERROR CalibrationType error (CalibrationType==" << CalibrationType2Str(calibType()) << ")"
              << " not decoding calibration data!"; 
        msg()(m_sx, "CalibModule::decode_data"); m_sx.str("");
        return;
    }
}

bool CalibModule::valid_xadc_sample(std::vector<uint32_t>& datagram)
{
    bool valid = true;
    if(vmm_type()==3) {
        if(datagram.size()>=2) {
            if(VMMDecoder::is_configuration_reply(datagram)) { valid = false; }
            valid = true;
        }
    }
    else if(vmm_type()==2) {
        bool found_non_empty = false;
        for(const auto & x : datagram) {
            if(!(x==0xffffffff) && !(x==0x0)) { found_non_empty = true; break; }
        }
        valid = found_non_empty;
    }
    return valid;
}

void CalibModule::decode_xadc_sampling(std::string& ip_string, std::vector<uint32_t>& datagram, int& counter)
{
    if(!valid_xadc_sample(datagram)) return;
    //cout << "CalibModule::decode_xadc_sampling    0x";
    //for(auto x : datagram) cout << std::hex << x;
    //cout << endl;

	// board id method
    std::string boardId = "";
    int board_id = -1;
    if(mapping() && mapHandler().mapLoaded()) {
	    board_id = mapHandler().boardIDfromIP(ip_string);
	    try{
	        std::stringstream b_id;
	        b_id << board_id;
	        boardId = b_id.str();
	    } // try
	    catch(std::exception& e) {
	        std::cout << "CalibModule::decode_xadc_sampling    Unable to convert board id (" << board_id << ") to string." << std::endl;
	        boardId="";
	    }//catch
    } // if mapping
    else {
        try {
            boardId = ip_string.substr(ip_string.find_last_of('.')+1);
            board_id = std::stoi(boardId);
        }
        catch(std::exception& e) {
            std::cout << "CalibModule::decode_xadc_sampling    Unable to get board id from board ip "
                << ip_string << endl;
            boardId = "";
        } // catch
    }
    
    std::vector<uint32_t> datagram_vector(datagram.begin(), datagram.end());
    std::vector<uint32_t> xadc_samples;

    //for loop over all data
    for(int i = 0; i<(int)datagram_vector.size();i+=2) {

        bool decode_ok = m_decoder.decode(std::vector<uint32_t>(datagram_vector.begin()+i, datagram_vector.begin()+i+2), false);
        //uint32_t frame_counter = datagram_vector.at(i);
        //if(frame_counter == 0xffffffff) break;
        if(!decode_ok) break;
        //if(m_decoder.frame() == 0xffffffff) break;

        m_chipid = m_decoder.chip_number();
        //cout << "Calib Calib  chip = " << m_chipid << endl;
        m_boardid = (boardId=="" ? -1 : std::stoi(boardId));
        //xadc_samples.clear();
        //for(auto x : m_decoder.xadc_samples())
        //    xadc_samples.push_back(x);
        for(auto sample : m_decoder.xadc_samples()) {
            fill_event(m_decoder.chip_number(), m_boardid, sample);
        }

        /*
        

        uint32_t frame_counter = datagram_vector.at(i);
        if(frame_counter == 0xffffffff) break;
        xadc_samples.clear();

        //uint32_t vmm_id_32 = (datagram_vector.at(i) & 0x000000f0) >> 4;
        //uint32_t read1_32 = ((datagram_vector.at(i) & 0x0000000f) << 8 | (datagram_vector.at(i) & 0x0000ff00) >> 8);
        //uint32_t read2_32 = ((datagram_vector.at(i) & 0xf0000000) >> 28 | (datagram_vector.at(i) & 0x00ff0000) >> 12) ;
        //uint32_t read3_32 = ((datagram_vector.at(i) & 0x0f000000) >> 16 | (datagram_vector.at(i+1) & 0x0000000ff));
        //uint32_t read4_32 = ((datagram_vector.at(i+1) & 0x0000ff00) >> 4 | (datagram_vector.at(i+1) & 0x00f00000) >> 20);
        //uint32_t read5_32 = ((datagram_vector.at(i+1) & 0xff000000) >> 24 | (datagram_vector.at(i+1) & 0x000f0000) >> 8);

        uint32_t vmm_id = (datagram_vector.at(i) & 0xf0) >> 4;
        uint32_t sample_0 = ( (datagram_vector.at(i) & 0xf) << 8 | (datagram_vector.at(i) & 0xff00) >> 8);
        uint32_t sample_1 = ( (datagram_vector.at(i) & 0xf0000000) >> 28 | (datagram_vector.at(i) & 0xff0000) >> 12);
        uint32_t sample_2 = ( (datagram_vector.at(i) & 0xf000000) >> 16 | (datagram_vector.at(i+1) & 0xff));
        uint32_t sample_3 = ( (datagram_vector.at(i+1) & 0xff00) >> 4 | (datagram_vector.at(i+1) & 0xf00000) >> 20); 
        uint32_t sample_4 = ( (datagram_vector.at(i+1) & 0xff000000) >> 24 | (datagram_vector.at(i+1) & 0xf0000) >> 8);


        //add the xadc data for creating the Ntuples
        m_chipid = vmm_id; 
        m_boardid = (boardId=="" ? -1 : std::stoi(boardId));

        xadc_samples.push_back(sample_0);
        xadc_samples.push_back(sample_1);
        xadc_samples.push_back(sample_2);
        xadc_samples.push_back(sample_3);
        xadc_samples.push_back(sample_4);

        //xadc_samples.push_back(m_decoder.xadc_samples()[0]);
        //xadc_samples.push_back(m_decoder.xadc_samples()[1]);
        //xadc_samples.push_back(m_decoder.xadc_samples()[2]);
        //xadc_samples.push_back(m_decoder.xadc_samples()[3]);
        //xadc_samples.push_back(m_decoder.xadc_samples()[4]);
        

        //if(m_decoder.xadc_samples()[0] != sample_0) cout << "SAMPLE MISMATCH : 0" << endl;
        //if(m_decoder.xadc_samples()[1] != sample_1) cout << "SAMPLE MISMATCH : 1" << endl;
        //if(m_decoder.xadc_samples()[2] != sample_2) cout << "SAMPLE MISMATCH : 2" << endl;
        //if(m_decoder.xadc_samples()[3] != sample_3) cout << "SAMPLE MISMATCH : 3" << endl;
        //if(m_decoder.xadc_samples()[4] != sample_4) cout << "SAMPLE MISMATCH : 4" << endl;
        
         for (auto sample : xadc_samples){
             fill_event(m_chipid, m_boardid, sample);
         }
        */
        
        // increment counter for each sample taken
        // this assumes that the xADC provides samples in multiples of 5
        n_push_back = n_push_back+5;
        counter = n_push_back;
    }//for loop
}

void CalibModule::decode_calib_data(std::string& ip_string, std::vector<uint32_t>& datagram, int& counter)
{

    // do not decode VMM SPI configuration replies from the FPGA
    if(datagram.size()>=2) {
        if(VMMDecoder::is_configuration_reply(datagram))
            return; 
    }

    size_t n_bytes_read = 0;

    //////////////////////////////////////
    // containers for the tree
    //////////////////////////////////////
    std::vector<int> _chipId_tree;
    std::vector<int> _boardId_tree;
    std::vector< std::vector<int> > _channelId_tree;
    std::vector< std::vector<int> > _pdo_tree;
    std::vector< std::vector<int> > _tdo_tree;
    std::vector< std::vector<int> > _threshold_tree;
    std::vector<int> _eventSize_tree;
    std::vector< std::vector<int> > _flag_tree;
    std::vector< std::vector<int> > _bcid_tree;
    std::vector< std::vector<int> > _grayDecoded_tree;
    std::vector< std::vector<int> > _neighbor_tree;

    std::string boardId = "";
    int board_id = -1;
    if(mapping() && mapHandler().mapLoaded()) {
	    board_id = mapHandler().boardIDfromIP(ip_string);
	    try{
	        std::stringstream b_id;
	        b_id << board_id;
	        boardId = b_id.str();
	    } // try
	    catch(std::exception& e) {
	        std::cout << "CalibModule::decode_calib_data    Unable to convert board id (" << board_id << ") to string." << std::endl;
	        boardId="";
	    }//catch
    } // mapping
    else {
        try {
            boardId = ip_string.substr(ip_string.find_last_of('.')+1);
            board_id = std::stoi(boardId);
        }
        catch(std::exception& e) {
            std::cout << "CalibModule::decode_calib_data    Unable to get board id from board ip "
                << ip_string << endl;
            boardId = "";
        } // catch
    }

    std::vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.end());
    std::vector<uint32_t> datagram_vector;

    for(const auto& data : datagram_vector_tmp) {
        datagram_vector.push_back(bits::endian_swap32()(data));
    }

    bool continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin(), datagram_vector.begin()+3), true);

    while(true) {
        //if(frame_counter == 0xffffffff) break;
        //if(m_decoder.frame() == 0xffffffff) break;
        if(!continue_decoding) break;

        //////////////////////////////////////
        // data container per chip
        //////////////////////////////////////
        std::vector<int> _tdo;
        std::vector<int> _pdo;
        std::vector<int> _channelId;
        std::vector<int> _threshold;
        std::vector<int> _flag;
        std::vector<int> _bcid;
        std::vector<int> _gray;
        std::vector<int> _neighbor;


        //uint32_t trig_counter = datagram_vector.at(0);
        //int32_t vmm_id = datagram_vector.at(1) & 0xff;

            int n_step = (m_do_l0 ? 1 : 2);
            for(int i = 3; i < (int)datagram_vector.size(); i+=n_step ) {
                continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin()+i, datagram_vector.begin()+i+n_step), false);
                //if(m_decoder.frame() == 0xffffffff) break;
                if(!continue_decoding) break;

                // register non-empty packets
                n_bytes_read += (m_do_l0 ? 1*4 : 2*4);

                // pdo
                _pdo.push_back(m_decoder.pdo());

                // tdo
                _tdo.push_back(m_decoder.tdo());

                // gray code
                _bcid.push_back(m_decoder.bcid());

                // decoded gray code
                _gray.push_back(m_decoder.decoded_bcid());

                if(!m_do_l0) {
                    // flag
                    _flag.push_back(m_decoder.flag());

                    // threshold
                    _threshold.push_back(m_decoder.pass_threshold());
                }

                // vmm channel
                _channelId.push_back(m_decoder.vmm_channel());

                // neighbor
                if(m_calib_state.channel < 0) {
                    _neighbor.push_back(0);
                }
                else {
                    _neighbor.push_back(!(static_cast<int>(m_calib_state.channel)==static_cast<int>(m_decoder.vmm_channel())));
                }

                // mapping
                int32_t mapped_channel = -1;
                int32_t feb_channel_no = -1;
                if(mapping() && mapHandler().mapLoaded() && n_bytes_read>0) {
                    if(board_id>=0) {
                        int chip_no = m_decoder.chip_number();
                        int vmm_chan = m_decoder.vmm_channel();
                        mapped_channel = mapHandler().elementNumber(board_id, chip_no, vmm_chan); 
                        feb_channel_no = mapHandler().boardChannel(board_id, chip_no, vmm_chan); 
                    } // valid board id
                } // mapping
            } // i

        // fill
        if(writeNtuple() && n_bytes_read > 0) {
            _chipId_tree.push_back(m_decoder.chip_number());
            _boardId_tree.push_back( (boardId=="" ? -1 : std::stoi(boardId)));

            _tdo_tree.push_back(_tdo);
            _pdo_tree.push_back(_pdo);
            _threshold_tree.push_back(_threshold);
            _channelId_tree.push_back(_channelId);
            _flag_tree.push_back(_flag);
            _bcid_tree.push_back(_bcid);
            _grayDecoded_tree.push_back(_gray);
            _eventSize_tree.push_back(n_bytes_read);
            _neighbor_tree.push_back(_neighbor);
        }
    } // while

    //if(frame_counter == 0xffffffff) {
    //if(m_decoder.frame() == 0xffffffff) {
    if(!continue_decoding) {
        if(writeNtuple() && (n_bytes_read>0 || calibType()==CalibrationType::Efficiency)) {

            // clear the global tree variables
            m_chipId.clear();
            m_boardId.clear();
            m_pdo.clear();
            m_tdo.clear();
            m_channelId.clear();
            m_threshold.clear();
            m_eventSize.clear();
            m_flag.clear();
            m_bcid.clear();
            m_grayDecoded.clear();
            //m_channelTrims.clear();
            m_neighbor_calib.clear();

            // assign the tree variables
            m_chipId = _chipId_tree;
            m_boardId = _boardId_tree;
            m_tdo = _tdo_tree;
            m_pdo = _pdo_tree;
            m_channelId = _channelId_tree;
            m_threshold = _threshold_tree;

            m_flag = _flag_tree;
            m_bcid = _bcid_tree;
            m_grayDecoded = _grayDecoded_tree;
            m_eventSize = _eventSize_tree;

            m_neighbor_calib = _neighbor_tree;


            // now fill the event
            fill_event();

            n_push_back++;
            //n_push_back += 64;
            counter = n_push_back;
        }

        if(!writeNtuple() && (n_bytes_read>0 || calibType()==CalibrationType::Efficiency)) {
            n_push_back++;
            counter = n_push_back;
        }

        if(!(n_bytes_read>0) && !(calibType()==CalibrationType::Efficiency)) {
            n_empty_events_received++;

            // channel may be dead
            if( (n_empty_events_received > 500) && m_skip_dead_channels) {
                // add enough that is sure to be more than any event sampling per
                // calibration loop
                n_push_back+=5000; 
                counter = n_push_back;
            }
        }

    } // if frame == 0xffffffff

    return;
}

void CalibModule::decode_calib_data_trig(std::string& ip_string, std::vector<uint32_t>& datagram,
    int& counter)
{
    if(datagram.size()>=2) {
        if(VMMDecoder::is_configuration_reply(datagram)) return;
    }

    size_t n_bytes_read = 0;
    if(m_trigger_events.ready_to_read()) {
        fill_trigger_event(counter/*,trigcounter*/);
    }

    std::string boardId = "";
    int board_id = -1;
    if(mapping() && mapHandler().mapLoaded()) {
	    board_id = mapHandler().boardIDfromIP(ip_string);
	    try{
	        std::stringstream b_id;
	        b_id << board_id;
	        boardId = b_id.str();
	    } // try
	    catch(std::exception& e) {
	        std::cout << "CalibModule::decode_calib_data    Unable to convert board id (" << board_id << ") to string." << std::endl;
	        boardId="";
	    }//catch
    } // mapping
    else {
        try {
            boardId = ip_string.substr(ip_string.find_last_of('.')+1);
            board_id = std::stoi(boardId);
        }
        catch(std::exception& e) {
            std::cout << "CalibModule::decode_calib_data    Unable to get board id from board ip "
                << ip_string << endl;
            boardId = "";
        } // catch
    }

    std::vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.end());
    std::vector<uint32_t> datagram_vector;
    for(const auto& data : datagram_vector_tmp) {
        datagram_vector.push_back(bits::endian_swap32()(data));
    }

    bool continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin(), datagram_vector.begin()+vmm_type()), true);

    while(true) {
        if(!continue_decoding) break;

        size_t trigger_count_pos = 1;
        if(vmm_type()==2) trigger_count_pos = 0;

        uint32_t trigger_count = datagram_vector.at(trigger_count_pos);

        VMMHit vmm_hit;
        int n_step = (m_do_l0 ? 1 : 2);
        if(vmm_type()==2) n_step = 2;

        for(unsigned int i = vmm_type(); i < datagram_vector.size(); i+=n_step) {
            continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin() + i,
                datagram_vector.begin() + i + n_step), false);

            if(!continue_decoding) break;

            vmm_hit.clear();

            vmm_hit.set_vmm_channel(m_decoder.vmm_channel());
            vmm_hit.set_pdo(m_decoder.pdo());
            vmm_hit.set_tdo(m_decoder.tdo());
            vmm_hit.set_graycode_bcid(m_decoder.bcid());
            vmm_hit.set_decoded_graycode_bcid(m_decoder.decoded_bcid());
            if(m_do_l0) {
                vmm_hit.set_relative_bcid(m_decoder.relbcid());
                vmm_hit.set_overflow_bit(m_decoder.vmm_overflow());
                vmm_hit.set_orbit_counter(m_decoder.orbit_counter());
            }
            else {
                vmm_hit.set_flag(m_decoder.flag());
                vmm_hit.set_pass_threshold(m_decoder.pass_threshold());
            }

            n_bytes_read += n_step * 4;

            vmm_hit.set_board_id(board_id);

            // store
            int32_t chip_no = m_decoder.chip_number();
            m_trigger_events.add_hit2(trigger_count, board_id, chip_no, vmm_hit);

        } // i

    } // while

}

void CalibModule::fill_trigger_event(int& hitcounts)
{
    bool trig_ok = true;
    uint32_t trigger_to_readout = m_trigger_events.trigger_to_read(trig_ok);
    if(!trig_ok) return;

    /*
    n_triggers++
    trigcounts = n_triggers
    */

    std::map<BoardChipPair, std::vector<VMMHit> > hitgroup = m_trigger_events.get_trigger2(trigger_to_readout);

    if(hitgroup.size()==0) {
        cout << "CalibModule::fill_trigger_event    WARNING Hit group empty for "
            << "trigger " << trigger_to_readout << " (0x" << std::hex << trigger_to_readout 
            << std::dec << ")" << endl;
        return;
    }

    bool write = writeNtuple();

    std::vector<int> board_id_tree;
    std::vector<int> chip_id_tree;
    std::vector<int> eventsize_tree;
    std::vector<int> trigcounter_tree;

    std::vector< std::vector<int> > vmm_channel_tree;
    std::vector< std::vector<int> > pdo_tree;
    std::vector< std::vector<int> > tdo_tree;
    std::vector< std::vector<int> > flag_tree;
    std::vector< std::vector<int> > threshold_tree;
    std::vector< std::vector<int> > bcid_tree;
    std::vector< std::vector<int> > graydecoded_tree;
    std::vector< std::vector<int> > relbcid_tree;
    std::vector< std::vector<int> > overflow_tree;
    std::vector< std::vector<int> > orbit_counter_tree;
    std::vector< std::vector<int> > neighbor_tree;

    std::vector<int> vmm_channel;
    std::vector<int> pdo;
    std::vector<int> tdo;
    std::vector<int> flag;
    std::vector<int> threshold;
    std::vector<int> bcid;
    std::vector<int> graydecoded;
    std::vector<int> relbcid;
    std::vector<int> overflow;
    std::vector<int> orbit_counter;
    std::vector<int> neighbor;

    size_t n_bytes_read = 0;
    for(const auto group : hitgroup) {
        BoardChipPair bcpair = group.first;    

        if(write) {
            vmm_channel.clear();
            pdo.clear();
            tdo.clear();
            flag.clear();
            threshold.clear();
            bcid.clear();
            graydecoded.clear();
            relbcid.clear();
            overflow.clear();
            orbit_counter.clear();
            neighbor.clear();
        } // write

            //n_push_back++;
            //hitcounts = n_push_back;
        std::vector<VMMHit> hits = hitgroup[bcpair];
        for(auto hit : hits) {
            ///n_hits++;
            ///hitcounts = n_hits;
            //hitcounts++;
            n_push_back++;
            hitcounts = n_push_back;

            if(write) {
                vmm_channel.push_back(hit.get_vmm_channel());
                pdo.push_back(hit.get_pdo());
                tdo.push_back(hit.get_tdo());
                flag.push_back(hit.get_flag());
                threshold.push_back(hit.get_pass_threshold());
                bcid.push_back(hit.get_graycode_bcid());
                graydecoded.push_back(hit.get_decoded_graycode_bcid());
                relbcid.push_back(hit.get_relative_bcid());
                overflow.push_back(hit.get_overflow_bit());
                orbit_counter.push_back(hit.get_orbit_counter());
            } // write

            int multiplier = (m_do_l0 ? 1 : 2);
            if(vmm_type()==2) multiplier = 2;
            n_bytes_read += multiplier * 4;
        } // hit

        if(n_bytes_read>0 && write) {
            trigcounter_tree.push_back(trigger_to_readout);
            board_id_tree.push_back(bcpair.first);
            chip_id_tree.push_back(bcpair.second);
            eventsize_tree.push_back(n_bytes_read);

            vmm_channel_tree.push_back(vmm_channel);
            pdo_tree.push_back(pdo);
            tdo_tree.push_back(tdo);
            flag_tree.push_back(flag);
            threshold_tree.push_back(threshold);
            bcid_tree.push_back(bcid);
            graydecoded_tree.push_back(graydecoded);
            relbcid_tree.push_back(relbcid);
            overflow_tree.push_back(overflow);
            orbit_counter_tree.push_back(orbit_counter);
        }
    } // gruop

    if(n_bytes_read>0 && write) {
        m_chipId.clear();
        m_boardId.clear();
        m_triggerCounter.clear();
        m_eventSize.clear();

        m_channelId.clear();
        m_pdo.clear();
        m_tdo.clear();
        m_flag.clear();
        m_threshold.clear();
        m_bcid.clear();
        m_grayDecoded.clear();
        m_relbcid.clear();
        m_overflow.clear();
        m_orbit_count.clear();

        m_chipId = chip_id_tree;
        m_boardId = board_id_tree;
        m_triggerCounter = trigcounter_tree;
        m_eventSize = eventsize_tree;

        m_channelId = vmm_channel_tree;
        m_pdo = pdo_tree;
        m_tdo = tdo_tree;
        m_flag = flag_tree;
        m_threshold = threshold_tree;
        m_bcid = bcid_tree;
        m_grayDecoded = graydecoded_tree;
        m_relbcid = relbcid_tree;
        m_overflow = overflow_tree;
        m_orbit_count = orbit_counter_tree;

        ///////////////////////////////////////
        // fill the ntuple branches
        ///////////////////////////////////////
        fill_event();
    }

    m_last_trigger_read = trigger_to_readout;
    m_trigger_events.remove_trigger(trigger_to_readout);

}

void CalibModule::setup_decoding(bool is_xadc, bool do_l0)
{
    m_decoder.clear(true); // clear data + header
    m_decoder.set_debug(dbg());

    int type_to_set = VMMDecoder::VMMEVENT;
    if(vmm_type()==2) do_l0 = false;

    m_do_l0 = false;
    
    if(is_xadc && !do_l0) {
        type_to_set = VMMDecoder::VMMXADC;
        m_do_l0 = false;
    }
    else if(!is_xadc && (do_l0 || vmm_type()==2)) {
        if(vmm_type()==2) {
            type_to_set = VMMDecoder::VMM2EVENT;
            m_do_l0 = false;
        }
        else if(vmm_type()==3) {
            type_to_set = VMMDecoder::VMMEVENTL0;
            m_do_l0 = true;
        }
    }
    //int type_to_set = ( is_xadc ? VMMDecoder::VMMXADC : VMMDecoder::VMMEVENT );
    m_decoder.set_type(type_to_set); 
}

void CalibModule::setupOutputTrees(bool do_l0_decoding)
{
    m_calib_tree = new TTree("calib", "calib");
    
    br_runNumber            = m_calib_tree->Branch("runNumber", &m_calib_state.run_number);
    br_gain                 = m_calib_tree->Branch("gain", &m_calib_state.gain); 
    br_tacSlope             = m_calib_tree->Branch("tacSlope", &m_calib_state.tac_slope);
    br_peakTime             = m_calib_tree->Branch("peakTime", &m_calib_state.peak_time);
    br_dacCounts            = m_calib_tree->Branch("dacCounts", &m_calib_state.dac_counts);
    br_tpSkew               = m_calib_tree->Branch("tpSkew", &m_calib_state.tp_skew);
    br_pulserCounts         = m_calib_tree->Branch("pulserCounts", &m_calib_state.test_pulse);
    br_subhysteresis        = m_calib_tree->Branch("subhysteresis", &m_calib_state.subhysteresis);
    br_neighbortrigger      = m_calib_tree->Branch("neighbor_trigger", &m_calib_state.neighbor_trigger);
    br_nExpected            = m_calib_tree->Branch("nExpected", &m_calib_state.n_expected_pulses);
    
    bool is_xadc = (calibType()==CalibrationType::ChannelTrim
                    || calibType()==CalibrationType::GlobalThresholdDAC
                    || calibType()==CalibrationType::GlobalTestPulseDAC
                    || calibType()==CalibrationType::ChannelBaseline);

    // initialize decoding
    setup_decoding(is_xadc, do_l0_decoding);

    
    if(is_xadc) {

        br_vmm_channel  = m_calib_tree->Branch("channel", &m_channel);
        br_boardid      = m_calib_tree->Branch("boardId", &m_boardid);
        br_chipid       = m_calib_tree->Branch("chip", &m_chipid);

        br_xadc_samples = m_calib_tree->Branch("samples", &m_xadc_samples);
        br_channelTrims = m_calib_tree->Branch("channelTrim", &m_channel_trim);
    }
    else if(!is_xadc) {

        br_chipId           = m_calib_tree->Branch("chip", "std::vector<int>", &m_chipId);
        br_boardid          = m_calib_tree->Branch("boardId", "std::vector<int>", &m_boardId);
        br_channelId        = m_calib_tree->Branch("channel", "std::vector< vector<int> >", &m_channelId);

        br_pdo              = m_calib_tree->Branch("pdo", "std::vector< vector<int> >", &m_pdo);
        br_tdo              = m_calib_tree->Branch("tdo", "std::vector< vector<int> >", &m_tdo);

        br_evSize           = m_calib_tree->Branch("eventSize", "std::vector<int>", &m_eventSize);
        br_thresh           = m_calib_tree->Branch("threshold","std::vector< vector<int> >", &m_threshold);
        br_flag             = m_calib_tree->Branch("flag", "std::vector< vector<int> >", &m_flag);
        br_bcid             = m_calib_tree->Branch("bcid", "std::vector< vector<int> >", &m_bcid);
        br_grayDecoded      = m_calib_tree->Branch("grayDecoded", "std::vector< vector<int> >", &m_grayDecoded);
        br_neighborCalib    = m_calib_tree->Branch("isNeighbor", "std::vector< vector<int> >", &m_neighbor_calib);
    }
    m_trees_setup = true;
}
void CalibModule::fill_event(uint32_t vmmid, int32_t board_id, uint32_t sample)
{
    if(sample==0) return;
    
    m_calibRootFile->cd();

    // get the xadc sample
    m_xadc_samples = sample;

    // VMM channel we are currently sampling
    m_channel = calibrationState().channel;

    // threshold trim for current channel
    try {
        m_channel_trim = calibrationState().channel_trims.at(m_channel);
    }
    catch(std::exception& e) {
        if(n_warnings<30) {
            cout << "CalibrationModule::fill_event    WARNING [" << (n_warnings+1) << "/30] Unable to store channel trim for "
                    << "board: " << board_id << "  chip: " << vmmid << "  channel: " << m_channel << endl;
            n_warnings++;
        }
    }
    m_calib_tree->Fill();
    clear_data();
}

uint32_t CalibModule::decodeGray(uint32_t gray)
{
    uint32_t mask;
    for( mask = gray >> 1; mask != 0; mask = mask >> 1) {
        gray = gray ^ mask;
    }
    return gray;
}

void CalibModule::fill_event()
{
    if(writeNtuple()) {
        m_calibRootFile->cd();
        m_calib_tree->Fill();
    }
}

void CalibModule::load_calibration_state(int channel, int gain, int run_number,
            int tac_slope, int neighbor_trigger, int subhyst, int peak_time, 
            int test_pulse_dac, int thresh_dac, std::vector<int> trims, double tp_skew,
            int n_expected_pulses)
{
    m_calib_state.clear();
    m_calib_state.channel                   = channel;
    m_calib_state.gain                      = gain;
    m_calib_state.run_number                = run_number;
    m_calib_state.tac_slope                 = tac_slope;
    m_calib_state.neighbor_trigger          = neighbor_trigger;
    m_calib_state.subhysteresis             = subhyst;
    m_calib_state.peak_time                 = peak_time;
    m_calib_state.test_pulse                = test_pulse_dac;
    m_calib_state.dac_counts                = thresh_dac;
    m_calib_state.channel_trims             = trims;
    m_calib_state.tp_skew                   = tp_skew; 
    m_calib_state.n_expected_pulses         = n_expected_pulses;

    if(dbg())
        m_calib_state.print();
}

void CalibModule::clear_data()
{
    // xadc
    m_channel = -1;
    m_xadc_samples = -1;
    m_channel_trim = 0;
    //m_channelTrims.clear();

    // non-xadc
    m_chipId.clear();
    m_boardId.clear();
    m_channelId.clear();
    m_pdo.clear();
    m_tdo.clear();
    m_threshold.clear();
    m_eventSize.clear();
    m_flag.clear();
    m_bcid.clear();
    m_grayDecoded.clear();
    m_neighbor_calib.clear();
}
void CalibModule::write_output(){
    m_sx.str("");

    if(!do_xadc()) {
        m_pulser_calib->write_output();
        return;
    }

    bool write_ok = true;
    m_calibRootFile->cd();
    m_calib_tree->Write("", TObject::kOverwrite);
    if(!m_calibRootFile->Write()) {
        write_ok = false;
        cout << "ERROR Writing output calib ROOT file" << endl;
    }

    TNamed user_comment_branch("comments", m_run_comment.c_str());
    user_comment_branch.Write();

    m_calibRootFile->Close();
    
    if(write_ok) {
        m_sx << "Calibration run " << m_run_number << " stored in file: "
            << m_output_rootfilename;
        //msg()(m_sx, "CalibModule::write_output"); m_sx.str("");
        cout << m_sx.str() << endl;
    }
// m_calib_tree->Close();
}
