// vmm
#include "run_module.h"
#include "data_handler.h"

// std/stl
#include <math.h>
#include <iostream>
using namespace std;

// qt
#include <QDataStream>
#include <QTimer>
#include <QDataStream>

//////////////////////////////////////////////////////////////////////////////
// ------------------------------------------------------------------------ //
//  RunModule
// ------------------------------------------------------------------------ //
//////////////////////////////////////////////////////////////////////////////
RunModule::RunModule(QObject *parent) :
    QObject(parent),
    m_dbg(false),
    m_vmm_type(3),
    m_msg(0),
    m_externalTrigger(false),
    m_pulseCount(0),
    m_initSocketHandler(false),
    m_initConfigHandler(false),
    m_socketHandler(0),
    m_configHandler(0)
{
}
// ------------------------------------------------------------------------ //
void RunModule::LoadMessageHandler(MessageHandler& m)
{
    m_msg = &m;
}
// ------------------------------------------------------------------------ //
RunModule& RunModule::LoadConfig(ConfigHandler& inconfig)
{
    m_configHandler = &inconfig;
    if(!m_configHandler) {
        msg()("FATAL ConfigHandler instance null", "RunModule::LoadConfig", true);
        exit(1);
    }
    else if(dbg()){
        msg()("ConfigHandler instance loaded", "RunModule::LoadConfig");
    }

    //depending on the run_mode set in the configuration
    //set whether doing a timed run or not
    bool setExternal = false;
    if(config().daqSettings().run_mode=="pulser")
        setExternal = false;
    else if(config().daqSettings().run_mode=="external") {
        setExternal = true;
    }
    setExternalTrig(setExternal);

    m_initConfigHandler = true;

    return *this;
}
// ------------------------------------------------------------------------ //
RunModule& RunModule::LoadSocket(SocketHandler& socket)
{
    m_socketHandler = &socket;
    //cout << "RunModule::LoadSocket   [" << boost::this_thread::get_id() << "] socket = " << m_socketHandler << endl;
    if(!m_socketHandler) {
        msg()("FATAL SocketHandler instance null","RunModule::LoadSocket",true);
        exit(1);
    }
    else if(dbg()){
        msg()("SocketHandler instance loaded","RunModule::LoadSocket");
        m_socketHandler->Print();
    }

    m_initSocketHandler = true;

    return *this;
}
// ------------------------------------------------------------------------ //
void RunModule::set_trigger_settings()
{
    bool send_ok = true;
    bool ok;
    stringstream sx;

    QByteArray datagram;

    // send trigger settings to VMMAPP port
    //int send_to_port = config().commSettings().vmmapp_port;
    // #warning HARDCODING CLOCKS
    int send_to_port = 6600;

    // second 32 bit word is currently not used in the FW
    QString second_32 = "AAFF";

    // register addresses
    QString address_latency = "05"; // trigger latency

    // register values
    uint32_t trigger_latency = config().daqSettings().trigger_latency;

    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        // update the command counter for new command
        socket().updateCommandCounter();

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        ////////////////////////////////////////
        // header
        ////////////////////////////////////////
                                                                // byte range
        out << (quint32)(socket().commandCounter())             // [0-3]
            << (quint32)(second_32.toUInt(&ok,16));             // [4-7]

        ////////////////////////////////////////
        // trigger settings
        ////////////////////////////////////////

        // latency
        out << (quint32)(address_latency.toUInt(&ok,16))        // [8-11]
            << (quint32)(trigger_latency);                      // [12-15]

        ////////////////////////////////////////
        // send the packet
        ////////////////////////////////////////
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::set_trigger_settings");
        if(send_ok) {
            bool read_ok = true;
            read_ok = socket().waitForReadyRead("fec");
            if(read_ok) {
                if(dbg()) msg()("Processing replies","RunModule::set_trigger_settings");
                socket().processReply("fec",ip);
            } // read_ok = 1
            else {
                sx.str("");
                sx << "Timeout while waiting for replies from VMM on board with IP: " << ip.toStdString();
                msg()(sx, "RunModule::set_trigger_settings"); sx.str("");
                socket().closeAndDisconnect("fec","RunModule::set_trigger_settings");
            }
        } // send_ok = 1
    } // ip
    socket().closeAndDisconnect("fec","RunModule::set_trigger_settings");
}
// ------------------------------------------------------------------------ //
/*
void RunModule::setTriggerAcqConstants()
{
    if(dbg()) msg()("Sending trigger ACQ constants...","RunModule::setTriggerAcqConstants");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send T/DAQ constants to VMMAPP port
    int send_to_port = config().commSettings().vmmapp_port;

    for(const auto& ip : socket().ipList()) {
        // UPDATE COUNTER, ETC... SHOULD NOW BE DONE
        // SOLELY IN SOCKETHANDLER TO WHICH WE PASS
        // THE DATAGRAMS

        socket().updateCommandCounter();

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); // rewind

        ///////////////////////////
        // header info
        ///////////////////////////
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd         = "AA";
        cmdType     = "AA";
        cmdLength   = "FFFF";
        msbCounter  = "0x80000000"; 
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint16) config().vmmMap().mask
            << (quint8)  cmd.toUInt(&ok,16) //[8]
            << (quint8)  cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok, 16); //[10,11]

        ///////////////////////////
        // trigger constants
        ///////////////////////////
        QString trigperiod = QString::fromStdString(config().daqSettings().trigger_period);
        out << (quint32) 0 //[12,15]
            #warning these descriptors no longer hold for VMM3
            //acq. sync
            << (quint32) 5 //[32,35]
            << (quint32) config().daqSettings().acq_sync //[36,39]
            //trigger period
            << (quint32) 2 //[16,19]
            << (quint32) trigperiod.toInt(&ok,16) //[20,23]
            //<< (quint32) config().daqSettings().trigger_period.toInt(&ok,16) //[20,23]
            //pulser delay
            << (quint32) 4 //[24,27]
            << (quint32) config().daqSettings().tp_delay //[28,31]
            //acq. window
            << (quint32) 6 //[40,43]
            << (quint32) config().daqSettings().acq_window //[44,47]
            //bcid reset
            << (quint32) 9 //[48,51]
            << (quint32) config().daqSettings().bcid_reset; //[52,55]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                "RunModule::setTriggerAcqConstants"); 

        if(send_ok) {

            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::setTriggerAcqConstants");
                socket().processReply("fec", ip);
            }
            else {
                msg()("Timeout while waiting for replies from VMM",
                            "RunModule::setTriggerAcqConstants");
                socket().closeAndDisconnect("fec","RunModule::setTriggerAcqConstants");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec","RunModule::setTriggerAcqConstants");
}
*/
// ------------------------------------------------------------------------ //
void RunModule::resetFPGA()
{
    if(vmm_type()==2) {
        msg()("WARNING No FPGA reset for VMM2 communication");
        return;
    }

    stringstream sx;
    bool ok;
    bool send_ok = true;
    QByteArray datagram;

    int send_to_port = 6600;

    // FW currently does not use second 32 bit word
    QString second_32 = "AAFF";

    // register address
    QString address = "AF";

    // register value
    QString value = "AA";

    if(dbg()) {
        sx << "Resetting FPGA(s)";
        msg()(sx, "RunModule::resetFPGA"); sx.str("");
    }

    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        sx.str("");

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        // update counter
        socket().updateCommandCounter();

        //////////////////////////////////
        // header 
        //////////////////////////////////
        out << (quint32)(socket().commandCounter())
            << (quint32)(second_32.toUInt(&ok, 16));

        /////////////////////////////////
        // FPGA reset
        /////////////////////////////////
        out << (quint32)(address.toUInt(&ok, 16))
            << (quint32)(value.toUInt(&ok, 16));

        /////////////////////////////////
        // send it
        /////////////////////////////////
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::resetFPGA");
        //if(send_ok) {
        //    bool read_ok = true;
        //    // don't expect replies after resetting FPGA
        //    //read_ok = socket().waitForReadyRead("fec");
        //    //if(read_ok) {
        //    //    if(dbg()) msg()("Processing replies", "RunModule::resetFPGA");
        //    //}
        //}
    } // ip
    socket().closeAndDisconnect("fec","RunModule::resetFPGA");
    
    
}
// ------------------------------------------------------------------------ //
void RunModule::setTriggerMode(bool enable_fixed_latency)
{

    stringstream sx;
    bool ok;
    bool send_ok = true;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    //int send_to_port = config().commSettings().vmmapp_port;
    //#warning HARDCODING PORTS
    int send_to_port = 6600;

    // FW currently does not use second 32 bit word
    QString second_32 = "AAFF";

    // register address
    QString address = "AB";
    QString address_latency = "05"; // trigger latency (we send this here as well)
    QString address_fixed_latency = "CD"; // fixed latency to stop TAC window


    // register values
    int mode_to_set = 7;
    string modestr = "INTERNAL";
    if(config().daqSettings().run_mode == "external")           { mode_to_set = 4; modestr = "EXTERNAL"; }
    //else if(config().daqSettings().run_mode == "externalBC")    { mode_to_set = 5; modestr = "EXTERNALBC"; }
    else {
        // default to internal
        mode_to_set = 7;
        modestr = "INTERNAL";
    }
    uint32_t set_fixed_latency_rm = (enable_fixed_latency ? 1 : 0);

    if(dbg()) {
        sx << "Setting trigger mode to " << modestr;
        if(set_fixed_latency_rm) sx << " with fixed TAC window";
        msg()(sx,"RunModule::setTriggerMode"); sx.str("");
    }

    // register values
    uint32_t trigger_latency = config().daqSettings().trigger_latency;


    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        sx.str("");

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        // update counter for new command
        socket().updateCommandCounter();


        ///////////////////////////////////////
        // header
        ///////////////////////////////////////
                                                            // byte range
        out << (quint32)(socket().commandCounter())         // [0-3]
            << (quint32)(second_32.toUInt(&ok,16));         // [4-7]

        ////////////////////////////////////////
        // trigger mode
        ////////////////////////////////////////
        out << (quint32)(address.toUInt(&ok, 16))           // [8-11]
            << (quint32)(mode_to_set);                      // [12-15]

        ////////////////////////////////////////
        // trigger latency
        ////////////////////////////////////////
        out << (quint32)(address_latency.toUInt(&ok, 16))
            << (quint32)(trigger_latency);

        ////////////////////////////////////////
        // fixed latency for TAC
        ////////////////////////////////////////
        out << (quint32)(address_fixed_latency.toUInt(&ok, 16))
            << (quint32)(set_fixed_latency_rm);

        // send to this IP
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::setTriggerMode");
        if(send_ok){
            bool read_ok = true;
            read_ok = socket().waitForReadyRead("fec");
            if(read_ok) {
                if(dbg()) msg()("Processing replies","RunModule::setTriggerMode");
                socket().processReply("fec", ip);
            } // read_ok = 1
            else {
                sx.str("");
                sx << "Timeout while waiting for replies from VMM on board with IP: " << ip.toStdString();
                msg()(sx, "RunModule::setTriggerMode");
                socket().closeAndDisconnect("fec","RunModule::setTriggerMode");
            }
        } // send_ok = 1
    } // ip
    socket().closeAndDisconnect("fec","RunModule::setTriggerMode");

/*
    if(dbg()) msg()("Setting trigger mode...","RunModule::setTriggerMode");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    int send_to_port = config().commSettings().vmmapp_port;

    for(const auto& ip : socket().ipList()) {
        // UPDATE COUNTER, ETC... SHOULD NOW BE DONE
        // SOLELY IN SOCKETHANDLER TO WHICH WE PASS
        // THE DATAGRAMS

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); // rewind

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd         = "AA";
        cmdType     = "AA";
        cmdLength   = "FFFF";
        msbCounter  = "0x80000000"; 

        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ///////////////////////////
        // trigger mode
        ///////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 0; //[16,19]
        if(config().daqSettings().run_mode == "external"){
            out << (quint32) 4; //[20,23]
            if(dbg()) msg()("Setting run mode to: EXTERNAL","RunModule::setTriggerMode");
        } // external
        else if(config().daqSettings().run_mode == "externalBC") {
            out << (quint32) 5; // [20,23]
            if(dbg()) msg()("Setting run mode to: EXTERNALBC","RunModule::setTriggerMode");
        } // externalBC
        else {
            out << (quint32) 7; //[20,23]
            if(dbg()) msg()("Setting run mode to: INTERNAL","RunModule::setTriggerMode");
        }
        
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                "RunModule::setTriggerMode");

        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::setTriggerMode");
                socket().processReply("fec", ip);
            }
            else {
                msg()("Timeout while waiting for replies from VMM",
                            "RunModule::setTriggerMode");
                socket().closeAndDisconnect("fec","RunModule::setTriggerMode");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec","RunModule::setTriggerMode");
*/
}
// ------------------------------------------------------------------------ //
void RunModule::send_tdaq_settings(int latency_steps, bool mode_internal, bool mode_external,
            bool do_external_bc, int deadtime, int n_ckbc_fixed_tac, int art_timeout,
            int extra_ckbc_latency)
{
    if(vmm_type()==2) {
        send_tdaq_settings_VMM2(mode_internal, mode_external);
        return;
    }

    stringstream sx;
    bool ok;
    bool send_ok = true;
    QByteArray datagram;

    int send_to_port = 6600;

    // FW currently does not use second 32 bit word
    QString second_32 = "AAFF";

    // register addresses
    QString address_latency = "05"; // trigger latency
    QString address_external_bc = "CD"; // set fixed latency to stop TAC window
    QString address_mode = "AB"; // setting of internal or external trigger mode
    QString address_deadtime = "C8"; // "simple deadtime"
    QString address_ckbc_count = "C7"; // number of CKBCs for fixed TAC
    QString address_art_timeout = "C9"; // art timeout
    QString address_extra_ckbc_latency = "CA";

    // register values
    int mode = -1;
    if(mode_internal)
        mode = 7;
    else if(mode_external)
        mode = 4;

    if(mode < 0) {
        sx << "ERROR Both Internal AND External run modes are attempting to be set, cannot send TDAQ settings as is!";
        msg()(sx, "RunModule::send_tdaq_settings"); sx.str("");
        return;
    }

    int fixed_tac = (do_external_bc ? 1 : 0);

    if(dbg()) {
        sx << "Sending TDAQ settings: [latency,mode,fixed-tac,deadtime,n_ckbc,art_timeout,extra_ckbc_latency]=[" << latency_steps
                << "," << (mode == 7 ? "internal" : "external") << ","
                << do_external_bc << ","<<deadtime << "," << n_ckbc_fixed_tac << "," << art_timeout << ","<<extra_ckbc_latency<<"]";
        msg()(sx,"RunModule::send_tdaq_settings"); sx.str("");
    }

    // begin sending
    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        sx.str("");

        // configure only the selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        // update the counter for the new command
        socket().updateCommandCounter();

        ///////////////////////////////////////
        // header
        ///////////////////////////////////////
        out << (quint32)(socket().commandCounter())
            << (quint32)(second_32.toUInt(&ok,16));

        ///////////////////////////////////////
        // latency
        ///////////////////////////////////////
        out << (quint32)(address_latency.toUInt(&ok,16))
            << (quint32)(latency_steps);

        ///////////////////////////////////////
        // fixed tac
        ///////////////////////////////////////
        out << (quint32)(address_external_bc.toUInt(&ok,16))
            << (quint32)(fixed_tac);

        ///////////////////////////////////////
        // run mode
        ///////////////////////////////////////
        out << (quint32)(address_mode.toUInt(&ok,16))
            << (quint32)(mode);

        ///////////////////////////////////////
        // deadtime settings
        ///////////////////////////////////////
        if(vmm_type()==3) {
            out << (quint32)(address_deadtime.toUInt(&ok,16))
                << (quint32)(deadtime);
        }

        ///////////////////////////////////////
        // n ckbc count
        ///////////////////////////////////////
        out << (quint32)(address_ckbc_count.toUInt(&ok,16))
            << (quint32)(n_ckbc_fixed_tac);

        ///////////////////////////////////////
        // art timeout window
        ///////////////////////////////////////
        out << (quint32)(address_art_timeout.toUInt(&ok,16))
            << (quint32)(art_timeout);

        ///////////////////////////////////////
        // extra CKBC latency
        ///////////////////////////////////////
        out << (quint32)(address_extra_ckbc_latency.toUInt(&ok,16))
            << (quint32)(extra_ckbc_latency);

        // SEND
        // ignore replies, FW only sends replies to SPI
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::send_tdaq_settings");
        if(!send_ok) {
            sx << "WARNING Error in sending TDAQ settings to board at IP " << ip.toStdString() << endl;
            msg()(sx,"RunModule::send_tdaq_settings"); sx.str("");
        }
    } // ip
    socket().closeAndDisconnect("fec","RunModule::send_tdaq_settings");


}
// ------------------------------------------------------------------------ //
void RunModule::send_tdaq_settings_VMM2(bool internal_trigger, bool external_trigger)
{
    stringstream sx;

    bool send_ok = true;
    bool ok;
    QByteArray datagram;

    int send_to_port = 6600;

    int mode = -1;
    if(internal_trigger) {
        mode = 7;
    }
    else if(external_trigger) {
        mode = 4;
    }

    int ip_idx = -1;
    for(const auto & ip : socket().ipList()) {
        sx.str("");

        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }
        datagram.clear();
        QDataStream out ( & datagram, QIODevice::WriteOnly );
        out.device()->seek(0);

        socket().updateCommandCounter();

        // header
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd = "AA";
        cmdType = "AA";
        cmdLength = "FFFF";
        msbCounter = "0x80000000"; 

        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16))
            << (quint16) 0
            << (quint16) config().vmmMap().mask
            << (quint8) cmd.toUInt(&ok,16)
            << (quint8) cmdType.toUInt(&ok,16)
            << (quint16) cmdLength.toUInt(&ok,16);

        // cmd
        out << (quint32) 0
            << (quint32) 0
            << (quint32) mode;

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::send_tdaq_settings_VMM2");
        if(!send_ok) {
            sx.str("");
            sx << "WARNING Error in sending TDAQ settings to board at IP " << ip.toStdString();
            msg()(sx,"RunModule::send_tdaq_settings_VMM2"); sx.str("");
        }
    } // ip
    socket().closeAndDisconnect("fec","RunModule::send_tdaq_settings_VMM2");
}
// ------------------------------------------------------------------------ //
void RunModule::ACQon()
{
    if(vmm_type()==2) {
        set_acquisition_VMM2(1);
        return;
    }
    set_acquisition(1);
    /*
    if(dbg()) msg()("Setting ACQ ON","RunModule::ACQon");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    int send_to_port = config().commSettings().vmmapp_port;

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd        = "AA";
        cmdType    = "AA";
        cmdLength  = "FFFF";
        msbCounter = "0x80000000"; 

        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0  //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ///////////////////////////
        // ACQ on
        ///////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 15 //[16,19]
            << (quint32) 1; //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                    "RunModule::ACQon");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...", "RunModule::ACQon");
                socket().processReply("fec", ip);
            } // readOK
            else {
                msg()("Timeout while waiting for replies from VMM","RunModule::ACQon");
                socket().closeAndDisconnect("fec","RunModule::ACQon");
            }
        }
    } // ip
    socket().closeAndDisconnect("fec", "RunModule::ACQon");
    */
}
// ------------------------------------------------------------------------ //
void RunModule::set_clocks_pulser(int cktk_max, int ckbc_frequency, int n_tp_pulses,
        int tp_skew_steps, int tp_period, int tp_width)
{
    configure_clocks(cktk_max, ckbc_frequency, n_tp_pulses, tp_skew_steps, tp_period, tp_width);
}
// ------------------------------------------------------------------------ //
void RunModule::calib_configuration_update(int current_calib_idx, int total_calib_loops)
{
    stringstream sx;
    sx << "Pulser calibration progress update [ "
        << std::setw(4) << current_calib_idx
        << " / "
        << std::setw(4) << total_calib_loops << " ] ("
        << std::setprecision(4)
        << std::setw(5)
        << (((float)(current_calib_idx) / (float)(total_calib_loops)) * 100.)
        << " %)";
    msg()(sx); sx.str("");
}
// ------------------------------------------------------------------------ //
void RunModule::configure_clocks(int cktk_max, int ckbc_frequency, int n_tp_pulses,
                                        int tp_skew_steps, int tp_period, int tp_width)
{

    stringstream sx;
    sx << "Configuring the clocks";
    if(dbg()) { msg()(sx,"RunModule::configure_clocks"); sx.str(""); }

    bool send_ok = true;
    bool ok;
    QByteArray datagram;

    //clock configuration commands go to VMMAPP port 
    //int send_to_port = config().commSettings().vmmapp_port;
    #warning HARDODING PORT NUMBERS
    int send_to_port = 6600;

    // FW does not currently use second 32 bit word
    QString second_32 = "AAFF";

    ///////////////////////////////////////////
    // addresses
    ///////////////////////////////////////////
    QString address_cktk_max    = "C1";
    QString address_ckbc_freq   = "C2";
    QString address_cktp_n      = "C3";
    QString address_cktp_skew   = "C4";
    QString address_cktp_period = "C5";
    QString address_cktp_width  = "C6";

    QString infinite_pulses = "FFFF";

    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        // update the command counter for the new packet
        socket().updateCommandCounter();

        ///////////////////////////////////////////
        // header
        ///////////////////////////////////////////
                                                            // byte range
        out << (quint32)(socket().commandCounter())         // [0-3]
            << (quint32)(second_32.toUInt(&ok,16));         // [4-7]

        ///////////////////////////////////////////
        // command
        ///////////////////////////////////////////

        // cktk max
        out << (quint32)(address_cktk_max.toUInt(&ok,16))   // [8,11]
            << (quint32)(cktk_max);                         // [12,15]

        // ckbc frequency
        out << (quint32)(address_ckbc_freq.toUInt(&ok,16))  // [16,19]
            << (quint32)(ckbc_frequency);                   // [20,23]

        // number of cktp pulses
        out << (quint32)(address_cktp_n.toUInt(&ok,16));     // [24,27]
        if(n_tp_pulses < 0) {                               // [28,31]
            out << (quint32)(infinite_pulses.toUInt(&ok,16));
        }
        else {
            out << (quint32)(n_tp_pulses);
        }

        // cktp skew
        out << (quint32)(address_cktp_skew.toUInt(&ok,16))  // [32,35]
            << (quint32)(tp_skew_steps);                    // [36,39]

        // cktp period
        out << (quint32)(address_cktp_period.toUInt(&ok,16)) // [40,43]
            << (quint32)(tp_period);                         // [44,47]

        // cktp width
        out << (quint32)(address_cktp_width.toUInt(&ok,16))  // [48,51]
            << (quint32)(tp_width);                          // [52,55]

        ///////////////////////////////////////////
        // send the packet
        ///////////////////////////////////////////
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::configure_clocks");
        if(send_ok) {
            bool read_ok = true;
            read_ok = socket().waitForReadyRead("fec");
            if(read_ok) {
                if(dbg()) msg()("Processing replies","RunModule::configure_clocks");
                socket().processReply("fec",ip);
            } // read_ok = 1
            else {
                sx.str("");
                sx << "Timeout while waiting for replies from VMM on board with IP: " << ip.toStdString();
                msg()(sx,"RunModule::configure_clocks"); sx.str("");
            }
        } // send_ok = 1
    } // ip
    socket().closeAndDisconnect("fec","RunModule::configure_clocks");
}
// ------------------------------------------------------------------------ //
void RunModule::set_acq_pulser(int on_or_off)
{
    set_acquisition(on_or_off);
}
// ------------------------------------------------------------------------ //
void RunModule::set_acquisition(int on_or_off)
{
    stringstream sx;
    sx << "Setting ACQ " << ( (on_or_off==1) ? "ON" : "OFF");
    if(dbg()) { msg()(sx, "RunModule::set_acquisition"); sx.str(""); }

    bool send_ok = true;
    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    //int send_to_port = config().commSettings().vmmapp_port;
    // #warning HARDCODING PORTS
    int send_to_port = 6600;

    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        datagram.clear();

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        // update the cmd counter for the new command
        socket().updateCommandCounter();

        /////////////////////////////////////////////////
        // header
        /////////////////////////////////////////////////
        QString second_32 = "AAFF"; // second 32 bit word ( | VMMID | CMD | ) is unused by the FW (currently)

        out << (quint32)(socket().commandCounter())     // [0-3]
            << (quint32)(second_32.toUInt(&ok,16));     // [4-7]

        // address for DAQ mode
        QString address = "0F";
        out << (quint32)(address.toUInt(&ok,16))        // [8-11]
            << (quint32)(on_or_off);                    // [12-15]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec", "RunModule::set_acquisition");
        if(send_ok) {
            bool read_ok = true;
            read_ok = socket().waitForReadyRead("fec");
            if(read_ok) {
                if(dbg()) msg()("Processing replies","RunModule::set_acquisition");
                socket().processReply("fec",ip);
            } // read_ok = 1
            else {
                sx << "Timeout while waiting for replies from VMM on board with IP: " << ip.toStdString();
                msg()(sx, "RunModule::set_acquisition"); sx.str("");
            }
        } // send_ok = 1
    } // ip
    socket().closeAndDisconnect("fec","RunModule::set_acquisition");
}
// ------------------------------------------------------------------------ //
void RunModule::set_acquisition_VMM2(int on_or_off)
{
    stringstream sx;
    sx << "Setting ACQ " << ( (on_or_off==1) ? "ON" : "OFF");
    if(dbg()) { msg()(sx,"RunModule::set_acquisition_VMM2"); sx.str(""); }

    bool send_ok = true;
    bool ok;
    QByteArray datagram;

    int send_to_port = 6600;
    int ip_idx = -1;
    for(const auto & ip : socket().ipList()) {
        datagram.clear();
        //configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }
        QDataStream out(&datagram, QIODevice::WriteOnly);
        out.device()->seek(0);

        socket().updateCommandCounter();

        // header
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd = "AA";
        cmdType = "AA";
        cmdLength = "FFFF";
        msbCounter = "0x80000000"; 

        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16))
            << (quint16) 0
            << (quint16)  config().vmmMap().mask // need to confirm that this 16 is used (in VMM2 this was the "HDMIChannelMap"
            << (quint8) cmd.toUInt(&ok,16)
            << (quint8) cmdType.toUInt(&ok,16)
            << (quint16) cmdLength.toUInt(&ok,16);

        // acq cmd
        out << (quint32) 0
            << (quint32) 15
            << (quint32) (on_or_off);

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                        "RunModule::set_acquisition_VMM2");

        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                socket().processReply("fec",ip);
            } // readOK 
            else {
                msg()("Timeout while waiting for replies from VMM", "RunModule::set_acquisition_VMM2");
                socket().closeAndDisconnect("fec","RunModule::set_acquisition_VMM2");
            }
        } // send_ok
    } // ip
    socket().closeAndDisconnect("fec","RunModule::set_acquisition_VMM2");

}
// ------------------------------------------------------------------------ //
void RunModule::ACQoff()
{
    if(vmm_type()==2) {
        set_acquisition_VMM2(0);
        return;
    }
    set_acquisition(0);
    /*
    if(dbg()) msg()("Setting ACQ OFF","RunModule::ACQoff");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    int send_to_port = config().commSettings().vmmapp_port;

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        QString cmd, cmdType, cmdLength, msbCounter;
        cmd         = "AA";
        cmdType     = "AA";
        cmdLength   = "FFFF";
        msbCounter  = "0x80000000";

        //stringstream sx;
        //sx << "AQCOFF command counter = " << socket().commandCounter();
        //msg()(sx);sx.str("");

        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ///////////////////////////
        // ACQ off
        ///////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 15 //[16,19]
            << (quint32) 0; //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::ACQoff");
        if(send_ok) {
        bool readOK = true;
        readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::ACQoff");
                QByteArray buffer;
                //buffer = socket().fecSocket().processReply(ip, 0, socket().commandCounter()); //.processReply("fec", ip);
                socket().processReply("fec", ip);

            }
            else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::ACQoff"); 
                socket().closeAndDisconnect("fec","RunModule::ACQoff");
            }
        }

    } // ip loop

    socket().closeAndDisconnect("fec", "RunModule::ACQoff");
    */
}
// ------------------------------------------------------------------------ //
/*
void RunModule::setEventHeaders(const int bld_info, const int bld_mode, bool highRes)
{
    if(dbg()) msg()("Setting event headers...","RunModule::setEventHeaders");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    int send_to_port = config().commSettings().vmmapp_port;

    // headers
    QString cmd, msbCounter;
    cmd = "AAAAFFFF";
    msbCounter = "0x80000000";

    // setup the word
    quint32 evbldinfo = 0;
    if(bld_info==0)             evbldinfo = 0;
    else if(bld_info==1)        evbldinfo = 256;
    else if(bld_info==2)        evbldinfo = 512;
    else if(bld_info==3)        evbldinfo = 768;
    else if(bld_info==4)        evbldinfo = 1024;
    else if(bld_info==5)        evbldinfo = 1280;
    //quint32 evbldinfo = (quint32) 256*bld_info;
    quint32 evbldmode = (quint32)bld_mode;

    //resolution
    quint32 resolutionBits = 0;
    if(highRes)
        resolutionBits = 32768;

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16) //[8,11]
            << (quint32) 0; //[12,15]

        ///////////////////////////
        // event header
        ///////////////////////////
        out << (quint32) 10 //[16,19]
            << (quint32) evbldmode //[20,23]
            << (quint32) 12 //[24,27]
            << (quint32) (evbldinfo + resolutionBits); //[28,31]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::setEventHeaders");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::setEventHeaders");
                socket().processReply("fec", ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::setEventHeaders");
                socket().closeAndDisconnect("fec","RunModule::setEventHeaders");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec", "RunModule::setEventHeaders");
}
*/
// ------------------------------------------------------------------------ //
void RunModule::resetASICs()
{
    msg()("Resetting VMMs...","RunModule::resetASICs");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send trigger mode to VMMAPP port
    int send_to_port = 6600;

    // headers
    QString cmd, cmdType, cmdLength, msbCounter;
    cmd = "AA";
    cmdType = "AA";
    cmdLength = "FFFF";
    msbCounter = "0x80000000"; 

    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind


        // only send to selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) continue;
        }

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5] 
            << (quint16) config().vmmMap().mask //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ///////////////////////////
        // reset
        ///////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 128 //[16,19]
            << (quint32) 2; //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::resetASICs");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::resetASICs");
                socket().processReply("fec", ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::resetASICs");
                socket().closeAndDisconnect("fec","RunModule::resetASICs");
            }
        }
    } // ip
    socket().closeAndDisconnect("fec", "RunModule::resetASICs");
}
// ------------------------------------------------------------------------ //
/*
void RunModule::resetFEC(bool do_reset)
{
    if(dbg()) msg()("Resetting FEC...","RunModule::resetFEC");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send reset call to FEC port
    int send_to_port = config().commSettings().fec_port;

    // headers
    QString cmd, cmdType, cmdLength, msbCounter;
    cmd = "AA";
    cmdType = "AA";
    cmdLength = "FFFF";
    msbCounter = "0x80000000";

    // setup
    QString address = "FFFFFFFF";
    QString value = "";
    if(do_reset) {
        value = "FFFF8000";
        if(dbg()) msg()("Rebooting FEC...","RunModule::resetFEC");
    } else {
        value = "FFFF0001";
        if(dbg()) msg()("WarmInit FEC...","RunModule::resetFEC");
    } 

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ///////////////////////////
        // header info
        ///////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3] 
            << (quint16) 0 //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ///////////////////////////
        // word
        ///////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) address.toUInt(&ok,16) //[16,19]
            << (quint32) value.toUInt(&ok,16); //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::resetFEC");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::resetFEC");
                socket().processReply("fec", ip);
            } else {
                if(dbg()) msg()("Timeout while waiting for replies from VMM",
                                "RunModule::resetFEC");
                socket().closeAndDisconnect("fec","RunModule::resetFEC");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec", "RunModule::resetFEC");
}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::setMask()
{
    if(dbg()) msg()("Setting VMM mask and ART...","RunModule::setMask");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to vmmapp port
    int send_to_port = config().commSettings().vmmapp_port;

    // header
    QString cmd, cmdType, cmdLength, msbCounter;
    cmd = "AA";
    cmdType = "AA";
    cmdLength = "FFFF";
    msbCounter = "0x80000000";

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 8 //[16,19]
            << (quint32) config().vmmMap().mask; //[20,23]
            //<< (quint32) config().getHDMIChannelMapART(); //[20,23]
          //  << (quint32) config().getHDMIChannelMap(); //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::setMask");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::setMask");
                socket().processReply("fec", ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::setMask");
                socket().closeAndDisconnect("fec", "RunModule::setMask");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec", "RunModule::setMask");

}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::enableART(bool enabling, bool holdoff)
{
    if(enabling)
        msg()("Enabling ART...","RunModule::enableART");
    else
        msg()("Disabling ART...","RunModule::enableART");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to vmmapp port
    int send_to_port = config().commSettings().s6_port;

    // header
    QString cmd, msbCounter;
    cmd = "AAAAFFFF";
    msbCounter = "0x80000000"; 

    int holdoff_enable = 233;
    if(holdoff) holdoff_enable = 232;

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32) (socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16); //[8,11]
          
        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 1 //[19,19]
            << (quint32) holdoff_enable; //[20,23]


        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::enableART");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::enableART");
                socket().processReply("fec",ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                            "RunModule::enableART");
                socket().closeAndDisconnect("fec","RunModule::enableART");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec","RunModule::enableART");

}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::checkLinkStatus()
{
    if(dbg()) msg()("Checking link status...","RunModule::checkLinkStatus");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to vmmapp port
    int send_to_port = config().commSettings().vmmapp_port;

    // header
    QString cmd = "BBAAFFFF";
    QString msbCounter = "0x80000000"; 

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16); //[8,11]

        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 16; //[16,19]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::checkLinkStatus");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                emit checkLinks();
                //if(dbg()) msg()("Processing replies...","RunModule::checkLinkStatus");
                //socket().processReply("fec", ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::checkLinkStatus");
                socket().closeAndDisconnect("fec", "RunModule::checkLinkStatus");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec", "RunModule::checkLinkStatus");

}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::resetLinks()
{
    if(dbg()) msg()("Resetting links...","RunModule::resetLinks");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to vmmapp port
    int send_to_port = config().commSettings().vmmapp_port;

    QString cmd, cmdType, cmdLength, msbCounter, cmdReset;
    cmd = "AA";
    cmdType = "AA";
    cmdLength = "FFFF";
    msbCounter = "0x80000000";
    cmdReset = "FF";

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header (1)
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint16) 0 //[4,5]
            << (quint16) config().vmmMap().mask //[6,7]
            //<< (quint16) config().getHDMIChannelMap() //[6,7]
            << (quint8) cmd.toUInt(&ok,16) //[8]
            << (quint8) cmdType.toUInt(&ok,16) //[9]
            << (quint16) cmdLength.toUInt(&ok,16); //[10,11]

        ////////////////////////////
        // command (1)
        ////////////////////////////
        out << (quint32) 13 //[12,15]
            << (quint32) cmdReset.toUInt(&ok,16); //[16,19]
        
        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::resetLinks");

        if(send_ok) {
            datagram.clear();
            out.device()->seek(0);
            socket().updateCommandCounter();
            ////////////////////////////
            // header (2)
            ////////////////////////////
            out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
                << (quint16) 0 //[4,5]
                << (quint16) config().vmmMap().mask //[6,7]
                //<< (quint16) config().getHDMIChannelMap() //[6,7]
                << (quint8) cmd.toUInt(&ok,16) //[8]
                << (quint8) cmdType.toUInt(&ok,16) //[9]
                << (quint16) cmdLength.toUInt(&ok, 16); //[10,11]

            ////////////////////////////
            // command (2)
            ////////////////////////////
            out << (quint32) 13 //[12,15]
                << (quint32) 0; //[16,19]

            send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                    "RunModule::resetLinks");
            if(!send_ok)
                msg()("Failed to send second packet", "RunModule::resetLinks");
        }
    } // ip

    socket().closeAndDisconnect("fec", "RunModule::resetLinks");

}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::s6clocks(int cktk, int ckbc, int ckbc_skew)
{
    if(dbg()) msg()("Setting S6 clocks...","RunModule::s6clocks");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to s6 port
    int send_to_port = config().commSettings().s6_port;

    QString cmd, msbCounter;
    cmd = "AAAAFFFF";
    msbCounter = "0x80000000";

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16); //[8,11]

        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 6 //[16,19]
            << (quint32) (cktk*16) //[20,23]
            << (quint32) 7 //[24,27]
            << (quint32) ( ckbc + (ckbc_skew*16) ); //[28,31]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                            "RunModule::s6clocks");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::s6clocks");
                socket().processReply("fec",ip);
            } else {
                msg()("Timout while waiting for replies from VMM",
                        "RunModule::s6clocks");
                socket().closeAndDisconnect("fec","RunModule::s6clocks");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec","RunModule::s6clocks");

}
*/
// ------------------------------------------------------------------------ //
/*
void RunModule::setS6Resets(int s6_tk_pulses, bool set_s6_autoReset, bool set_s6_fecReset,
                                    int s6_fec_periodRest)
{
    if(dbg()) msg()("Setting s6 reset settings...","RunModule::setS6Resets");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to s6 port
    int send_to_port = config().commSettings().s6_port;

    // header
    QString cmd, msbCounter;
    cmd = "AAAAFFFF";
    msbCounter = "0x80000000"; 
    int s6_auto_reset = 0;
    int s6_fec_reset = 0;
    bool fec_reset = false;
    if(set_s6_autoReset) s6_auto_reset = 8;
    if(set_s6_fecReset) { s6_fec_reset = 32; fec_reset = true; }

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16); //[8,11]

        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0; //[12,15]

        out << (quint32) 9 //[16,19]
            << (quint32)( s6_tk_pulses + s6_auto_reset + s6_fec_reset); //[20,23]

    //    int s6_auto = 0;
    //    int s6_fec = 0;
    //    if(set_s6_autoReset) s6_auto = 8;
    //    if(set_s6_fecReset) s6_fec = 16;

    //    out << (quint32) 9 //[16,19]
    //        << (quint32)( s6_tk_pulses + s6_auto + s6_fec); //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::setS6Resets");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...", "RunModule::setS6Resets");
                socket().processReply("fec", ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                                        "RunModule::setS6Resets");
                socket().closeAndDisconnect("fec","RunModule::setS6Resets");
            }

            ////////////////////////////////
            // set periodic reset
            ////////////////////////////////

            bool resetSeek = out.device()->reset();
            if(resetSeek) {
                socket().updateCommandCounter();
                out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16))
                    << (quint32) config().vmmMap().mask
                    //<< (quint32) config().getHDMIChannelMap()
                    << (quint32) cmd.toUInt(&ok,16);

                /////////////////////
                // command
                /////////////////////
                out << (quint32) 9
                    << (quint32) s6_fec_periodRest;

                send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                    "RunModule::setS6Resets");

                if(send_ok) {
                    readOK = socket().waitForReadyRead("fec");
                    if(readOK) {
                        if(dbg()) msg()("Processing replies [2]...", "RunModule::setS6Resets");
                        socket().processReply("fec", ip);
                    } else {
                        msg()("Timeout while waiting for replies from VMM [2]",
                                            "RunModule::setS6Resets");
                        socket().closeAndDisconnect("fec","RunModule::setS6Resets");
                    } // readok
                } // send_ok
            } //resetSeek
            else {
                msg()("Error upon resetting datastream seek. Unable to send period reset command for FEC",
                                            "RunModule::setS6Resets");
            }
        } // send_ok
    } // ip

    socket().closeAndDisconnect("fec","RunModule::setS6Resets");

}

*/
// ------------------------------------------------------------------------ //
/*
void RunModule::configTP(int tpskew, int tpwidth, int tppolarity)
{
    if(dbg()) msg()("Configuring the pulser...","RunModule::configTP");

    bool send_ok = true;

    bool ok;
    QByteArray datagram;

    // send call to s6 port
    int send_to_port = config().commSettings().s6_port;

    QString cmd, msbCounter;
    cmd = "AAAAFFFF";
    msbCounter = "0x80000000"; 

    for(const auto& ip : socket().ipList()) {
        datagram.clear();
        QDataStream out (&datagram, QIODevice::WriteOnly);
        out.device()->seek(0); //rewind

        socket().updateCommandCounter();

        ////////////////////////////
        // header
        ////////////////////////////
        out << (quint32)(socket().commandCounter() + msbCounter.toUInt(&ok,16)) //[0,3]
            << (quint32) config().vmmMap().mask //[4,7]
            //<< (quint32) config().getHDMIChannelMap() //[4,7]
            << (quint32) cmd.toUInt(&ok,16); //[8,11]

        ////////////////////////////
        // command
        ////////////////////////////
        out << (quint32) 0 //[12,15]
            << (quint32) 2 //[16,19]
            << (quint32) (tpskew + (tpwidth*16) + (tppolarity*128)); //[20,23]

        send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "RunModule::configTP");
        if(send_ok) {
            bool readOK = true;
            readOK = socket().waitForReadyRead("fec");
            if(readOK) {
                if(dbg()) msg()("Processing replies...","RunModule::configTP");
                socket().processReply("fec",ip);
            } else {
                msg()("Timeout while waiting for replies from VMM",
                        "RunModule::configTP");
                socket().closeAndDisconnect("fec","RunModule::configTP");
            }
        }
    } // ip

    socket().closeAndDisconnect("fec","RunModule::configTP");

}
*/












