#ifndef DATA_HANDLER_H
#define DATA_HANDLER_H

/////////////////////////////////////////
//
// data_handler
//
// Tool that interfaces between the GUI
// and the DAQ server
//
// daniel.joseph.antrim@cern.ch
// August 2016
//
//////////////////////////////////////////

#include <QObject>

//nsw
#include "daq_server.h"
#include "calibration_type.h"

//boost
#include <boost/shared_ptr.hpp>

//std/stdl
#include <iostream>

class Configuration;
class RunModule;
struct PulserCalibDef;
struct PulserCalibRunProperties;
class SocketHandler;

class GlobalSetting;
class FPGAClocks;
class VMMMap;
class Channel;

class MapHandler;
class OnlineMonTool;
class MessageHandler;
class CalibrationState;

class DataHandler : public QObject
{
    Q_OBJECT

    public :
        explicit DataHandler(QObject *parent = 0);
        virtual ~DataHandler(){};

        void LoadMessageHandler(MessageHandler& msg);
        MessageHandler& msg() { return *m_msg; }

        void LoadModulesForCalibration(Configuration& config, RunModule& run);//, SocketHandler& socket);
        bool load_pulser_calib(PulserCalibDef def);
        bool load_pulser_calib_run_properties(PulserCalibRunProperties props);
        void send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& vmmMap,
                std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq);
        
        void setDebug(bool dbg);
        bool dbg() { return m_dbg; }

        void set_vmm_type(int type);
        int vmm_type() { return m_vmm_type; }

        bool writeRaw() { return m_writeRaw; }
        bool writeNtuple() { return m_writeNtuple; }

        void resetCounts(); 
        int getCounts();

        void initialize();
        bool initialized() { return m_initialized; }

        void setOutputDir(std::string output_dir);
        bool initializeRun(bool writeRaw, bool writeNtuple, std::string dir, int run_number, int events_to_process, bool do_calibration=false, bool is_L0_decoding=false, bool is_xadc=false);
        void set_cktp_limit(int num_pulses_expected = -1);

        bool initializeCalibration(int vmm_id, int n_samples, int sample_period, CalibrationType type, bool skip_dead_channels, bool doL0);
        void start_xadc_sampling(std::string ip_address); 

        // load the DAQ mapping
        bool loadMapping(std::string filename);
        void disableMapping();
        MapHandler& mapHandler() { return *m_mapHandler; }
        bool mappingOK() { return m_mappingSetup; }
        std::string getFirstIP();
        int getNumberOfFecs();

        // setup the online monitoring
        void setupMonitoring(bool doit, std::string ip, int port, bool is_middle_of_run=false);
        void set_monitoring(bool mon_on=true);
        bool sendMonitoringConfiguration();
        void resetMonitoringConfiguration();

        void setMMFE8(bool do_mmfe8);

        void fillRunProperties(int gain, int tac_slope, int peak_time, int dac_threshold,
                int dac_amplitude, int angle, int tp_skew, int ckbc);
        void fillRunComment(std::string comment);
        void startGathering();
        bool begin_pulser_calibration();

        int checkForExistingFiles(std::string dir, int run_number);
        bool setOutputFile(std::string output_dir, int run_number, bool do_calibration);
        bool checkRootFile(std::string filename);

        void setDoMonitoring(bool doit);
        void setCalibrationRun(bool is_calibration);
        void updateCalibrationState(double gain, int dac_threshold, int dac_amplitude,
                double tp_skew, int peakTime);
        //void loadCalibrationState(CalibrationState state);
        void loadCalibrationState(int channel, int gain, int run_number, int tac_slope,
                int neighbor_trigger, int subhyst, int peak_time, int test_pusle_dac,
                int thresh_dac, std::vector<int> channel_trims, double tp_skew,
                int n_expected_pulses);


        // misc bit manipulation on Qt objects (QStrings) -- should be moved elsewhere
        static quint32 ValueToReplaceHEX32(QString hex, int bitToChange, bool newBitValue);
        static QByteArray bitsToBytes(QBitArray);
        static QBitArray bytesToBits(QByteArray bytes);
        static QString QBitArrayToString(const QBitArray& array);
        static quint32 reverse32(QString hex);
        

    private :
        MessageHandler* m_msg;
        std::stringstream m_sx;
        bool m_dbg;
        int m_vmm_type;
        bool m_initialized;

        std::string m_output_dir;
        std::string m_output_filename;
        std::string m_output_fullfilename;
        int n_total_events_to_process;
        int m_current_run_number;

        // run configuration
        bool m_writeRaw;
        bool m_writeNtuple;
        bool m_do_monitoring;
        bool m_is_calibration_run;

        // server to listen for and handle incoming UDP packets
        DaqServer* m_server;

        // tool for handling online monitoring
        OnlineMonTool* m_monTool;
        bool m_monitoringSetup;

        // tool for handling the data mapping
        MapHandler* m_mapHandler;
        bool m_mappingSetup;

        int times_updated;

    signals :
        void badOutputDir();
        void updateRunNumber(int);
        void updateCountsSend(int, int, unsigned int, unsigned int, bool);
        void update_empty_count_send(unsigned int);
        void occupancy_check_send(QString, int, int);
        void trigger_delta_check_send(int, int);
        void eventCountReachedSend();
        void effCountReachedSend();
        void updateCalibration();
        void daqHangObserved();
        void setup_monitoring(bool);
        void resetCountsSignal();
        void stop_pulser_calib_run();

    public slots :
        void testMon();
        void updateCounts(int, int, unsigned int, unsigned int, bool);
        void update_empty_count(unsigned int);
        void occupancy_check(QString, int, int);
        void trigger_delta_check(int, int);
        void toggle_occupancy_check(bool);
        void setCalibrationChannel(int);
        void eventCountReached();
        void effCountReached();
        void updateCalibrationState();
        void sendDaqHangSignal();
        void endRun(bool);
        void recv_setup_monitoring(bool, bool);
        void recv_send_monitor_config();
        void update_monitoring_rate(int);
        void pulser_calib_complete();

};

#endif
