#ifndef PULSER_CALIBRATION_H
#define PULSER_CALIBRATION_H

/////////////////////////////////////////////////////////////////
//
// pulser_calibration
//
// handles calibration loops based on the VMM digital data
// from internal pulser
//
// daniel.joseph.antrim@cern.ch
// July 2017
//
/////////////////////////////////////////////////////////////////

// Qt
#include <QObject>
#include <QTime>

// vmm
class Configuration;
class RunModule;
class ConfigHandler;
class SocketHandler;

#include "global_setting.h"
#include "vmm_map.h"
#include "channel.h"
#include "fpga_clocks.h"
#include "trigger_daq.h"

#include "vmm_decoder.h"
#include "trigger_event.h"
#include "calibration_type.h"

// std/stl
#include <string>
#include <vector>
#include <tuple>
#include <map>
#include <cstdint> // uintX_t

// ROOT
#include "TFile.h"
#include "TTree.h"
#include "TBranch.h"

struct PulserCalibHolder
{
    PulserCalibHolder(int index) :
        loop_idx(index) { clear(); }

    int idx() { return loop_idx; }
    void clear();
    std::string str();
    int loop_idx;

    int board_number;
    int vmm_mask;

    int gain_idx;

    int peak_time_idx;

    int tac_idx;

    int tp_skew;

    int threshold_dac_val;

    int pulser_dac_val;

    float analog_pulser_val;

};

struct PulserCalibDef
{
    PulserCalibDef() { clear(); }
    void clear();
    std::string str();

    CalibrationType type;

    int n_samples_per_channel;
    int n_expected;

    int channel_low;
    int channel_high;

    std::vector<int> board_numbers;
    int board_selection;
    int vmm_mask;

    int gain_idx_start;
    int gain_idx_stop;

    int peak_time_idx_start;
    int peak_time_idx_stop;

    int tac_idx_start;
    int tac_idx_stop;
    int tac_idx_step;

    float tp_time_per_step; // ns
    int tp_skew_start;
    int tp_skew_stop;
    int tp_skew_step;

    int threshold_dac_val_start;
    int threshold_dac_val_stop;
    int threshold_dac_val_step;

    int pulser_dac_val_start;
    int pulser_dac_val_stop;
    int pulser_dac_val_step;

    float analog_dac_val_start;
    float analog_dac_val_stop;
    float analog_dac_val_step;

};

struct PulserCalibRunProperties {

    PulserCalibRunProperties() { clear(); }
    void clear();
    std::string str();

    int run_number;
    std::string filename;
    std::string full_filename;
    std::string output_dir;
    std::string run_comment;
    
    bool l0_decoding;

    int subhysteresis;
    int neighbor_enable;
    std::vector<int> channel_trims;

    int cktk_max;
    int ckbc_frequency;
    int cktp_period;
    int cktp_width;
};

class PulserCalib : public QObject
{
    Q_OBJECT 

    public :
        explicit PulserCalib(QObject* parent = 0);
        virtual ~PulserCalib(){};

        typedef std::tuple<int, int, int> BoardChipChannelTuple;

        void set_debug(bool doit) { m_dbg = doit; }
        bool dbg() { return m_dbg; }

        void set_vmm_type(int type);
        void set_value_maps(int type);
        int vmm_type() { return m_vmm_type; }

        void LoadModulesForCalibration(Configuration& conf, RunModule& run);//, SocketHandler& socket);

        ConfigHandler* config_handle() { return m_config_handle; }
        bool load_calibration_loop(PulserCalibDef definition);
        bool load_run_properties(PulserCalibRunProperties props);
        bool initialize_output_file();
        bool init_tfile();
        bool init_ttree();
        void write_output();

        void load_flags(bool* flags);
        void build_calibration_quota();
        void build_calibration_loop();
        void load_initial_config(GlobalSetting& global, VMMMap& vmmMap,
            std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq);

        int n_loop();
        bool calib_loop(int i, PulserCalibHolder& holder);

        ///////////////////////////////////////////
        // data gathering
        ///////////////////////////////////////////
        bool start();
        void set_configuration(int loop_idx);
        bool is_valid_sample(std::vector<uint32_t>& datagram);
        void decode_pulser_data(std::string& ip_string, std::vector<uint32_t>& datagram, int& counter);
        void fill_trigger_event();
        void fill_event();
        bool above_quota(PulserCalib::BoardChipChannelTuple tuple);
        bool ready_for_next_step();
        void check_for_static_channels(int board_id);
        bool move_to_next_calib_step();
        void shutdown_channel(PulserCalib::BoardChipChannelTuple tuple);
        bool tuple_is_shutdown(BoardChipChannelTuple tuple);
        void print_quota();
        void reset_quota_map();
        void stop_pulser_calibration();

        void start_analog_pulser(float pulser_val);//, int loop_idx);

        bool is_valid_tuple(PulserCalib::BoardChipChannelTuple tuple);
        bool is_valid_board(int board);
        bool is_valid_chip(int chip);
        bool is_valid_channel(int channel);
        std::vector<int> m_loaded_boards;
        std::vector<int> m_loaded_chips;
        std::vector<int> m_loaded_channels;

    private :
        bool m_dbg;
        int m_vmm_type;
        int m_run_number;
        bool* m_accept_data;
        bool m_accept;

        std::map<int, float> m_gain_vals;
        std::map<int, float> m_tac_vals;
        std::map<int, float> m_peak_time_vals;

        Configuration* m_config_module;
        RunModule* m_run_module;
        SocketHandler* m_socket_handler;

        ConfigHandler *m_config_handle;

        GlobalSetting  m_def_global_settings;
        FPGAClocks  m_def_fpga_clocks_settings;
        VMMMap  m_def_vmm_map;
        std::vector<Channel> m_def_channel_settings;

        std::vector<std::string> m_board_ips;
        uint32_t m_vmm_mask;

        PulserCalibDef m_def;
        std::vector<PulserCalibHolder> m_calibration_loop_holder;
        float m_current_gain;
        float m_current_gain_val;
        float m_current_tac;
        float m_current_tac_val;
        float m_current_peak_time;
        float m_current_peak_time_val;
        float m_current_threshold_dac;
        float m_current_pulser_dac;
        float m_current_tp_skew;
        float m_current_tp_skew_ns;
        float m_current_analog_dac_val;
        

        PulserCalibRunProperties m_props;

        int m_current_loop_idx;
        bool m_calib_complete;

        int n_unique_channels;
        int n_shutdown;
        std::map<BoardChipChannelTuple, int> m_quota_map;
        std::map<BoardChipChannelTuple, int> m_last_count_check;
        std::map<BoardChipChannelTuple, int> m_static_count;
        std::vector<BoardChipChannelTuple> m_ignore_list;
        std::vector<BoardChipChannelTuple> m_shutdown_list;
        bool move_to_next_test;

        VMMDecoder m_decoder;
        TriggerEvent m_trigger_events;
        uint32_t m_last_trigger_read;

        QTime m_start_run_time;

        ////////////////////////////////////////////
        // ROOT STUFF
        ////////////////////////////////////////////
        TFile* m_root_file;
        std::string m_output_filename;

        // variables for branches
        std::vector<int> m_boardId;
        std::vector<int> m_chipId;
        std::vector< std::vector<int> > m_pdo;
        std::vector< std::vector<int> > m_tdo;
        std::vector< std::vector<int> > m_channelId;
        std::vector< std::vector<int> > m_threshold;
        std::vector<int> m_triggerCounter;
        std::vector<int> m_eventSize;
        std::vector< std::vector<int> > m_flag;
        std::vector< std::vector<int> > m_bcid;
        std::vector< std::vector<int> > m_grayDecoded;
        std::vector< std::vector<int> > m_neighbor_calib;
        std::vector< std::vector<int> > m_overflow;
        std::vector< std::vector<int> > m_orbit_count;
        std::vector< std::vector<int> > m_relbcid;

        // TTrees and TBranches
        TTree* m_tree;

        // run-state
        TBranch* br_runNumber;
        TBranch* br_gain;
        TBranch* br_tacSlope;
        TBranch* br_peakTime;
        TBranch* br_dacCounts;
        TBranch* br_tpSkew;
        TBranch* br_pulserCounts;
        TBranch* br_analogPulser;
        TBranch* br_subhysteresis;
        TBranch* br_neighbortrigger;
        TBranch* br_nExpected;
        TBranch* br_channelTrims;
        TBranch* br_neighborCalib; 

        // data
        TBranch* br_boardId;
        TBranch* br_chipId;
        TBranch* br_channelId;
        TBranch* br_tdo;
        TBranch* br_pdo;
        TBranch* br_thresh;
        TBranch* br_flag;
        TBranch* br_grayDecoded;
        TBranch* br_bcid;
        TBranch* br_evSize;

    signals :
        void send_configuration();
        void set_clocks_pulser(int, int, int, int, int, int);
        void set_acq_pulser(int);
        void pulser_calib_complete();
        void calib_configuration_update(int, int);

    public slots :


}; // class PulserCalib


#endif
