#ifndef DAQ_SERVER_H
#define DAQ_SERVER_H

/////////////////////////////////////////
//
// daq_monitor
//
// Tool for monitoring the DAQ socket (the
// UDP socket listening for and responding to
// VMM data)
//
//  - at pre-set intervals, if the DAQ is stopped
//    due to buggy QUdpSocket, this tool will
//    initiate a socket shutdown and restart
//  - Qt QUdpSocket has known issues before
//    Qt version 5.5/5.6
//
// daniel.joseph.antrim@cern.ch
// August 2016
//
//////////////////////////////////////////

#include <QObject>
#include <QTime>

//std/stl
#include <sstream>

//boost
//DaqBuffer
//#include <boost/array.hpp>
#include <boost/thread.hpp>
#include <boost/asio.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/bind.hpp>
//#include <boost/atomic.hpp>

//nsw
#include "event_builder.h"
#include "data_array_types.h" // RawDataArray
#include "calibration_module.h"
#include "daq_monitor.h"

class Configuration;
class RunModule;
class SocketHandler;
struct PulserCalibDef;
struct PulserCalibRunProperties;
class GlobalSetting;
class VMMMap;
class Channel;
class FPGAClocks;
class TriggerDAQ;

class MapHandler;
class OnlineMonTool;
class MessageHandler;
class CalibrationState;
class DaqBuffer;

//#define MAXBUFLEN 65507


class DaqServer : public QObject 
{

    Q_OBJECT

    public :
        explicit DaqServer(QObject *parent=0);
        virtual ~DaqServer(){};

        void setDebug(bool dbg);
        bool dbg() { return m_dbg; }

        void set_vmm_type(int type); 
        int vmm_type() { return m_vmm_type; }

        void LoadMessageHandler(MessageHandler& msg);
        MessageHandler& msg() { return *m_msg; }

        void LoadModulesForCalibration(Configuration& config, RunModule& run); //, SocketHandler& socket);
        bool load_pulser_calib(PulserCalibDef def);
        bool load_pulser_calib_run_properties(PulserCalibRunProperties props);
        bool begin_pulser_calibration();
        void send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& vmmMap,
                std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq);

        void loadMappingTool(MapHandler& maptool);
        void disableMapping();

        void loadMonitoringTool(OnlineMonTool& montool);
        void setMonitoringStatus(bool status);
        void startMonitoring();
        void stopMonitoring();

        void initialize();

        bool initializeRun(bool writeRaw, bool writeNtuple, std::string filename, int run_number, int num_events_to_process, bool do_calibration=false, bool do_L0_decoding=false, bool is_xadc = false);
        void set_cktp_limit(int n_pulses_expected = -1);
        bool initializeCalibration(int vmm_id, int samples, int sample_period, CalibrationType type, bool skip_dead_channels, bool doL0decoding);
        void start_xadc_sampling(std::string ip); 

        void setMMFE8(bool do_mmfe8);

        void setDoMonitoring(bool do_mon);
        void setCalibrationRun(bool calib_run);

        void setCalibrationChannel(int channel);
        void updateCalibrationState(double gain, int dac_thresh, int dac_ampl,
                    double tp_skew, int peakTime);
        //void loadCalibrationState(CalibrationState state);
        void loadCalibrationState(int channel, int gain, int run_number,
                int tac_slope, int neighbor_trigger, int subhyst,
                int peak_time, int test_pulse_dac, int thresh_dac, std::vector<int> trims, double tp_skew,
                int n_expected_pulses);

        void fillRunProperties(int gain, int tac_slope, int peak_time, int dac_threshold,
                int dac_amplitude, int angle, int tp_skew, int ckbc);
        void fillRunComment(std::string comment);

        int getCounts();
        int getTrigCounts();

        void listen();


        void handle_data();

        //void decode_data(int& daq_count, const boost::system::error_code error, std::size_t size_);
        void gather_data(const boost::system::error_code error, std::size_t size_);

        //DaqBuffer
        void read_data(int& daq_count, int& trig_count);

        void stop_listening();
        void flush();

        void stop_server();
        bool is_stopped();

        void write_output();

        void update_monitoring_rate(int rate);

    private :
        MessageHandler* m_msg;
        bool m_dbg;
        int m_vmm_type;
        bool m_do_l0;
        std::stringstream m_sx;

        bool m_use_pulser_calib;

        boost::shared_ptr<bool> m_continue_gathering;
        boost::shared_ptr<bool> m_flushing;
        boost::shared_ptr<bool> m_continue_calibration_gathering;
        bool m_mini2;
        int m_daq_port;
        int m_calibration_port;
        bool m_is_calibration_run;
        int m_run_number;
        int m_total_events_to_process;

        //dataflow
        unsigned int n_input_packets;
        unsigned int n_input_empty_packets;
        //occupancy
        bool m_do_occupancy_check;
        QTime m_run_time;
        double m_last_occ_check;
        int m_empty_event;
        int m_full_event;
        int* m_trigger_delta;
        int* m_trigger_board_delta;
        

        boost::mutex hitcount_mutex;
        bool do_flush;
        bool m_do_eff;
        int n_effCount;
        bool reported;
        int n_pulse_limit;
        int n_daqCount;
        bool* m_accept_pulser_data;
        boost::mutex trigcount_mutex;
        int n_trigCount;

        CalibrationType m_current_caltype;
        int n_cal_check;

        int n_received;
        int m_thread_count;
        boost::thread_group m_thread_group;
        RawDataArray m_data_buffer;
        //DaqBuffer
        //boost::array<uint32_t, MAXBUFLEN> m_data_buffer;

        bool m_monitor_status;
        bool* m_monitor;

        boost::shared_ptr<boost::asio::ip::udp::socket> m_socket;
        boost::shared_ptr<boost::asio::io_service> m_io_service;
        boost::shared_ptr<boost::asio::io_service::work> m_idle_work;
        boost::asio::ip::udp::endpoint m_remote_endpoint;

        // DaqBuffer
        boost::shared_ptr<DaqBuffer> m_buffer;

        boost::shared_ptr<boost::timed_mutex> m_mutex;
        boost::shared_ptr<boost::condition_variable_any> m_fill_condition;

        boost::shared_ptr<EventBuilder> m_event_builder;
        //boost::shared_ptr<CalibModule> m_calibration_module;
        CalibModule* m_calibration_module;
        DaqMonitor* m_daqMonitor;
        bool m_set_daq_monitor;



    signals :
        void eventCountReached();
        void effCountReached();
        void updateCounts(int,int, unsigned int, unsigned int, bool); // signal to update the GUI counter
        void update_empty_count(unsigned int);
        void perform_occupancy_check(QString, int, int); // [IP, is full, is empty]
        void perform_trigger_delta_check(int, int); // board #, delta w.r.t. expected trigger
        // send signal that says expected number of calibration samples
        // have been taken
        void updateCalibrationState();
        void resetCalibrationCounts();
        //void load_calibration(CalibrationState state);
        void load_calibration(int, int, int, int, int, int, int, int, int);
        void initializeCalibrationSignal(int, int, int, int, bool, bool);
        void start_xadc(QString);
        void stopTimer();
        void daqHangObserved();
        void setup_monitoring(bool);

        void begin_pulser_calibration_signal();
        void pulser_calib_complete_signal();

    public slots :
        void sendDaqHangSignal();
        void toggleCalibrationSocket();
        //void updateCounter();
        void recv_setup_monitoring(bool);
        void resetCounts();
        void pulser_calib_complete();

        // occupancy check
        void toggle_occupancy_check(bool);

        

};

#endif
