
#include "OnlineMonTool.h"

//qt
#include <QtNetwork>
#include <qlocalserver.h>
#include <qlocalsocket.h>
#include <QByteArray>
#include <QDataStream>

//std/stl
#include <iostream>
using std::cout;
using std::endl;

// boost
//#include <boost/mutex.hpp>
boost::mutex output_stream_lock;
using boost::asio::ip::udp;
#include <boost/algorithm/string/split.hpp>                                      
#include <boost/algorithm/string.hpp>
namespace ip = boost::asio::ip;


//////////////////////////////////////////////////////////////////////////////
// ------------------------------------------------------------------------ //
//  OnlineMonTool
// ------------------------------------------------------------------------ //
//////////////////////////////////////////////////////////////////////////////
OnlineMonTool::OnlineMonTool(QObject *parent) :
    QObject(parent),
    m_dbg(false),
    n_warnings(0),
    m_refresh_rate(500),
    m_bind_port(2222),
    m_monitor_ip("127.0.0.2"),
    m_monitor_port(2224),
    m_accept_events(true),
    m_configuration_sent(false),
    m_is_started(false),
    m_timer(0)
{
    QHostAddress address = QHostAddress::LocalHost;
    std::string ip_address = address.toString().toStdString();

    m_monitor_port = 2224;
    m_monitor_ip = ip_address;
    m_remote_endpoint.address(ip::address::from_string(m_monitor_ip));
    m_remote_endpoint.port(m_monitor_port);

    std::cout << "OnlineMonTool::OnlineMonTool    INFO Assuming that monitoring application "
            << "receives at IP: " << m_monitor_ip << ", port: " << m_monitor_port << std::endl;
}

// ------------------------------------------------------------------------ //
bool OnlineMonTool::initialize(boost::shared_ptr<boost::asio::io_service> io_service)
{
    bool init_ok = true;
    if(m_socket) {
        if(m_socket->is_open()) {
            m_socket->close();
            boost::system::error_code ec;
            m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
        }
    }

    m_io_service = io_service;
    m_socket = boost::shared_ptr<boost::asio::ip::udp::socket>(new ip::udp::socket(*m_io_service,
            ip::udp::endpoint(ip::udp::v4(), m_bind_port)));


    if(!m_socket->is_open()) {
        if(dbg()) {
            output_stream_lock.lock();
            std::cout << "OnlineMonTool::initialize    ERROR Monitoring UDP socket unable to be setup properly" << std::endl;
            output_stream_lock.unlock();
        }
        init_ok = false;
    }

    return init_ok;
}
// ------------------------------------------------------------------------ //
void OnlineMonTool::setMonitorAddress(std::string ip, int port)
{
    m_monitor_ip = ip;
    m_monitor_port = port;

    m_remote_endpoint.address(ip::address::from_string(ip));
    m_remote_endpoint.port(port);

    if(dbg()) {
        output_stream_lock.lock();
        std::cout << "OnlineMonTool::setMonitorAddress    Setting monitoring address to: "
                << "IPv4:Port : " << ip << ":" << port << std::endl;
        output_stream_lock.unlock();
    }
}
// ------------------------------------------------------------------------ //
void OnlineMonTool::closeMonitoringSocket()
{
    if(dbg()) {
        output_stream_lock.lock();
        std::cout << "OnlineMonTool::closeMonitoringSocket    Shutting down communication with monitoring application... " << std::endl;
        output_stream_lock.unlock();
    }

    if(m_socket) {
        m_socket->close();
        boost::system::error_code ec;
        m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
    }


}
// ------------------------------------------------------------------------ //
void OnlineMonTool::addEventString(std::string event_string)
{
    if(m_accept_events)
        m_event_string_vector.push_back(event_string);
    else {
        return;
    }
}

// ------------------------------------------------------------------------ //
void OnlineMonTool::testMon()
{
    std::cout << "online mon tool testMon" << std::endl;
    std::string test = "VMM2.00.01 1 2 3 4 5";
    for(int i = 0; i < 10; i++)
        addEventString(test);

}
void OnlineMonTool::start()
{
    if(dbg()) {
        output_stream_lock.lock();
        std::cout << "OnlineMonTool::start    Starting monitoring loop..." << std::endl;
        output_stream_lock.unlock();
    }
    n_warnings = 0;
    
    //timer
    //if(!m_timer) {
    //m_timer = boost::shared_ptr<boost::asio::deadline_timer>(new boost::asio::deadline_timer(*m_io_service, boost::posix_time::milliseconds(10)));
    m_timer = boost::shared_ptr<boost::asio::deadline_timer>(new boost::asio::deadline_timer(*m_io_service, boost::posix_time::milliseconds(m_refresh_rate)));
    m_timer->async_wait(boost::bind(&OnlineMonTool::send, this));

    m_is_started = true;

}

void OnlineMonTool::restart()
{
    m_timer->expires_at(m_timer->expires_at() + boost::posix_time::seconds(100));
    m_timer->async_wait(boost::bind(&OnlineMonTool::send, this));
    m_is_started = true;
}

void OnlineMonTool::pause()
{
    if(m_is_started) {
        m_timer->cancel();
        m_is_started = false;
    }
}

void OnlineMonTool::stop()
{

    if(m_is_started) {
        if(dbg()) {
            output_stream_lock.lock();
            std::cout << "OnlineMonTool::stop    Stopping monitoring loop and closing socket..." << std::endl;
            output_stream_lock.unlock();
        }
        m_timer->cancel();
    }
    closeMonitoringSocket();

    m_is_started = false;
}

//void OnlineMonTool::stopTimer()
//{
//    m_timer->cancel();
//}

void OnlineMonTool::resetMonitoringConfiguration()
{
    m_configuration_sent = false;
}

void OnlineMonTool::sendConfiguration(std::vector<std::string> words)
{
//    if(m_is_started) {
//        m_timer->expires_at(m_timer->expires_at() + boost::posix_time::seconds(3));
//        m_timer->async_wait(boost::bind(&OnlineMonTool::send, this));
//    }
    if(words.size()) {
        for(unsigned int i = 0; i < words.size(); i++) {
            sendString(words.at(i), true);
        }
    }

    m_configuration_sent = true;
}


void OnlineMonTool::send_trig()
{

    return;
}
void OnlineMonTool::send()
{
    // dummy strings for testing purposes
    //for(int i = 0; i < 5; i++) {
    //    std::stringstream sx;
    //    sx << i << "  VMM2.00.0 444 222 111";
    //    m_event_string_vector.push_back(sx.str());
    //    sx.str("");
    //}

    if(m_event_string_vector.size()) {
        m_accept_events = false;
        std::vector<std::vector<std::string>> out_vector;

        size_t n_per_send = 60;
        size_t n_total = m_event_string_vector.size();
        std::vector<std::string> tmpVector; tmpVector.clear();
        for(size_t i = 0; i < n_total; i++) {
            tmpVector.push_back(m_event_string_vector.at(i));
            bool pass_through = (n_total - (i+1)) < n_per_send;
            if(i%n_per_send==0){ // || pass_through) {
                out_vector.push_back(tmpVector);
                tmpVector.clear();
                pass_through = false;
            }
            if(i==(n_total-1) && pass_through) {
                out_vector.push_back(tmpVector);
                tmpVector.clear();
                pass_through = false;
            }
        }

        std::stringstream send_string;
        for(int i = 0; i < (int)out_vector.size(); i++) {
            for(int j = 0; j < (int)out_vector.at(i).size(); j++) {
                if(out_vector.at(i).at(j)=="") continue;
                send_string << out_vector.at(i).at(j) << "  ";
            } // j
            sendString(send_string.str(), false);

            // reset the stream to prevent overflow
            send_string.str("");
        } // i
        out_vector.clear();
    }
    // we've sent all of the strings, now clear the buffer
    m_event_string_vector.clear();
    m_accept_events = true;

    // keep the timer loop going
    //m_timer->expires_at(m_timer->expires_at() + boost::posix_time::seconds(1));
    m_timer->expires_at(m_timer->expires_at() + boost::posix_time::milliseconds(m_refresh_rate));
    m_timer->async_wait(boost::bind(&OnlineMonTool::send, this));
}
// ------------------------------------------------------------------------ //
void OnlineMonTool::sendString(std::string event_string, bool is_config)
{
    size_t n_bytes_sent;
    bool ok = true;

    if(is_config) {

        boost::asio::io_service tmp_service;
        boost::asio::ip::udp::socket* tmp_socket = new boost::asio::ip::udp::socket(tmp_service, boost::asio::ip::udp::endpoint(ip::udp::v4(), m_bind_port+1));
        try {
            n_bytes_sent = tmp_socket->send_to(boost::asio::buffer(event_string), m_remote_endpoint);
        }
        catch(boost::system::system_error err) {
            output_stream_lock.lock();
            std::cout << "OnlineMonTool::sendString    Exception caught: " << err.what() << std::endl;
            std::cout << "OnelineMonTool::sendString    > Unable to send config packet to "
                << "IP: " << m_remote_endpoint.address().to_string() << ", " << m_remote_endpoint.port() << std::endl;
            output_stream_lock.unlock();
            ok = false;
        }
        delete tmp_socket;
        return;
    }


    if(!m_socket) {
        cout << "OnlineMonTool::sendString    Monitoring UDP socket is null. Stopping monitoring loop." << endl;
        m_timer->cancel();
        return;
    }

    try {
        n_bytes_sent = m_socket->send_to(boost::asio::buffer(event_string), m_remote_endpoint);
        //output_stream_lock.lock();
        //std::cout << "OnlineMonTool::sendString   sending str: " << event_string << std::endl;
        //output_stream_lock.unlock();
    } // try
    catch(boost::system::system_error err) {
        if(n_warnings < 20) {
            output_stream_lock.lock();
            std::cout << "OnlineMonTool::sendString    WARNING [" << n_warnings << "/20]    Exception caught, missed sending monitoring packet: " << err.what() << std::endl;
            output_stream_lock.unlock();
            n_warnings++;
        }
        ok = false;
    }
    if(!n_bytes_sent && ok) {
        if(dbg()) {
            ip::address address = m_remote_endpoint.address();
            int port = m_remote_endpoint.port();
            output_stream_lock.lock();
            std::cout << "OnlineMonTool::sendString    Failed to send " << (is_config ? "config" : "event") << " string to IPv4:port : "
                << address.to_string() << ":" << port << std::endl;
            output_stream_lock.unlock();
        }
    }
}


//// ------------------------------------------------------------------------ //
//void OnlineMonTool::sendEventStrings()
//{
//    if(m_event_string_vector.size()) {
//        m_accept_events = false;
//        std::stringstream eventstring;
//        for(int i = 0; i < (int)m_event_string_vector.size(); i++) {
//            eventstring << m_event_string_vector.at(i);
//            if(i!=((int)m_event_string_vector.size()-1)) eventstring << "  ";
//        }
//        send(eventstring.str());
//    }
//    else { m_accept_events = true; }
//}
//// ------------------------------------------------------------------------ //
//void OnlineMonTool::send(std::string event_string)
//{
//    size_t n_bytes_sent;
//    bool ok = true;
//    try {
//        n_bytes_sent = m_socket->send_to(boost::asio::buffer(event_string), m_remote_endpoint);
//    } // try
//    catch(boost::system::system_error err) {
//        output_stream_lock.lock();
//        std::cout << "OnlineMonTool::send    Exception caught: " << err.what() << std::endl;
//        output_stream_lock.unlock();
//        ok = false;
//    }
//    if(!n_bytes_sent && ok) {
//        if(dbg()) {
//            ip::address address = m_remote_endpoint.address();
//            int port = m_remote_endpoint.port();
//            output_stream_lock.lock();
//            std::cout << "OnlineMonTool::send    Failed to send event string to IPv4:port : "
//                    << address.to_string() << ":" << port << std::endl;
//            output_stream_lock.unlock();
//        }
//    }
//
//    m_accept_events = true;
//}
