#ifndef CONFIG_HANDLER_H
#define CONFIG_HANDLER_H

/////////////////////////////////////////
//
// config_handler
//
// containers for global configuration
// (global, VMM channels, IP/socket information)
//
// daniel.joseph.antrim@cern.ch
// March 2016
//
//////////////////////////////////////////

// qt
#include <QObject>
#include <QList>
#include <QStringList>

// boost
#include <boost/property_tree/ptree.hpp>
#include <boost/property_tree/xml_parser.hpp>

// std/stdl
#include <iostream>

// vmm
#include "message_handler.h"
#include "trigger_daq.h"
#include "comm_info.h"
#include "global_setting.h"
#include "vmm_map.h"
#include "channel.h"
#include "fpga_clocks.h"

// ---------------------------------------------------------------------- //
//  Main Configuration Handler tool
// ---------------------------------------------------------------------- //
class ConfigHandler : public QObject
{
    Q_OBJECT

    public :
        explicit ConfigHandler(QObject *parent = 0);
        virtual ~ConfigHandler(){};

        ConfigHandler& setDebug(bool dbg) { m_dbg = dbg; return *this;}
        bool dbg() { return m_dbg; }

        //addmmfe8
        void setMMFE8(bool set_to_run_mmfe8);
        bool mmfe8() { return m_mmfe8; }

        void LoadMessageHandler(MessageHandler& msg);
        MessageHandler& msg() { return *m_msg; }

        bool LoadConfig(const QString &filename);
        void WriteConfig(QString filename);

        CommInfo LoadCommInfo(const boost::property_tree::ptree& p);
        void LoadCommInfo(const CommInfo& info);

        GlobalSetting LoadGlobalSettings(const boost::property_tree::ptree& p);

        TriggerDAQ LoadDAQConfig(const boost::property_tree::ptree& p);

        //std::vector<VMMMap> LoadHDMIChannels(const boost::property_tree::ptree& p);
        VMMMap LoadVMMMap(const boost::property_tree::ptree& p);
        bool setVMMMap();
        quint16 getVMMMask() { return m_vmmMask; }

        std::vector<Channel> LoadVMMChannelConfig(const boost::property_tree::ptree& p);

        FPGAClocks LoadFPGAClockSettings(const boost::property_tree::ptree& p);
        void LoadFPGAClocksConfiguration(FPGAClocks& clocks);

        // methods for GUI interaction
        void LoadBoardConfiguration(GlobalSetting& global,
             VMMMap& vmmMap, std::vector<Channel>& channels);
        void LoadTDAQConfiguration(TriggerDAQ& daq);

        bool EnabledOrDisabled(int val, std::string& out, std::string reg_name, bool invert_logic);
        bool isEnabledOrDisabled(std::string reg, int& idx, std::string reg_name, bool invert_logic);
        int isOn(std::string onOrOff = "", std::string where ="");
        std::string isOnOrOff(int onOrOf, std::string node);
        std::string isEnaOrDis(int enaOrDis, std::string node);

        QString getIPList() { return m_commSettings.ip_list; }

        // retrieve the objects
        CommInfo& commSettings()                { return m_commSettings; }
        TriggerDAQ& daqSettings()               { return m_daqSettings; }
        GlobalSetting& globalSettings()         { return m_globalSettings; }
        FPGAClocks& clocksSettings()            { return m_fpgaClockSettings; }
        VMMMap& hdmiChannelSettings(int i)      { return m_channelmap[i]; }
        VMMMap& vmmMap()                        { return m_vmmMap; }
        Channel& channelSettings(int i)         { return m_channels[i]; }

        // set the currently selected board(s)
        void setSelectedBoards(int board_selection); 
        int boardSelection() { return m_board_selection; }

        ////////////////////////////
        // expected settings
        ////////////////////////////
        //static const QStringList all_gains;
        //static const QList<int>  all_peakTimes;
        //static const QStringList all_s6TPskews;
        //static const QList<int>  all_TACslopes;
        //static const QStringList all_polarities;
        //static const QStringList all_ARTmodes;
        //static const QStringList all_directTimeModes;
        //static const QStringList all_ADC10bits;
        //static const QStringList all_ADC8bits;
        //static const QStringList all_ADC6bits;
        //// s6
        //static const QStringList all_CKTK;
        //static const QStringList all_CKBC;
        //static const QStringList all_CKBC_SKEW;

    private :
        bool m_dbg;
        int m_board_selection;
        //addmmfe8
        bool m_mmfe8;

        CommInfo                m_commSettings;
        TriggerDAQ              m_daqSettings;
        GlobalSetting           m_globalSettings;
        FPGAClocks              m_fpgaClockSettings;
        std::vector<VMMMap>     m_channelmap;
        VMMMap                  m_vmmMap;
        quint16                 m_vmmMask;
        std::vector<Channel>    m_channels;

        MessageHandler* m_msg;

}; // class ConfigHandler



#endif
