#ifndef EVENT_BUILDER_H
#define EVENT_BUILDER_H

/////////////////////////////////////////
//
// event_builder
//
//  Tool that handles the decoding of the
//  data coming from the front-end as
//  received by the DaqServer. Handles also
//  the storage of the data to its output
//  files
//
//    - Currenty decodes mini2 and mmfe8 data
//    - Outputs only to ROOT ntuple
//
// daniel.joseph.antrim@cern.ch
// August 2016
//
//////////////////////////////////////////

//std/stl
#include <string>
#include <iostream>
#include <vector>
#include <sstream>

//boost
#include <boost/shared_ptr.hpp>
//DaqBuffer
//#include <boost/array.hpp>
#include <boost/thread.hpp>
#include <boost/asio.hpp>

//nsw
class MapHandler;
class OnlineMonTool;
class MessageHandler;
#include "data_array_types.h" // RawDataArray
#include "trigger_event.h"
#include "raw_data_dumper.h"
#include "vmm_decoder.h"

//ROOT
class TFile;
class TTree;
class TBranch;

//#define MAXBUFLEN 65507

class EventBuilder 
{

    public :
        EventBuilder();
        virtual ~EventBuilder(){};

        void load_mon_flag(bool* mon);

        void LoadMessageHandler(MessageHandler& msg);
        MessageHandler& msg() { return *m_msg; }

        void setDebug(bool dbg) { m_dbg = dbg; }
        bool dbg() { return m_dbg; }

        void set_vmm_type(int type);// { m_vmm_type = type; }
        int vmm_type() { return m_vmm_type; }

        void resetCounts(); // { n_push_back = 0; m_last_trigger_read = 0xffffffff; }

        // load the mapping tool
        void loadMappingTool(MapHandler& maptool);
        void disableMapping();
        MapHandler& mapHandler() { return *m_mapHandler; }
        bool mapping() { return m_mapping_setup; }

        // load the monitoring tool
        void loadMonitoringTool(OnlineMonTool& montool);
        void initializeMonitoring(boost::shared_ptr<boost::asio::io_service> service);
        OnlineMonTool& monTool() { return *m_monTool; }
        void setMonitoringStatus(bool status);
        //bool monitoring() { return m_monitoringStatus; }
        bool monitoring() { return (*m_do_do_monitoring); }
        void startMonitoring();
        void stopMonitoring();
        void set_monitor_rate(int rate); // { m_monitor_rate = rate; }

        // initialize the output file and trees
        bool initializeRun(bool writeRaw, bool writeNtuple, std::string filename, int run_number,
                        bool do_L0_decoding=false);

        void setCalibrationRun(bool is_calib) { m_calibRun = is_calib; }
        void setCalibrationChannel(int channel) { m_calib_channel = channel; }
        void updateCalibrationState(double gainIdx, int dacThreshold, int dacAmplitude,
                                        double tpSkewIdx, int peakTimeIdx);

        //void get_sync_items(boost::mutex& data_mutex, boost::condition_variable_any data_condition);
        //void get_sync_items(boost::shared_ptr<boost::timed_mutex> data_mutex, boost::shared_ptr<boost::condition_variable_any> data_condition);

        //void loadCounter(boost::shared_ptr<int> counter);
        void resetCounter() { n_daqCount = 0; }
        int getDAQCounts() { return (n_daqCount); }

        // obtain the run properties
        void fillRunProperties(int gain, int tac_slope, int peak_time, int dac_threshold,
                int dac_pulser, int angle, int tp_skew, int ckbc);
        void fillRunComment(std::string comment) { m_run_comment = comment; }

        void setupOutputTrees();

        bool writeRaw() { return m_writeRaw; }
        bool writeNtuple() { return m_writeNtuple; }
        bool calibrationRun() { return m_calibRun; }

        void print_data(std::string msg, int& daq_counter);

        void receive_event_data(std::string& ip_string, std::vector<uint32_t>& data, uint32_t length, int& counter, int& trigger_counter);
        void store_raw_data(std::string& ip_string, std::vector<uint32_t>& data, int& counter);
        void decode_event(std::string& ip_string, std::vector<uint32_t>& data, int& counter);
        bool is_valid_packet(uint32_t length);
        void decode_event_trig(std::string& ip_string, std::vector<uint32_t>& data, uint32_t length, int& counter, int& trig_counter);
        void fill_trigger_event(int& hitcounter, int& trigcounter);
        uint32_t ip_str_to_int(std::string& ip_string);
        int ip_str_to_id(std::string& ip_string);



        uint32_t decodeGray(uint32_t gray);

        void fill_event();
        void flush_remaining_events(int& hits, int& trigs);
        void write_output();

        void clearData();

        //int n_push_back;
        bool m_is_first_fill;
        int n_hits;
        int n_triggers;
        
        bool m_calibRun; // this is a calibration run

        //dataflow
        int* m_trigger_for_delta;
        int* m_board_for_delta;
        void get_delta_params(int* trigger_counter, int* board_counter);

        // dummy test
        void print_mon_flag();

    private :

        MessageHandler* m_msg;
        std::stringstream m_sx;
        std::stringstream m_monitoring_stream;

        bool m_dbg;
        int m_vmm_type;

        // vmm data decoding
        VMMDecoder m_decoder;

        // for storing the raw data
        RawDataDumper* m_raw_dumper;

        // mapping
        MapHandler* m_mapHandler;
        bool m_mapping_setup;

        // online monitoring tool
        OnlineMonTool* m_monTool;
        bool m_monitoringStatus;
        bool* m_do_do_monitoring;
        bool m_current_monitoring_state;
        int m_monitor_rate;
        int n_monitor_strings;

        std::string m_output_rootfilename;
        std::string m_output_calib_rootfilename;
        int m_run_number;

        bool m_writeRaw;
        bool m_writeNtuple; // flag for whether we are writing an output ntuple (ROOT) file
        bool m_do_L0_decoding;
        bool m_fill_event;

        bool m_xADC; //is this ADC stuff 
        bool m_pulse_bool;
        bool m_chan_thres_bool;
        bool m_thres_range_bool;
        int m_calib_channel;

        // event counter
        int n_daqCount;

        // mutex for event data access between threads
        boost::shared_ptr<boost::timed_mutex> m_data_mutex;
        //boost::mutex m_data_mutex;

        // condition for accessing 
        boost::shared_ptr<boost::condition_variable_any> m_event_fill_condition;
        //boost::condition_variable_any m_event_fill_condition;
        

        // the file to hold the output data ntuple
        TFile* m_daqRootFile;

        ///////////////////////////////////////////////////////////////
        ///////////////////////////////////////////////////////////////
        // EVENT DATA CONTAINERS
        ///////////////////////////////////////////////////////////////
        ///////////////////////////////////////////////////////////////

        // global run properties
        int m_gain;
        int m_tacSlope;
        int m_peakTime;
        int m_dacCounts;
        int m_pulserCounts;
        int m_tpSkew; // in steps
        int m_angle;
        int m_ckbc; // CKBC frequency, used for converted TP skew

        // vmm event data (on the fly containers)
        std::vector<int> _pdo;
        std::vector<int> _tdo;
        std::vector<int> _bcid;
        std::vector<int> _relbcid; // L0
        std::vector<int> _overflow; // L0
        std::vector<int> _orbit_count; // L0
        std::vector<int> _grayDecoded;
        std::vector<int> _channelId;
        std::vector<int> _febChannelId;
        std::vector<int> _mappedChannelId;
        std::vector<int> _flag;
        std::vector<int> _threshold;
        std::vector<int> _neighbor;

        // data to store in output ntuple per event 
        int m_eventNumberFAFA;
        int m_daqCnt;

        std::vector<int> m_boardId;
        std::vector<int> m_triggerTimeStamp;
        std::vector<int> m_triggerCounter;
        std::vector<int> m_chipId;
        int m_mini2_eventsize_per_packet;
        std::vector<int> m_eventSize;
        std::vector<int> m_art; // ART
        std::vector<int> m_art_valid; // ART
        std::vector<int> m_art_trigger; // ART

        std::vector< std::vector<int> > m_pdo;
        std::vector< std::vector<int> > m_tdo;
        std::vector< std::vector<int> > m_flag;
        std::vector< std::vector<int> > m_threshold;
        std::vector< std::vector<int> > m_bcid;
        std::vector< std::vector<int> > m_relbcid; // L0
        std::vector< std::vector<int> > m_overflow; // L0
        std::vector< std::vector<int> > m_orbit_count; // L0
        std::vector< std::vector<int> > m_grayDecoded;
        std::vector< std::vector<int> > m_channelId;
        std::vector< std::vector<int> > m_febChannelId;
        std::vector< std::vector<int> > m_mappedChannelId; // detector element strip number

        int m_pulserCounts_calib;
        double m_gain_calib;
        int m_peakTime_calib;
        int m_dacCounts_calib;
        double m_tpSkew_calib;
        std::vector< std::vector<int> > m_neighbor_calib;

        ///////////////////////////////////////////////////////////////
        ///////////////////////////////////////////////////////////////
        // EVENT OUTPUT NTUPLES BELOW
        ///////////////////////////////////////////////////////////////
        ///////////////////////////////////////////////////////////////

        // the TTrees for the various data
        TTree* m_runProperties_tree; // global run properties tree
        TTree* m_vmm_tree; // vmm event data

        std::string m_run_comment; // comment/info for run from user

        // branches for the run properties tree
        TBranch *br_runNumber;
        TBranch *br_gain;
        TBranch *br_tacSlope;
        TBranch *br_peakTime;
        TBranch *br_dacCounts; // TP threshold
        TBranch *br_pulserCounts; // TP amplitude
        TBranch *br_TPskew;
        TBranch *br_ckbc;
        TBranch *br_angle; // incident angle (relative angle of chamber and beam)
        TBranch *br_calibrationRun; // flag for if the run is calibration

        // branches for the vmm tree
        TBranch *br_eventNumberFAFA;
        TBranch *br_triggerTimeStamp;
        TBranch *br_triggerCounter;
        TBranch *br_boardIp;
        TBranch *br_boardId;
        TBranch *br_chipId;
        TBranch *br_art_valid;
        TBranch *br_art_address;
        TBranch *br_art_trigger;
        TBranch *br_evSize;
        TBranch *br_tdo;
        TBranch *br_pdo;
        TBranch *br_flag;
        TBranch *br_thresh;
        TBranch *br_bcid;
        TBranch *br_relbcid; // L0
        TBranch *br_overflow; // L0
        TBranch *br_orbit_count; // L0
        TBranch *br_grayDecoded;
        TBranch *br_channelId;
        TBranch *br_febChannelId;
        TBranch *br_mappedChannelId;
        TBranch *br_pulserCalib;
        TBranch *br_gainCalib;
        TBranch *br_peakTimeCalib;
        TBranch *br_threshCalib;
        TBranch *br_TPskewCalib;
        TBranch *br_calibRun;
        TBranch *br_neighborCalib;

        // trigger events
        TriggerEvent m_trigger_events;
        uint32_t m_last_trigger_read;
        int n_warning;

};

#endif
