#include "dataflowwindow.h"
#include "ui_dataflowwindow.h"

#include <QMargins>

#include <iostream>
#include <iomanip>
#include <math.h>
using namespace std;

DataFlowWindow::DataFlowWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::DataFlowWindow),
    dataflow_monitoring_stopped(false),
    counter_input_empty_rate(0),
    counter_input_hit_rate(0),
    counter_record_hit_rate(0),
    counter_record_trigger_rate(0),
    last_count_input_hit_rate(0),
    last_count_record_hit_rate(0),
    last_count_record_trigger_rate(0),
    rate_input_hit_rate(0),
    rate_record_hit_rate(0),
    rate_record_trigger_rate(0),
    max_rate_seen(0.0),
    is_logy(false),
    use_khz(true),
    update_paused(false),
    lastPointKey(0.0),
    occupancy_update_paused(false),
    trig_delta_update_paused(false)
{
    test_counter = 0;
    ui->setupUi(this);

    ui->customPlot->replot();
    //QCPAxisRect *volumeAxisRect = new QCPAxisRect(ui->customPlot);
    axes = new QCPAxisRect(ui->customPlot);


    initialize_graphs();

    connect(&dataTimer, SIGNAL(timeout()), this, SLOT(update_monitor()));
    connect(&sendTimer, SIGNAL(timeout()), this, SLOT(increment_counter()));

    // log y
    connect(ui->set_logy_button, SIGNAL(clicked()), this, SLOT(set_logy()));

    // pause scrolling of x-axis
    connect(ui->pause_update_button, SIGNAL(clicked()), this, SLOT(pause_update()));


    logTicker = QSharedPointer<QCPAxisTickerLog>(new QCPAxisTickerLog);
    fixedTicker = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);

    //////////////////////////////////////////////
    // occupancy plot
    //////////////////////////////////////////////
    occPlot = ui->occupancyPlot;
    fullchart = new QCPBars(occPlot->xAxis, occPlot->yAxis);
    nullchart = new QCPBars(occPlot->xAxis, occPlot->yAxis);

    occLogTicker = QSharedPointer<QCPAxisTickerLog>(new QCPAxisTickerLog);
    occFixedTicker = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);

    connect(ui->pause_update_occupancy_button, SIGNAL(clicked()), this, SLOT(pause_occupancy_update()));
    connect(ui->set_logy_occupancy_button, SIGNAL(clicked()),
                this, SLOT(set_logy_occupancy()));

    //////////////////////////////////////////////
    // trigger delta plot
    //////////////////////////////////////////////
    trigDeltaPlot = ui->trigDeltaPlot;
    deltachartup = new QCPBars(trigDeltaPlot->xAxis, trigDeltaPlot->yAxis);
    deltachartdown = new QCPBars(trigDeltaPlot->xAxis, trigDeltaPlot->yAxis);
    logTicker = QSharedPointer<QCPAxisTickerLog>(new QCPAxisTickerLog);
    fixedTicker = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);

    trigDeltaLogTicker = QSharedPointer<QCPAxisTickerLog>(new QCPAxisTickerLog);
    trigDeltaFixedTicker = QSharedPointer<QCPAxisTickerFixed>(new QCPAxisTickerFixed);

    connect(ui->refresh_trigDelta_button, SIGNAL(pressed()), this, SLOT(refresh_trig_delta()));
    connect(ui->pause_update_trigDelta_button, SIGNAL(clicked()),
                            this, SLOT(pause_trig_delta_update()));
    connect(ui->set_logy_trigDelta_button, SIGNAL(clicked()),
                            this, SLOT(set_logy_trigDelta()));

}

DataFlowWindow::~DataFlowWindow()
{
    delete ui;
}

void DataFlowWindow::initialize_graphs()
{

    // graph input hit rate
    ui->customPlot->addGraph();
    QPen pen2;
    pen2.setColor(QColor(204, 95, 95));
    pen2.setWidth(2);
    pen2.setStyle(Qt::DotLine);
    ui->customPlot->graph(0)->setPen(pen2);
    ui->customPlot->graph(0)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssDisc, 6));
    ui->customPlot->graph(0)->setName("Input hit rate");

    //graph record hit rate
    ui->customPlot->addGraph();
    QPen pen0;
    pen0.setColor(QColor(40, 110, 255, 155));
    pen0.setWidth(2);
    ui->customPlot->graph(1)->setPen(pen0);
    ui->customPlot->graph(1)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssCircle, 6));
    ui->customPlot->graph(1)->setName("Record hit rate");

    // graph record trigger rate
    ui->customPlot->addGraph();
    QPen pen1;
    pen1.setColor(QColor(133, 175, 93));
    pen1.setWidth(2);
    ui->customPlot->graph(2)->setPen(pen1);
    ui->customPlot->graph(2)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssDiamond, 6));
    ui->customPlot->graph(2)->setName("Record trigger rate");

    
    // graph input empty rate
    ui->customPlot->addGraph();
    QPen pen3;
    pen3.setColor(QColor(105, 112, 129));
    pen3.setWidth(2);
    pen3.setStyle(Qt::DotLine);
    ui->customPlot->graph(3)->setPen(pen3);
    ui->customPlot->graph(3)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssDiamond, 4));
    ui->customPlot->graph(3)->setName("Empty packet rate");

    ui->customPlot->yAxis->setPadding(10);

    // minimum size

    // grid
    ui->customPlot->yAxis->grid()->setSubGridVisible(true);
    ui->customPlot->xAxis->grid()->setSubGridVisible(true);

    // axis move
    QSharedPointer<QCPAxisTickerTime> time_ticker(new QCPAxisTickerTime);
    time_ticker->setTimeFormat("%h:%m:%s");
    ui->customPlot->xAxis->setTicker(time_ticker);
    ui->customPlot->axisRect()->setupFullAxesBox();
    ui->customPlot->yAxis->setRange(-1, 10);

    // axis labels
    stringstream label;
    label << "Elapsed Time of Run";
    ui->customPlot->xAxis->setLabel(QString::fromStdString(label.str()));
    label.str("");
    label << "Rate " << (use_khz ? "[kHz]" : "[Hz]");
    ui->customPlot->yAxis->setLabel(QString::fromStdString(label.str()));

    // axis responds to mouse
    ui->customPlot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom);// | QCP::iSelectPlottables);

    connect(ui->customPlot->xAxis, SIGNAL(rangeChanged(QCPRange)),
                ui->customPlot->xAxis2, SLOT(setRange(QCPRange)));
    connect(ui->customPlot->yAxis, SIGNAL(rangeChanged(QCPRange)),
                ui->customPlot->yAxis2, SLOT(setRange(QCPRange)));

    // legend
    ui->customPlot->legend->setVisible(true);
    ui->customPlot->legend->setBrush(QBrush(QColor(255, 255, 255, 150)));
    ui->customPlot->axisRect()->insetLayout()->setInsetAlignment(0, Qt::AlignRight|Qt::AlignTop);

    // plot
    ui->customPlot->replot();
    

}

void DataFlowWindow::initialize_occupancy_plots(int starting_id, int n_boards)
{

    number_of_boards = n_boards;

    //fullchart->setAntialiased(false);
    //nullchart->setAntialiased(false);

    // names and colors
    fullchart->setName("Board Occupancy");
    fullchart->setPen(QPen(QColor(70, 117, 221).lighter(150)));
    nullchart->setName("Board Null");
    nullchart->setPen(QPen(QColor(212, 93, 78).lighter(130)));

    occTicks.clear();
    occLabels.clear();
    stringstream labels_stream;

    board_idx_map.clear();
    int board_idx = 0;
    int board_number = starting_id;
    for(int i = 1; i <= 2*number_of_boards; i+=2) {
        occTicks << i;
        labels_stream.str("");
        labels_stream << board_number;
        occLabels << QString::fromStdString(labels_stream.str());

        occTicks << (i+1);
        labels_stream.str("");
        labels_stream << "";
        occLabels << QString::fromStdString(labels_stream.str());

        board_idx_map[board_number] = board_idx;

        board_number++;
        board_idx++;
    }

    // x-axis
    occTextTicker = QSharedPointer<QCPAxisTickerText>(new QCPAxisTickerText);
    occTextTicker->addTicks(occTicks, occLabels);
    occPlot->xAxis->setTicker(occTextTicker);
    occPlot->xAxis->setLabel("Board #");
    occPlot->xAxis->setSubTicks(false);
    occPlot->xAxis->setTickLength(0,4);
    occPlot->xAxis->setRange(0, number_of_boards + 1);
    occPlot->xAxis->grid()->setVisible(true);

    // y-axis
    occPlot->yAxis->setRange(0, 1.2);
    occPlot->yAxis->setPadding(5);
    occPlot->yAxis->setLabel("Sample Fraction");
    occPlot->yAxis->grid()->setSubGridVisible(true);
    occPlot->yAxis->setNumberPrecision(4);

    // legend
    occPlot->legend->setVisible(true);
    occPlot->axisRect()->insetLayout()->setInsetAlignment(0, Qt::AlignTop|Qt::AlignRight);
    occPlot->legend->setBrush(QBrush(QColor(255, 255, 255, 120)));
    QFont legendFont = font();
    legendFont.setPointSize(10);
    occPlot->legend->setFont(legendFont);

    // interactions
    //occPlot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom);

    // replot
    occPlot->replot();

    // clear data
    data_occupancy.clear();
    data_null.clear();

    // clear denominator
    total_number_of_samples = 0.0;

    for(int i = 0; i < number_of_boards; i++) {
        data_occupancy.push_back(0.0);
        data_null.push_back(0.0);
    }

    QVector<double> data, nulldata;
    for(int i = 0; i < number_of_boards; i++) {
        data << 0.0 << 0.0;
        nulldata << 0.0 << 0.0;
    }

    fullchart->setData(occTicks, data);
    nullchart->setData(occTicks, nulldata);

    occPlot->xAxis->setRange(starting_id - 1.5, 2*number_of_boards+1);
    occPlot->replot();

}

void DataFlowWindow::set_logy()
{

    if(ui->set_logy_button->isChecked()) {
        float upper = 1e5;
        max_rate_seen = get_max_rate_seen();
        if(max_rate_seen != 0.0) {
            upper = max_rate_seen*1e1;
        }
        ui->customPlot->yAxis->setRange(1e-1, upper);
        ui->customPlot->replot();
        ui->customPlot->yAxis->setScaleType(QCPAxis::stLogarithmic);
        ui->customPlot->yAxis2->setScaleType(QCPAxis::stLogarithmic);
      //  QSharedPointer<QCPAxisTickerLog> logTicker(new QCPAxisTickerLog);
        //QCPAxisTickerLog logTicker2;
        ui->customPlot->yAxis->setTicker(logTicker);
        ui->customPlot->yAxis2->setTicker(logTicker);
        ui->customPlot->yAxis->setNumberFormat("eb");
        ui->customPlot->yAxis->setNumberPrecision(0);
        ui->customPlot->replot();
    }
    else {
        set_lineary();
    }
}

float DataFlowWindow::get_max_rate_seen()
{
    while(priority_queue_for_max.size())
        priority_queue_for_max.pop();

    for(int i = 0; i < (int)queue_for_max.size(); i++)
        priority_queue_for_max.push(queue_for_max.front());

    //cout << "get_max_rate_seen : " << priority_queue_for_max.top() << endl;
    return priority_queue_for_max.top();
}

void DataFlowWindow::set_lineary()
{
        

       // ui->customPlot->graph(0)->rescaleKeyAxis();
        //QSharedPointer<QCPAxisTickerFixed> fixedTicker(new QCPAxisTickerFixed);
        //QCPAxisTickerFixed fixedTicker2;
        //ui->customPlot->yAxis->setRange(0, 5000);

        //ui->customPlot->replot();
        ui->customPlot->yAxis->setNumberPrecision(4);
        ui->customPlot->yAxis->setScaleType(QCPAxis::stLinear);
        ui->customPlot->yAxis2->setScaleType(QCPAxis::stLinear);
        fixedTicker->setTickStep(0.01);
        fixedTicker->setScaleStrategy(QCPAxisTickerFixed::ScaleStrategy::ssMultiples);
        ui->customPlot->yAxis->setTicker(fixedTicker);
        ui->customPlot->yAxis2->setTicker(fixedTicker);
        ui->customPlot->replot();


//        ui->customPlot->yAxis->setRange(1, 50);
//        ui->customPlot->yAxis->setTicker(fixedTicker);
//        ui->customPlot->yAxis2->setTicker(fixedTicker);
//        ui->customPlot->replot();
//        ui->customPlot->yAxis->setScaleType(QCPAxis::stLinear);
//        ui->customPlot->yAxis2->setScaleType(QCPAxis::stLinear);
//        ui->customPlot->replot();
        float upper = 50;
        max_rate_seen = get_max_rate_seen();

        int max_int = (int)max_rate_seen;
        upper = (max_int + (10 - max_int % 10) );
        //if(max_rate_seen != 0.0) {
        //    upper =  * max_rate_seen;
        //    //upper = max_rate_seen + 1000;
        //}
        //else {
        //    upper = 50;
        //}

        //upper = pow(10, ceil(log(upper)/log(10)));

        int step_size = 20;
        if(upper != 0)
            step_size = upper / 20;
        if(step_size < 20)
            step_size = 5;
        fixedTicker->setTickStep(step_size);

        //cout << "max_rate_seen = " << max_rate_seen << "  upper: " << upper << endl;
        ui->customPlot->yAxis->setRange(-0.5, upper);
        ui->customPlot->yAxis->setNumberFormat("g");
        ui->customPlot->yAxis->setNumberPrecision(4);
        ui->customPlot->replot();
}

void DataFlowWindow::start_dataflow_monitor(int run_number, QTime start_time)
{

    dataflow_monitoring_stopped = false;

    while(queue_for_max.size())
        queue_for_max.pop();

    // propagate run number for x-axis label
    stringstream label;
    label << "Elapsed Time of Run #" << run_number;
    ui->customPlot->xAxis->setLabel(QString::fromStdString(label.str()));
    
    //// clear the graphs
    QVector<double> x;
    x.clear();
    QVector<double> y;
    y.clear();
    ui->customPlot->graph(0)->setData(x, y);
    ui->customPlot->graph(1)->setData(x, y);
    ui->customPlot->graph(2)->setData(x, y);
    ui->customPlot->graph(3)->setData(x, y);

    lastPointKey = 0.0;
    last_count_input_empty_rate = 0.0;
    last_count_input_hit_rate = 0.0;
    last_count_record_hit_rate = 0.0;
    last_count_record_trigger_rate = 0.0;


    start_run_time = start_time;

    dataTimer.start(0);
}

void DataFlowWindow::stop_dataflow_monitor()
{
    dataflow_monitoring_stopped = true;

    dataTimer.stop();
    //sendTimer.stop();

    counter_input_empty_rate = 0;
    counter_input_hit_rate = 0;
    counter_record_hit_rate = 0;
    counter_record_trigger_rate = 0;

    // occupancy
    total_number_of_samples = 0;
    data_occupancy.clear();
    data_null.clear();
    qdata_occupancy.clear();
    qdata_null.clear();

    // trigger delta
    data_delta_zero.clear();
    data_delta_up.clear();
    data_delta_down.clear();
    qdata_delta_up.clear();
    qdata_delta_down.clear();
    test_counter = 0;
    
    
}

void DataFlowWindow::increment_counter()
{
    counter_input_empty_rate++;
    counter_input_hit_rate++;
    counter_record_hit_rate++;
    counter_record_trigger_rate++;
}

void DataFlowWindow::update_counters(int record_hit_rate, int trig_counter,
                unsigned int input_hit_rate, unsigned int input_empty_hit_rate)
{

    //cout << "update_counters: " << input_empty_hit_rate << " "
    //                        << input_hit_rate << " " << record_hit_rate << " " << trig_counter << endl;
    counter_input_empty_rate = input_empty_hit_rate;
    counter_input_hit_rate = input_hit_rate;
    counter_record_hit_rate = record_hit_rate;
    counter_record_trigger_rate = trig_counter;
}

void DataFlowWindow::update_empty_count(unsigned int empty_count)
{
    counter_input_empty_rate = empty_count;
}

void DataFlowWindow::update_monitor()
{

    if(dataflow_monitoring_stopped) return;

    //static QTime time(QTime::currentTime());
    double key = start_run_time.elapsed()/1000.0; // time elapsed since start of monitoring, in seconds
    //static double lastPointKey = 0;


    if(key - lastPointKey > 1) {

        // input empty rate
        rate_input_empty_rate = (counter_input_empty_rate - last_count_input_empty_rate) /
                                        (key - lastPointKey);

        // input rate
        rate_input_hit_rate = (counter_input_hit_rate - last_count_input_hit_rate) / (key - lastPointKey);

        //cout << "input empty count: " << counter_input_empty_rate << "  input count: " << counter_input_hit_rate << "   record_hit : " << counter_record_hit_rate
        //    << "  recrod trig: " << counter_record_trigger_rate << endl;
        
        // record input rate
        rate_record_hit_rate = (counter_record_hit_rate - last_count_record_hit_rate) / (key - lastPointKey);

        // record trigger rate
        rate_record_trigger_rate = (counter_record_trigger_rate - last_count_record_trigger_rate) / (key - lastPointKey);

        //cout << "counter_record_hit_rate = " << counter_record_hit_rate
        //            << "  last: " << last_count_record_hit_rate << "  rate: " << rate_record_hit_rate << endl;
        //cout << "counter_record_trig_rate = " << counter_record_trigger_rate
        //            << "  last: " << last_count_record_trigger_rate << "  rate: " << rate_record_trigger_rate << endl;
        

        // set graph value
        float divisor = (use_khz ? 1000.0 : 1.0);

        rate_input_hit_rate = (rate_input_hit_rate / divisor);
        rate_record_hit_rate = (rate_record_hit_rate / divisor);
        rate_record_trigger_rate = (rate_record_trigger_rate / divisor);
        rate_input_empty_rate = (rate_input_empty_rate / divisor);

        ui->customPlot->graph(0)->addData(key, rate_input_hit_rate);
        ui->customPlot->graph(1)->addData(key, rate_record_hit_rate);
        ui->customPlot->graph(2)->addData(key, rate_record_trigger_rate);
        ui->customPlot->graph(3)->addData(key, rate_input_empty_rate);

        // set last
        lastPointKey = key;

        last_count_input_empty_rate = counter_input_empty_rate;
        last_count_input_hit_rate = counter_input_hit_rate;
        last_count_record_hit_rate = counter_record_hit_rate;
        last_count_record_trigger_rate = counter_record_trigger_rate;

        if(rate_input_empty_rate > max_rate_seen) max_rate_seen = rate_input_empty_rate;
        if(rate_input_hit_rate > max_rate_seen) max_rate_seen = rate_input_hit_rate;
        if(rate_record_hit_rate > max_rate_seen) max_rate_seen = rate_record_hit_rate;
        if(rate_record_trigger_rate > max_rate_seen) max_rate_seen = rate_record_trigger_rate;

        

        if(queue_for_max.size()>5) queue_for_max.pop();
        queue_for_max.push(max_rate_seen);

    }


    /*
    static int last_count = 0;
    if(key - lastPointKey > 1) {
        rate = (counter - last_count) / (key - lastPointKey); // Hz
        ui->customPlot->graph(0)->addData(key, rate);
        //ui->customPlot->graph(0)->addData(key, (counter - last_count) / (key - lastPointKey));

        cout << "counter: " << counter << "  last_count: " << last_count
            << "  key: " << key << "  lastPointKey: " << lastPointKey 
            << "  rate: " << rate << endl;
            //<< "  rate: " << (counter - last_count) / (key - lastPointKey) << endl; 
        lastPointKey = key;
        last_count = counter;
        if(rate > max_rate_seen) max_rate_seen = rate;
    }
    */


    if(!update_paused) {
        ui->customPlot->graph(0)->rescaleKeyAxis();
        ui->customPlot->graph(1)->rescaleKeyAxis();
        ui->customPlot->graph(2)->rescaleKeyAxis();
        ui->customPlot->graph(3)->rescaleKeyAxis();

        ui->customPlot->xAxis->setRange(key+4, 10, Qt::AlignRight);
        //ui->customPlot->xAxis->setRange(key+8, key+2, Qt::AlignRight);
        ui->customPlot->replot();
    }

    status_message.str("");
    status_message << "[kHz] Input hit rate: " << std::setprecision(3)
                    << (rate_input_hit_rate)
                    << "  Record hit rate: "
                    << (rate_record_hit_rate)
                    << "  Record trig rate: "
                    << (rate_record_trigger_rate)
                    << "  Empty packet rate: "
                    << (rate_input_empty_rate)
                    << ", Record efficiency: " << std::setprecision(4)
                    << (rate_record_hit_rate / rate_input_hit_rate);
    ui->statusbar->showMessage(QString::fromStdString(status_message.str()), 0);

    
}

void DataFlowWindow::pause_update()
{
    update_paused = ui->pause_update_button->isChecked();
}

void DataFlowWindow::pause_occupancy_update()
{
    occupancy_update_paused = ui->pause_update_occupancy_button->isChecked();
}

void DataFlowWindow::perform_occupancy_check(QString ip, int is_full, int is_empty)
{

    if(dataflow_monitoring_stopped) return;

    //cout << "DataFlowWindow::perform_occupancy_check    " << ip.toStdString() 
    //        << "  is_full: " << is_full << "  is_empty: " << is_empty << endl;
    QStringList ip_split = ip.split(".");
    int board_number_to_add = -1;
    try {
        board_number_to_add = std::stoi(ip_split[3].toStdString());
    }
    catch(std::exception& e) {
        cout << "DataFlowWindow::perform_occupancy_check    Unable to obtain board ID number from IP '"
                    << ip.toStdString() << "'" << endl;
        board_number_to_add = -1;
    }

    int board_idx = board_idx_map[board_number_to_add];

    //// testing multi-board
    //if(rand()%2==0)
    //    board_idx = board_idx+=1;

    data_occupancy.at(board_idx) += is_full;
    data_null.at(board_idx) += is_empty;

    qdata_occupancy.clear();
    qdata_null.clear();

    // increment denominator
    total_number_of_samples++;

    float maxy = -1;
    for(int i = 0; i < number_of_boards; i++) {
        qdata_occupancy << (data_occupancy.at(i) / total_number_of_samples) << 0.0;
        qdata_null << 0.0 << (data_null.at(i) / total_number_of_samples);

        if(data_occupancy.at(i) > maxy) maxy = data_occupancy.at(i);
        if(data_null.at(i) > maxy) maxy = data_null.at(i);
    }

    if(!occupancy_update_paused) {
        fullchart->setData(occTicks, qdata_occupancy);
        nullchart->setData(occTicks, qdata_null);
        float lower = 0.0;
        maxy = 1.5*maxy;
        maxy_occupancy_global = maxy_occupancy_global; 

        //occPlot->yAxis->setRange(lower, maxy);
        
        if(ui->set_logy_occupancy_button->isChecked()) {
            occPlot->yAxis->setRange(1e-2, 10);
        }
        else {
            occPlot->yAxis->setRange(0, 1.2);
        }
        occPlot->replot();
    }

    return;
}

void DataFlowWindow::set_logy_occupancy()
{

    if(ui->set_logy_occupancy_button->isChecked())
    {
        occPlot->yAxis->setRange(1e-2,10);
        occPlot->replot();
        occPlot->yAxis->setScaleType(QCPAxis::stLogarithmic);
        occPlot->yAxis2->setScaleType(QCPAxis::stLogarithmic);
        occPlot->yAxis->setTicker(occLogTicker);
        occPlot->yAxis2->setTicker(occLogTicker);
        occPlot->yAxis->setNumberFormat("eb");
        occPlot->yAxis->setNumberPrecision(0);
        occPlot->replot();
    }
    else {
        occPlot->yAxis->setNumberPrecision(3);
        occPlot->yAxis->setScaleType(QCPAxis::stLinear);
        occPlot->yAxis2->setScaleType(QCPAxis::stLinear);
        occFixedTicker->setTickStep(0.01);
        occFixedTicker->setScaleStrategy(QCPAxisTickerFixed::ScaleStrategy::ssMultiples);
        occPlot->yAxis->setTicker(occFixedTicker);
        occPlot->yAxis2->setTicker(occFixedTicker);
        occPlot->replot();
        occPlot->yAxis->setRange(0,1.2);
        occPlot->yAxis->setNumberFormat("g");
        occPlot->replot();
    }
}

void DataFlowWindow::initialize_trig_delta_plots(int starting_id, int n_boards)
{

    // names and colors
    deltachartup->setName("Delta +");
    deltachartup->setPen(QPen(QColor(70, 117, 221).lighter(150)));
    deltachartdown->setName("Delta -");
    deltachartdown->setPen(QPen(QColor(212, 93, 78).lighter(150)));

    deltaTicks.clear();
    deltaLabels.clear();
    stringstream labels_stream;

    int board_number = starting_id;
    for(int i = 1; i <= 2*number_of_boards; i+=2) {
        deltaTicks << i;
        labels_stream.str("");
        labels_stream << board_number;
        deltaLabels << QString::fromStdString(labels_stream.str());

        deltaTicks << (i+1);
        labels_stream.str("");
        labels_stream << "";
        deltaLabels << QString::fromStdString(labels_stream.str());

        board_number++;
    }

    // x-axis
    deltaTextTicker = QSharedPointer<QCPAxisTickerText>(new QCPAxisTickerText);
    deltaTextTicker->addTicks(deltaTicks, deltaLabels);
    trigDeltaPlot->xAxis->setTicker(deltaTextTicker);
    trigDeltaPlot->xAxis->setLabel("Board #");
    trigDeltaPlot->xAxis->setSubTicks(false);
    trigDeltaPlot->xAxis->setTickLength(0,4);
    trigDeltaPlot->xAxis->setRange(0, number_of_boards + 1);
    trigDeltaPlot->xAxis->grid()->setVisible(true);

    // y-axis
    trigDeltaPlot->yAxis->setRange(0,1.2);
    trigDeltaPlot->yAxis->setPadding(5);
    trigDeltaPlot->yAxis->setLabel("Delta Fraction");
    trigDeltaPlot->yAxis->grid()->setSubGridVisible(true);
    trigDeltaPlot->yAxis->setNumberPrecision(2);

    // legend
    trigDeltaPlot->legend->setVisible(true);
    trigDeltaPlot->axisRect()->insetLayout()->setInsetAlignment(0, Qt::AlignTop|Qt::AlignRight);
    trigDeltaPlot->legend->setBrush(QBrush(QColor(255, 255, 255, 120)));
    QFont legendFont = font();
    legendFont.setPointSize(10);
    trigDeltaPlot->legend->setFont(legendFont);

    // interactions?
    //trigDeltaPlot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom | QCP::iSelectAxes); 

    // replote
    trigDeltaPlot->replot();

    // clear data
    data_delta_zero.clear();
    data_delta_up.clear();
    data_delta_down.clear();

    for(int i = 0; i < number_of_boards; i++) {
        data_delta_zero.push_back(0.0);
        data_delta_up.push_back(0.0);
        data_delta_down.push_back(0.0);
    }

    // re-set denominator
    test_counter = 0.0;

    QVector<double> up, down;
    for(int i = 0; i < number_of_boards; i++) {
        up << 0.0 << 0.0;
        down << 0.0 << 0.0;
    }

    deltachartup->setData(deltaTicks, up);
    deltachartdown->setData(deltaTicks, down);

    trigDeltaPlot->xAxis->setRange(starting_id - 1.5, 2*number_of_boards+1);
    trigDeltaPlot->replot();

}

void DataFlowWindow::perform_trigger_delta_check(int board, int delta)
{

    if(dataflow_monitoring_stopped) return;

    int board_idx = board_idx_map[board];

    //if(rand()%10==0) delta = -1;
    //if(test_counter < 100) { test_counter++; delta = 1; }
    //else if(test_counter >= 100 && test_counter <200) { test_counter++; delta = -1; }
    //else if (test_counter >= 200) { delta = 0; test_counter++; }

    test_counter++;

    if(delta==0) {
        data_delta_zero.at(board_idx) += 1;
    }
    else if(delta>0) {
        data_delta_up.at(board_idx) += 1;
    }
    else if(delta<0) {
        data_delta_down.at(board_idx) += 1;
    }

    qdata_delta_up.clear();
    qdata_delta_down.clear();


    //float maxy = -1;
    for(int i = 0; i < number_of_boards; i++) {
        float zero_value = data_delta_zero.at(i);
        float up_value = data_delta_up.at(i);
        float down_value = data_delta_down.at(i);

        //int up_num = (zero_value - up_value);
        //float up = ((float)(up_num) / (float)(zero_value + up_value) * 1.0);
        //up = 0.5*(1 - up);

        //int down_num = (zero_value - down_value);
        //float down = ((float)(down_num) / (float)(zero_value + down_value) * 1.0);
        //down = 0.5*(1 - down);
        float up;
        float down;
        up = (float)up_value * 1.0 / test_counter;
        down = (float)down_value * 1.0 / test_counter;

        //if(up < 0) up = -1 * up;
        //if(down > 0) down = -1 * down;

        // not doing up and down on chart
        if(up < 0) up = -1 * up;
        if(down < 0) down = -1 * down;

        //if(delta==0) up = 0.0;
        //if(delta==0) down = 0.0;

        qdata_delta_up << up << 0.0;
        qdata_delta_down << 0.0 << down;

        //up = fabs(up);
        //down = fabs(down);

        //if(up > maxy) maxy = up;
        //if(down > maxy) maxy = down;

    } 

    if(!trig_delta_update_paused) {
        deltachartup->setData(deltaTicks, qdata_delta_up);
        deltachartdown->setData(deltaTicks, qdata_delta_down);

        //maxy = 1.1*maxy;
        //trigDeltaPlot->yAxis->setRange(-maxy, maxy);
        trigDeltaPlot->replot();
    }

}

void DataFlowWindow::refresh_trig_delta()
{
    qdata_delta_up.clear();
    qdata_delta_down.clear();

    data_delta_zero.clear();
    data_delta_up.clear();
    data_delta_down.clear();

    for(int i = 0; i < number_of_boards; i++) {
        qdata_delta_up << 0.0 << 0.0;
        qdata_delta_down << 0.0 << 0.0;
        data_delta_zero.push_back(0.0);
        data_delta_up.push_back(0.0);
        data_delta_down.push_back(0.0);
    }

    // denominator
    test_counter = 0.0;

    deltachartup->setData(deltaTicks, qdata_delta_up);
    deltachartdown->setData(deltaTicks, qdata_delta_down);

    if(ui->set_logy_trigDelta_button->isChecked()) {
        trigDeltaPlot->yAxis->setRange(1e-2, 10);
    }
    else {
        trigDeltaPlot->yAxis->setRange(0, 1.2);
    }

    trigDeltaPlot->replot();
}

void DataFlowWindow::pause_trig_delta_update()
{
    trig_delta_update_paused = ui->pause_update_trigDelta_button->isChecked();
}

void DataFlowWindow::set_logy_trigDelta()
{
    if(ui->set_logy_trigDelta_button->isChecked()) {
        float max = 10;
        trigDeltaPlot->yAxis->setRange(1e-2, 10);
        trigDeltaPlot->replot();
        trigDeltaPlot->yAxis->setScaleType(QCPAxis::stLogarithmic);
        trigDeltaPlot->yAxis2->setScaleType(QCPAxis::stLogarithmic);
        trigDeltaPlot->yAxis->setTicker(trigDeltaLogTicker);
        trigDeltaPlot->yAxis2->setTicker(trigDeltaLogTicker);
        trigDeltaPlot->yAxis->setNumberFormat("eb");
        trigDeltaPlot->yAxis->setNumberPrecision(0);
        trigDeltaPlot->replot();
    }
    else {
        trigDeltaPlot->yAxis->setNumberPrecision(3);
        trigDeltaPlot->yAxis->setScaleType(QCPAxis::stLinear);
        trigDeltaPlot->yAxis2->setScaleType(QCPAxis::stLinear);
        trigDeltaFixedTicker->setTickStep(0.01);
        trigDeltaFixedTicker->setScaleStrategy(QCPAxisTickerFixed::ScaleStrategy::ssMultiples);
        trigDeltaPlot->yAxis->setTicker(trigDeltaFixedTicker);
        trigDeltaPlot->yAxis2->setTicker(trigDeltaFixedTicker);
        trigDeltaPlot->replot();
        trigDeltaPlot->yAxis->setRange(0, 1.2);
        trigDeltaPlot->yAxis->setNumberFormat("g");
        trigDeltaPlot->replot();
    }

}

void DataFlowWindow::closeEvent(QCloseEvent* event)
{
    emit close_dataflow();
    QWidget::closeEvent(event);
}
