#include "event_builder.h"
#include "bit_manip.h"
#include <iostream>
using namespace std;
using namespace boost::chrono;

//std/stl
#include <string>
#include <iostream>
#include <bitset>
#include <inttypes.h>
#include <sstream>

//boost
//#include <boost/thread.hpp>
//#include <boost/asio.hpp>
#include <boost/foreach.hpp>
#include <boost/chrono.hpp>
#include <boost/dynamic_bitset.hpp>

//ROOT
#include "TFile.h"
#include "TTree.h"
#include "TBranch.h"

//nsw
#include "map_handler.h"
#include "OnlineMonTool.h"
#include "message_handler.h"

boost::mutex stream_lock;

uint32_t MMFE8TRAILER = 0xffffffff;

bool verbose = false;

EventBuilder::EventBuilder() :
    n_hits(0),
    m_msg(NULL),
    m_dbg(false),
    m_vmm_type(3),
    m_mapHandler(NULL),
    m_mapping_setup(false),
    m_monTool(NULL),
    m_monitoringStatus(false),
    m_do_do_monitoring(0),
    m_current_monitoring_state(false),
    m_monitor_rate(50),
    n_monitor_strings(0),
    m_output_rootfilename(""),
    m_run_number(0),
    m_writeRaw(false),
    m_writeNtuple(false),
    m_tree_flush(-1),
    m_do_L0_decoding(false),
    m_fill_event(false),
    m_calibRun(false),
    m_calib_channel(-1),
    n_daqCount(0),
    m_daqRootFile(NULL),
    m_runProperties_tree(NULL),
    m_vmm_tree(NULL),
    m_last_trigger_read(0xffffffff),
    n_warning(0)
{
    m_sx.str("");
    // trigger events
    m_trigger_events.clear();

    m_do_do_monitoring = new bool();
    *m_do_do_monitoring = false;
}
void EventBuilder::set_vmm_type(int type)
{
    m_vmm_type = type;
}
void EventBuilder::set_monitor_rate(int rate)
{
    //std::cout << "EventBuilder::set_monitor_rate    setting monitor sampling rate to " << rate << std::endl;
    m_monitor_rate = rate;
}

void EventBuilder::print_mon_flag()
{
    stream_lock.lock(); 
    std::cout << "EventBuilder::print_mon_flag    Online monitoring flag : " << (*m_do_do_monitoring) << std::endl;
    stream_lock.unlock();
}

void EventBuilder::load_mon_flag(bool* flag)
{
    m_do_do_monitoring = flag;
}

void EventBuilder::resetCounts()
{
    n_hits = 0;
    n_triggers = 0;
    m_last_trigger_read = 0xffffffff;
}

void EventBuilder::LoadMessageHandler(MessageHandler& msg)
{
    m_msg = &msg;
}

void EventBuilder::loadMappingTool(MapHandler& maptool)
{
    m_mapHandler = &maptool;
   
    if(dbg()) {
        stream_lock.lock(); 
        std::cout << "EventBuilder::loadMappingTool    Mapping tool loaded. Maps built? " << (mapHandler().mapLoaded() ? "YES" : "NO") << std::endl;
        stream_lock.unlock();
    }
    m_mapping_setup = true;
}
void EventBuilder::disableMapping()
{
    if(dbg()) {
        stream_lock.lock();
        std::cout << "EventBuilder::disableMapping    Disabling DAQ configuration mapping" << std::endl;
        stream_lock.unlock();
    }

    m_mapping_setup = false;
}

void EventBuilder::loadMonitoringTool(OnlineMonTool& montool)
{
    if(!m_monTool) {
        m_monTool = &montool;
        if(dbg()) {
            stream_lock.lock();
            std::cout << "EventBuilder::loadMonitoringTool    Monitoring tool loaded" << std::endl;
            stream_lock.unlock();
        }
    }
}
void EventBuilder::setMonitoringStatus(bool status)
{
    m_monitoringStatus = status;
    if(dbg()) {
        stream_lock.lock();
        std::cout << "EventBuilder::setMonitoringStatus    Monitoring status: " << (m_monitoringStatus ? "TRUE" : "FALSE") << std::endl;
        stream_lock.unlock();
    }
}
void EventBuilder::startMonitoring()
{
    m_monTool->start();
}
void EventBuilder::stopMonitoring()
{
    m_monTool->stop();
}

bool EventBuilder::initializeRun(bool writeRaw_, bool writeNtuple_, std::string filename,
                                 int run_number, int tree_flush, bool do_L0_decoding)
{
    m_writeRaw = writeRaw_;
    m_writeNtuple = writeNtuple_;

    // clear
    m_trigger_events.clear();

    std::stringstream sx;
    m_run_number = run_number;

    //////////////////////////////////////////////////////
    // data decoder
    //////////////////////////////////////////////////////
    m_decoder.clear(true); // clear all (data + header)
    m_decoder.set_debug(dbg());
    if(!do_L0_decoding || vmm_type()==2) {
        if(vmm_type()==3) {
            m_decoder.set_type(VMMDecoder::VMMEVENT);
        }
        else if(vmm_type()==2) {
            m_decoder.set_type(VMMDecoder::VMM2EVENT);
        }
    }
    else {
        m_decoder.set_type(VMMDecoder::VMMEVENTL0);
    }
    m_do_L0_decoding = do_L0_decoding;
    if(vmm_type()==2) m_do_L0_decoding = false;

    //////////////////////////////////////////////////////
    // Ntuple
    //////////////////////////////////////////////////////
    if(writeNtuple()) {
        m_daqRootFile = new TFile(filename.c_str(), "UPDATE");
        if(m_daqRootFile->IsZombie()) {
            m_sx << "ERROR DAQ ROOT file unable to be opened";
            msg()(m_sx,"EventBuilder::initializeRun"); m_sx.str("");
            delete m_daqRootFile;
            return false;
        }
        m_output_rootfilename = filename;

        // setup trees and tree structure
        setupOutputTrees();
    }
    m_tree_flush = tree_flush;

    //////////////////////////////////////////////////////
    // Raw File
    //////////////////////////////////////////////////////
    if(writeRaw()) {
        m_raw_dumper = new RawDataDumper();
        if(!m_raw_dumper->open_output_file(filename, run_number)) {
            cout << "EventBuilder::initializeRun    ERROR Unable to create output file for raw data!" << endl;
            m_writeRaw = false;
        }
        
    } // writeRaw

    // reset the warning counters for the new run
    if(mapping()) {
        mapHandler().resetWarningCounters();
    }

    // timer
    m_start_time = 0;
    auto now = time_point_cast<seconds>(system_clock::now());
    m_start_time = now.time_since_epoch().count();

    return true;
}
void EventBuilder::initializeMonitoring(boost::shared_ptr<boost::asio::io_service> service)
{
   m_monTool->initialize(service);
}

//void EventBuilder::get_sync_items(boost::shared_ptr<boost::timed_mutex> data_mutex, boost::shared_ptr<boost::condition_variable_any> data_condition)
//{
//    m_data_mutex = data_mutex;
//    m_event_fill_condition = data_condition;
//}

void EventBuilder::setupOutputTrees()
{
    // clear the data containers
    clearData();

    if(writeNtuple()) {

        //m_daqRootFile->cd();

        // run properties
        m_runProperties_tree = new TTree("run_properties", "run_properties");
        br_runNumber            = m_runProperties_tree->Branch("runNumber", &m_run_number);
        br_gain                 = m_runProperties_tree->Branch("gain", &m_gain); 
        br_tacSlope             = m_runProperties_tree->Branch("tacSlope", &m_tacSlope);
        br_peakTime             = m_runProperties_tree->Branch("peakTime", &m_peakTime);
        br_dacCounts            = m_runProperties_tree->Branch("dacCounts", &m_dacCounts);
        br_pulserCounts         = m_runProperties_tree->Branch("pulserCounts", &m_pulserCounts);
        br_TPskew               = m_runProperties_tree->Branch("tpSkew", &m_tpSkew);
        br_ckbc                 = m_runProperties_tree->Branch("ckbc", &m_ckbc);
        br_angle                = m_runProperties_tree->Branch("angle", &m_angle);
        br_calibrationRun       = m_runProperties_tree->Branch("calibrationRun", &m_calibRun);

        // vmm event data
        m_vmm_tree = new TTree("vmm", "vmm");
        br_eventNumberFAFA      = m_vmm_tree->Branch("eventFAFA", &m_eventNumberFAFA);
        br_triggerTimeStamp     = m_vmm_tree->Branch("triggerTimeStamp", "std::vector<int>", &m_triggerTimeStamp);
        br_triggerCounter       = m_vmm_tree->Branch("triggerCounter", "std::vector<int>", &m_triggerCounter); 
        br_boardId              = m_vmm_tree->Branch("boardId", "std::vector<int>", &m_boardId);
        br_chipId               = m_vmm_tree->Branch("chip", "std::vector<int>", &m_chipId);
        br_evSize               = m_vmm_tree->Branch("eventSize", "std::vector<int>", &m_eventSize);
        br_daq_timestamp_s      = m_vmm_tree->Branch("daq_timestamp_s",  "std::vector<int>", &m_daq_timestamp_s);
        br_daq_timestamp_ns     = m_vmm_tree->Branch("daq_timestamp_ns", "std::vector<int>", &m_daq_timestamp_ns);
        br_tdo                  = m_vmm_tree->Branch("tdo", "std::vector< vector<int> >", &m_tdo);
        br_pdo                  = m_vmm_tree->Branch("pdo", "std::vector< vector<int> >", &m_pdo);
        br_flag                 = m_vmm_tree->Branch("flag", "std::vector< vector<int> >", &m_flag);
        br_thresh               = m_vmm_tree->Branch("threshold", "std::vector< vector<int> >", &m_threshold);
        br_bcid                 = m_vmm_tree->Branch("bcid", "std::vector< vector<int> >", &m_bcid);
        br_relbcid              = m_vmm_tree->Branch("relbcid", "std::vector< vector<int> >", &m_relbcid);
        br_overflow             = m_vmm_tree->Branch("overflow", "std::vector< vector<int> >", &m_overflow);
        br_orbit_count          = m_vmm_tree->Branch("orbitCount", "std::vector< vector<int> >", &m_orbit_count);
        br_grayDecoded          = m_vmm_tree->Branch("grayDecoded", "std::vector< vector<int> >", &m_grayDecoded);
        br_channelId            = m_vmm_tree->Branch("channel", "std::vector< vector<int> >", &m_channelId);
        br_febChannelId         = m_vmm_tree->Branch("febChannel", "std::vector< vector<int> >", &m_febChannelId);
        br_mappedChannelId      = m_vmm_tree->Branch("mappedChannel", "std::vector< vector<int> >", &m_mappedChannelId);
        br_art_valid            = m_vmm_tree->Branch("art_valid", "std::vector<int>", &m_art_valid);
        br_art_address          = m_vmm_tree->Branch("art", "std::vector<int>", &m_art);
        br_art_trigger          = m_vmm_tree->Branch("art_trigger", "std::vector<int>", &m_art_trigger);

        if(calibrationRun()) {
            br_calibRun         = m_vmm_tree->Branch("calibrationRun", &m_calibRun);
            br_pulserCalib      = m_vmm_tree->Branch("pulserCounts", &m_pulserCounts_calib);
            br_threshCalib      = m_vmm_tree->Branch("dacCounts", &m_dacCounts_calib);
            br_gainCalib        = m_vmm_tree->Branch("gain", &m_gain_calib);
            br_peakTimeCalib    = m_vmm_tree->Branch("peakTime", &m_peakTime_calib);
            br_TPskewCalib      = m_vmm_tree->Branch("tpSkew", &m_tpSkew_calib);
            br_neighborCalib    = m_vmm_tree->Branch("neighbor", "std::vector< vector<int> >", &m_neighbor_calib);
        } // calibration run  

    } // writing output ntuple

}
void EventBuilder::clearData()
{
    m_gain              = -999;
    m_tacSlope          = -999;
    m_peakTime          = -999;
    m_dacCounts         = -999;
    m_pulserCounts      = -999;
    m_tpSkew            = -999;
    m_angle             = -999;

    // event data
    m_eventNumberFAFA = 0;
    m_triggerTimeStamp.clear();
    m_triggerCounter.clear();
    m_boardId.clear();
    m_chipId.clear();
    m_mini2_eventsize_per_packet = 0;
    m_eventSize.clear();
    m_daq_timestamp_s.clear();
    m_daq_timestamp_ns.clear();
    m_art.clear();
    m_art_valid.clear();
    m_art_trigger.clear();
    m_tdo.clear();
    m_pdo.clear();
    m_flag.clear();
    m_threshold.clear();
    m_bcid.clear();
    m_grayDecoded.clear();
    m_channelId.clear();
    m_febChannelId.clear();
    m_mappedChannelId.clear();

    // calib
    m_neighbor_calib.clear();
}

void EventBuilder::fillRunProperties(int gain, int tac_slope, int peak_time,
        int dac_threshold, int dac_pulser, int angle, int tp_skew, int ckbc)
{
    m_gain          = gain;
    m_tacSlope      = tac_slope;
    m_peakTime      = peak_time;
    m_dacCounts     = dac_threshold;
    m_pulserCounts  = dac_pulser;
    m_angle         = angle;
    m_tpSkew        = tp_skew;
    m_ckbc          = ckbc;

    if(writeNtuple()) {
        m_daqRootFile->cd();
        if(!m_runProperties_tree) {
            stream_lock.lock();
            msg()("runProperties tree is null, unable to store run properties","EventBuilder::fillRunProperties");
            stream_lock.unlock();
        }
        else {
            m_runProperties_tree->Fill();
            m_runProperties_tree->Write("", TObject::kOverwrite);
        }
        delete m_runProperties_tree;
    } // writeNtuple

    if(writeRaw()) {
        m_raw_dumper->fill_run_properties(gain, tac_slope, peak_time, dac_threshold,
                dac_pulser, angle, tp_skew);
    } // writeRaw
}
//DaqBuffer
uint32_t EventBuilder::ip_str_to_int(std::string& ip_string)
{

    uint32_t m0 = 0xff;
    stringstream ss(ip_string);
    vector<uint32_t> tokens;
    string token;
    while (std::getline(ss, token, '.')) {
        if(!token.empty()) {
            int val = std::stoi(token);
            uint32_t val32 = val;
            tokens.push_back(val32);
        }
    }

    uint32_t out = 0;
    out |= (tokens.at(0) & m0); out = (out << 8);
    out |= (tokens.at(1) & m0); out = (out << 8);
    out |= (tokens.at(2) & m0); out = (out << 8);
    out |= (tokens.at(3) & m0);

    return out;
}

int EventBuilder::ip_str_to_id(std::string& ip_string)
{
    stringstream ss(ip_string);
    vector<uint32_t> tokens;
    string token;
    while (std::getline(ss, token, '.')) {
        if(!token.empty()) {
            int val = std::stoi(token);
            uint32_t val32 = val;
            tokens.push_back(val32);
        }
    }

    return tokens.at(3);
}

void EventBuilder::receive_event_data(std::string& ip_string, std::vector<uint32_t>& datagram, uint32_t length, 
    int& counter, int& trig_counter)
{
    if(writeRaw()) {
        store_raw_data(ip_string, datagram, counter);
    }
    if(writeNtuple() || monitoring()) {
        decode_event_trig(ip_string, datagram, length, counter, trig_counter);
        //decode_event(ip_string, datagram, counter);
    }
}
void EventBuilder::store_raw_data(std::string& ip_string, std::vector<uint32_t>& datagram,
    int& counter)
{
    uint32_t ip_int = ip_str_to_int(ip_string);
    m_raw_dumper->write_data(ip_int);
    for(auto data : datagram) {
        m_raw_dumper->write_data(data);
        if(data==0xffffffff && !writeNtuple()) {
            n_hits++;
            counter = n_hits;
        }
    }
}
void EventBuilder::decode_event(std::string& ip_string, std::vector<uint32_t>& datagram,
    int& counter)
{
    //std::cout << datagram.at(0) << std::endl;
    //return;

    //size_t n_bytes_for_empty = 12;
    size_t n_bytes_read = 0;
    //DaqBuffer
    //if( (num_bytes>n_bytes_for_empty) && !m_fill_event)
    //    m_fill_event = true;

    auto now = time_point_cast<nanoseconds>(system_clock::now());
    uint64_t now_count = now.time_since_epoch().count();
    uint64_t now_s     = now_count / 1e9;
    uint64_t now_ns    = now_count - (now_s * 1e9);

    /////////////////////////////////////////////////////
    // these are the data containers for the tree
    /////////////////////////////////////////////////////
    std::vector<int> _boardId_tree;
    std::vector<int> _trigTimeStamp_tree;
    std::vector<int> _trigCounter_tree;
    std::vector<int> _chipId_tree;
    std::vector<int> _eventSize_tree;
    std::vector<int> _daq_timestamp_s_tree;
    std::vector<int> _daq_timestamp_ns_tree;

    std::vector< std::vector<int> > _pdo_tree;
    std::vector< std::vector<int> > _tdo_tree; 
    std::vector< std::vector<int> > _flag_tree;
    std::vector< std::vector<int> > _threshold_tree;
    std::vector< std::vector<int> > _bcid_tree;
    std::vector< std::vector<int> > _grayDecoded_tree;
    std::vector< std::vector<int> > _channelId_tree;
    std::vector< std::vector<int> > _febChannelId_tree;
    std::vector< std::vector<int> > _mappedChannelId_tree;
    std::vector< std::vector<int> > _neighbor_tree;

    // get the board id from the packet ip
    std::string boardId = "";
    int board_id = -1;
    if(mapping() && mapHandler().mapLoaded()) {
	    board_id = mapHandler().boardIDfromIP(ip_string);
	    try{
	        std::stringstream b_id;
	        b_id << board_id;
	        boardId = b_id.str();
	    } // try
	    catch(std::exception& e) {
	        std::cout << "EventBuilder::decode_event    Unable to convert board id (" << board_id << ") to string." << std::endl;
	        boardId="";
	    }//catch
    }
    else {
        try {
            boardId = ip_string.substr(ip_string.find_last_of('.')+1);
            board_id = std::stoi(boardId);
        } // try
        catch(std::exception& e) {
            std::cout << "EventBuilder::decode_event    Unable to get board id from board ip "
                    << ip_string << std::endl;
            boardId = "";
        }
    }

//    std::cout << "EventBuilder::decode_event    Incoming packet: " << datagram.data() << "  from: " << ip_string << std::endl;

    //std::vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.begin()+num_bytes/sizeof(uint32_t));
    //DaqBuffer
    std::vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.end());
    std::vector<uint32_t> datagram_vector; 
    //DaqBuffer
    //#warning not reversing 32 bits
    #warning check that reversal is needed in real use case
    for(const auto& data : datagram_vector_tmp) {
        datagram_vector.push_back(bits::endian_swap32()(data));
    }
    
    //for(auto& x : datagram_vector) {
    //    std::cout << std::hex << x << std::endl;
    //}
    //std::cout << "+++++++++" << std::endl;
    //bool hit_trailer = false;
    //for(auto& x : datagram) {
    //    if(!hit_trailer)
    //    std::cout << std::hex << x << std::endl;
    //    if(x==0xffffffff) hit_trailer=true;
    //}
    //std::cout << "++++++++++++" << std::endl;
    //return;
    //std::reverse(std::begin(datagram_vector), std::end(datagram_vector));
    //for(auto& x : datagram_vector) {
    //    std::cout << std::hex << x << std::endl;
    //}
    

    uint32_t frame_counter = datagram_vector.at(0);

    //for(auto& x : datagram_vector) {
    //    std::cout << std::hex << x << std::endl;
    //}
    //std::cout << "=============" << std::endl;
   
    

    while(true) {
        //std::cout << "frame: " << std::hex << frame_counter << std::endl;
        if(frame_counter == 0xffffffff) { break; }

        /////////////////////////////////////////////////
        // these are the data containers per chip
        /////////////////////////////////////////////////
        std::vector<int> _tdo;
        std::vector<int> _pdo;
        std::vector<int> _flag;
        std::vector<int> _threshold;
        std::vector<int> _bcid;
        std::vector<int> _grayDecoded;
        std::vector<int> _channelId;
        std::vector<int> _febChannelId;
        std::vector<int> _mappedChannelId;
        std::vector<int> _neighbor;

        boost::dynamic_bitset<> full_event_data(32*datagram_vector.size(), 0);
        for(int i = 0; i < (int)datagram_vector.size(); i++) {
            boost::dynamic_bitset<> tmp(32*datagram_vector.size(), datagram_vector.at(i));
            full_event_data = (full_event_data << (i==0 ? 0 : 32)) | tmp;
        }

        uint32_t trig_counter = datagram_vector.at(0);
        int32_t vmm_id = datagram_vector.at(1) & 0xff;
        uint32_t trig_timestamp = ( datagram_vector.at(1) & 0xffff00 ) >> 8;  
        #warning what are the precision bits?
        uint32_t precision = ( datagram_vector.at(1) & 0xff000000 ) >> 24;


        if(dbg()) {
        //if(true) {
            m_sx << "********************************************************\n"
               << " Data from board #  : " << (boardId=="" ? "mapping not loaded" : boardId) << "\n"
               << "   > IP             : " << ip_string << "\n"
               << "   > VMM ID         : " << vmm_id << "\n"
               << "   > Data           : " << full_event_data << "\n"
               << "********************************************************";
            //stream_lock.lock();
            std::cout << m_sx.str() << std::endl; m_sx.str("");
            //stream_lock.unlock();
        }

        for(int i = 2; i < (int)datagram_vector.size(); ) {
            frame_counter = datagram_vector.at(i);
            //std::cout << "in loop i  : " << std::hex << datagram_vector.at(i) << std::endl;
            //std::cout << "in loop i+1: " << std::hex << datagram_vector.at(i+1) << std::endl;
            if(frame_counter == 0xffffffff) { break; }

            uint32_t data0 = datagram_vector.at(i);
            uint32_t data1 = datagram_vector.at(i+1);

            //std::cout << "data0 : " << std::hex << data0 << std::endl;
            //std::cout << "data1 : " << std::hex << data1 << std::endl;
            //std::cout << "---------------" << std::endl;

            // ----------- pdo ----------- //
            uint32_t pdo = (data0 & 0x3ff);
            boost::dynamic_bitset<> pdo_bits(32, pdo);
            _pdo.push_back(pdo);

            // ------ gray (bcid) -------- //
            uint32_t gray = (data0 & 0x3ffc00) >> 10;
            _bcid.push_back(gray);

            // ---- gray decoded bcid ---- // 
            uint32_t decoded = decodeGray(gray);
            _grayDecoded.push_back(decoded);

            // ----------- tdo ----------- //
            uint32_t tdo = (data0 & 0x3fc00000) >> 22;
            _tdo.push_back(tdo);
            
            // ---------- flag ----------- //
            uint32_t flag = (data1 & 0x1);
            _flag.push_back(flag);

            // ------- threshold --------- //
            uint32_t threshold = (data1 & 0x2) >> 1;
            _threshold.push_back(threshold);

            // ------  vmm channel ------ //
            int32_t vmm_channel = (data1 & 0xfc) >> 2;
            _channelId.push_back(vmm_channel);

            int32_t mapped_channel = -1;
            int32_t feb_channel_no = -1;
            //std::cout << "HARD CODING BOARD ID" << std::endl;
            //boardId = "1";
            //std::string vmm_id_str = "5";
            //std::string vmm_chan_str = "20";
            //if(mapping() && mapHandler().mapLoaded() && num_bytes>13) {
            //DaqBuffer
            if(mapping() && mapHandler().mapLoaded() && n_bytes_read>0) {
                //if(!(boardId=="")) {
		        if(board_id>=0) {
                    int32_t vmm_id_to_send = vmm_id;
                    mapped_channel = mapHandler().elementNumber(board_id, vmm_id_to_send, vmm_channel);
	 	    //std::cout << "board id: " << boardId << " vmmid: " << vmm_id << "  chan: " << vmm_channel << "   mapped to--> " << mapped_channel << endl;
                    feb_channel_no = mapHandler().boardChannel(board_id, vmm_id, vmm_channel);
		    //feb_channel_no = vmm_channel + 64*vmm_id;
                }
            }
            _mappedChannelId.push_back(mapped_channel);
            _febChannelId.push_back(feb_channel_no);

            if(calibrationRun()) {
                if(m_calib_channel < 0) {
                    //stream_lock.lock();
                    std::cout << "EventBuilder::decode_event    Channel for calibration not set. Will store as 0." << std::endl;
                    //stream_lock.unlock();
                    _neighbor.push_back(0);
                }
                else {
                    _neighbor.push_back(!(static_cast<int>(m_calib_channel) == static_cast<int>(vmm_channel)));
                }
            }

            if(dbg() && verbose) {
            //if(true) {
                m_sx << "[" << counter << ", " << i << "] pdo   : " <<  (pdo) <<     "    tdo  : " << tdo << "\n"
                   << "                                 gray  : " <<  (gray) <<    "    bcid : " << decoded << "  flag  : " << flag << "\n"
                   << "                                 thresh: " <<  (threshold)  << " vmmchan : " << vmm_channel;
                //stream_lock.lock();
                std::cout << m_sx.str() << std::endl; m_sx.str("");
                //stream_lock.unlock();
            }

            // online monitoring
            //DaqBuffer
            if(monitoring() && mapping() &&  mapHandler().mapLoaded() && !(board_id<0) && n_bytes_read>0) {
                if(rand()%50==0) {
                    std::stringstream monitoring_string;
                    monitoring_string << n_hits
                                      << " " << mapHandler().chipName(board_id, vmm_id)
                                      << " " << vmm_channel
                                      << " " << pdo
                                      << " " << tdo
                                      << " " << gray;
                    monTool().addEventString(monitoring_string.str());
                    //monTool().send(monitoring_string.str());
                }
            }

            // move 2*32 bits forward 
            //DaqBuffer
            n_bytes_read += 2*4;  // each element in datagram is 32 bit

            i += 2;
        }

        //fill
        //don't store events that are simply a header+trailer
        //if(writeNtuple() && (num_bytes > n_bytes_for_empty)) {
        if(writeNtuple() && n_bytes_read>0) {
            _chipId_tree.push_back(vmm_id);
            _boardId_tree.push_back( (boardId=="" ? -1 : stoi(boardId)));
            _eventSize_tree.push_back(n_bytes_read);
            _trigTimeStamp_tree.push_back(trig_timestamp);
            _trigCounter_tree.push_back(trig_counter);
            _daq_timestamp_s_tree.push_back((uint32_t)(now_s));
            _daq_timestamp_ns_tree.push_back((uint32_t)(now_ns));

            _tdo_tree.push_back(_tdo);
            _pdo_tree.push_back(_pdo);
            _flag_tree.push_back(_flag);
            _threshold_tree.push_back(_threshold);
            _bcid_tree.push_back(_bcid);
            _grayDecoded_tree.push_back(_grayDecoded);
            _channelId_tree.push_back(_channelId);
            _mappedChannelId_tree.push_back(_mappedChannelId);
            _febChannelId_tree.push_back(_febChannelId);

            if(calibrationRun())
                _neighbor_tree.push_back(_neighbor);
        }


    } // while (continues until hitting the trailer)

    if(frame_counter == 0xffffffff) {
        //std::cout << "fafafa " << std::hex << frame_counter << std::endl;


        if(writeNtuple() && n_bytes_read>0) { //m_fill_event) {
            //boost::unique_lock<boost::timed_mutex> lock(*m_data_mutex, boost::try_to_lock);
            //if(lock.owns_lock() || lock.try_lock_for(boost::chrono::milliseconds(100))) {
                //usleep(500000);

                ////////////////////////////////////////////////////
                // clear the global tree variables
                ////////////////////////////////////////////////////
                m_chipId.clear();
                m_boardId.clear();
                m_triggerTimeStamp.clear();
                m_triggerCounter.clear();
                m_eventSize.clear();
                m_daq_timestamp_s.clear();
                m_daq_timestamp_ns.clear();
                m_art.clear();
                m_art_valid.clear();
                m_art_trigger.clear();

                m_tdo.clear();
                m_pdo.clear();
                m_flag.clear();
                m_threshold.clear();
                m_bcid.clear();
                m_grayDecoded.clear();
                m_channelId.clear();
                m_mappedChannelId.clear();
                m_febChannelId.clear();

                m_neighbor_calib.clear();

                ////////////////////////////////////////////////////
                // assign the tree variables
                ////////////////////////////////////////////////////

                m_eventNumberFAFA = n_hits;

                m_chipId = _chipId_tree;
                m_boardId = _boardId_tree;
                m_triggerTimeStamp = _trigTimeStamp_tree;
                m_triggerCounter = _trigCounter_tree;
                m_eventSize = _eventSize_tree;
                m_daq_timestamp_s = _daq_timestamp_s_tree;
                m_daq_timestamp_ns = _daq_timestamp_ns_tree;

                m_tdo = _tdo_tree;
                m_pdo = _pdo_tree;
                m_flag = _flag_tree;
                m_threshold = _threshold_tree;
                m_bcid = _bcid_tree;
                m_grayDecoded = _grayDecoded_tree;
                m_channelId = _channelId_tree;
                m_mappedChannelId = _mappedChannelId_tree;
                m_febChannelId = _febChannelId_tree;

                if(calibrationRun())
                    m_neighbor_calib = _neighbor_tree; 

                //////////////////////////////////////////////
                // fill the tree branches
                //////////////////////////////////////////////
                fill_event();

                //boost::timed_mutex *m = lock.release();
                //m->unlock();

                m_fill_event = false;
            //} // while
            //else {
            //    //stream_lock.lock();
            //    std::cout << "EventBuilder::decode_event     [" << boost::this_thread::get_id() << "] Lock timed out. Missed filling event " << (n_hits) << std::endl;
            //    //stream_lock.unlock();

            //    m_fill_event = false;
            //}
        } // writeNtuple


        //DaqBuffer
        #warning not incrementing counter in decode_data
        //update daq counter
        n_hits++;
        counter = n_hits;

    } // at trailer

    //stream_lock.unlock();
}

bool EventBuilder::is_valid_packet(uint32_t length)
{
    bool is_valid = true;
    if(vmm_type()==2) {
        uint32_t header_size = 2;
        uint32_t trailer_size = 1;
        uint32_t vmm_data_length = (length - (header_size + trailer_size));
        if(!(length >= (header_size + trailer_size))) {
            is_valid = false;
        }
        else if(!(vmm_data_length%2==0)) {
            is_valid = false;
        }
    }
    else if(vmm_type()==3) {
        if(m_do_L0_decoding) {
            uint32_t header_size = 3;
            uint32_t trailer_size = 1;
            if(!(length >= (header_size + trailer_size))) {
                is_valid = false;
            }
        }
        else if(!m_do_L0_decoding) {
            uint32_t header_size = 3;
            uint32_t trailer_size = 1;
            uint32_t vmm_data_length = (length - (header_size + trailer_size));
            if(!(length >= (header_size + trailer_size))) {
                is_valid = false;
            }
            else if(!(vmm_data_length%2==0)) {
                is_valid = false;
            }
        }
    }

    //if(!is_valid) {
    //    cout << "EventBuilder::is_valid_packet    Invalid packet found" << endl;
    //}

    return is_valid;
}

void EventBuilder::decode_event_trig(std::string& ip_string, std::vector<uint32_t>& datagram, uint32_t data_length, int& counter, int& trigcounter)
{
    size_t n_bytes_read = 0;

    //cout << "EventBuilder::decode_event_trig    Received data (vmm_type=" << vmm_type() << ")  [length=" << data_length<<"] ";
    //for(auto x : datagram) {
    //    cout << " " << std::hex << x;
    //    //if(x!=0x0) cout << " " << std::hex << x;
    //}
    //cout << std::dec << endl;

    if(!is_valid_packet(data_length)) return;

    ////////////////////////////////////////////////
    // get the board ID from the packet IP
    ////////////////////////////////////////////////
    std::string boardId = "";
    int board_id = -1;
    std::stringstream b_id;
    if(mapping() && mapHandler().mapLoaded()) {
        board_id = mapHandler().boardIDfromIP(ip_string);
        try{
            b_id.str("");
            b_id << board_id;
            boardId = b_id.str();
        }
        catch(std::exception& e) {
            std::cout << "EventBuilder::decode_event_trig    [" << boost::this_thread::get_id() << "]"
                    << " Unable to convert board id (" << board_id << ") to string" << endl;
            boardId = "";
        } // catch
    }
    else {
        try {
            boardId = ip_string.substr(ip_string.find_last_of('.')+1);
            board_id = std::stoi(boardId);
        }
        catch(std::exception& e) {
            std::cout << "EventBuilder::decode_event_trig    [" << boost::this_thread::get_id() << "]"
                    << " Unable to get board id from board ip " << ip_string << endl;
            boardId = "";
        } // catch
    }

    // reverse each of the 32-bit words
    std::vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.end());
    std::vector<uint32_t> datagram_vector;
    for(const auto& data : datagram_vector_tmp) {
        datagram_vector.push_back(bits::endian_swap32()(data));
    }

    // frame counter
    bool continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin(), datagram_vector.begin() + vmm_type()), true);
    //uint32_t frame_counter = datagram_vector.at(0);

    while(true) {
        //if(frame_counter == MMFE8TRAILER) {
        //if(m_decoder.frame() == MMFE8TRAILER) {
        if(!continue_decoding) {
            break; // store the built up hits
        }

        size_t trigger_count_pos = 0;
        if(vmm_type()==3) trigger_count_pos = 1;
        uint32_t trigger_count = datagram_vector.at(trigger_count_pos);


        //stream_lock.lock();
        //std::cout << "trigger count: " << trigger_count << "  [" << std::hex << trigger_count << "]" << std::dec << std::endl;
        //stream_lock.unlock();

        // container to store hit data
        VMMHit vmm_hit;

        unsigned int n_empty_data = 0;

        if(!m_do_L0_decoding) {
            for(unsigned int i = vmm_type(); i < datagram_vector.size(); i+=2) {

                continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin() + i, datagram_vector.begin() + i + 2), false);

                // if we have seen multiple periods of empty data, break
                if(datagram_vector.at(i) == 0x0 && datagram_vector.at(i+1) == 0x0) n_empty_data++; 
                if(n_empty_data >= 4) continue_decoding = false;

                if(!continue_decoding) {
                    break;
                }
                vmm_hit.clear();

                // check if we have hit the trailer
                //frame_counter = datagram_vector.at(i);
                //frame_counter = m_decoder.frame();
                //if(m_decoder.frame() == MMFE8TRAILER) { break; } // break out of loop i to hit end of while loop

                // trigger count
                vmm_hit.set_trigger_counter(trigger_count);

                // trigger timestamp
                vmm_hit.set_trigger_timestamp(m_decoder.trigger_timestamp());

                // vmm channel
                vmm_hit.set_vmm_channel(m_decoder.vmm_channel());

                // mapping
                int32_t mapped_channel = -1;
                int32_t feb_channel_no = -1;
                if(mapping() && mapHandler().mapLoaded()) {
                    if(board_id>=0) {
                        int chip_no = m_decoder.chip_number();
                        int vmm_chan = m_decoder.vmm_channel();
                        mapped_channel = mapHandler().elementNumber(board_id, chip_no, vmm_chan); 
                        feb_channel_no = mapHandler().boardChannel(board_id, chip_no, vmm_chan); 
                    } // valid board
                } // mapping
                vmm_hit.set_mapped_channel(mapped_channel);
                vmm_hit.set_feb_channel(feb_channel_no);

                // pdo
                vmm_hit.set_pdo(m_decoder.pdo());

                // tdo
                vmm_hit.set_tdo(m_decoder.tdo());

                // gray code
                vmm_hit.set_graycode_bcid(m_decoder.bcid());

                // gray decoded
                vmm_hit.set_decoded_graycode_bcid(m_decoder.decoded_bcid());

                // flag
                vmm_hit.set_flag(m_decoder.flag());

                // signal passes threshold
                vmm_hit.set_pass_threshold(m_decoder.pass_threshold());

                // ART
                vmm_hit.set_art(m_decoder.art_address());
                vmm_hit.set_art_valid(m_decoder.art_valid());
                vmm_hit.set_art_trigger(m_decoder.art_trigger());


                /////////////////////////////////////////////
                // register that we read some VMM hit data
                /////////////////////////////////////////////
                n_bytes_read += 2*4; // VMM payload is 2 32-bit words

                /////////////////////////////////////////////
                // store 'meta' info
                /////////////////////////////////////////////
                vmm_hit.set_board_id(board_id);

                /////////////////////////////////////////////
                // store the hit
                /////////////////////////////////////////////
                //m_trigger_events.add_hit(trigger_count, vmm_hit);
                int32_t chip_no = m_decoder.chip_number();
                m_trigger_events.add_hit2(trigger_count, board_id, chip_no, vmm_hit); 

                if(vmm_type()==3) {
                    //ArtPair art_pair(m_decoder.art_valid(), m_decoder.art_address());
                    //m_trigger_events.add_art(trigger_count, board_id, chip_no, art_pair); 
                    ArtTuple art_tuple(m_decoder.art_valid(), m_decoder.art_address(), m_decoder.art_trigger());
                    m_trigger_events.add_art2(trigger_count, board_id, chip_no, art_tuple);
                }

            } // i
        } // not L0 decoding
        else if(m_do_L0_decoding) {
            for(unsigned int i = 3; i < datagram_vector.size(); i+=1) {
                continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin() + i,
                                    datagram_vector.begin() + i + 1), false);

                // if we have seen multiple periods of empty data, break
                if(datagram_vector.at(i) == 0x0) n_empty_data++; 
                if(n_empty_data >= 4) continue_decoding = false;

                if(!continue_decoding) {
                    break;
                }
                vmm_hit.clear();

                // vmm channel
                vmm_hit.set_vmm_channel(m_decoder.vmm_channel());
                // mapping
                int32_t mapped_channel = -1;
                int32_t feb_channel_no = -1;
                if(mapping() && mapHandler().mapLoaded()) {
                    if(board_id>=0) {
                        int chip_no = m_decoder.chip_number();
                        int vmm_chan = m_decoder.vmm_channel();
                        mapped_channel = mapHandler().elementNumber(board_id, chip_no, vmm_chan); 
                        feb_channel_no = mapHandler().boardChannel(board_id, chip_no, vmm_chan); 
                    } // valid board
                } // mapping
                vmm_hit.set_mapped_channel(mapped_channel);
                vmm_hit.set_feb_channel(feb_channel_no);

                // trigger count
                vmm_hit.set_trigger_counter(trigger_count);

                // pdo
                vmm_hit.set_pdo(m_decoder.pdo());

                // tdo
                vmm_hit.set_tdo(m_decoder.tdo());

                // gray code
                vmm_hit.set_graycode_bcid(m_decoder.bcid());

                // gray decoded
                vmm_hit.set_decoded_graycode_bcid(m_decoder.decoded_bcid());

                // relative bcid
                vmm_hit.set_relative_bcid(m_decoder.relbcid());

                // vmm overflow bit
                vmm_hit.set_overflow_bit(m_decoder.vmm_overflow());

                // orbit counter
                vmm_hit.set_orbit_counter(m_decoder.orbit_counter());

                // ART
                vmm_hit.set_art(m_decoder.art_address());
                vmm_hit.set_art_valid(m_decoder.art_valid());
                vmm_hit.set_art_trigger(m_decoder.art_trigger());

                /////////////////////////////////////////////
                // register that we read some VMM hit data
                /////////////////////////////////////////////
                n_bytes_read += 1*4; // VMM L0 payload is 1 32-bit words

                //std::cout << "_trig   orbit: " << std::dec << m_decoder.orbit_counter()
                //<< "  bcid: " << m_decoder.bcid() << "  decoded: " << m_decoder.decoded_bcid() << std::endl;

                /////////////////////////////////////////////
                // store 'meta' info
                /////////////////////////////////////////////
                vmm_hit.set_board_id(board_id);

                /////////////////////////////////////////////
                // store the hit
                /////////////////////////////////////////////
                //m_trigger_events.add_hit(trigger_count, vmm_hit);
                int32_t chip_no = m_decoder.chip_number();
                m_trigger_events.add_hit2(trigger_count, board_id, chip_no, vmm_hit); 

                //ArtPair art_pair(m_decoder.art_valid(), m_decoder.art_address());
                //m_trigger_events.add_art(trigger_count, board_id, chip_no, art_pair); 

                ArtTuple art_tuple(m_decoder.art_valid(), m_decoder.art_address(), m_decoder.art_trigger());
                m_trigger_events.add_art2(trigger_count, board_id, chip_no, art_tuple);

            } // i

        } // L0 decoding
    } // while
    if(m_trigger_events.ready_to_read()) {
        fill_trigger_event(counter, trigcounter);
    }

}

void EventBuilder::get_delta_params(int* trigger, int* board)
{
    m_trigger_for_delta = trigger;
    m_board_for_delta = board;
}

void EventBuilder::fill_trigger_event(int& hitcounts, int& trigcounts)
{
    bool trig_ok = true;
    uint32_t trigger_to_readout = m_trigger_events.trigger_to_read(trig_ok);
    //m_trigger_events.print();
    //stream_lock.lock();
    //cout << "EventBuilder::fill_trigger_event    trig_ok? " << trig_ok << "  trig: " << trigger_to_readout << "   " << std::hex << trigger_to_readout << endl;
    //stream_lock.unlock();
    if(!trig_ok) return; 

    auto now = time_point_cast<nanoseconds>(system_clock::now());
    uint64_t now_count = now.time_since_epoch().count();
    uint64_t now_s     = now_count / 1e9;
    uint64_t now_ns    = now_count - (now_s * 1e9);
    
    n_triggers++;
    //n_hits++;
    //counter = n_hits;
    trigcounts = n_triggers;
    //stream_lock.lock();
    //std::cout << "FLUSHING trigger_to_readout : " << trigger_to_readout << std::endl;
    ////std::cout << "EventBuilder::fill_trigger_event    trigger_to_readout = "
    ////        << std::dec << trigger_to_readout << "  (last trigger read = "
    ////        << m_last_trigger_read << std::endl;
    //stream_lock.unlock();

   // if( (trigger_to_readout != (m_last_trigger_read+1))) {
   //     stream_lock.lock();
   //     std::cout << "EventBuilder::fill_trigger_event    WARNING Mismatch in trigger being "
   //         << "readout versus expected trigger! (current=" << trigger_to_readout
   //         << " expected=" << (m_last_trigger_read+1) << ")" << std::endl;
   //     stream_lock.unlock();
   // }

    //n_triggers++;
    //trigcounts = n_triggers;

    ////////////////////////////////////////////////////
    // grab the hits associated with the trigger
    // to be read out
    ////////////////////////////////////////////////////
    //std::vector<VMMHit> hit = m_trigger_events.get_trigger(trigger_to_readout);
    std::map<BoardChipPair, std::vector<VMMHit> > hitgroup = m_trigger_events.get_trigger2(trigger_to_readout);

    //std::map<BoardChipPair, ArtPair> artgroup;
    //if(vmm_type()==3)
    //    artgroup = m_trigger_events.get_art(trigger_to_readout);
    std::map<BoardChipPair, ArtTuple> artgroup;
    if(vmm_type()==3)
        artgroup = m_trigger_events.get_art2(trigger_to_readout);

    if(hitgroup.size()==0) {
        cout << "EventBuilder::fill_trigger_event    WARNING Hit group empty for "
            << "trigger " << trigger_to_readout << " (0x" << std::hex
            << trigger_to_readout << std::dec << ")" << endl;
        return;
    }
    //stream_lock.lock();
    //for(const auto h : hit) {
    //    BoardChipPair bcpair = h.first;
    //    std::cout << "EventBuilder::fill_trigger_event  board# " << bcpair.first
    //            << "  chip# " << bcpair.second << " ";
    //    for(auto vmm : h.second) {
    //        std::cout << " [chan: " << vmm.get_vmm_channel() << " board #" << vmm.get_board_id()
    //            << "] ";
    //    }
    //    std::cout << std::endl;
    //}
    //stream_lock.unlock();


    //stream_lock.lock();
    //std::cout << "EventBuilder::fill_trigger_event    number of hits for trigger "
    //        << std::dec << trigger_to_readout << "  " << hit.size() << "  ";
    //for(auto h : hit) {
    //    std::cout << "[chan: " << h.get_vmm_channel() << " board #" << h.get_board_id() << "]  ";
    //}
    //std::cout << std::endl;
    //stream_lock.unlock();

    bool write = writeNtuple();
    bool mon = (monitoring() && mapping() && mapHandler().mapLoaded() && (rand()%m_monitor_rate==0));
    //if(mon && !m_current_monitoring_state) {
    //        m_current_monitoring_state = true;
    //        m_monTool->start();
    //}
    //else if(!mon && m_current_monitoring_state) {
    //    m_current_monitoring_state = false;
    //    m_monTool->stop();
    //}

    //if(write) {
    /////////////////////////////////////////////////////
    // containers to store
    /////////////////////////////////////////////////////
    std::vector<int> board_id_tree;
    std::vector<int> chip_id_tree;
    std::vector<int> eventsize_tree;
    std::vector<int> trigcounter_tree;
    std::vector<int> trigtimestamp_tree;
    std::vector<int> art_tree;
    std::vector<int> art_valid_tree;
    std::vector<int> art_trigger_tree;
    std::vector<int> daq_timestamp_s_tree;
    std::vector<int> daq_timestamp_ns_tree;

    std::vector< std::vector<int> > vmm_channel_tree; 
    std::vector< std::vector<int> > pdo_tree;
    std::vector< std::vector<int> > tdo_tree;
    std::vector< std::vector<int> > flag_tree;
    std::vector< std::vector<int> > threshold_tree;
    std::vector< std::vector<int> > bcid_tree;
    std::vector< std::vector<int> > graydecoded_tree;
    std::vector< std::vector<int> > relbcid_tree;
    std::vector< std::vector<int> > overflow_tree;
    std::vector< std::vector<int> > orbit_counter_tree;
    std::vector< std::vector<int> > febchannel_tree;
    std::vector< std::vector<int> > mappedchannel_tree;
    std::vector< std::vector<int> > neighbor_tree;

    std::vector<int> vmm_channel;
    std::vector<int> pdo;
    std::vector<int> tdo;
    std::vector<int> flag;
    std::vector<int> threshold;
    std::vector<int> bcid;
    std::vector<int> graydecoded;
    std::vector<int> relbcid;
    std::vector<int> overflow;
    std::vector<int> orbit_counter;
    std::vector<int> febchannel;
    std::vector<int> mappedchannel;
    std::vector<int> neighbor;
    //} // write

    if(mon) {
        m_monitoring_stream.str(""); // clear monitoring string

    }

    //vector<int> boards_readout;
    size_t n_bytes_read = 0;

    bool got_timestamp = false;
    uint32_t timestamp = 0;

    for(const auto group : hitgroup) {
        // collect hits from each (board,chip) pair
        BoardChipPair bcpair = group.first;

        //boards_readout.push_back(bcpair.first);
        (*m_board_for_delta) = bcpair.first; 
        (*m_trigger_for_delta) = (trigger_to_readout - (m_last_trigger_read+1));

        if(write) {
            // clear the vmm hit data containers
            vmm_channel.clear();
            pdo.clear();
            tdo.clear();
            flag.clear();
            threshold.clear();
            bcid.clear();
            graydecoded.clear();
            relbcid.clear();
            overflow.clear();
            orbit_counter.clear();
            febchannel.clear();
            mappedchannel.clear();
            neighbor.clear();
        } // write

        std::vector<VMMHit> hits = hitgroup[bcpair];
        for(auto hit : hits) {
            n_hits++;
            hitcounts = n_hits;

            if(hit.get_trigger_counter() != trigger_to_readout) {
                cout << "EventBuilder::fill_trigger_event    **** VMMHit trigger counter is not correct: VMMHit = " << hit.get_trigger_counter() << "  Expected = " << trigger_to_readout << " **** " << endl;
            }

            if(!got_timestamp) { timestamp = hit.get_trigger_timestamp(); got_timestamp = true; }

            if(write) {
                vmm_channel.push_back(hit.get_vmm_channel());
                pdo.push_back(hit.get_pdo());
                tdo.push_back(hit.get_tdo());
                flag.push_back(hit.get_flag());
                threshold.push_back(hit.get_pass_threshold());
                bcid.push_back(hit.get_graycode_bcid());
                graydecoded.push_back(hit.get_decoded_graycode_bcid());
                relbcid.push_back(hit.get_relative_bcid());
                overflow.push_back(hit.get_overflow_bit());
                orbit_counter.push_back(hit.get_orbit_counter());
                febchannel.push_back(hit.get_feb_channel());
                mappedchannel.push_back(hit.get_mapped_channel());

                // debug
                //cout << "board id: " << bcpair.first << "  "
                //     << "chip id: " << bcpair.second << "  "
                //     << "channel: " << hit.get_vmm_channel() << "  "
                //     << "pdo: " << hit.get_pdo() << "  "
                //     << "tdo: " << hit.get_tdo() << "  "
                //     << "rel bcid: " << hit.get_relative_bcid() << endl;


            } // write
    

            if(mon) {
                string chip_name = mapHandler().chipName(bcpair.first, bcpair.second);
                //chip_name = "VMM2.00.0";
                if(chip_name == "" && n_warning < 20) {
                    n_warning++;
                    stream_lock.lock();
                    std::cout << "EventBuilder::fill_trigger_event    [" << n_warning
                        << "/20] ERROR MapHandler returns "
                        << "invalid VMM name '" << chip_name << "' when provided "
                        << "[board,chip]=[" << bcpair.first << ","<<bcpair.second<<"] --> will stop monitoring "
                        << "(is your mapping configuration file OK?)" << std::endl;
                    stream_lock.unlock();
                    if(n_warning >= 20) {
                        mon = false;
                        (*m_do_do_monitoring) = false;
                        //m_current_monitoring_state = false;
                    }
                }
                else {
                    // add the trigger counter at the start
                    m_monitoring_stream << " " <<  trigger_to_readout;
                    //int pdo = (rand()%45 + 1);
                    //int tdo = (rand()%75 + 1);
                    m_monitoring_stream << " " << chip_name
                    
                    //m_monitoring_stream << " " << mapHandler().chipName(bcpair.first, bcpair.second)
                                        << " " << hit.get_vmm_channel()
                                        << " " << hit.get_pdo()
                                        << " " << hit.get_tdo()
                                        //<< " " << pdo
                                        //<< " " << tdo
                                        << " " << hit.get_decoded_graycode_bcid();
                    n_monitor_strings++;
                    if(n_monitor_strings > 10) {
                        monTool().addEventString(m_monitoring_stream.str());
                        m_monitoring_stream.str("");
                        n_monitor_strings = 0;
                    }
                }
            } // mon

            int multiplier = (m_do_L0_decoding ? 1 : 2);
            n_bytes_read += multiplier*4;
        } // hit

        if(n_bytes_read>0 && write) {
            trigcounter_tree.push_back(trigger_to_readout);
            trigtimestamp_tree.push_back(timestamp);
            board_id_tree.push_back(bcpair.first);
            chip_id_tree.push_back(bcpair.second);
            eventsize_tree.push_back(n_bytes_read);
            daq_timestamp_s_tree.push_back((uint32_t)(now_s));
            daq_timestamp_ns_tree.push_back((uint32_t)(now_ns));

            //ART
            if(vmm_type()==3) {
                //art_valid_tree.push_back(artgroup[bcpair].first);
                //art_tree.push_back(artgroup[bcpair].second);
                int valid = std::get<0>(artgroup[bcpair]);
                int address = std::get<1>(artgroup[bcpair]);
                int art_trig = std::get<2>(artgroup[bcpair]);

                //cout << "EventBuilder::fill_trigger_evnet   art valid = " << valid << "  address = " << address << "  trigger = " << art_trig << endl;
                art_valid_tree.push_back(valid);
                art_tree.push_back(address);
                art_trigger_tree.push_back(art_trig);
            }

            vmm_channel_tree.push_back(vmm_channel);
            pdo_tree.push_back(pdo);
            tdo_tree.push_back(tdo);
            flag_tree.push_back(flag);
            threshold_tree.push_back(threshold);
            bcid_tree.push_back(bcid);
            graydecoded_tree.push_back(graydecoded);
            relbcid_tree.push_back(relbcid);
            overflow_tree.push_back(overflow);
            orbit_counter_tree.push_back(orbit_counter);
            febchannel_tree.push_back(febchannel);
            mappedchannel_tree.push_back(mappedchannel);
        }
    } // group

    //cout << "EventBuilder::fill_trigger_event    Boards readout for trigger 0x" << std::hex << trigger_to_readout << "  (" << std::dec << trigger_to_readout << ")   ";
    //for(auto x : boards_readout) cout << " " << x;
    //cout << endl;

    if(n_bytes_read>0 && write) {
        m_chipId.clear();
        m_boardId.clear();
        m_triggerCounter.clear();
        m_triggerTimeStamp.clear();
        m_eventSize.clear();
        m_daq_timestamp_s.clear();
        m_daq_timestamp_ns.clear();

        m_art.clear();
        m_art_valid.clear();
        m_art_trigger.clear();

        m_channelId.clear();
        m_pdo.clear();
        m_tdo.clear();
        m_flag.clear();
        m_threshold.clear();
        m_bcid.clear();
        m_grayDecoded.clear();
        m_relbcid.clear();
        m_overflow.clear();
        m_orbit_count.clear();
        m_mappedChannelId.clear();
        m_febChannelId.clear();

        m_chipId = chip_id_tree;
        m_boardId = board_id_tree;
        m_triggerCounter = trigcounter_tree;
        m_triggerTimeStamp = trigtimestamp_tree;
        m_eventSize = eventsize_tree;
        m_daq_timestamp_s = daq_timestamp_s_tree;
        m_daq_timestamp_ns = daq_timestamp_ns_tree;

        if(vmm_type()==3) {
            m_art = art_tree;
            m_art_valid = art_valid_tree;
            m_art_trigger = art_trigger_tree;
        }

        m_channelId = vmm_channel_tree;
        m_pdo = pdo_tree;
        m_tdo = tdo_tree;
        m_flag = flag_tree;
        m_threshold = threshold_tree;
        m_bcid = bcid_tree;
        m_grayDecoded = graydecoded_tree;
        m_relbcid = relbcid_tree;
        m_overflow = overflow_tree;
        m_orbit_count = orbit_counter_tree;
        m_mappedChannelId = mappedchannel_tree;
        m_febChannelId = febchannel_tree;

        ///////////////////////////////////////
        // fill the ntuple branches
        ///////////////////////////////////////
        fill_event();
    }

    if(n_bytes_read>0 && mon) {
        // send this stream to the monitoring tool to buffer
        monTool().addEventString(m_monitoring_stream.str());
        n_monitor_strings = 0;
    }

    /*
    if(monitoring() && mapping() && mapHandler().mapLoaded() && (rand()%50==0)) {
        m_monitoring_stream.str("");
        // add the trigger counter at the start
        m_monitoring_stream << trigger_to_readout; 
        for(const auto group : hitgroup) {

            // collect hits from each (board,chip) pair
            BoardChipPair bcpair = group.first;

            int board_id = bcpair.first; 
            int chip_id = bcpair.second; 

            std::vector<VMMHit> hits = hitgroup[bcpair];
            for(auto hit : hits) {
                m_monitoring_stream << " " << mapHandler().chipName(board_id, chip_id)
                                    << " " << hit.get_vmm_channel()
                                    << " " << hit.get_pdo()
                                    << " " << hit.get_tdo()
                                    << " " << hit.get_decoded_graycode_bcid();
            } // hit
        } // group
        // send this stream to the monitoring tool to buffer
        monTool().addEventString(m_monitoring_stream.str());

    }
    */

    //for(const auto group : hitgroup) {
    //    // collect hits from each (board,chip) pair
    //    BoardChipPair bcpair = group.first;
    //    std::vector<VMMHit> hits = hitgroup[bcpair];
    //    stream_lock.lock();
    //    std::cout << "hits for (board# " << bcpair.first << ", chip# " << bcpair.second
    //        << "): ";
    //    for(auto hit : hits) {
    //        std::cout << " chan# " << hit.get_vmm_channel() << " ";
    //    }
    //    std::cout << std::endl;
    //    stream_lock.unlock();


    //} // hitgroup

    m_last_trigger_read = trigger_to_readout;
    //m_trigger_events.remove_first_trigger();
    m_trigger_events.remove_trigger(trigger_to_readout);
}

uint32_t EventBuilder::decodeGray(uint32_t gray)
{
    uint32_t mask;
    for( mask = gray >> 1; mask != 0; mask = mask >> 1) {
        gray = gray ^ mask;
    }
    return gray;
}

void EventBuilder::fill_event()
{
    if(writeNtuple()) {
        if(dbg()) {
            stream_lock.lock();
            msg()("Filling ROOT trees...","EventBuilder::fill_event");
            stream_lock.unlock();
        }
        
        m_daqRootFile->cd();
        m_vmm_tree->Fill();

        // option to write to file at a fixed interval
        if (m_tree_flush > 0 && m_vmm_tree->GetEntries() % m_tree_flush == 0)
          m_vmm_tree->AutoSave("SaveSelf");

    }
}

void EventBuilder::flush_remaining_events(int& hits, int& trigs)
{
    //size_t n_remaining = m_trigger_events.size();
    //while(n_remaining > 0) {
    for(int i = 0; i < (int)m_trigger_events.capacity(); i++) {
        //stream_lock.lock();
        //std::cout << "EventBuilder::flush_remaining_events() " << n_remaining
        //        << std::endl;
        //stream_lock.unlock();
        fill_trigger_event(hits, trigs);
        //n_triggers++;
        ////n_hits++;
        ////counter = n_hits;
        //trigs = n_triggers;
        //n_remaining--;
    }
    return;
}

void EventBuilder::write_output()
{
    if(writeNtuple()) {
        //if(m_trigger_events.size()) {
        //    stream_lock.lock();
        //    std::cout << "EventBuilder::write_output    Writing output when "
        //        " have " << m_trigger_events.size() << " triggers" << std::endl;
        //    stream_lock.unlock();
        //}
        //cout << "EventBuilder::write_output    trigger_events in buffer: "
        //        << m_trigger_events.size() << endl;
        //while(m_trigger_events.size()) {
        ////while(!m_trigger_events.empty()) {
        //    stream_lock.lock();
        //    std::cout << "EventBuilder::write_output    Writing remaining (pending) events" << std::endl;
        //    stream_lock.unlock();
        //    fill_trigger_event(n_hits, n_triggers);
        //}
        //m_trigger_events.clear();


        if(dbg()) {
            m_sx << "[" << boost::this_thread::get_id() << "]    Writing ROOT files";
            stream_lock.lock();
            msg()(m_sx,"EventBuilder::write_output"); m_sx.str("");
            stream_lock.unlock();
        }
        m_daqRootFile->cd();

        m_vmm_tree->Write("", TObject::kOverwrite);
        if(!m_daqRootFile->Write()) {
            msg()("ERROR Writing output DAQ ROOT file","EventBuilder::write_output");
        }

        // write the user comments about this run
        TNamed user_comment_branch("comments", m_run_comment.c_str());
        user_comment_branch.Write();

        m_daqRootFile->Close();

        stream_lock.lock();
        m_sx << "Run " << m_run_number << " stored in file:\n    " << m_output_rootfilename;
        msg()(m_sx, "EventBuilder::write_output"); m_sx.str("");
        stream_lock.unlock();
    }
    if(writeRaw()) {
        m_raw_dumper->close_output_file();
    }
}
void EventBuilder::updateCalibrationState(double gain, int dacThreshold, int dacAmplitude,
                                double tp_skew, int peakTime)
{
    m_gain_calib = gain;
    m_dacCounts_calib = dacThreshold;
    m_pulserCounts_calib = dacAmplitude;
    m_tpSkew_calib = tp_skew;
    m_peakTime_calib = peakTime;
}

