// vmm
#include "config_handler.h"
//#include "string_utils.h"

// std/stl
#include <bitset> // debugging
#include <exception>
#include <sstream>
using namespace std;

// boost
#include <boost/format.hpp>



//////////////////////////////////////////////////////////////////////////////
// ------------------------------------------------------------------------ //
//  ConfigHandler -- Constructor
// ------------------------------------------------------------------------ //
//////////////////////////////////////////////////////////////////////////////
ConfigHandler::ConfigHandler(QObject *parent) :
    QObject(parent),
    m_dbg(false),
    m_board_selection(-1),
    m_msg(0)
{
    m_channelmap.clear();
    m_channels.clear();
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::setSelectedBoards(int board_selection)
{
    m_board_selection = board_selection;
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::LoadMessageHandler(MessageHandler& m)
{
    m_msg = &m;
}
//// ------------------------------------------------------------------------ //
bool ConfigHandler::LoadConfig(const QString &filename)
{

    m_commSettings.config_filename = filename;

    using boost::property_tree::ptree;
    using namespace boost::property_tree::xml_parser;
    ptree pt;
    read_xml(filename.toStdString(), pt, trim_whitespace | no_comments);

    // load the communication settings (UDP ports, board IPs, etc...)
    m_commSettings = LoadCommInfo(pt);
    bool comm_ok = m_commSettings.ok;
    if(!comm_ok) {
        msg()("Problem loading CommInfo", "ConfigHandler::LoadConfig");
    }

    // Load the global configuration
    m_globalSettings = LoadGlobalSettings(pt);
    bool global_ok = m_globalSettings.ok;
    if(!global_ok) {
        msg()("Problem loading GlobalSettings", "ConfigHandler::LoadConfig");
    }

    // Load the VMM channel configuration
    m_channels = LoadVMMChannelConfig(pt);
    bool vmmchan_ok = (m_channels.size() ? true : false);
    for(int ichan = 0; ichan < (int)m_channels.size(); ichan++) {
        bool chan_ok = m_channels.at(ichan).ok;
        if(!chan_ok) {
            stringstream sx;
            sx << "Problem loading channel " << ichan;
            msg()(sx, "ConfigHandler::LoadConfig");
            vmmchan_ok = false;
            break;
        }
    }

    // load the VMM chip map
    m_vmmMap = LoadVMMMap(pt);
    if(m_vmmMap.ok) {
        m_vmmMap.ok = setVMMMap();
    }


    // load the trigger setup
    m_daqSettings = LoadDAQConfig(pt);
    bool daq_ok = m_daqSettings.ok;
    if(!daq_ok) {
        msg()("Problem loading DAQ settings", "ConfigHandler::LoadConfig");
    }

    // load Clocks settings
    m_fpgaClockSettings = LoadFPGAClockSettings(pt);
    bool clocks_ok = m_fpgaClockSettings.ok;
    if(!clocks_ok) {
        msg()("Problem loading FPGA clocks settings", "ConfigHandler::LoadConfig");
    }

    if(!comm_ok) cout << "comm_ok" << endl;
    if(!global_ok) cout << "global_ok" << endl;
    if(!daq_ok) cout << "daq_ok" << endl;
    if(!m_vmmMap.ok) cout << "m_vmmMap" << endl;
    if(!vmmchan_ok) cout << "vmmchan_ok" << endl;
    if(!clocks_ok) cout << "clocks_ok" << endl;

    if(!(comm_ok && global_ok && daq_ok && m_vmmMap.ok && vmmchan_ok && clocks_ok)) {
        msg()("Configuration loaded unsucessfully","ConfigHandler::LoadConfig");
        if(!comm_ok)        cout << "comm_ok" << endl;
        if(!global_ok)      cout << "global_ok" << endl;
        if(!daq_ok)         cout << "daq_ok" << endl;
        if(!m_vmmMap.ok)    cout << "m_vmmMap" << endl;
        if(!vmmchan_ok)     cout << "vmmchan_ok" << endl;
        if(!clocks_ok)      cout << "clocks_ok" << endl;
    }

    return (comm_ok && global_ok && daq_ok && m_vmmMap.ok && vmmchan_ok && clocks_ok);
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::WriteConfig(QString filename)
{
    using boost::format;
    using boost::property_tree::ptree;
    using namespace boost::property_tree::xml_parser;
    ptree outpt;

    ptree out_root;

    // ----------------------------------------------- //
    //  udp_setup
    // ----------------------------------------------- //
    ptree out_udpsetup;
    out_udpsetup.put("fec_port", commSettings().fec_port);
    out_udpsetup.put("daq_port", commSettings().daq_port);
    out_udpsetup.put("vmmasic_port", commSettings().vmmasic_port);
    out_udpsetup.put("vmmapp_port", commSettings().vmmapp_port);
    out_udpsetup.put("clocks_port", commSettings().clocks_port);

    // ----------------------------------------------- //
    // trigger_daq
    // ----------------------------------------------- //
    bool tdaq_ok = true;

    ptree out_tdaq;
    out_tdaq.put("trigger_latency", daqSettings().trigger_latency);
    out_tdaq.put("run_mode", daqSettings().run_mode);

    // ----------------------------------------------- //
    //  VMM map
    // ----------------------------------------------- //
    ptree out_vmm;
    stringstream ss;
    stringstream valss;
    int map_mask = vmmMap().mask; 
    ss.str("");
    ss << "map"; 
    valss.str("");
    valss << map_mask;
    out_vmm.put(ss.str(), valss.str());

    // ----------------------------------------------- //
    //  global settings
    // ----------------------------------------------- //
    bool global_ok = true;

    ptree out_global;
    out_global.put("sdp_dac", globalSettings().sdp_dac);
    out_global.put("sdt_dac", globalSettings().sdt_dac);
    out_global.put("st",
        GlobalSetting::all_peakTimes[globalSettings().st]);
    out_global.put("sm5", globalSettings().sm5);
    //EnabledOrDisabled(int ena_or_dis, std::string& value, std::string from, bool invert)      
    string val = "";
    bool write_ok = EnabledOrDisabled(globalSettings().scmx, val, "scmx", false);
    if(write_ok) {
        out_global.put("scmx", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sbmx, val, "sbmx", false);
    if(write_ok) {
        out_global.put("sbmx", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slg, val, "slg", true);
    if(write_ok) {
        out_global.put("slg", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sp, val, "sp", false);
    if(write_ok) {
        if(val=="ENABLED") { val = "POSITIVE"; }
        else if(val=="DISABLED") { val = "NEGATIVE"; }
        out_global.put("sp", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sng, val, "sng", false);
    if(write_ok) {
        out_global.put("sng", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sdrv, val, "sdrv", false);
    if(write_ok) {
        out_global.put("sdrv", val);
    } else { global_ok = false; }
    out_global.put("sg",
        GlobalSetting::all_gains[globalSettings().sg].toStdString());
    out_global.put("stc",
        GlobalSetting::all_TACslopes[globalSettings().stc]);
    write_ok = EnabledOrDisabled(globalSettings().sfm, val, "sfm", false);
    if(write_ok) {
        out_global.put("sfm", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().ssh, val, "ssh", false);
    if(write_ok) {
        out_global.put("ssh", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sdp, val, "sdp", false);
    if(write_ok) {
        out_global.put("sdp", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sbft, val, "sbft", false);
    if(write_ok) {
        out_global.put("sbft", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sbfp, val, "sbfp", false);
    if(write_ok) {
        out_global.put("sbfp", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sbfm, val, "sbfm", false);
    if(write_ok) {
        out_global.put("sbfm", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().s6b, val, "s6b", false);
    if(write_ok) {
        out_global.put("s6b", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().s10b, val, "s10b", false);
    if(write_ok) {
        out_global.put("s10b", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sttt, val, "sttt", false);
    if(write_ok) {
        out_global.put("sttt", val);
    } else { global_ok = false; }

    #warning check that correct direct timing mode is chosen
    int mode = 0;
    mode |= (globalSettings().stpp << 1);
    mode |= (globalSettings().stot);
    out_global.put("direct_timing_mode",
        GlobalSetting::all_directTimeModes[mode].toStdString());
    
    write_ok = EnabledOrDisabled(globalSettings().sfa, val, "sfa", false);
    if(write_ok) {
        out_global.put("sfa", val);
    } else { global_ok = false; }
    if(globalSettings().sfam == 0) val = "THRESHOLD";
    else if(globalSettings().sfam == 1) val = "PEAK";
    out_global.put("sfam", val);

    out_global.put("sc10b", globalSettings().sc10b);
    out_global.put("sc08b", globalSettings().sc8b);
    out_global.put("sc06b", globalSettings().sc6b);
    write_ok = EnabledOrDisabled(globalSettings().sdcks, val, "sdcks", false);
    if(write_ok) {
        out_global.put("sdcks", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sdcka, val, "sdcka", false);
    if(write_ok) {
        out_global.put("sdcka", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sdck6b, val, "sdck6b", false);
    if(write_ok) {
        out_global.put("sdck6b", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvs, val, "slvs", false);
    if(write_ok) {
        out_global.put("slvs", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().stlc, val, "stlc", false);
    if(write_ok) {
        out_global.put("stlc", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().stcr, val, "stcr", false);
    if(write_ok) {
        out_global.put("stcr", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().ssart, val, "ssart", false);
    if(write_ok) {
        out_global.put("ssart", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().srec, val, "srec", false);
    if(write_ok) {
        out_global.put("srec", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sfrst, val, "sfrst", false);
    if(write_ok) {
        out_global.put("sfrst", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().s32, val, "s32", false);
    if(write_ok) {
        out_global.put("s32", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sbip, val, "sbip", false);
    if(write_ok) {
        out_global.put("sbip", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().srat, val, "srat", false);
    if(write_ok) {
        out_global.put("srat", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvsbc, val, "slvsbc", false);
    if(write_ok) {
        out_global.put("slvsbc", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvsart, val, "slvsart", false);
    if(write_ok) {
        out_global.put("slvsart", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvstp, val, "slvstp", false);
    if(write_ok) {
        out_global.put("slvstp", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvstki, val, "slvstki", false);
    if(write_ok) {
        out_global.put("slvstki", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvstk, val, "slvstk", false);
    if(write_ok) {
        out_global.put("slvstk", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvsena, val, "slvsena", false);
    if(write_ok) {
        out_global.put("slvsena", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvsdt, val, "slvsdt", false);
    if(write_ok) {
        out_global.put("slvsdt", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().slvs6b, val, "slvs6b", false);
    if(write_ok) {
        out_global.put("slvs6b", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sL0ena, val, "sL0ena", false);
    if(write_ok) {
        out_global.put("sL0ena", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sL0enaV, val, "sL0enaV", false);
    if(write_ok) {
        out_global.put("sL0enaV", val);
    } else { global_ok = false; }
    out_global.put("slh", globalSettings().slh);
    out_global.put("slxh", globalSettings().slxh);
    out_global.put("stgc", globalSettings().stgc);

    out_global.put("l0offset", globalSettings().l0offset);
    out_global.put("offset", globalSettings().offset);
    out_global.put("rollover", globalSettings().rollover);
    out_global.put("window", globalSettings().window);
    out_global.put("truncate", globalSettings().truncate);
    out_global.put("nskip", globalSettings().nskip);
    write_ok = EnabledOrDisabled(globalSettings().sL0cktest, val, "sL0cktest", false);
    if(write_ok) {
        out_global.put("sL0cktest", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sL0ckinv, val, "sL0ckinv", false);
    if(write_ok) {
        out_global.put("sL0ckinv", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().sL0dckinv, val, "sL0dckinv", false);
    if(write_ok) {
        out_global.put("sL0dckinv", val);
    } else { global_ok = false; }
    write_ok = EnabledOrDisabled(globalSettings().nskipm, val, "nskipm", false);
    if(write_ok) {
        out_global.put("nskipm", val);
    } else { global_ok = false; }


    // ----------------------------------------------- //
    //  Clocks settings
    // ----------------------------------------------- //
    bool clocks_ok = true;

    ptree out_clocks;
    out_clocks.put("cktk_max", clocksSettings().cktk_max_number); 
    out_clocks.put("ckbc",
        FPGAClocks::all_CKBC[clocksSettings().ckbc_frequency].toStdString());
    out_clocks.put("cktp_max", clocksSettings().cktp_max_number);
    out_clocks.put("cktp_skew", clocksSettings().cktp_skew_steps);
    out_clocks.put("cktp_period", clocksSettings().cktp_period);
    out_clocks.put("cktp_width", clocksSettings().cktp_width);
   
    // ----------------------------------------------- //
    // stitch together the fields
    // ----------------------------------------------- //
    out_root.add_child("global_registers", out_global);


    // ----------------------------------------------- //
    //  individual VMM channels
    // ----------------------------------------------- //
    // get the "global" channel masks
    uint64_t channel_sc = 0;
    uint64_t channel_sl = 0;
    uint64_t channel_sth = 0;
    uint64_t channel_st = 0;
    uint64_t channel_sm = 0;
    uint64_t channel_smx = 0;

    bool channels_ok = true;

    for(int i = 0; i < 64; i++) {
        if((int)channelSettings(i).number != i) {
            stringstream sx;
            sx << "WARNING VMM Channel numbers out of sync! Expecting VMM channel "
               << i << " but at this index we have VMM " << channelSettings(i).number;
            msg()(sx, "ConfigHandler::WriteConfig"); sx.str("");
            sx << "WARNING Setting output config channel " << i << " to default values";
            msg()(sx, "ConfigHandler::WriteConfig"); sx.str("");
            channels_ok = false;
        }
        uint64_t tmp_sc = (channelSettings(i).sc==1 ? 1 : 0);
        uint64_t tmp_sl = (channelSettings(i).sl==1 ? 1 : 0);
        uint64_t tmp_sth= (channelSettings(i).sth==1? 1 : 0);
        uint64_t tmp_st = (channelSettings(i).st==1 ? 1 : 0);
        uint64_t tmp_sm = (channelSettings(i).sm==1 ? 1 : 0);
        uint64_t tmp_smx= (channelSettings(i).smx==1? 1 : 0);

        channel_sc |= (tmp_sc << i);
        channel_sl |= (tmp_sl << i);
        channel_sth |= (tmp_sth << i);
        channel_st |= (tmp_st << i);
        channel_sm |= (tmp_sm << i);
        channel_smx |= (tmp_smx << i);
    } // i

    QString hex_start = "0x";
    QString sc_str =  hex_start + QString::number(channel_sc, 16);
    QString sl_str =  hex_start + QString::number(channel_sl, 16);
    QString sth_str = hex_start + QString::number(channel_sth, 16);
    QString st_str =  hex_start + QString::number(channel_st, 16);
    QString sm_str =  hex_start + QString::number(channel_sm, 16);
    QString smx_str = hex_start + QString::number(channel_smx, 16);

    ptree out_channel_registers;

    ptree out_channel_masks;
    out_channel_masks.put("sc",  sc_str.toStdString());
    out_channel_masks.put("sl",  sl_str.toStdString());
    out_channel_masks.put("sth", sth_str.toStdString());
    out_channel_masks.put("st",  st_str.toStdString());
    out_channel_masks.put("sm",  sm_str.toStdString());
    out_channel_masks.put("smx", smx_str.toStdString());
    out_channel_registers.add_child("channel_register_mask", out_channel_masks);


    stringstream chnode;
    for(int i = 0; i < 64; i++) {
        ptree out_channel;
        
        if((int)channelSettings(i).number != i) {
            stringstream sx;
            sx << "WARNING VMM Channel numbers out of sync! Expecting VMM channel "
               << i << " but at this index we have VMM " << channelSettings(i).number;
            msg()(sx, "ConfigHandler::WriteConfig"); sx.str("");
            sx << "WARNING Setting output config channel " << i << " to default values";
            msg()(sx, "ConfigHandler::WriteConfig"); sx.str("");
            channels_ok = false;
        }

        ss.str("");
        ss << "channel_" << format("%02i") % i; 

        // trim
        //chnode << ss.str() << ".sd";        
        chnode << "sd";
        out_channel.put(chnode.str(), channelSettings(i).sd);
        chnode.str("");

        // sz010b
        //chnode << ss.str() << ".sz010b";
        chnode << "sz010b";
        out_channel.put(chnode.str(), channelSettings(i).sz10b);
        chnode.str("");

        // sz08b
        //chnode << ss.str() << ".sz08b";
        chnode << "sz08b";
        out_channel.put(chnode.str(), channelSettings(i).sz8b);
        chnode.str("");

        // sz06b
        //chnode << ss.str() << ".sz06b";
        chnode << "sz06b";
        out_channel.put(chnode.str(), channelSettings(i).sz6b);
        chnode.str("");

        out_channel_registers.add_child(ss.str(), out_channel);
        //out_root.add_child("channel_registers", out_channel);
    } // i

    out_root.add_child("channel_registers", out_channel_registers);


    // add the rest of the nodes
    out_root.add_child("udp_setup", out_udpsetup);
    out_root.add_child("trigger_daq", out_tdaq);
    out_root.add_child("fpga_clocks", out_clocks);
    out_root.add_child("vmm_map", out_vmm);

    // put everything under a global node
    outpt.add_child("configuration", out_root);


    stringstream sx;
    try {
        #if BOOST_VERSION >= 105800
        write_xml(filename.toStdString(), outpt, std::locale(),
                boost::property_tree::xml_writer_make_settings<std::string>('\t',1));
        #else
        write_xml(filename.toStdString(), outpt, std::locale(),
                boost::property_tree::xml_writer_make_settings<char>('\t',1));
        #endif

        sx.str("");
        sx << "Configuration written successfully to file: \n";
        sx << " > " << filename.toStdString();
        msg()(sx,"ConfigHandler::WriteConfig");
    }
    catch(std::exception& e) {
        sx.str("");
        sx << "ERROR Unable to write output configuration XML file: " << e.what();
        msg()(sx,"ConfigHandler::WriteConfig");
        return;
    }

}
//// ------------------------------------------------------------------------ //
CommInfo ConfigHandler::LoadCommInfo(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;

    CommInfo comm;

    try
    {
        for(const auto& conf : pt.get_child("configuration")) {
            if(!(conf.first == "udp_setup"))
                continue;

            if(conf.first=="udp_setup") {
                // FEC port
                comm.fec_port = conf.second.get<int>("fec_port");
                // DAQ port
                comm.daq_port = conf.second.get<int>("daq_port");
                // VMMASIC port
                comm.vmmasic_port = conf.second.get<int>("vmmasic_port");
                // VMMAPP port
                comm.vmmapp_port = conf.second.get<int>("vmmapp_port");
                // clocks port
                comm.clocks_port = conf.second.get<int>("clocks_port");

                comm.ok = true;
            } // udp_setup
        }
    }
    catch(std::exception &e)
    {
        stringstream sx;
        sx << "!! -------------------------------------------- !! \n"
           << "  ERROR CONF: " << e.what() << "\n"
           << "!! -------------------------------------------- !!"; 
        msg()(sx,"ConfigHandler::LoadCommInfo"); 
        comm.ok = false;
    }

    return comm;
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::LoadCommInfo(const CommInfo& info)
{
    m_commSettings = info;
    if(dbg())
        m_commSettings.print();
}
//// ------------------------------------------------------------------------ //
GlobalSetting ConfigHandler::LoadGlobalSettings(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;

    stringstream sx;

    GlobalSetting g;

    bool outok = true;

    try{
        for(const auto& conf : pt.get_child("configuration")) {
            if(!(conf.first == "global_registers")) continue;

            // pulser DAC
            g.sdp_dac = conf.second.get<int>("sdp_dac");

            // threshold DAC
            g.sdt_dac = conf.second.get<int>("sdt_dac");

            // integration time
            int peak_time = conf.second.get<int>("st");
            if(GlobalSetting::all_peakTimes.indexOf(peak_time)>=0) {
                g.st = GlobalSetting::all_peakTimes.indexOf(peak_time);
            }
            else {
                sx << "ERROR Invalid peak time [st] requested in config file: " << peak_time << "\n";
                sx << "ERROR Peak time must be one of: ";
                for(auto& i : GlobalSetting::all_peakTimes) sx << " " << i << " ";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str(""); 
                outok = false;
            }

            // sm5 (channel monitor/common mode)
            int ch = conf.second.get<int>("sm5");
            if(ch>=0 && ch<64) {
                g.sm5 = ch;
            }
            else {
                sx << "ERROR Invalid channel for sm5: " << ch << "\n";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // scmx
            int idx = 0;
            bool ok = isEnabledOrDisabled(conf.second.get<string>("scmx"), idx, "scmx", false);
            if(ok) {
                g.scmx = idx;
            }
            else {
                outok = false;
            }

            // sbmx
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sbmx"), idx, "sbmx", false);
            if(ok) {
                g.sbmx = idx;
            }
            else {
                outok = false;
            }

            // slg
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slg"), idx, "slg", true);
            if(ok) {
                g.slg = idx;
            }
            else {
                outok = false;
            }

            // sp
            std::string polarity = conf.second.get<string>("sp");
            if(polarity=="POSITIVE") {
                g.sp = 1;
            }
            else if(polarity=="NEGATIVE") {
                g.sp = 0;
            }
            else {
                sx << "ERROR Invalid channel polarity [sp] provided by config: " << polarity;
                sx << "\nERROR Expect either 'POSITIVE' or 'NEGATIVE'";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // sng
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sng"), idx, "sng", false);
            if(ok) {
                g.sng = idx;
            }
            else {
                outok = false;
            }

            // sdrv
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sdrv"), idx, "sdrv", false);
            if(ok) {
                g.sdrv = idx;
            }
            else {
                outok = false;
            }

            // sg
            string gain = conf.second.get<string>("sg");
            if(GlobalSetting::all_gains.indexOf(QString::fromStdString(gain))>=0) {
                g.sg = GlobalSetting::all_gains.indexOf(QString::fromStdString(gain));
            }
            else {
                sx << "ERROR Invalid gain [sg] value requested in config file: " << gain << "\n";
                sx << "ERROR Gain must be one of: ";
                for(auto& i : GlobalSetting::all_gains) sx << " " << i.toStdString() << " ";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // stc
            int tac = conf.second.get<int>("stc");
            if(GlobalSetting::all_TACslopes.indexOf(tac)>=0) {
                g.stc = GlobalSetting::all_TACslopes.indexOf(tac);
            }
            else {
                sx << "ERROR Invalid TAC slope [stc] requested in config file: " << tac << "\n";
                sx << "ERROR TAC slope must be one of: ";
                for(auto& i : GlobalSetting::all_TACslopes) sx << " " << i << " ";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
           } 

            // sfm
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sfm"), idx, "sfm", false);
            if(ok) {
                g.sfm = idx;
            }
            else {
                outok = false;
            }

            // ssh
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("ssh"), idx, "ssh", false);
            if(ok) {
                g.ssh = idx;
            }
            else {
                outok = false;
            }

            // sdp
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sdp"), idx, "sdp", false);
            if(ok) {
                g.sdp = idx;
            }
            else {
                outok = false;
            }

            // sbft
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sbft"), idx, "sbft", false);
            if(ok) {
                g.sbft = idx;
            }
            else {
                outok = false;
            }

            // sbfp
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sbfp"), idx, "sbfp", false);
            if(ok) {
                g.sbfp = idx;
            }
            else {
                outok = false;
            }

            // sbfm
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sbfm"), idx, "sbfm", false);
            if(ok) {
                g.sbfm = idx;
            }
            else {
                outok = false;
            }

            // s6b
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("s6b"), idx, "s6b", false);
            if(ok) {
                g.s6b = idx;
            }
            else {
                outok = false;
            }

            // s10b
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("s10b"), idx, "s10b", false);
            if(ok) {
                g.s10b = idx;
            }
            else {
                outok = false;
            }

            // sttt
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sttt"), idx, "sttt", false);
            if(ok) {
                g.sttt = idx;
            }
            else {
                outok = false;
            }

            // direct_timing_mode
            string timing_mode = conf.second.get<string>("direct_timing_mode");
            if(GlobalSetting::all_directTimeModes.indexOf(QString::fromStdString(timing_mode))>=0) {
                QString mode = QString::fromStdString(timing_mode);
                int time_index = GlobalSetting::all_directTimeModes.indexOf(mode);
                QString tmp = QString("%1").arg(time_index, 2, 2, QChar('0'));
                g.stpp = static_cast<int>(tmp.at(0).digitValue());
                g.stot = static_cast<int>(tmp.at(1).digitValue());
            }
            else {
                sx << "ERROR Invalid direct time mode requested in config file: " << timing_mode << "\n"; 
                sx << "ERROR Direct time mode must be one of: ";
                for(auto& i : GlobalSetting::all_directTimeModes) sx << " " << i.toStdString() << " ";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // sfa
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sfa"), idx, "sfa", false);
            if(ok) {
                g.sfa = idx;
            }
            else {
                outok = false;
            }

            // sfam
            string art_detect_mode = conf.second.get<string>("sfam");
            if(art_detect_mode=="THRESHOLD") {
                g.sfam = 0;
            }
            else if(art_detect_mode=="PEAK") {
                g.sfam = 1;
            }
            else {
                sx << "ERROR Invalid ART detect mode [sfam] requested in config file: " << art_detect_mode << "\n";
                sx << "ERROR ART detect mode must be either 'THRESHOLD' or 'PEAK'";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // sc10b
            int conv = conf.second.get<int>("sc10b");
            if(conv==0 || conv==1 || conv==2 || conv==3) {
                g.sc10b = conv;
            }
            else {
                sx << "ERROR Invalid sc10b value requested in config file: " << conv << "\n";
                sx << "ERROR Value can only take 0 or 1";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }
            
            // sc08b
            conv = conf.second.get<int>("sc08b");
            if(conv==0 || conv==1 || conv==2 || conv==3) {
                g.sc8b = conv;
            }
            else {
                sx << "ERROR Invalid sc08b value requested in config file: " << conv << "\n";
                sx << "ERROR Value can only take 0 or 1";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }
            
            // sc06b
            conv = conf.second.get<int>("sc06b");
            if(conv==0 || conv==1 || conv==2) {
                g.sc6b = conv;
            }
            else {
                sx << "ERROR Invalid sc06b value requested in config file: " << conv << "\n";
                sx << "ERROR Value can only take 0, 1, or 2";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // sdcks
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sdcks"), idx, "sdcks", false);
            if(ok) {
                g.sdcks = idx;
            }
            else {
                outok = false;
            }

            // sdcka
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sdcka"), idx, "sdcka", false);
            if(ok) {
                g.sdcka = idx;
            }
            else {
                outok = false;
            }

            // sdck6b
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sdck6b"), idx, "sdck6b", false);
            if(ok) {
                g.sdck6b = idx;
            }
            else {
                outok = false;
            }

            // slvs
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvs"), idx, "slvs", false);
            if(ok) {
                g.slvs = idx;
            }
            else {
                outok = false;
            }

            // stlc
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("stlc"), idx, "stlc", false);
            if(ok) {
                g.stlc = idx;
            }
            else {
                outok = false;
            }

            // stcr
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("stcr"), idx, "stcr", false);
            if(ok) {
                g.stcr = idx;
            }
            else {
                outok = false;
            }

            // ssart
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("ssart"), idx, "ssart", false);
            if(ok) {
                g.ssart = idx;
            }
            else {
                outok = false;
            }

            // srec
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("srec"), idx, "srec", false);
            if(ok) {
                g.srec = idx;
            }
            else {
                outok = false;
            }

            // sfrst
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sfrst"), idx, "sfrst", false);
            if(ok) {
                g.sfrst = idx;
            }
            else {
                outok = false;
            }

            // s32
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("s32"), idx, "s32", false);
            if(ok) {
                g.s32 = idx;
            }
            else {
                outok = false;
            }

            // sbip
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sbip"), idx, "sbip", false);
            if(ok) {
                g.sbip = idx;
            }
            else {
                outok = false;
            }

            // srat
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("srat"), idx, "srat", false);
            if(ok) {
                g.srat = idx;
            }
            else {
                outok = false;
            }

            // slvsbc
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvsbc"), idx, "slvsbc", false);
            if(ok) {
                g.slvsbc = idx;
            }
            else {
                outok = false;
            }

            // slvsart
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvsart"), idx, "slvsart", false);
            if(ok) {
                g.slvsart = idx;
            }
            else {
                outok = false;
            }

            // slvstp
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvstp"), idx, "slvstp", false);
            if(ok) {
                g.slvstp = idx;
            }
            else {
                outok = false;
            }

            // slvstki
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvstki"), idx, "slvstki", false);
            if(ok) {
                g.slvstki = idx;
            }
            else {
                outok = false;
            }

            // slvstk
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvstk"), idx, "slvstk", false);
            if(ok) {
                g.slvstk = idx;
            }
            else {
                outok = false;
            }

            // slvsena
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvsena"), idx, "slvsena", false);
            if(ok) {
                g.slvsena = idx;
            }
            else {
                outok = false;
            }

            // slvsdt
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvsdt"), idx, "slvsdt", false);
            if(ok) {
                g.slvsdt = idx;
            }
            else {
                outok = false;
            }

            // slvs6b
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("slvs6b"), idx, "slvs6b", false);
            if(ok) {
                g.slvs6b = idx;
            }
            else {
                outok = false;
            }

            // sL0ena
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sL0ena"), idx, "sL0ena", false);
            if(ok) {
                g.sL0ena = idx;
            }
            else {
                outok = false;
            }

            // sL0enaV
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sL0enaV"), idx, "sL0enaV", false);
            if(ok) {
                g.sL0enaV = idx;
            }
            else {
                outok = false;
            }

            // l0offset
            int offset = conf.second.get<int>("l0offset");
            if(offset>=0 && offset<4096) {
                g.l0offset = offset;
            }
            else {
                sx << "ERROR Invalid [l0offset] value requested in config file: " << offset << "\n";
                sx << "ERROR [l0offset] must take values [0,4095] inclusively";
                msg()(sx, "ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // offset
            offset = conf.second.get<int>("offset");
            if(offset>=0 && offset<4096) {
                g.offset = offset;
            }
            else {
                sx << "ERROR Invalid [offset] value requested in config file: " << offset << "\n";
                sx << "ERROR [offset] must take values [0,4095] inclusively";
                msg()(sx, "ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // rollover
            int rollover = conf.second.get<int>("rollover");
            if(rollover>=0 && rollover<4096) {
                g.rollover = rollover;
            }
            else {
                sx << "ERROR Invalid [rollover] value requested in config file: " << offset << "\n";
                sx << "ERROR [rollover] must take values [0,4095] inclusively";
                msg()(sx, "ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // window
            int window = conf.second.get<int>("window");
            if(window>=0 && window<8) {
                g.window = window;
            }
            else {
                sx << "ERROR Invalid trigger window size [window] requested in "
                        << "config file: " << window << "\n";
                sx << "ERROR Window size [window] must take values [0,7] inclusively";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // truncate
            int truncate = conf.second.get<int>("truncate");
            if(truncate>=0 && truncate<64) {
                g.truncate = truncate;
            }
            else {
                sx << "ERROR Max hits per L0 [truncate] requested in "
                        << "config file is invalid: " << truncate << "\n";
                sx << "ERROR [truncate] must take values [0,63] inclusively";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // nskip
            int nskip = conf.second.get<int>("nskip");
            if(nskip>=0 && nskip<128) {
                g.nskip = nskip;
            }
            else {
                sx << "ERROR Number of L0 to skip on o'flow [nskip] requested in "
                    << "config file is invalid: " << nskip << "\n";
                sx << "ERROR [nskip] must take values [0,127] inclusively";
                msg()(sx,"ConfigHandler::LoadGlobalSettings"); sx.str("");
                outok = false;
            }

            // sL0cktest
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sL0cktest"), idx, "sL0cktest", false);
            if(ok) {
                g.sL0cktest = idx;
            }
            else {
                outok = false;
            }

            // sL0ckinv
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sL0ckinv"), idx, "sL0ckinv", false);
            if(ok) {
                g.sL0ckinv = idx;
            }
            else {
                outok = false;
            }

            // sL0dckinv
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("sL0dckinv"), idx, "sL0dckinv", false);
            if(ok) {
                g.sL0dckinv = idx;
            }
            else {
                outok = false;
            }

            // nskipm
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("nskipm"), idx, "nskipm", false);
            if(ok) {
                g.nskipm = idx;
            }
            else {
                outok = false;
            }
            
        } // conf

    } // try
    catch(std::exception &e){
        stringstream sx;
        sx << "Exception in loading global registers: " << e.what();
        msg()(sx,"ConfigHandler::LoadGlobalSettings");
        outok = false;
    } // catch

    g.ok = outok;

    return g;
/*

    try
    {
        for(const auto& conf : pt.get_child("configuration")) {
            if( !(conf.first == "global_settings") ) continue;

            // ------------------------------------------------------------- //
            // global channel polarity
            string pol = conf.second.get<string>("ch_polarity"); 
            if(GlobalSetting::all_polarities.indexOf(QString::fromStdString(pol))>=0)
                g.polarity = GlobalSetting::all_polarities.indexOf(QString::fromStdString(pol));
            else {
                stringstream sx;
                sx << "ERROR ch_polarity value must be one of: [";
                for(auto& i : GlobalSetting::all_polarities) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << pol;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // global channel leakage current ena/dis
            string leak = conf.second.get<string>("ch_leakage_current");
            g.leakage_current = (bool)isOn(leak, "ch_leakage_current");
       
            // ------------------------------------------------------------- //
            // analog tristates
            string tristates = conf.second.get<string>("analog_tristates");
            g.analog_tristates = isOn(tristates, "analog_tristates");

            // ------------------------------------------------------------- //
            // double leakage
            string dleak = conf.second.get<string>("double_leakage");
            g.double_leakage = isOn(dleak, "double_leakage");

            // ------------------------------------------------------------- //
            // gain
            string gain = conf.second.get<string>("gain");
            if(GlobalSetting::all_gains.indexOf(QString::fromStdString(gain))>=0)
                g.gain = GlobalSetting::all_gains.indexOf(QString::fromStdString(gain));
            else {
                stringstream sx;
                sx << "ERROR gain value must be one of: [";
                for(auto& i : GlobalSetting::all_gains)  sx << " " << i.toStdString() << " ";
                sx << "] mV/fC\nYou have provided: " << gain;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }
            
            // ------------------------------------------------------------- //
            // peak_time
            int ptime = conf.second.get<int>("peak_time");
            if(GlobalSetting::all_peakTimes.indexOf(ptime)>=0)
                g.peak_time = GlobalSetting::all_peakTimes.indexOf(ptime);
            else {
                stringstream sx;
                sx << "ERROR peak_time value must be one of: [";
                for(auto& i : GlobalSetting::all_peakTimes) sx << " " << i << " ";
                sx << "] ns\nYou have provided: " << ptime;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // neighbor trigger
            string neighbor = conf.second.get<string>("neighbor_trigger");
            g.neighbor_trigger = isOn(neighbor, "neighbor_trigger");

            // ------------------------------------------------------------- //
            // TAC slope
            int tacslope = conf.second.get<int>("TAC_slop_adj");
            if(GlobalSetting::all_TACslopes.indexOf(tacslope)>=0)
                g.tac_slope = GlobalSetting::all_TACslopes.indexOf(tacslope);
            else {
                stringstream sx;
                sx << "ERROR TAC_slope_adj value must be one of: [";
                for(auto& i : GlobalSetting::all_TACslopes) sx << " " << i << " ";
                sx << "] ns\nYou have provided: " << tacslope;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // disable at peak
            string disable = conf.second.get<string>("disable_at_peak");
            g.disable_at_peak = isOn(disable, "disable_at_peak");

            // ------------------------------------------------------------- //
            // ART
            string art = conf.second.get<string>("ART");
            g.art = isOn(art, "ART");

            // ------------------------------------------------------------- //
            // ART mode
            string artmode = conf.second.get<string>("ART_mode");
            if( GlobalSetting::all_ARTmodes.indexOf(QString::fromStdString(artmode))>=0) {
                QString mode = QString::fromStdString(artmode);
                g.art_mode = GlobalSetting::all_ARTmodes.indexOf(mode);
            }
            else {
                stringstream sx;
                sx << "ERROR ART_mode value must be one of: [";
                for(auto& i : GlobalSetting::all_ARTmodes) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << artmode;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // dual clock ART
            string dcart = conf.second.get<string>("dual_clock_ART");
            g.dual_clock_art = isOn(dcart, "dual_clock_ART");

            // ------------------------------------------------------------- //
            // out buffer mo
            string obmo = conf.second.get<string>("out_buffer_mo");
            g.out_buffer_mo = isOn(obmo, "out_buffer_mo");

            // ------------------------------------------------------------- //
            // out buffer pdo
            string obpdo = conf.second.get<string>("out_buffer_pdo");
            g.out_buffer_pdo = isOn(obpdo, "out_buffer_pdo");

            // ------------------------------------------------------------- //
            // out buffer tdo
            string obtdo = conf.second.get<string>("out_buffer_tdo");
            g.out_buffer_tdo = isOn(obtdo, "out_buffer_tdo");

            // ------------------------------------------------------------- //
            // channel for monitoring
            g.channel_monitor = conf.second.get<int>("channel_monitoring");

            // ------------------------------------------------------------- //
            // monitoring control
            string mcontrol = conf.second.get<string>("monitoring_control");
            g.monitoring_control = isOn(mcontrol, "monitoring_control");

            // ------------------------------------------------------------- //
            // monitor pdo out
            string mpdo = conf.second.get<string>("monitor_pdo_out");
            g.monitor_pdo_out = isOn(mpdo, "monitor_pdo_out");

            // ------------------------------------------------------------- //
            // ADCs
            string adcs = conf.second.get<string>("ADCs");
            g.adcs = isOn(adcs, "ADCs");

            // ------------------------------------------------------------- //
            // sub hysteresis
            string subhyst = conf.second.get<string>("sub_hyst_discr");
            g.sub_hysteresis = isOn(subhyst, "sub_hyst_discr");

            // ------------------------------------------------------------- //
            // direct time
            string direct = conf.second.get<string>("direct_time");
            g.direct_time = isOn(direct, "direct_time");

            // ------------------------------------------------------------- //
            // direct time mode
            string dtmode = conf.second.get<string>("direct_time_mode");
            if(GlobalSetting::all_directTimeModes.indexOf(QString::fromStdString(dtmode))>=0)
            {
                QString mode = QString::fromStdString(dtmode);
                int dtindex = GlobalSetting::all_directTimeModes.indexOf(mode);
                QString tmp = QString("%1").arg(dtindex, 2, 2, QChar('0'));
                g.direct_time_mode  = dtindex;
                g.direct_time_mode0 = tmp.at(0).digitValue();
                g.direct_time_mode1 = tmp.at(1).digitValue();
            }
            else {
                stringstream sx;
                sx << "ERROR direct_time_mode value must be one of: [";
                for(auto& i : GlobalSetting::all_directTimeModes) sx <<" "<< i.toStdString()<<" ";
                sx << "]\nYou have provided: " << dtmode;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            } 

            // ------------------------------------------------------------- //
            // conv mode 8bit
            string cmode8bit = conf.second.get<string>("conv_mode_8bit");
            g.conv_mode_8bit = isOn(cmode8bit, "conv_mode_8bit");

            // ------------------------------------------------------------- //
            // enable 6bit
            string ena6 = conf.second.get<string>("enable_6bit");
            g.enable_6bit = isOn(ena6, "enable_6bit");

            // ------------------------------------------------------------- //
            // ADC 10bit
            string adc10 = conf.second.get<string>("ADC_10bit");
            if(GlobalSetting::all_ADC10bits.indexOf(QString::fromStdString(adc10))>=0)
                g.adc_10bit = GlobalSetting::all_ADC10bits.indexOf(QString::fromStdString(adc10));
            else {
                stringstream sx;
                sx << "ERROR ADC_10bit value must be one of: [";
                for(auto& i : GlobalSetting::all_ADC10bits) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << adc10;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // ADC 8bit
            string adc8 = conf.second.get<string>("ADC_8bit");
            if(GlobalSetting::all_ADC8bits.indexOf(QString::fromStdString(adc8))>=0)
                g.adc_8bit = GlobalSetting::all_ADC8bits.indexOf(QString::fromStdString(adc8));
            else {
                stringstream sx;
                sx << "ERROR ADC_8bit value must be one of: [";
                for(auto& i : GlobalSetting::all_ADC8bits) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << adc8;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // ADC 6bit
            string adc6 = conf.second.get<string>("ADC_6bit");
            if(GlobalSetting::all_ADC6bits.indexOf(QString::fromStdString(adc6))>=0)
                g.adc_6bit = GlobalSetting::all_ADC6bits.indexOf(QString::fromStdString(adc6));
            else {
                stringstream sx;
                sx << "ERROR ADC_6bit value must be one of: [";
                for(auto& i : GlobalSetting::all_ADC6bits) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << adc6;
                msg()(sx,"ConfigHandler::LoadGlobalSettings");
                outok = false;
            }

            // ------------------------------------------------------------- //
            // dual clock data
            string dcdata = conf.second.get<string>("dual_clock_data");
            g.dual_clock_data = isOn(dcdata, "dual_clock_data");

            // ------------------------------------------------------------- //
            // dual clock 6bit
            string dc6 = conf.second.get<string>("dual_clock_6bit");
            g.dual_clock_6bit = isOn(dc6, "dual_clock_6bit");

            // ------------------------------------------------------------- //
            // threshold DAC
            g.threshold_dac = conf.second.get<int>("threshold_DAC");
            
            // ------------------------------------------------------------- //
            // test pulse DAC
            g.test_pulse_dac = conf.second.get<int>("test_pulse_DAC");
        }
    }
    catch(std::exception &e)
    {
        stringstream sx;
        sx << "!! -------------------------------------------- !! \n"
           << "  ERROR GLOBAL: " << e.what() << "\n"
           << "!! -------------------------------------------- !!"; 
        msg()(sx,"ConfigHandler::LoadCommInfo"); 
        outok = false;
    }
    g.ok = outok;

    return g;

*/
}
//// ------------------------------------------------------------------------ //
TriggerDAQ ConfigHandler::LoadDAQConfig(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;

    TriggerDAQ daq;

    bool outok = true;

    stringstream sx;

    try
    {
        for(const auto& conf : pt.get_child("configuration")) {
            if( !(conf.first == "trigger_daq") ) continue;

            // trigger_latency
            daq.trigger_latency = conf.second.get<int>("trigger_latency");

            // run_mode
            daq.run_mode = conf.second.get<string>("run_mode");
        }
    }
    catch(std::exception &e)
    {
        stringstream sx;
        sx << "!! --------------------------------- !!\n"
           << "    ERROR DAQ: " << e.what() << "\n"
           << "!! --------------------------------- !!";
        msg()(sx, "ConfigHandler::LoadDAQConfig"); 
        outok = false;
    }

    daq.ok = outok;

    return daq;
}
//// ------------------------------------------------------------------------ //
VMMMap ConfigHandler::LoadVMMMap(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;
    using boost::format;
    stringstream sx;

    // clear
    m_vmmMap.mask = 0;
    VMMMap outmap;

    try{
        for(const auto& conf : pt.get_child("configuration")) {
            if(!(conf.first == "vmm_map")) continue;
            int map = conf.second.get<int>("map");
            if(map>=0 && map<256) {
                outmap.mask = map;
                outmap.ok = true;
            }
            else {
                sx << "ERROR Invalid VMM mask requested in config file: " << map << "\n";
                sx << "ERROR Expecting 8 bit value, [0,255] inclusive";
                msg()(sx,"ConfigHandler::LoadVMMMap"); sx.str("");
                outmap.ok = false;
            }
        } // conf
    } // try
    catch(std::exception& e) {
        sx << "ERROR Unable to define VMMMap, exception thrown: " << e.what();
        msg()(sx,"ConfigHandler::LoadVMMMap"); sx.str("");
        outmap.ok = false;
    }

    return outmap;
}
//// ------------------------------------------------------------------------ //
/*
std::vector<VMMMap> ConfigHandler::LoadHDMIChannels(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;
    using boost::format;

    // clear the channelmap vector
    m_channelmap.clear();
    std::vector<VMMMap> outmap;

    stringstream ss;
    stringstream val;
    try
    {
        for(const auto& conf : pt.get_child("configuration")) {
            if( !(conf.first == "channel_map") ) continue;
            for(int iHDMI = 1; iHDMI < 9; iHDMI++) {
                ss.str("");
                ss << "hdmi_" << format("%1i") % iHDMI;
                val.str("");

                // number
                VMMMap chMap;
                chMap.hdmi_no = iHDMI;

                // HDMI switch
                val << ss.str() << ".switch";
                chMap.on = isOn(conf.second.get<string>(val.str()));
                // vmm0 on/off
                val.str("");
                val << ss.str() << ".first";
                chMap.first = bool(conf.second.get<int>(val.str()));
                // vmm1 on/off
                val.str("");
                val << ss.str() << ".second";
                chMap.second = bool(conf.second.get<int>(val.str()));
                // art on/off
                val.str("");
                val << ss.str() << ".art";
                chMap.second = bool(conf.second.get<int>(val.str()));

                if(dbg())
                    chMap.print();
                chMap.ok = true;
                outmap.push_back(chMap);

            } // iHDMI
        }
    }
    catch(std::exception &e)
    {
        stringstream sx;
        sx << "!! --------------------------------- !!\n"
           << "ERROR HDMI: " << e.what() << "\n"
           << "!! --------------------------------- !!"; 
        msg()(sx, "ConfigHandler::LoadHDMIChannels");
        for(int ichan = 0; ichan < outmap.size(); ichan++) {
            outmap.at(ichan).ok = false;
        }
    }
    return outmap;
}
*/
//// ------------------------------------------------------------------------ //
bool ConfigHandler::setVMMMap()
{
    bool outok = true;
    bool ok;
    QString chMapString = "0000000000000000";
    chMapString = QString("%1").arg(m_vmmMap.mask, 16, 2, QChar('0'));
    m_vmmMask = (quint16)chMapString.toInt(&ok,2);
    return outok;
}
/*
bool ConfigHandler::setHDMIChannelMap()
{
    bool outok = true;

    stringstream sx;
    if(m_channelmap.size()==0) {
        sx.str("");
        sx << "ERROR Channel map vector is empty!";
        msg()(sx, "ConfigHandler::setHDMIChannelMap");
        outok = false;
    }
    if(m_channelmap.size()!=8) {
        sx.str("");
        sx << "ERROR Channel map vector must have 8 entries.\n"
           << "      Current vector has " << m_channelmap.size() << " entries.";
        msg()(sx, "ConfigHandler::setHDMIChannelMap");
        outok = false;
    }
    ///////////////////////////////////////////////////////////////////
    // build the channel (VMMID) map
    ///////////////////////////////////////////////////////////////////
    QString chMapString = "";
    bool ok;
    //addmmfe8
    if(!mmfe8()) {
        chMapString = "000000000000000000000000"; //24bit (16 vmm + 8 art)
        for(int i = 0; i < (int)m_channelmap.size(); ++i) {
            QString first, second, art;
            first  = m_channelmap[i].first ? "1" : "0";
            second = m_channelmap[i].second ? "1" : "0";
            art    = m_channelmap[i].art ? "1" : "0";
            if(m_channelmap[i].on) {
                chMapString.replace(23-2*i, 1, first);
                chMapString.replace(22-2*i, 1, second);
                chMapString.replace(7-i, 1, art);

//                chMapString.replace(15-2*i, 1, first);
//                chMapString.replace(14-2*i, 1, second);
            }
        } // i
        m_vmmMap =    (quint16)chMapString.toInt(&ok,2);
        m_vmmMapART = (quint32)chMapString.toInt(&ok,2); 
    } // mini2
    ///////////////////////////////////////////////////////////////////
    // build the channel (VMMID) map
    ///////////////////////////////////////////////////////////////////
    else if(mmfe8()) {
        chMapString = "";
        for(int i = 0; i < (int)m_channelmap.size(); ++i) {
            if(m_channelmap[i].on) {
                chMapString.append(QString::number(i+1));
                break;
            }
        } // i
        m_vmmMap    = (quint16)chMapString.toInt(&ok,10);
        m_vmmMapART = (quint32)0;
    }

    return outok;
}
*/
//// ------------------------------------------------------------------------ //
std::vector<Channel> ConfigHandler::LoadVMMChannelConfig(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;
    using boost::format;

    // clear the current channels
    m_channels.clear();
    std::vector<Channel> channels;
    stringstream sx;
    stringstream ss;
    stringstream where;

    uint64_t channel_sc = 0;
    uint64_t channel_sl = 0;
    uint64_t channel_sth = 0;
    uint64_t channel_st = 0;
    uint64_t channel_sm = 0;
    uint64_t channel_smx = 0;

    int channel_reg = true;

    try {
        for(const auto& conf : pt.get_child("configuration.channel_registers")) {
            if(!(conf.first == "channel_register_mask")) continue;

            // sc
            string sc_ = conf.second.get<string>("sc");
            QString sc_hex = QString::fromStdString(sc_); 
            bool ok;
            uint64_t sc = sc_hex.toULong(&ok,16);
            if(sc<=0xffffffffffffffff) {
                channel_sc = sc;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [sc] in requested configuration file: " << sc << "\n";
                sx << "ERROR [sc] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }

            // sl
            string sl_ = conf.second.get<string>("sl");
            QString sl_hex = QString::fromStdString(sl_);
            uint64_t sl = sl_hex.toULong(&ok,16);
            if(sl<=0xffffffffffffffff) {
                channel_sl = sl;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [sl] in requested configuration file: " << sc << "\n";
                sx << "ERROR [sl] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }

            // sth
            string sth_ = conf.second.get<string>("sth");
            QString sth_hex = QString::fromStdString(sth_);
            uint64_t sth = sth_hex.toULong(&ok,16);
            if(sth<=0xffffffffffffffff) {
                channel_sth = sth;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [sth] in requested configuration file: " << sc << "\n";
                sx << "ERROR [sth] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }

            // st
            string st_ = conf.second.get<string>("st");
            QString st_hex = QString::fromStdString(st_);
            uint64_t st = st_hex.toULong(&ok,16);
            if(st<=0xffffffffffffffff) {
                channel_st = st;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [st] in requested configuration file: " << sc << "\n";
                sx << "ERROR [st] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }

            // sm
            string sm_ = conf.second.get<string>("sm");
            QString sm_hex = QString::fromStdString(sm_);
            uint64_t sm = sm_hex.toULong(&ok,16);
            if(sm<=0xffffffffffffffff) {
                channel_sm = sm;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [sm] in requested configuration file: " << sc << "\n";
                sx << "ERROR [sm] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }

            // smx
            string smx_ = conf.second.get<string>("smx");
            QString smx_hex = QString::fromStdString(smx_);
            uint64_t smx = smx_hex.toULong(&ok,16);
            if(smx<=0xffffffffffffffff) {
                channel_smx = smx;
            }
            else {
                sx.str("");
                sx << "ERROR Invalid [smx] in requested configuration file: " << sc << "\n";
                sx << "ERROR [smx] takes an 8-bit value, [0,255] inclusively";
                msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                channel_reg = false;
            }
                
        } // conf

    } // try
    catch(std::exception& e) {
        sx.str("");
        sx << "ERROR Loading VMM registers from requested config file, exception thrown: "
            << e.what();
        msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
        for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
            channels.at(ichan).ok = false;
        }
    } // catch


    try {
        for(int iChan = 0; iChan < 64; iChan++) {
            bool channel_ok = true;
            ss.str(""); 
            ss << "channel_" << format("%02i") % iChan;
            for(const auto& conf : pt.get_child("configuration.channel_registers")) {
                size_t find = conf.first.find(ss.str());
                if(find==string::npos) continue;
                where.str("");
                where << "(ch. " << format("%02i") % iChan;
                string value = "";

                Channel chan;
                chan.number = iChan;


                // sd
                int trim = conf.second.get<int>("sd");
                if(trim>=0 && trim<32) {
                    chan.sd = trim;
                }
                else {
                    sx.str("");
                    sx << "ERROR Trim [sd] value for channel " << iChan << " invalid: " << trim << "\n";
                    sx << "ERROR Trim [sd] may take values between [0,31] inclusively";
                    msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                    channel_ok = false;
                }

                // sz010b
                int sz10 = conf.second.get<int>("sz010b");
                if(sz10>=0 && sz10<32) {
                    chan.sz10b = sz10;
                }
                else {
                    sx.str("");
                    sx << "ERROR 10 bit ADC zero [sz010b] value for channel " << iChan << " invalid: " << sz10 << "\n";
                    sx << "ERROR [sz010b] may take values between [0,31] inclusively";
                    msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                    channel_ok = false;
                }

                // sz08b
                int sz8 = conf.second.get<int>("sz08b");
                if(sz8>=0 && sz8<16) {
                    chan.sz8b = sz8;
                }
                else {
                    sx.str("");
                    sx << "ERROR 8 bit ADC zero [sz08b] value for channel " << iChan << " invalid: " << sz8 << "\n";
                    sx << "ERROR [sz08b] may take values between [0,15] inclusively";
                    msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                    channel_reg = false;
                }
                
                // sz06b
                int sz6 = conf.second.get<int>("sz06b");
                if(sz6>=0 && sz6<8) {
                    chan.sz6b = sz6;
                }
                else {
                    sx.str("");
                    sx << "ERROR 8=6 bit ADC zero [sz06b] value for channel " << iChan << " invalid: " << sz6 << "\n";
                    sx << "ERROR [sz06b] may take values between [0,7] inclusively";
                    msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
                    channel_ok = false;
                }

                chan.ok = channel_ok;
                channels.push_back(chan);
            } // conf
        } // iChan

    } // try
    catch(std::exception& e) {
        sx.str("");
        sx << "ERROR Loading VMM registers from requested config file, exception thrown: "
            << e.what();
        msg()(sx,"ConfigHandler::LoadVMMChannelConfig"); sx.str("");
        for(int ichan = 0; ichan < (int)channels.size(); ichan++) {
            channels.at(ichan).ok = false;
        }
    } // catch

    if(!channel_reg || channels.size()!=64) {
        for(auto& chan : channels) chan.ok = false;
    }
    else {
        for(int i = 0; i < 64; i++) {
            for(auto& chan : channels) {
                if(!(chan.number==i)) continue;
                uint64_t tmp = 1;
                tmp = (tmp << i);

                chan.sc = (channel_sc & tmp) >> i;
                chan.sl = (channel_sl & tmp) >> i;
                chan.sth = (channel_sth & tmp) >> i;
                chan.st = (channel_st & tmp) >> i;
                chan.sm = (channel_sm & tmp) >> i;
                chan.smx = (channel_smx & tmp) >> i;

                chan.ok = true;

            } // chan
        } 
    }

    return channels;



/*
    // clear the current channels
    m_channels.clear();
    std::vector<Channel> channels;

    stringstream ss;
    stringstream where;
    stringstream sx;

    try {
        for(int iChan = 0; iChan < 64; iChan++) {
            bool channel_ok = true;
            ss.str("");
            ss << "channel_" << format("%02i") % iChan;
            for(const auto& conf : pt.get_child("configuration")) {
                size_t find = conf.first.find(ss.str());
                if(find==string::npos) continue;
                where.str("");
                where << "(ch. " << iChan << ")";
                string value = "";

                // fill a Channel struct
                Channel chan;
                // channel number
                chan.number = iChan;
                // channel polarity type
                string polarity = conf.second.get<string>("polarity");
                if(GlobalSetting::all_polarities.indexOf(QString::fromStdString(polarity))>=0) {
                    QString pol = QString::fromStdString(polarity);
                    chan.polarity = GlobalSetting::all_polarities.indexOf(pol);
                }
                else {
                    sx.str("");
                    sx << "ERROR polarity value for channel " << iChan << " "
                         << " must be one of: [";
                    for(auto& i : GlobalSetting::all_polarities) sx <<" "<<i.toStdString()<<" ";
                    sx << "]\nYou have provided: " << polarity;
                    msg()(sx,"ConfigHandler::LoadVMMChannelConfig");
                    channel_ok = false;
                }
                // capacitance
                string cap = conf.second.get<string>("capacitance");
                value = "capacitance " + where.str();
                chan.capacitance = (bool)isOn(cap, value);
                // leakage_current
                string leakage = conf.second.get<string>("leakage_current");
                value = "leakage_current " + where.str();
                chan.leakage_current = (bool)isOn(leakage, value);
                // test pulse
                string testpulse = conf.second.get<string>("test_pulse");
                value = "test_pulse " + where.str();
                chan.test_pulse = (bool)isOn(testpulse, value);
                // hidden mode
                string hidden = conf.second.get<string>("hidden_mode");
                value = "hidden_mode " + where.str();
                chan.hidden_mode = (bool)isOn(hidden, value);
                // trim
                chan.trim = conf.second.get<int>("trim");
                #warning what values can channel trim take?
                // monitor mode
                string mon = conf.second.get<string>("monitor_mode");
                value = "monitor_mode " + where.str();
                chan.monitor = (bool)isOn(mon, value);
                // 10bADC time set
                chan.s10bitADC = conf.second.get<int>("s10bADC_time_set");
                // 8bADC time set
                chan.s8bitADC = conf.second.get<int>("s08bADC_time_set");
                // 6bADC time set
                chan.s6bitADC = conf.second.get<int>("s06bADC_time_set");

                if(dbg())
                    chan.print();
                chan.ok = channel_ok;
                channels.push_back(chan);
            }
        }
    }
    catch(std::exception &e)
    {
        sx.str("");
        sx << "!! --------------------------------- !!\n"
           << "ERROR VMM Channel: " << e.what() << "\n"
           << "!! --------------------------------- !!";
        msg()(sx,"ConfigHandler::LoadVMMChannelConfig");
        for(int ichan = 0; ichan < channels.size(); ichan++) {
            channels.at(ichan).ok = false;
        }
    }
    return channels;
*/
}
//// ------------------------------------------------------------------------ //
FPGAClocks ConfigHandler::LoadFPGAClockSettings(const boost::property_tree::ptree& pt)
{
    using boost::property_tree::ptree;
    FPGAClocks clocks;
    bool outok = true;

    try
    {
        for(const auto& conf : pt.get_child("configuration")) {
            if(!(conf.first == "fpga_clocks")) continue;

            /////////////////////////////////////////////////
            // cktk max
            clocks.cktk_max_number = conf.second.get<int>("cktk_max");
            
            /////////////////////////////////////////////////
            // ckbc frequency
            string ckbc_in = conf.second.get<string>("ckbc");
            if(FPGAClocks::all_CKBC.indexOf(QString::fromStdString(ckbc_in))>=0) {
                clocks.ckbc_frequency = FPGAClocks::all_CKBC.indexOf(QString::fromStdString(ckbc_in));
            }
            else {
                stringstream sx;
                sx << "ERROR ckbc frequency value must be one of: [";
                for(auto& i : FPGAClocks::all_CKBC) sx << " " << i.toStdString();
                sx << " ]\nYou have provided: " << ckbc_in;
                msg()(sx,"ConfigHandler::LoadFPGAClockSettings");
                outok = false;
            }

            /////////////////////////////////////////////////
            // cktp max
            clocks.cktp_max_number = conf.second.get<int>("cktp_max");
            
            /////////////////////////////////////////////////
            // cktp skew
            clocks.cktp_skew_steps = conf.second.get<int>("cktp_skew");

            /////////////////////////////////////////////////
            // cktp period
            clocks.cktp_period = conf.second.get<int>("cktp_period");
    
            /////////////////////////////////////////////////
            // cktp width
            clocks.cktp_width = conf.second.get<int>("cktp_width");



            /*
            /////////////////////////////////////////////////
            // cktk period
            string cktk_in = conf.second.get<string>("cktk");
            if(s6Setting::all_CKTK.indexOf(QString::fromStdString(cktk_in))>=0)
                s6.cktk = s6Setting::all_CKTK.indexOf(QString::fromStdString(cktk_in));
            else {
                stringstream sx;
                sx << "ERROR cktk value must be one of: [";
                for(auto& i : s6Setting::all_CKTK) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << cktk_in;
                msg()(sx,"ConfigHandler::LoadFPGAClockSettings");
                outok = false;
            }
            /////////////////////////////////////////////////
            // ckbc frequency
            string ckbc_in = conf.second.get<string>("ckbc");
            if(s6Setting::all_CKBC.indexOf(QString::fromStdString(ckbc_in))>=0)
                s6.ckbc = s6Setting::all_CKBC.indexOf(QString::fromStdString(ckbc_in));
            else {
                stringstream sx;
                sx << "ERROR ckbc value must be one of: [";
                for(auto& i : s6Setting::all_CKBC) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << ckbc_in;
                msg()(sx,"ConfigHandler::LoadFPGAClockSettings");
                outok = false;
            }
            /////////////////////////////////////////////////
            // ckbc skew
            string ckbc_skew_in = conf.second.get<string>("ckbc_skew");
            if(s6Setting::all_CKBC_SKEW.indexOf(QString::fromStdString(ckbc_skew_in))>=0)
                s6.ckbc_skew = s6Setting::all_CKBC_SKEW.indexOf(QString::fromStdString(ckbc_skew_in));
            else {
                stringstream sx;
                sx << "ERROR ckbc_skew value must be one of: [";
                for(auto& i : s6Setting::all_CKBC_SKEW) sx << " " << i.toStdString() << " ";
                sx << "]\nYou have provided: " << ckbc_skew_in;
                msg()(sx,"ConfigHandler::LoadFPGAClockSettings");
                outok = false;
            }
            /////////////////////////////////////////////////
            // vmm auto reset
            //string auto_reset = conf.second.get<string>("vmm_auto_reset");
            //s6.do_auto_reset = isOn(auto_reset, "vmm_auto_reset");
            int idx = 0;
            bool ok = isEnabledOrDisabled(conf.second.get<string>("vmm_auto_reset"), idx, "vmm_auto_reset", false);
            if(ok) {
                s6.do_auto_reset = idx;
            }
            else {
                outok = false;
            }
            /////////////////////////////////////////////////
            // fec period reset
            //string fec_reset = conf.second.get<string>("fec_reset");
            //s6.do_fec_reset = isOn(fec_reset, "fec_reset");
            idx = 0;
            ok = isEnabledOrDisabled(conf.second.get<string>("vmm_auto_reset"), idx, "fec_reset", false);
            if(ok) {
                s6.do_fec_reset = idx;
            }
            else {
                outok = false;
            }

            /////////////////////////////////////////////////
            // tk pulses before reset
            s6.tk_pulses = conf.second.get<int>("tk_pulses");
            /////////////////////////////////////////////////
            // fec reset period
            s6.fec_reset_period = conf.second.get<int>("fec_reset_period");
            */
            
        } // conf

    } // try
    catch(std::exception &e)
    {
        stringstream sx;
        sx << "!! --------------------------------- !!\n"
           << "    ERROR FPGAClocks: " << e.what() << "\n"
           << "!! --------------------------------- !!";
        msg()(sx, "ConfigHandler::LoadFPGAClockSettings");
        outok = false;
    } // catch
    clocks.ok = outok;

    return clocks;
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::LoadBoardConfiguration(GlobalSetting& global,
                VMMMap& vmmMap, std::vector<Channel>& channels)
{
    m_globalSettings = global;
 //   bool ok;
 //   QString tmp = "";
 //   tmp.append(QString::number(global.direct_time_mode0));
 //   tmp.append(QString::number(global.direct_time_mode1));
 //   m_globalSettings.direct_time_mode = tmp.toUInt(&ok,2);

    m_vmmMap = vmmMap;
    
    //m_channelmap.clear();
    //for(int i = 0; i < (int)chMap.size(); i++) {
    //    m_channelmap.push_back(chMap[i]);
    //    //m_channelmap[i].print();
    //}
    //// build the HDMI channel map bit word
    //setHDMIChannelMap();

    m_channels.clear();
    for(int i = 0; i < (int)channels.size(); i++) {
        //cout << "channels[" << i << "] st = " << channels[i].st << "  sm = " << channels[i].sm << endl;
        m_channels.push_back(channels[i]);
        //m_channels[i].print();
    }
    //m_globalSettings.print();
}
//// ------------------------------------------------------------------------ //
void ConfigHandler::LoadTDAQConfiguration(TriggerDAQ& daq)
{
    m_daqSettings = daq; 
    if(dbg())
        m_daqSettings.print();
}

void ConfigHandler::LoadFPGAClocksConfiguration(FPGAClocks& clocks)
{
    m_fpgaClockSettings = clocks;
    if(dbg())
        m_fpgaClockSettings.print();
}
//// ------------------------------------------------------------------------ //
bool ConfigHandler::EnabledOrDisabled(int ena_or_dis, std::string& value, std::string from, bool invert)
{
    stringstream sx;
    int is_enabled = 1;
    int is_disabled = 0;
    if(invert) {
        is_enabled = 0;
        is_disabled = 1;
    }
    if(ena_or_dis==is_enabled) {
        value = "ENABLED";
    }
    else if(ena_or_dis==is_disabled) {
        value = "DISABLED";
    }
    else {
        string where = "";
        if(from!="") where = " [for " + from + "]";
        sx << "ERROR Invalid input, expecting 0 or 1 but you have: " << ena_or_dis;
        msg()(sx,"ConfigHandler::EnabledOrDisabled"); sx.str("");
        value = "";
        return false;
    }
    return true;
}
//// ------------------------------------------------------------------------ //
bool ConfigHandler::isEnabledOrDisabled(std::string ena_or_dis, int& value, std::string from, bool invert)
{
    stringstream sx;
    int is_enabled = 1;
    int is_disabled = 0;
    if(invert) {
        is_enabled = 0;
        is_disabled = 1;
    }

    if(ena_or_dis=="ENABLED") {
        value = is_enabled;
    }
    else if(ena_or_dis=="DISABLED") {
        value = is_disabled;
    }
    else {
        std::string where = "";
        if(from!="") where = " [for " + from + "]";
        sx << "ERROR Expecting only 'ENABLED' or 'DISABLED' " << where << " but your config provides: "
           << ena_or_dis;
        msg()(sx,"ConfigHandler::isEnabledOrDisabled"); sx.str("");
        return false; 
    }
    return true;
}
//// ------------------------------------------------------------------------ //
int ConfigHandler::isOn(std::string onOrOff, std::string where)
{
    stringstream sx;

    if(onOrOff=="on" || onOrOff=="disabled")
    //if(onOrOff=="on" || onOrOff=="enabled")
        return 1;
    else if(onOrOff=="off" || onOrOff=="enabled")
    //else if(onOrOff=="off" || onOrOff=="disabled")
        return 0;
    else {
        std::string from = "";
        if(where!="") from = " [for " + where + "]";
        sx << "Expecting either 'on'/'enabled' or 'off'/'disabled'" << from << ".\n"
           << ">>> Returning 'off'/'disabled'.";
        msg()(sx,"ConfigHandler::isOn");
        return 0;
    } 
}
//// ------------------------------------------------------------------------ //
std::string ConfigHandler::isOnOrOff(int onOrOff, std::string node)
{
    stringstream sx;

    if(onOrOff==1)
        return "on";
    else if(onOrOff==0)
        return "off";
    else {
        sx << "WARNING Expect only '0' or '1' as input. You have provided: " << onOrOff;
        msg()(sx,"ConfigHandler::isOnOrOff"); sx.str("");
        sx << "WARNING Setting node \"" << node << "\" to \"off\"";
        msg()(sx,"ConfigHandler::isOnOrOff"); sx.str("");
        string out = "off";
        return out;
    }
}
//// ------------------------------------------------------------------------ //
std::string ConfigHandler::isEnaOrDis(int enaOrDis, std::string node)
{
    stringstream sx;
    if(enaOrDis==0)
    //if(enaOrDis==1)
        return "enabled";
    else if(enaOrDis==1)
    //else if(enaOrDis==0)
        return "disabled";
    else {
        sx << "WARNING Expect only '0' or '1' as input. You have provided: " << enaOrDis;
        msg()(sx,"ConfigHandler::isEnaOrDis"); sx.str("");
        sx << "WARNING Setting node \"" << node << "\" to \"disabled\"";
        msg()(sx, "ConfigHandler::isEnaOrDis"); sx.str("");
        string out = "disabled";
        return out;
    }
}
