
// vmm
#include "configuration_module.h"
#include "data_handler.h" // for reverse32

// std/stl
#include <iostream>
#include <bitset>
using namespace std;

// Qt
#include <QString>
#include <QDataStream>
#include <QByteArray>
#include <QBitArray>
#include <QStringList>

// boost
#include <boost/format.hpp>

//////////////////////////////////////////////////////////////////////////////
// ------------------------------------------------------------------------ //
//  Configuration
// ------------------------------------------------------------------------ //
//////////////////////////////////////////////////////////////////////////////
Configuration::Configuration(QObject *parent) :
    QObject(parent),
    m_dbg(false),
    m_vmm_type(3),
    m_socketHandler(0),
    m_configHandler(0),
    m_msg(0)
{
}
// ------------------------------------------------------------------------ //
void Configuration::LoadMessageHandler(MessageHandler& m)
{
    m_msg = &m;
}
// ------------------------------------------------------------------------ //
Configuration& Configuration::LoadConfig(ConfigHandler& config)
{
    m_configHandler = &config;
    if(!m_configHandler) {
        msg()("FATAL ConfigHandler instance is null", "Configuration::LoadConfig", true);
        exit(1);
    }
    else if(dbg()) {
        msg()("ConfigHandler instance loaded", "Configuration::LoadConfig");
    }
    return *this;
}
// ------------------------------------------------------------------------ //
Configuration& Configuration::LoadSocket(SocketHandler& socket)
{
    m_socketHandler = &socket;
    //cout << "Configuration::LoadSocket    [" << boost::this_thread::get_id() << "] socket = " << m_socketHandler << endl;
    if(!m_socketHandler) {
        msg()("FATAL SocketHandler instance is null", "Configuration::LoadSocket", true);
        exit(1);
    }
    else if(dbg()) {
        msg()("SocketHandler instance loaded", "Configuration::LoadSocket");
        m_socketHandler->Print();
    }
        
    return *this;
}
// ------------------------------------------------------------------------ //
quint32 Configuration::reverse32(QString input_bin)
{
    bool ok;
    QBitArray received(32,false);
    for(int i = 0; i < input_bin.size(); i++) {
        QString bit = input_bin.at(i);
        received.setBit(32-input_bin.size()+i, bit.toUInt(&ok,10));
    } // i

    QBitArray reversed(32,false);
    for(int j = 0; j < 32; j++) {
        reversed.setBit(31-j, received[j]);
    } // j

    QByteArray reversed_byte = DataHandler::bitsToBytes(reversed);
    return reversed_byte.toHex().toUInt(&ok,16);
}
// ------------------------------------------------------------------------ //
QString Configuration::reverseString(QString string)
{
    QString tmp = string;
    QByteArray ba = tmp.toLocal8Bit();
    char *d = ba.data();
    std::reverse(d,d+tmp.length());
    tmp = QString(d);
    return tmp;
}
// ------------------------------------------------------------------------ //
void Configuration::send_configuration_pulser()
{
    SendConfig();
}
// ------------------------------------------------------------------------ //
void Configuration::SendConfig()
{
    ///////////////////////////////////////////////////
    // check VMM type
    ///////////////////////////////////////////////////
    if(vmm_type()==2) {
        SendConfig_VMM2();
        return;
    }

    stringstream sx;
    bool send_ok = true;
    bool ok;

    #warning hard coding port
    int send_to_port = 6008;

    //////////////////////////////////////////////////
    // build the configuration word(s) to be send to
    // the front ends
    //////////////////////////////////////////////////

    //////////////////////////////////////////////////
    // Global Registers
    //////////////////////////////////////////////////
    std::vector<QString> globalRegisters;
    globalRegisters.clear();
    fillGlobalRegisters(globalRegisters);
    

    //////////////////////////////////////////////////
    // channel registers
    //////////////////////////////////////////////////
    std::vector<QString> channelRegisters;
    channelRegisters.clear();
    fillChannelRegisters(channelRegisters);

    //////////////////////////////////////////////////
    // begin to send the word
    //////////////////////////////////////////////////
    QByteArray datagram;
    QDataStream out(&datagram, QIODevice::WriteOnly);
    out.device()->seek(0);

    ////////////////////////////////////////////////////////////
    // Now prepare and send the word
    ////////////////////////////////////////////////////////////
    int ip_idx = -1;
    for(const auto& ip : socket().ipList()) {

        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) {
                continue;
            }
        }

        // send configuration individually to each VMM
        int vmm_mask = config().vmmMap().mask;
        for(int i = 0; i < 1; i++) { // testing bit mask, so only need one loop
            int tmp = 1;
            //tmp = tmp << i;
            tmp = vmm_mask;
            if(vmm_mask & tmp) {
                datagram.clear();
                QDataStream out(&datagram, QIODevice::WriteOnly);
                out.device()->seek(0);

                // update the serial command
                socket().updateCommandCounter();
                if(dbg()) {
                    sx << "Sending command SPI at command #: " << socket().commandCounter();
                    msg()(sx,"Configuration::SendConfig"); sx.str("");
                }

                #warning header for configuration is set to be the same as VMM2-based MMFE8
                QString h1 = "ffbb";
                QString h2 = "aadd";
                QString h3 = "bb00";

                out << (quint16)h1.toUInt(&ok,16)
                    << (quint16)h2.toUInt(&ok,16)
                    //<< (quint16)(i+1) // vmm id (starts at 1, ends at 8)
                    << (quint8)(0)
                    << (quint8)(vmm_mask)
                    << (quint16)h3.toUInt(&ok,16);

                // first bit to write is the last bit of global bank 2
                out << (quint32)(reverseString(globalRegisters.at(5)).toUInt(&ok,2));
                out << (quint32)(reverseString(globalRegisters.at(4)).toUInt(&ok,2));
                out << (quint32)(reverseString(globalRegisters.at(3)).toUInt(&ok,2));

                // send channels
                for(int i = 63; i >= 0; i--) {
                    QString first8bits = channelRegisters.at(i).mid(8,8); // 00000000|sc|sl|...
                    QString second16bits = channelRegisters.at(i).mid(16,16);

                    // |nu|sz6|sz8|...|sl||sc|
                    out << (quint8) (first8bits).toUInt(&ok,2);
                    out << (quint16)(second16bits).toUInt(&ok,2);
                } // i

                // last bit to write is first bit of global bank 1
                out << (quint32)(reverseString(globalRegisters.at(2)).toUInt(&ok,2));
                out << (quint32)(reverseString(globalRegisters.at(1)).toUInt(&ok,2));
                out << (quint32)(reverseString(globalRegisters.at(0)).toUInt(&ok,2));

                //cout << "----------------------------------------------------" << endl;
                //cout << "configuration packet vmm " << i << endl;
                //qDebug() << datagram.toHex();
                //cout << endl;

                bool readOK = true;
                send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                                "Configuration::SendConfig");
                if(send_ok) {
                    readOK = socket().waitForReadyRead("fec");
                    if(readOK) {
                        if(dbg()) msg()("Processing replies...","Configuration::SendConfig");
                        socket().processReply("fec",ip);
                    }
                    else {
                        sx.str("");
                        sx << "Timeout while waiting for replies from VMM " << i;
                        msg()(sx,"Configuration::SendConfig");
                        socket().closeAndDisconnect("fec", "Configuration::SendConfig");
                    }
                }

            } // send to this VMM
        } // per VMM
    } // ip
    

}
// ------------------------------------------------------------------------ //
void Configuration::fillGlobalRegisters(std::vector<QString>& global)
{
    stringstream sx;
    if(dbg()) msg()("Loading global registers","Configuration::fillGlobalRegisters");
    global.clear();
    int pos = 0;


    QString bit32_empty = "00000000000000000000000000000000"; 
    QString tmp;

    //////////////////////////////////////////////////////////////////
    // Global Bank 1
    //////////////////////////////////////////////////////////////////


    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-1 [sp:sdt3]
    QString spi1_0 = bit32_empty;
    pos = 0;

    // sp
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sp));
    pos++;

    // sdp
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sdp));
    pos++;

    // sbmx
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sbmx));
    pos++;

    // sbft
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sbft));
    pos++;

    // sbfp
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sbfp));
    pos++;
    
    // sbfm
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sbfm));
    pos++;
    
    // slg
    spi1_0.replace(pos,1,QString::number(config().globalSettings().slg));
    pos++;

    // sm5-sm0
    tmp = QString("%1").arg(config().globalSettings().sm5,6,2,QChar('0'));
    spi1_0.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // scmx
    spi1_0.replace(pos,1,QString::number(config().globalSettings().scmx));
    pos++;

    // sfa
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sfa));
    pos++;

    // sfam
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sfam));
    pos++;

    // st
    tmp = QString("%1").arg(config().globalSettings().st,2,2,QChar('0'));
    spi1_0.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sfm
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sfm));
    pos++;

    // sg
    tmp = QString("%1").arg(config().globalSettings().sg,3,2,QChar('0'));
    spi1_0.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sng
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sng));
    pos++;

    // stot
    spi1_0.replace(pos,1,QString::number(config().globalSettings().stot));
    pos++;

    // sttt
    spi1_0.replace(pos,1,QString::number(config().globalSettings().sttt));
    pos++;

    // ssh
    spi1_0.replace(pos,1,QString::number(config().globalSettings().ssh));
    pos++;

    // stc
    tmp = QString("%1").arg(config().globalSettings().stc,2,2,QChar('0'));
    spi1_0.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // first 4 bits of sdt
    uint32_t sdt_tmp = 0;
    sdt_tmp |= ( (0x3C0 & config().globalSettings().sdt_dac) >> 6 );
    //sdt_tmp |= ( 0xF & config().globalSettings().sdt_dac );
    tmp = QString("%1").arg(sdt_tmp,4,2,QChar('0'));
    //tmp = reverseString(tmp);
    spi1_0.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();


    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-1 [sd4:res00]
    QString spi1_1 = bit32_empty;
    pos = 0;

    // last 6 bits of sdt
    sdt_tmp = 0;
    sdt_tmp |= ((config().globalSettings().sdt_dac & 0x3F));// & 0x3F;
    //sdt_tmp |= ((config().globalSettings().sdt_dac & 0x3F0) >> 4);// & 0x3F;
    tmp = QString("%1").arg(sdt_tmp,6,2,QChar('0'));
    //tmp = reverseString(tmp);
    spi1_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sdp
    tmp = QString("%1").arg(config().globalSettings().sdp_dac,10,2,QChar('0'));
    //tmp = reverseString(tmp);
    spi1_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sc010b:sc110b
    tmp = QString("%1").arg(config().globalSettings().sc10b,2,2,QChar('0'));
    tmp = reverseString(tmp);
    spi1_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sc08b:sc18b
    tmp = QString("%1").arg(config().globalSettings().sc8b,2,2,QChar('0'));
    tmp = reverseString(tmp);
    spi1_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // sc06b:sc26b
    tmp = QString("%1").arg(config().globalSettings().sc6b,3,2,QChar('0'));
    tmp = reverseString(tmp);
    spi1_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // s8b
    spi1_1.replace(pos,1,QString::number(config().globalSettings().s8b));
    pos++;

    // s6b
    spi1_1.replace(pos,1,QString::number(config().globalSettings().s6b));
    pos++;

    // s10b
    spi1_1.replace(pos,1,QString::number(config().globalSettings().s10b));
    pos++;

    // sdcks
    spi1_1.replace(pos,1,QString::number(config().globalSettings().sdcks));
    pos++;

    // sdcka
    spi1_1.replace(pos,1,QString::number(config().globalSettings().sdcka));
    pos++;

    // sdck6b
    spi1_1.replace(pos,1,QString::number(config().globalSettings().sdck6b));
    pos++;

    // sdrv
    spi1_1.replace(pos,1,QString::number(config().globalSettings().sdrv));
    pos++;

    // stpp
    spi1_1.replace(pos,1,QString::number(config().globalSettings().stpp));
    pos++;

    // res00

    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-2 [res0:reset]
    QString spi1_2 = bit32_empty;
    pos = 0;

    pos += 4; // first 4 reserved

    // slvs
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvs));
    pos++;

    // s32
    spi1_2.replace(pos,1,QString::number(config().globalSettings().s32));
    pos++;

    // stcr
    spi1_2.replace(pos,1,QString::number(config().globalSettings().stcr));
    pos++;

    // ssart
    spi1_2.replace(pos,1,QString::number(config().globalSettings().ssart));
    pos++;

    // srec
    spi1_2.replace(pos,1,QString::number(config().globalSettings().srec));
    pos++;

    // stlc
    spi1_2.replace(pos,1,QString::number(config().globalSettings().stlc));
    pos++;

    // sbip
    spi1_2.replace(pos,1,QString::number(config().globalSettings().sbip));
    pos++;

    // srat
    spi1_2.replace(pos,1,QString::number(config().globalSettings().srat));
    pos++;

    // sfrst
    spi1_2.replace(pos,1,QString::number(config().globalSettings().sfrst));
    pos++;

    // slvsbc
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvsbc));
    pos++;

    // slvstp
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvstp));
    pos++;

    // slvstk
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvstk));
    pos++;

    // slvsdt
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvsdt));
    pos++;

    // slvsart
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvsart));
    pos++;

    // slvstki
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvstki));
    pos++;

    // slvsena
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvsena));
    pos++;

    // slvs6b
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slvs6b));
    pos++;

    // sL0enaV
    spi1_2.replace(pos,1,QString::number(config().globalSettings().sL0enaV));
    pos++; // we have 8 nu before the reset bits

    //slh
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slh));
    pos++;

    //slxh
    spi1_2.replace(pos,1,QString::number(config().globalSettings().slxh));
    pos++;

    //stgc
    spi1_2.replace(pos,1,QString::number(config().globalSettings().stgc));
    pos+=6;
    // reset
    uint32_t reset_flags = 0;
    if(config().globalSettings().reset == 1) {
        // raise last 2 bits high if doing reset
        reset_flags = 3;
    }
    tmp = QString("%1").arg(reset_flags,2,2,QChar('0'));
    //tmp = reverseString(tmp);
    spi1_2.replace(pos, tmp.size(),tmp);


    //////////////////////////////////////////////////////////////////
    // Global Bank 2
    //////////////////////////////////////////////////////////////////

    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-0 [nu:nskipm]
    QString spi2_0 = bit32_empty;
    pos = 0;

    // bits [0:30] are not-used
    pos+=31;

    // nskipm
    spi2_0.replace(pos,1,QString::number(config().globalSettings().nskipm));
    

    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-1 [sL0cktest:rollover]

    QString spi2_1 = bit32_empty;
    pos = 0;

    // sL0cktest
    spi2_1.replace(pos,1,QString::number(config().globalSettings().sL0cktest));
    pos++;

    // sL0dckinv
    spi2_1.replace(pos,1,QString::number(config().globalSettings().sL0dckinv));
    pos++;

    // sL0ckinv
    spi2_1.replace(pos,1,QString::number(config().globalSettings().sL0ckinv));
    pos++;

    // sL0ena
    spi2_1.replace(pos,1,QString::number(config().globalSettings().sL0ena));
    pos++;

    // truncate
    tmp = QString("%1").arg(config().globalSettings().truncate,6,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // nskip
    tmp = QString("%1").arg(config().globalSettings().nskip,7,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // window
    tmp = QString("%1").arg(config().globalSettings().window,3,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_1.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // rollover
    tmp = QString("%1").arg(config().globalSettings().rollover,12,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_1.replace(pos,tmp.size(),tmp);
    

    ////////////////////////////////////////////////////////////////////
    ////////////////// 32-2 [l0offset:offset]

    QString spi2_2 = bit32_empty;
    pos = 0;

    // l0offset
    tmp = QString("%1").arg(config().globalSettings().l0offset,12,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_2.replace(pos,tmp.size(),tmp);
    pos+=tmp.size();

    // offset
    tmp = QString("%1").arg(config().globalSettings().offset,12,2,QChar('0'));
    tmp = reverseString(tmp);
    spi2_2.replace(pos,tmp.size(),tmp);
    


/*
    uint32_t spi20 = 0;
    sequence = 31; // [0:30] are not-used
    spi20 |= (config().globalSettings().nskipm << sequence); // 31

    uint32_t spi21 = 0;
    sequence = 0;
    spi21 |= (config().globalSettings().sL0cktest);             sequence++; // 0
    spi21 |= (config().globalSettings().sL0dckinv << sequence); sequence++; // 1
    spi21 |= (config().globalSettings().sL0ckinv << sequence);  sequence++; // 2
    spi21 |= (config().globalSettings().sL0ena << sequence);    sequence++; // 3
    spi21 |= (config().globalSettings().truncate << sequence);  sequence += 6; // 9
    spi21 |= (config().globalSettings().nskip << sequence);     sequence += 7; // 16
    spi21 |= (config().globalSettings().window << sequence);    sequence += 3; // 19
    spi21 |= (config().globalSettings().rollover << sequence);  sequence += 12; // 31

    uint32_t spi22 = 0;
    sequence = 0;
    spi22 |= (config().globalSettings().l0offset);              sequence += 12; // [0:11]
    spi22 |= (config().globalSettings().offset << sequence);    sequence += 12; // [12:23]
    // [24:31] are not-used

*/

/*

    //QString spi1_0 = bit32_empty; 
    //QString spi1_1 = bit32_empty; 
    //QString spi1_2 = bit32_empty; 
    QString spi2_0 = bit32_empty; 
    QString spi2_1 = bit32_empty; 
    QString spi2_2 = bit32_empty; 


    //////////////////////////////////////////////////////////////////
    // Global Bank 1
    //////////////////////////////////////////////////////////////////

    uint32_t spi10 = 0;
    spi10 |= config().globalSettings().sp;                  sequence++; // 0
    spi10 |= (config().globalSettings().sdp << sequence);   sequence++; // 1
    spi10 |= (config().globalSettings().sbmx << sequence);  sequence++; // 2
    spi10 |= (config().globalSettings().sbft << sequence);  sequence++; // 3
    spi10 |= (config().globalSettings().sbfp << sequence);  sequence++; // 4
    spi10 |= (config().globalSettings().sbfm << sequence);  sequence++; // 5
    spi10 |= (config().globalSettings().slg << sequence);   sequence++; // 6
    spi10 |= (config().globalSettings().sm5 << sequence);   sequence += 6; // 12
    spi10 |= (config().globalSettings().scmx << sequence);  sequence++; // 13
    spi10 |= (config().globalSettings().sfa << sequence);   sequence++; // 14
    spi10 |= (config().globalSettings().sfam << sequence);  sequence++; // 15
    spi10 |= (config().globalSettings().st << sequence);    sequence += 2; // 17
    spi10 |= (config().globalSettings().sfm << sequence);   sequence++; // 18
    spi10 |= (config().globalSettings().sg << sequence);    sequence += 3; // 21
    spi10 |= (config().globalSettings().sng << sequence);   sequence++; // 22
    spi10 |= (config().globalSettings().stot << sequence);  sequence++; // 23
    spi10 |= (config().globalSettings().sttt << sequence);  sequence++; // 24
    spi10 |= (config().globalSettings().ssh << sequence);   sequence++; // 25
    spi10 |= (config().globalSettings().stc << sequence);   sequence += 2; // 27
    spi10 |= ( (0xF & config().globalSettings().sdt_dac) << sequence); sequence += 4; // 31

    sequence = 0;
    uint32_t spi11 = 0;
    spi11 |= (0x3F0 & config().globalSettings().sdt_dac)>>4;    sequence += 6; // 5
    spi11 |= (config().globalSettings().sdp_dac << sequence);   sequence += 10; // 15
    spi11 |= (config().globalSettings().sc10b << sequence);     sequence += 2; // 17
    spi11 |= (config().globalSettings().sc8b << sequence);      sequence += 2; // 19
    spi11 |= (config().globalSettings().sc6b << sequence);      sequence += 3; // 22
    spi11 |= (config().globalSettings().s8b << sequence);       sequence++; // 23
    spi11 |= (config().globalSettings().s6b << sequence);       sequence++; // 24
    spi11 |= (config().globalSettings().s10b << sequence);      sequence++; // 25
    spi11 |= (config().globalSettings().sdcks << sequence);     sequence++; // 26
    spi11 |= (config().globalSettings().sdcka << sequence);     sequence++; // 27
    spi11 |= (config().globalSettings().sdck6b << sequence);    sequence++; // 28
    spi11 |= (config().globalSettings().sdrv << sequence);      sequence++; // 29
    spi11 |= (config().globalSettings().stpp << sequence);      sequence++; // 30
    spi11 |= (0 << sequence); sequence++; //res00 // 31

    sequence = 4; // first 4 bits are reserved
    uint32_t spi12 = 0;
    spi12 |= (config().globalSettings().slvs << sequence);      sequence++; // 4
    spi12 |= (config().globalSettings().s32 << sequence);       sequence++; // 5
    spi12 |= (config().globalSettings().stcr << sequence);      sequence++; // 6
    spi12 |= (config().globalSettings().ssart << sequence);     sequence++; // 7
    spi12 |= (config().globalSettings().srec << sequence);      sequence++; // 8
    spi12 |= (config().globalSettings().stlc << sequence);      sequence++; // 9
    spi12 |= (config().globalSettings().sbip << sequence);      sequence++; // 10
    spi12 |= (config().globalSettings().srat << sequence);      sequence++; // 11
    spi12 |= (config().globalSettings().sfrst << sequence);     sequence++; // 12
    spi12 |= (config().globalSettings().slvsbc << sequence);    sequence++; // 13
    spi12 |= (config().globalSettings().slvstp << sequence);    sequence++; // 14
    spi12 |= (config().globalSettings().slvstk << sequence);    sequence++; // 15
    spi12 |= (config().globalSettings().slvsdt << sequence);    sequence++; // 16
    spi12 |= (config().globalSettings().slvsart << sequence);   sequence++; // 17
    spi12 |= (config().globalSettings().slvstki << sequence);   sequence++; // 18
    spi12 |= (config().globalSettings().slvsena << sequence);   sequence++; // 19
    spi12 |= (config().globalSettings().slvs6b << sequence);    sequence++; // 20
    spi12 |= (config().globalSettings().sL0enaV << sequence);   sequence++; // 21
    // [22:29] are reserved
    if(config().globalSettings().reset == 1) {
        // raise last 2 bits high if doing reset
        uint32_t reset_flags = 3;
        spi12 |= (reset_flags << 30);
    }
    
    //////////////////////////////////////////////////////////////////
    // Global Bank 2
    //////////////////////////////////////////////////////////////////
    uint32_t spi20 = 0;
    sequence = 31; // [0:30] are not-used
    spi20 |= (config().globalSettings().nskipm << sequence); // 31

    uint32_t spi21 = 0;
    sequence = 0;
    spi21 |= (config().globalSettings().sL0cktest);             sequence++; // 0
    spi21 |= (config().globalSettings().sL0dckinv << sequence); sequence++; // 1
    spi21 |= (config().globalSettings().sL0ckinv << sequence);  sequence++; // 2
    spi21 |= (config().globalSettings().sL0ena << sequence);    sequence++; // 3
    spi21 |= (config().globalSettings().truncate << sequence);  sequence += 6; // 9
    spi21 |= (config().globalSettings().nskip << sequence);     sequence += 7; // 16
    spi21 |= (config().globalSettings().window << sequence);    sequence += 3; // 19
    spi21 |= (config().globalSettings().rollover << sequence);  sequence += 12; // 31

    uint32_t spi22 = 0;
    sequence = 0;
    spi22 |= (config().globalSettings().l0offset);              sequence += 12; // [0:11]
    spi22 |= (config().globalSettings().offset << sequence);    sequence += 12; // [12:23]
    // [24:31] are not-used


    //cout << "spi1_0: " << std::bitset<32>(spi10) << endl;
    //cout << "spi1_1: " << std::bitset<32>(spi11) << endl;
    //cout << "spi1_2: " << std::bitset<32>(spi12) << endl;
    //cout << "spi2_0: " << std::bitset<32>(spi20) << endl;
    //cout << "spi2_1: " << std::bitset<32>(spi21) << endl;
    //cout << "spi2_2: " << std::bitset<32>(spi22) << endl;

    global.push_back(spi10);
    global.push_back(spi11);
    global.push_back(spi12);
    global.push_back(spi20);
    global.push_back(spi21);
    global.push_back(spi22);

*/
/*

    // [31:28]
    int32_t sdt_0to3 = 0xF & config().globalSettings().sdt_dac; // 0xF = 1111
    tmp = QString("%1").arg(sdt_0to3,4,2,QChar('0'));
    spi1_0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [27:26]
*/    
 


/*
    //////////////////////////////////////////////////////////////////
    // Global Bank 2
    //////////////////////////////////////////////////////////////////

    QString tmp;

    // we are filling from MSB --> LSB direction, so "nu" is LSB and first out
    // fill in "correct" order (i.e. not order of sending)

    // [0:11] -- l0offset
    tmp = QString("%1").arg(config().globalSettings().l0offset, 12, 2, QChar('0'));
    spi1_0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [12:23] -- offset
    tmp = QString("%1").arg(config().globalSettings().offset, 12, 2, QChar('0'));
    spi1_0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [24:31] -- nu
    tmp = QString("%1").arg(0,7,2,QChar('0'));
    spi1_0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();
  
    // -----------------------------
    sequence = 0;

    // [0] -- sL0cktest
    spi1_1.replace(sequence,1,QString::number(config().globalSettings().sL0cktest));
    sequence++;

    // [1] -- sL0dckinv
    spi1_1.replace(sequence,1,QString::number(config().globalSettings().sL0dckinv));
    sequence++;

    // [2] -- sL0ckinv
    spi1_1.replace(sequence,1,QString::number(config().globalSettings().sL0ckinv));
    sequence++;

    // [3] -- sL0ena
    spi1_1.replace(sequence,1,QString::number(config().globalSettings().sL0ena));
    sequence++;

    // [4:9] -- truncate
    tmp = QString("%1").arg(config().globalSettings().truncate, 6, 2, QChar('0'));
    spi1_1.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [10:16] -- nskip
    tmp = QString("%1").arg(config().globalSettings().nskip, 7, 2, QChar('0'));
    spi1_1.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [17:19] -- window
    tmp = QString("%1").arg(config().globalSettings().window, 3, 2, QChar('0'));
    spi1_1.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [20:31] -- rollover
    tmp = QString("%1").arg(config().globalSettings().rollover, 12, 2, QChar('0'));
    spi1_1.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // -----------------------------
    sequence = 0;

    // [0:30] -- nu
    tmp = QString("%1").arg(0,31,2,QChar('0'));
    spi1_2.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    // [31] -- nskipm
    spi1_2.replace(sequence,1,QString::number(config().globalSettings().nskipm));
    sequence++;

*/ 
    //////////////////////////////////////////////////////////////////
    // Global Bank 1
    //////////////////////////////////////////////////////////////////

    



/*

    // sdt [9:6]
    int32_t sdt_9to6 = 0x3C0 & config().globalSettings().sdt_dac; // 0x3C0 = 0000 0000 0000 0000 0000 0011 1100 0000
    sdt_9to6 = (sdt_9to6 >> 6);
    tmp = QString("%1").arg(sdt_9to6,4,2,QChar('0'));
    bank1_0.replace(sequence, tmp.size(), tmp);
    
    

    tmp = QString("%1").arg(config().globalSettings().sp, 1, 2, QChar('0'));
    bank1_0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();
*/

 //   qDebug() << "spi1_0: " << spi1_0;
 //   qDebug() << "spi1_1: " << spi1_1;
 //   qDebug() << "spi1_2: " << spi1_2;
 //   qDebug() << "spi2_0: " << spi2_0;
 //   qDebug() << "spi2_1: " << spi2_1;
 //   qDebug() << "spi2_2: " << spi2_2;
    

    //global.push_back(reverseString(spi1_0));
    //global.push_back(reverseString(spi1_1));
    //global.push_back(reverseString(spi1_2));
    //global.push_back(reverseString(spi2_0));
    //global.push_back(reverseString(spi2_1));
    //global.push_back(reverseString(spi2_2));
    global.push_back(spi1_0);
    global.push_back(spi1_1);
    global.push_back(spi1_2);
    global.push_back(spi2_0);
    global.push_back(spi2_1);
    global.push_back(spi2_2);
    
    //QString spi1_0 = bit32_empty; 
    //QString spi1_1 = bit32_empty; 
    //QString spi1_2 = bit32_empty; 





}
// ------------------------------------------------------------------------ //
void Configuration::fillChannelRegisters(std::vector<QString>& channels)
{
    stringstream sx;
    if(dbg()) msg()("Loading channel configuration", "Configuration::fillChannelRegisters");

    channels.clear();
    int pos = 0;
    QString tmp;
    QString reg;
    QString bit32_empty = "00000000000000000000000000000000"; 

    for(int i = 0; i < 64; ++i) {
        reg = bit32_empty;
        pos = 0;
      //  pos = 8;

        //cout << "Configuration::fillChannelRegisters    st = "
        //        << config().channelSettings(i).st
        //        << "   sm = " << config().channelSettings(i).sm << endl;

        // sc
        reg.replace(pos,1,QString::number(config().channelSettings(i).sc)); 
        pos++;

        // sl
        reg.replace(pos,1,QString::number(config().channelSettings(i).sl));
        pos++;

        // st
        reg.replace(pos,1,QString::number(config().channelSettings(i).st));
        pos++;

        // sth
        reg.replace(pos,1,QString::number(config().channelSettings(i).sth));
        pos++;

        // sm
        reg.replace(pos,1,QString::number(config().channelSettings(i).sm));
        pos++;

        // smx
        reg.replace(pos,1,QString::number(config().channelSettings(i).smx));
        pos++;

        // sd (trim)
        tmp = QString("%1").arg(config().channelSettings(i).sd,5,2,QChar('0')); 
        // sd0:sd4 -- is reversal still needed for trim in VMM3?
        tmp = reverseString(tmp);
        reg.replace(pos,tmp.size(),tmp);
        pos+=tmp.size();

        // sz10b 10 bit adc lsb
        tmp = QString("%1").arg(config().channelSettings(i).sz10b,5,2,QChar('0'));
        tmp = reverseString(tmp);
        reg.replace(pos,tmp.size(),tmp);
        pos+=tmp.size();

        // sz8b 8 bit adc lsb
        tmp = QString("%1").arg(config().channelSettings(i).sz8b,4,2,QChar('0'));
        tmp = reverseString(tmp);
        reg.replace(pos,tmp.size(),tmp);
        pos+=tmp.size();

        // sz6b 6 bit adc lsb
        tmp = QString("%1").arg(config().channelSettings(i).sz6b,3,2,QChar('0'));
        tmp = reverseString(tmp);
        reg.replace(pos,tmp.size(),tmp);
        pos+=tmp.size();

        // push back
        reg = reverseString(reg);
        channels.push_back(reg);

    } // i

/*
    channels.clear();
    int sequence = 0;

    for(int i = 0; i < 64; i++) {
        sequence = 0;
        uint32_t channel_register = 0;
        sequence = 0;
        channel_register |= (config().channelSettings(i).sc << sequence);   sequence++; // 1
        channel_register |= (config().channelSettings(i).sl << sequence);   sequence++; // 2
        channel_register |= (config().channelSettings(i).st << sequence);   sequence++; // 3
        channel_register |= (config().channelSettings(i).sth << sequence);  sequence++; // 4
        channel_register |= (config().channelSettings(i).sm << sequence);   sequence++; // 5
        channel_register |= (config().channelSettings(i).sd << sequence);   sequence += 5; // [6,10]
        channel_register |= (config().channelSettings(i).smx << sequence);  sequence++; // 11
        channel_register |= (config().channelSettings(i).sz10b << sequence); sequence += 5; // [12:16]
        channel_register |= (config().channelSettings(i).sz8b << sequence); sequence += 4; // [17:20]
        channel_register |= (config().channelSettings(i).sz6b << sequence); sequence +=3; //[21:24]
        channels.push_back(channel_register);
    } // i
*/
    

}
// ------------------------------------------------------------------------ //
void Configuration::SendConfig_VMM2()
{

    cout << "Configuration::SendConfig" << endl;
    stringstream sx;
    bool send_ok = true;
    bool ok;

    int send_to_port = 6008;

    ////////////////////////////////////////////////////
    // GLOBAL
    ////////////////////////////////////////////////////
    vector<QString> globalRegisters;
    fillGlobalRegisters_VMM2(globalRegisters);
    if(globalRegisters.size()!=3) {
        sx << "ERROR Global registers in VMM SPI is not length 3, unable to send SPI configuration";
        msg()(sx,"Configuration::SendConfig_VMM2");sx.str("");
        return;
    }

    ////////////////////////////////////////////////////
    // CHANNEL
    ////////////////////////////////////////////////////
    vector<QString> channelRegisters;
    fillChannelRegisters_VMM2(channelRegisters);
    if(channelRegisters.size()!=64) {
        sx << "ERROR Channel registers in VMM SPI is not 64 channels long, unable to send SPI configuration";
        msg()(sx,"Configuration::SendConfig_VMM2");sx.str("");
        return;
    }

    ////////////////////////////////////////////////////
    // now prepare to send the word, worrrdd
    ////////////////////////////////////////////////////

    QByteArray datagram;
    QDataStream out(&datagram, QIODevice::WriteOnly);
    out.device()->seek(0);

    int ip_idx = -1;
    for(const auto & ip : socket().ipList()) {
        // configure only selected boards
        ip_idx++;
        if(config().boardSelection()>=0) {
            if(config().boardSelection() != ip_idx) {
                continue;
            }
        }

        int vmm_mask = config().vmmMap().mask;
        for(int i = 0; i < 1; i++) { 
            int tmp = 1;
            tmp = vmm_mask;
            if(vmm_mask & tmp) {
                datagram.clear();
                QDataStream out(&datagram, QIODevice::WriteOnly);
                out.device()->seek(0);

                // update count
                socket().updateCommandCounter();
                if(dbg()) {
                    sx << "Sending command SPI at command #: " << socket().commandCounter();
                    msg()(sx,"Configuration::SendConfig_VMM2"); sx.str("");
                }

                QString h1 = "ffbb";
                QString h2 = "aadd";
                QString h3 = "bb00";

                out << (quint16)h1.toUInt(&ok,16)
                    << (quint16)h2.toUInt(&ok,16)
                    << (quint16)(vmm_mask)
                    << (quint16)h3.toUInt(&ok,16);


                // SEND CHANNELS

                for(int ichan = 63; ichan >= 0; ichan--) {
                    // only send 24bits per channel with current MMFE8 VMM2 FW (i.e. no padding)
                    QString first8bits = channelRegisters[ichan].mid(8,8);
                    QString second16bits = channelRegisters[ichan].mid(16,16);
                    out << (quint16)reverseString(second16bits).toUInt(&ok,2);
                    out << (quint8)reverseString(first8bits).toUInt(&ok,2);
                } // i

                // SEND GLOBAL
                
                for(size_t ig = 0; ig < 3; ig++) {
                    out << (quint32)Configuration::reverse32(globalRegisters.at(ig));
                }

                bool readOK = true;
                send_ok = socket().SendDatagram(datagram, ip, send_to_port, "fec",
                                "Configuration::SendConfig_VMM2");

                if(send_ok) {
                    readOK = socket().waitForReadyRead("fec");
                    if(readOK) {
                        socket().processReply("fec", ip);
                    }
                    else {
                        sx.str("");
                        sx << "Timeout while waiting for replies from VMM " << i;
                        msg()(sx,"Configuration::SendConfig_VMM2");sx.str("");
                        socket().closeAndDisconnect("fec","Configuration::SendConfig");
                    }
                }
            } // vmm mask is good
        } // i
    } // ip

}
// ------------------------------------------------------------------------ //
void Configuration::fillGlobalRegisters_VMM2(vector<QString>& global)
{
    stringstream sx;
    global.clear();
    int sequence = 0;

    //////////////////////////////////////////////////////
    // SPI 0
    //////////////////////////////////////////////////////
    QString spi0 = "00000000000000000000000000000000"; 
    QString tmp;

    //10bit ADC
    tmp = QString("%1").arg(config().globalSettings().sc10b,2,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().adc_10bit,2,2,QChar('0'));
    spi0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    //8bit ADC
    tmp = QString("%1").arg(config().globalSettings().sc8b,2,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().adc_8bit,2,2,QChar('0'));
    spi0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    //6bit
    tmp = QString("%1").arg(config().globalSettings().sc6b,3,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().adc_6bit,3,2,QChar('0'));
    spi0.replace(sequence, tmp.size(), tmp);
    sequence += tmp.size();

    //8-bit enable
    spi0.replace(sequence,1,QString::number(config().globalSettings().s8b));
    //spi0.replace(sequence,1,QString::number(config().globalSettings().conv_mode_8bit));
    sequence++;

    //6-bit enable
    spi0.replace(sequence,1,QString::number(config().globalSettings().s6b));
    //spi0.replace(sequence,1,QString::number(config().globalSettings().enable_6bit));
    sequence++;

    //ADC enable
    spi0.replace(sequence,1,QString::number(1)); // just hard code this -- it has never been anything else and there is no UI to change it
    //spi0.replace(sequence,1,QString::number(config().globalSettings().adcs));
    sequence++;

    //dual clock serialized
    spi0.replace(sequence,1,QString::number(config().globalSettings().sdcks));
    sequence++;

    //dual clock ART
    spi0.replace(sequence,1,QString::number(config().globalSettings().sdcka));
    //spi0.replace(sequence,1,QString::number(config().globalSettings().dual_clock_art));
    sequence++;

    //dual clock 6-bit
    spi0.replace(sequence,1,QString::number(config().globalSettings().sdck6b));
    sequence++;

    //analog tri-states
    spi0.replace(sequence,1,QString::number(config().globalSettings().sdrv));
    sequence++;

    //timing out 2
    spi0.replace(sequence,1,QString::number(config().globalSettings().stpp));
    //spi0.replace(sequence,1,QString::number(config().globalSettings().direct_time_mode0));

    if(m_dbg)
    {
        sx.str("");
        sx << "SPI[0]: "<<spi0.toStdString();
        msg()(sx,"Configuration::fillGlobalRegisters");
    }

    global.push_back(spi0);
    
    
    //////////////////////////////////////////////////////////////////////
    // GLOBAL SPI 1
    //////////////////////////////////////////////////////////////////////

    QString spi1 = "00000000000000000000000000000000"; 
    sequence = 0;

    //peak_time
    // [0,1]
    tmp = QString("%1").arg(config().globalSettings().st,2,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().peak_time,2,2,QChar('0'));
    spi1.replace(sequence, tmp.size(),tmp);
    sequence += tmp.size();

    //double leakage (doubles the leakage current)
    // [2]
    spi1.replace(sequence,1,QString::number(config().globalSettings().sfm));
    //spi1.replace(sequence,1,QString::number(config().globalSettings().double_leakage));
    sequence++;

    //gain
    // [3,5]
    tmp = QString("%1").arg(config().globalSettings().sg,3,2,QChar('0'));
    spi1.replace(sequence,tmp.size(),tmp);
    sequence += tmp.size();

    //neighbor trigger
    // [6]
    spi1.replace(sequence,1,QString::number(config().globalSettings().sng));
    sequence++;

    //direct outputs settings
    // [7]
    spi1.replace(sequence,1,QString::number(config().globalSettings().stot));
    sequence++;

    //direct timing
    // [8]
    spi1.replace(sequence,1,QString::number(config().globalSettings().sttt));
    sequence++;

    //sub-hysteresis
    // [9]
    spi1.replace(sequence,1,QString::number(config().globalSettings().ssh));
    sequence++;

    //TAC slope adjustment
    // [10,11]
    tmp = QString("%1").arg(config().globalSettings().stc,2,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().tac_slope,2,2,QChar('0'));
    spi1.replace(sequence,tmp.size(),tmp);
    sequence += tmp.size();

    //threshold DAC
    // [12,21]
    tmp = QString("%1").arg(config().globalSettings().sdt_dac,10,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().threshold_dac,10,2,QChar('0'));
    spi1.replace(sequence,tmp.size(),tmp);
    sequence += tmp.size();

    //pulse DAC
    // [22,31]
    tmp = QString("%1").arg(config().globalSettings().sdp_dac,10,2,QChar('0'));
    //tmp = QString("%1").arg(config().globalSettings().test_pulse_dac,10,2,QChar('0'));
    spi1.replace(sequence,tmp.size(),tmp);
    sequence += tmp.size();

    cout << "Configuration::fillGlobalRegisters_VMM2    sdp_dac = " << config().globalSettings().sdp_dac << endl;

    if(m_dbg)
    {
        sx.str("");
        sx << "SPI[1]: " << spi1.toStdString();
        msg()(sx,"Configuration::fillGlobalRegisters");
    }

    global.push_back(spi1);

    //////////////////////////////////////////////////////////////////////
    // GLOBAL SPI 2
    //////////////////////////////////////////////////////////////////////

    QString spi2 = "00000000000000000000000000000000"; 
    sequence = 16;

    //polarity
    //[16]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sp));
    sequence++;

    //disable at peak
    //[17]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sdp));
    sequence++;

    //analog monitor to pdo
    //[18]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sbmx));
    sequence++;

    //tdo buffer
    //[19]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sbft));
    sequence++;

    //pdo buffer
    //[20]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sbfp));
    sequence++;

    //mo buffer
    //[21]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sbfm));
    sequence++;

    //leakage current
    //[22]
    spi2.replace(sequence,1,QString::number(config().globalSettings().slg));
    sequence++;

    //channel to monitor
    //[23,28]
    tmp = QString("%1").arg(config().globalSettings().sm5,6,2,QChar('0'));
    spi2.replace(sequence,tmp.size(),tmp);
    sequence += tmp.size();

    //multiplexer
    //[29]
    spi2.replace(sequence,1,QString::number(config().globalSettings().scmx));
    sequence++;

    //ART enable
    //[30]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sfa));
    sequence++;

    //ART mode
    //[31]
    spi2.replace(sequence,1,QString::number(config().globalSettings().sfam));
    sequence++;

    if(m_dbg)
    {
        sx.str("");
        sx << "SPI[2]: "<<spi2.toStdString();
        msg()(sx,"Configuration::fillGlobalRegisters");
    }

    global.push_back(spi2);

}
// ------------------------------------------------------------------------ //
void Configuration::fillChannelRegisters_VMM2(vector<QString>& registers)
{
    stringstream sx;

    registers.clear();
    int sequence;
    QString tmp;
    QString reg;

    bool do_check = false;

    for(int i = 0; i < 64; ++i) {
        sequence=8;
        reg = "00000000000000000000000000000000";

        //SP [8]
        // for VMM2 use STH bits from the UI
        reg.replace(sequence,1,QString::number(config().channelSettings(i).sth));
        sequence++; 
        if(do_check) std::cout << " SP : " << reg.toStdString() << std::endl;

        //SC [9]
        reg.replace(sequence,1,QString::number(config().channelSettings(i).sc));
        //reg.replace(sequence,1,QString::number(config().channelSettings(i).capacitance));
        sequence++;
        if(do_check) std::cout << " SC : " << reg.toStdString() << std::endl;

        //SL [10]
        reg.replace(sequence,1,QString::number(config().channelSettings(i).sl));
        //reg.replace(sequence,1,QString::number(config().channelSettings(i).leakage_current));
        sequence++;
        if(do_check) std::cout << " SL : " << reg.toStdString() << std::endl;

        //ST [11]
        reg.replace(sequence,1,QString::number(config().channelSettings(i).st));
        sequence++;
        if(do_check) std::cout << " ST : " << reg.toStdString() << std::endl;

        //SM [12]
        reg.replace(sequence,1,QString::number(config().channelSettings(i).sm));
        sequence++;
        if(do_check) std::cout << " SM : " << reg.toStdString() << std::endl;

        //trim [13,16]
        tmp = "0000";
        tmp = QString("%1").arg(config().channelSettings(i).sd,4,2,QChar('0'));
        std::reverse(tmp.begin(),tmp.end()); //bug in VMM2, needs to be reversed
        reg.replace(sequence, tmp.size(), tmp);
        sequence += tmp.size();
        if(do_check) std::cout << " TRIM : " << reg.toStdString() << std::endl;

        //SMX [17]
        reg.replace(sequence,1,QString::number(config().channelSettings(i).smx));
        sequence++;
        if(do_check) std::cout << " SMX : " << reg.toStdString() << std::endl;

        //10 bit adc lsb [18,22]
        tmp = QString("%1").arg(config().channelSettings(i).sz10b,5,2,QChar('0'));
        reg.replace(sequence,tmp.size(),tmp);
        sequence += tmp.size();
        if(do_check) std::cout << " 10bit : " << reg.toStdString() << std::endl;

        //8 bit adc lsb [23,26]
        tmp = QString("%1").arg(config().channelSettings(i).sz8b,4,2,QChar('0'));
        reg.replace(sequence,tmp.size(),tmp);
        sequence += tmp.size();
        if(do_check) std::cout << " 8bit : " << reg.toStdString() << std::endl;

        //6 bit adc lsb [27,29]
        tmp = QString("%1").arg(config().channelSettings(i).sz6b,3,2,QChar('0'));
        reg.replace(sequence,tmp.size(),tmp);
        sequence += tmp.size();
        if(do_check) std::cout << " 6bit : " << reg.toStdString() << std::endl;
        
        if(m_dbg) {
            using boost::format; 
            std::stringstream chan;
            chan.str("");
            chan << " Chan["<< format("%02i") % i <<"]: " << reg.toStdString();
            msg()(chan, "Configuration::fillChannelRegisters");
            //chan << format("%02i") % i;
            //
            //std::cout << "-----------------------------------------------" << std::endl;
            //std::cout << " Channel [" << chan.str() << "] register "
            //     << reg.toStdString() << std::endl;
        } 

        registers.push_back(reg);

    } // i

} 
// ------------------------------------------------------------------------ //
std::string Configuration::configureBoardIP(std::string current_ip, std::string new_ip,
                                                std::string daq_ip)
{


    //#warning HARDCODING PORT
    int port_to_send_to = 6604;

    ////////////////////////////////////////////////////////
    // check that all of the IP's are valid
    ////////////////////////////////////////////////////////
    QStringList current_ip_split;
    current_ip_split << QString::fromStdString(current_ip).split(".");
    std::vector<std::string> current_ip_values;
    for(auto sp : current_ip_split) {
        current_ip_values.push_back(sp.toStdString());
    }

    std::string status = checkIpFormat(current_ip, current_ip_values, "INPUT");
    if(status!="")
        return status;

    QStringList new_ip_split;
    new_ip_split << QString::fromStdString(new_ip).split(".");
    std::vector<std::string> new_ip_values;
    for(auto sp : new_ip_split) {
        new_ip_values.push_back(sp.toStdString());
    }

    status = checkIpFormat(new_ip, new_ip_values, "OUTPUT");
    if(status!="")
        return status;

    QStringList daq_ip_split;
    daq_ip_split << QString::fromStdString(daq_ip).split(".");
    std::vector<std::string> daq_ip_values;
    for(auto sp : daq_ip_split) {
        daq_ip_values.push_back(sp.toStdString());
    }

    status = checkIpFormat(daq_ip, daq_ip_values, "DAQ");
    if(status!="")
        return status;

    ////////////////////////////////////////////////////////
    // now send the command
    ////////////////////////////////////////////////////////

    ////////////////////////////////////////////////////////
    // register addresses
    ////////////////////////////////////////////////////////
    QString address_dest_ip         = "B1";
    QString address_source_ip       = "B2";
    QString address_source_mac_1    = "B3";
    QString address_source_mac_2    = "B4";


    QString first_32 = "AAFF"; // for setting board IP do not care about serial number (currently)
    QString second_32 = "AAFF"; // second 32 bits are not used by FW

    //////////////////////////////////////////////////////
    // build up the packet
    //////////////////////////////////////////////////////
    QByteArray datagram;
    datagram.clear();
    QDataStream out (&datagram, QIODevice::WriteOnly);
    out.device()->seek(0);

  
    bool ok;
                                                            // byte range
    // header
    out << (quint32)(first_32.toUInt(&ok,16))               // [0,3]
        << (quint32)(second_32.toUInt(&ok,16));             // [4,7]

    // destination IP (this is the IP of the DAQ)
    out << (quint32)(address_dest_ip.toUInt(&ok,16))        // [8,11]
        << (quint8)(daq_ip_split[0].toUInt(&ok,10))
        << (quint8)(daq_ip_split[1].toUInt(&ok,10))
        << (quint8)(daq_ip_split[2].toUInt(&ok,10))
        << (quint8)(daq_ip_split[3].toUInt(&ok,10));        // [12,15]

    // new IP of board
    out << (quint32)(address_source_ip.toUInt(&ok,16))      // [16,19]
        << (quint8)(new_ip_split[0].toUInt(&ok,10))
        << (quint8)(new_ip_split[1].toUInt(&ok,10))
        << (quint8)(new_ip_split[2].toUInt(&ok,10))
        << (quint8)(new_ip_split[3].toUInt(&ok,10));        // [20,23]

    // MAC is based on new IP of board, first 32 bits are dummy
    // new MAC of board (1)
    out << (quint32)(address_source_mac_1.toUInt(&ok,16))   // [24,27]
        << (quint16)0
        << (quint16)0xFAFA;                                 // [28,31]

    out << (quint32)(address_source_mac_2.toUInt(&ok,16))   // [32,35]
        << (quint8)(new_ip_split[0].toUInt(&ok,10))
        << (quint8)(new_ip_split[1].toUInt(&ok,10))
        << (quint8)(new_ip_split[2].toUInt(&ok,10))
        << (quint8)(new_ip_split[3].toUInt(&ok,10));        // [36,39]


    /*
    msg()("IP Configuration is in non-standard form for testing purposes!");

    // for tests
    QString b1 = "B1";
    QString b2 = "B2";
    QString b3 = "B3";
    QString b4 = "B4";
    QString b5 = "B5";
    QString b6 = "B6";
    QString b7 = "B7";

    // header
    out << (quint32)(first_32.toUInt(&ok,16))
        << (quint32)(second_32.toUInt(&ok,16));

    // b1
    out << (quint32)(b1.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(daq_ip_split[0].toUInt(&ok,10))
        << (quint8)(daq_ip_split[1].toUInt(&ok,10));

    // b2
    out << (quint32)(b2.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(daq_ip_split[2].toUInt(&ok,10))
        << (quint8)(daq_ip_split[3].toUInt(&ok,10));

    // b3
    out << (quint32)(b3.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(new_ip_split[0].toUInt(&ok,10))
        << (quint8)(new_ip_split[1].toUInt(&ok,10));

    // b4
    out << (quint32)(b4.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(new_ip_split[2].toUInt(&ok,10))
        << (quint8)(new_ip_split[3].toUInt(&ok,10));

    // b5
    out << (quint32)(b5.toUInt(&ok,16))
        << (quint16)0
        << (quint16)0xFAFA;

    // b6
    out << (quint32)(b6.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(new_ip_split[0].toUInt(&ok,10))
        << (quint8)(new_ip_split[1].toUInt(&ok,10));

    // b7
    out << (quint32)(b7.toUInt(&ok,16))
        << (quint16)0
        << (quint8)(new_ip_split[2].toUInt(&ok,10))
        << (quint8)(new_ip_split[3].toUInt(&ok,10));
    */
    // don't send the word yet until format is confirmed
    //qDebug() << "Board IP configuration packet: " << datagram.toHex();
    //status = "IP allocation not enabled. Board IP configuration format not confirmed. Not sending configuration packet.";
    //return status;
    
    socket().SendDatagram(datagram, QString::fromStdString(current_ip), port_to_send_to, "fec",
                                                "Configuration::configureBoardIP");
    return "";


}
// ------------------------------------------------------------------------ //
std::string Configuration::checkIpFormat(std::string full_ip, std::vector<std::string> ip_values, std::string from)
{

    std::stringstream sx;
    std::string output_status = "";

    //std::string from = (input_or_output ? "Input" : "Output");

    // check format
    for(auto ip : ip_values) {
        if(ip=="") {
            sx << from << " IP (" << full_ip << ") has unset value. Unable to configure IP.";
            output_status = sx.str(); sx.str("");
            return output_status;
        }
    }

    for(auto sp : ip_values) {
        try {
            int value = std::stoi(sp);
            if(!(value>=0 && value<256)) {
                sx << from << " IP (" << full_ip << ") has invalid value (outside of [0,255]). Unable to configure board IP.";
                output_status = sx.str(); sx.str("");
                return output_status;
            }
        }
        catch(std::exception& e) {
            sx << from << " IP (" << full_ip << ") has non-integer value set. Unable to configure board IP.";
            output_status = sx.str(); sx.str("");
            return output_status;
        }
    }

    if(ip_values.at(0)!="") {
        if(std::stoi(ip_values.at(0))==0) {
            sx << from << " IP (" << full_ip << ") is not appropriately defined. First"
               << " number is zero. Unable to configure board IP.";
            output_status = sx.str(); sx.str("");
            return output_status;
        }
    }

    bool negative_found = false;
    for(auto ip : ip_values) {
        if(ip!="") {
            if(std::stoi(ip)<0) { negative_found = true; break; }
        }
    }
    if(negative_found) {
        sx << from << " IP (" << full_ip << ") has negative value set. Unable to configure board IP."; 
        output_status = sx.str(); sx.str("");
        return output_status;
    }

    bool not_all_zero = false;
    for(auto ip : ip_values) {
        if(ip!="") {
            if(std::stoi(ip)>0) { not_all_zero = true; break; }
        }
    }
    if(!not_all_zero) {
        sx << from << " IP (" << full_ip << ") has not been set. Unable to configure board IP.";
        output_status = sx.str(); sx.str("");
    }

    return output_status;

}

