#include "daq_server.h"
#include <string>
#include <iostream>
using namespace std;

//nsw
#include "map_handler.h"
#include "message_handler.h"
#include "daq_buffer.h"



namespace ip = boost::asio::ip;

boost::mutex global_stream_lock;

DaqServer::DaqServer(QObject* parent) :
    QObject(parent),
    m_msg(NULL),
    m_dbg(false),
    m_vmm_type(3),
    m_do_l0(false),
    m_use_pulser_calib(false),
    m_continue_gathering(new bool()),
    m_flushing(new bool()),
    m_continue_calibration_gathering(new bool()),
    m_daq_port(6008),
    m_calibration_port(6008),
    m_is_calibration_run(false),
    m_run_number(0),
    m_total_events_to_process(-1),
    m_tree_flush(-1),
    n_input_packets(0),
    n_input_empty_packets(0),
    m_do_occupancy_check(false),
    m_last_occ_check(0),
    m_empty_event(0),
    m_full_event(0),
    m_trigger_delta(new int()),
    m_trigger_board_delta(new int()),
    do_flush(true),
    m_do_eff(false),
    n_effCount(0),
    reported(false),
    n_pulse_limit(-1), // -1 for ifinite
    n_daqCount(0),
    n_trigCount(0),
    n_cal_check(0),
    m_current_caltype(CalibrationType::Invalid),
    n_received(0),
    m_thread_count(2), // 2 threads: one producer, one consumer
    m_monitor_status(false),
    m_monitor(0),
    m_socket(NULL),
    m_io_service(NULL),
    m_idle_work(NULL),
    //DaqBuffer
    m_buffer(NULL),
    m_set_daq_monitor(false)
{
    m_sx.str("");
    m_monitor = new bool();
    *m_monitor = false;

    m_accept_pulser_data = new bool();
    *m_accept_pulser_data = true;

    // initialize the io_service here so that we have it at the beginning
    // (we need it regardless of starting a run or not if we are to setup the monitoring
    // independently of running status)
    // config
    //m_io_service = boost::shared_ptr<boost::asio::io_service>(new boost::asio::io_service());

}
void DaqServer::set_vmm_type(int type)
{
    m_vmm_type = type;
    m_event_builder->set_vmm_type(type);
    m_calibration_module->set_vmm_type(type);
}
void DaqServer::LoadMessageHandler(MessageHandler& msg)
{
    m_msg = &msg;
    m_event_builder->LoadMessageHandler(*m_msg);
    if(m_calibration_module)
        m_calibration_module->LoadMessageHandler(*m_msg);
    if(m_daqMonitor && m_set_daq_monitor)
        m_daqMonitor->LoadMessageHandler(*m_msg);
}
void DaqServer::LoadModulesForCalibration(Configuration& config, RunModule& run) //, SocketHandler& socket)
{
    m_calibration_module->LoadModulesForCalibration(config, run);//, socket);
}
bool DaqServer::load_pulser_calib(PulserCalibDef def)
{
    m_calibration_module->load_pulser_calib_flags(m_accept_pulser_data);
    return m_calibration_module->load_pulser_calib(def);
}
bool DaqServer::load_pulser_calib_run_properties(PulserCalibRunProperties props)
{
    return m_calibration_module->load_pulser_calib_run_properties(props);
}
void DaqServer::send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& vmmMap,
        std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq)
{
    m_calibration_module->send_initial_config_to_pulser_calib(global, vmmMap,
        channels, clocks, daq);
}
void DaqServer::setDebug(bool dbg)
{
    m_dbg = dbg;
    m_event_builder->setDebug(dbg);
    if(m_calibration_module) m_calibration_module->setDebug(dbg);
}

void DaqServer::setDoMonitoring(bool do_mon)
{
    #warning set up monitoring flags!
}
void DaqServer::setCalibrationRun(bool is_calib_run)
{
    m_event_builder->setCalibrationRun(is_calib_run);
}

void DaqServer::setCalibrationChannel(int channel)
{
    m_event_builder->setCalibrationChannel(channel);
}
void DaqServer::updateCalibrationState(double gain, int dacThreshold, int dacAmplitude,
            double tp_skew, int peakTime)
{
    m_event_builder->updateCalibrationState(gain, dacThreshold, dacAmplitude, tp_skew, peakTime);
}

//void DaqServer::loadCalibrationState(CalibrationState state)
void DaqServer::loadCalibrationState(int channel, int gain, int run_number,
        int tac_slope, int neighbor_trigger, int subhyst, int peak_time,
        int test_pulse_dac, int thresh_dac, std::vector<int> trims, double tp_skew, int n_expected_pulses)
{
    m_calibration_module->load_calibration_state(channel, gain, run_number, tac_slope, neighbor_trigger,
                subhyst, peak_time, test_pulse_dac, thresh_dac, trims, tp_skew, n_expected_pulses);
    //emit load_calibration(channel, gain, run_number, tac_slope, neighbor_trigger,
    //            subhyst, peak_time, test_pulse_dac, thresh_dac);
    //emit load_calibration(state);
    //m_calibration_module->loadCalibrationState(state);
}
void DaqServer::loadMappingTool(MapHandler& maptool)
{
    m_event_builder->loadMappingTool(maptool);
}
void DaqServer::disableMapping()
{
    m_event_builder->disableMapping();
}

void DaqServer::loadMonitoringTool(OnlineMonTool& montool)
{
    m_event_builder->loadMonitoringTool(montool);
}

void DaqServer::setMonitoringStatus(bool status)
{
    m_monitor_status = status;
    m_event_builder->setMonitoringStatus(status);

    // monitoring
    //config
//    if(m_monitor_status) {
//        if(is_stopped()) {
//            //std::cout << "DaqServer::setMonitoringStatus    io service is stopped..." << std::endl;
//            m_io_service = boost::shared_ptr<boost::asio::io_service>(new boost::asio::io_service());
//        }
//        m_event_builder->initializeMonitoring(m_io_service);
//
//    }

}

void DaqServer::recv_setup_monitoring(bool do_monitoring)
{
    *m_monitor = do_monitoring;
    if(dbg())
        m_event_builder->print_mon_flag();
}

void DaqServer::initialize()
{
    m_event_builder = boost::shared_ptr<EventBuilder>(new EventBuilder());
    m_event_builder->load_mon_flag(m_monitor);

    //dataflow
    (*m_trigger_delta) = 0;
    (*m_trigger_board_delta) = 0;
    m_event_builder->get_delta_params(m_trigger_delta, m_trigger_board_delta);

    m_buffer = boost::shared_ptr<DaqBuffer>(new DaqBuffer());

    m_calibration_module = new CalibModule(this);
    m_calibration_module->set_vmm_type(vmm_type());
    connect(this, SIGNAL(start_xadc(QString)), m_calibration_module, SLOT(start_xadc_sampling(QString)), Qt::DirectConnection);
    connect(this, SIGNAL(initializeCalibrationSignal(int, int, int, int, bool, bool)), m_calibration_module, SLOT(initialize_calibration(int, int, int, int, bool, bool)), Qt::DirectConnection);
    connect(this, SIGNAL(resetCalibrationCounts()), m_calibration_module, SLOT(reset_counts()));
    connect(this, SIGNAL(begin_pulser_calibration_signal()), m_calibration_module, SLOT(begin_pulser_calibration()));
    connect(m_calibration_module, SIGNAL(pulser_calib_complete_signal()), this, SLOT(pulser_calib_complete()));

    // daq monitor for checking hanging calibration run
    m_daqMonitor = new DaqMonitor();
    connect(this, SIGNAL(stopTimer()), m_daqMonitor, SLOT(stopTimer()));
    connect(m_daqMonitor, SIGNAL(daqHangObserved()), this, SLOT(sendDaqHangSignal()));
    connect(m_daqMonitor, SIGNAL(toggleCalibrationSocket()), this, SLOT(toggleCalibrationSocket()));

}
void DaqServer::sendDaqHangSignal()
{
    emit daqHangObserved();
}

void DaqServer::set_cktp_limit(int pulse_limit)
{
    n_pulse_limit = pulse_limit;
}

void DaqServer::update_monitoring_rate(int rate)
{
    if(!m_event_builder) return;
    m_event_builder->set_monitor_rate(rate);
}

bool DaqServer::initializeRun(bool writeRaw, bool writeNtuple, std::string filename, int run_number, int num_events_to_process, int tree_flush, bool do_calibration, bool do_L0_decoding, bool is_xadc)
{
    *m_continue_gathering = true;
    *m_flushing = false;
    m_is_calibration_run = do_calibration;

    m_do_l0 = do_L0_decoding;

    if(dbg()) {
        m_sx << "[" << boost::this_thread::get_id() << "] initializing run " << run_number;
        msg()(m_sx, "DaqServer::initializeRun"); m_sx.str("");
    }


    if(do_calibration) {
        *m_continue_calibration_gathering = true;
        *m_flushing = false;
        m_daq_port = m_calibration_port;
    }
    // if we have stopped a run previously, then the io_service was stopped
    // restart it if that is the case
    //config
//    if(is_stopped()) {
//        m_io_service = boost::shared_ptr<boost::asio::io_service>(new boost::asio::io_service());
//
//        // and give the newly started io_service to the monitoring tool
//        if(m_monitor_status)
//            m_event_builder->initializeMonitoring(m_io_service);
//    }

    
    //if(!m_io_service)
        m_io_service = boost::shared_ptr<boost::asio::io_service>(new boost::asio::io_service());

    ///cout << "WARNING BINDING TO LOCAL LOOPPACK PORT" << endl;
    ///int port_to_bind_to = 2226;
    int port_to_bind_to = (m_is_calibration_run ? m_calibration_port : m_daq_port);
#ifndef __linux__
    m_idle_work = boost::shared_ptr<boost::asio::io_service::work>(new boost::asio::io_service::work(*m_io_service));
#else
    m_idle_work = boost::shared_ptr<boost::asio::io_service::work>(new boost::asio::io_service::work(*m_io_service));

#endif

    if(m_socket) {
        if(m_socket->is_open()) {
            m_socket->close();
            boost::system::error_code ec;
            m_socket->shutdown(ip::udp::socket::shutdown_both, ec);
        }
    }
    m_socket = boost::shared_ptr<boost::asio::ip::udp::socket>(new ip::udp::socket(*m_io_service, ip::udp::endpoint(ip::udp::v4(), port_to_bind_to)));
    if(!m_socket->is_open()) {
        m_sx << "ERROR DAQ socket not setup up at requested port: " << m_daq_port;
        msg()(m_sx,"DaqServer::initializeRun"); m_sx.str("");
        return false;
    }

    //if(m_monitor_status)
    //    m_event_builder->initializeMonitoring(m_io_service);
    // initalize the monitoring tool always
    m_event_builder->initializeMonitoring(m_io_service);

    // DaqBuffer
    m_buffer->initialize(1000);

    
    // initialize output
    if(!do_calibration) {
        if(!m_event_builder->initializeRun(writeRaw, writeNtuple, filename, run_number, tree_flush, do_L0_decoding))
            return false;
    }
    //#warning not handing calibration module the IO service (needed for xADC!)
    else {
        m_calibration_module->initializeSocket(m_io_service);
        if(!m_calibration_module->setOutputFile(filename, run_number, is_xadc))
            return false;
    }

    // event filling conditions
    //m_mutex = boost::shared_ptr<boost::timed_mutex>(new boost::timed_mutex());
    //m_fill_condition = boost::shared_ptr<boost::condition_variable_any>(new boost::condition_variable_any());
    //m_event_builder->get_sync_items(m_mutex, m_fill_condition);

    m_run_number = run_number;
    m_total_events_to_process = num_events_to_process; 
    m_tree_flush = tree_flush;

    return true;

}
void DaqServer::toggleCalibrationSocket()
{
    m_calibration_module->toggleSocket();
}

bool DaqServer::initializeCalibration(int vmm_id, int n_samples, int sample_period, CalibrationType  type, bool skip_dead_channels, bool do_l0_decoding)
{
    m_use_pulser_calib = false;
    m_do_l0 = do_l0_decoding;

    m_is_calibration_run = true;
    bool ok = true;

    m_current_caltype = type;

    if(m_calibration_module) {
        msg()(m_sx, "DaqServer::initializeCalibration");

        //m_calibration_module->InitializeCalibration(m_is_calibration_run, vmm_id, n_samples, sample_period, type);
        m_calibration_module->initialize_calibration(vmm_id, n_samples, sample_period, CalibrationType2Int(type), skip_dead_channels, do_l0_decoding);
        //emit initializeCalibrationSignal(vmm_id, n_samples, sample_period, CalibrationType2Int(type), skip_dead_channels, do_l0_decoding);

        if(type==CalibrationType::GlobalThresholdDAC || type==CalibrationType::ChannelTrim
                || type==CalibrationType::GlobalTestPulseDAC || type==CalibrationType::ChannelBaseline) {
            m_total_events_to_process = (vmm_id == 0 ? (8*n_samples) : n_samples);
        }
        else {
            int multiplier = 1;
            m_total_events_to_process = multiplier*n_samples;

            if(type==CalibrationType::Efficiency) {
                m_do_eff = true;
            }
            else {
                m_do_eff = false;
            }
        }
    }
    else {
        ok = false;
        m_sx << "[" << boost::this_thread::get_id() << "] Attempting a calibration run but CalibModule object is not initialized!";
        msg()(m_sx, "DaqServer::initializeRun"); m_sx.str("");
        return false;
    }
    if(ok) {
        *m_continue_calibration_gathering = true;

    }
    return true;
}

void SPSCThread(boost::shared_ptr< boost::asio::io_service> io_service)
{
    //global_stream_lock.lock();
    //std::cout << "DaqServer::SPSCThread    thread start [" << boost::this_thread::get_id() << "]" << std::endl;
    //global_stream_lock.unlock();

    io_service->run();

    //global_stream_lock.lock();
    //std::cout << "DaqServer::SPSCThread    thread finish [" << boost::this_thread::get_id() << "]" << std::endl;
    //global_stream_lock.unlock();

}
void DaqServer::fillRunProperties(int gain, int tac_slope, int peak_time, int dac_threshold,
        int dac_amplitude, int angle, int tp_skew, int ckbc)
{
    m_event_builder->fillRunProperties(gain, tac_slope, peak_time, dac_threshold,
                            dac_amplitude, angle, tp_skew, ckbc);
}
void DaqServer::fillRunComment(std::string comment)
{
    if(!m_is_calibration_run)
        m_event_builder->fillRunComment(comment);
    else if(m_is_calibration_run && m_calibration_module) {
        m_calibration_module->fillRunComment(comment);
    }
}

void DaqServer::startMonitoring()
{
    m_event_builder->startMonitoring();
}
void DaqServer::stopMonitoring()
{
    m_event_builder->stopMonitoring();
}

void DaqServer::listen()
{
    if(dbg()) msg()("IO service posting...","DaqServer::listen");

    // reset
    if(!m_io_service->stopped())
        m_io_service->stop();
    m_io_service->reset();

    for(int i = 0; i < m_thread_count; i++) {
        m_thread_group.create_thread(boost::bind(SPSCThread, m_io_service));
    }

    m_io_service->post(boost::bind(&DaqServer::handle_data, this));
    m_io_service->post(boost::bind(&DaqServer::read_data, this, boost::ref(n_daqCount),
                                                                    boost::ref(n_trigCount)));

    // daq mon
    if(m_daqMonitor && m_set_daq_monitor)
        m_daqMonitor->setCounter(n_daqCount);

    // time
    m_run_time = QTime::currentTime();
    

}
bool DaqServer::begin_pulser_calibration()
{
    emit begin_pulser_calibration_signal();
    m_is_calibration_run = true;
    m_use_pulser_calib = true;
    return true;
    //return m_calibration_module->begin_pulser_calibration();
}
void DaqServer::pulser_calib_complete()
{
    stringstream sx;
    sx << " *** Pulser calibration complete ***";
    msg()(sx); sx.str("");
    emit pulser_calib_complete_signal();
}
void DaqServer::start_xadc_sampling(std::string ip)
{
    if(m_is_calibration_run && m_socket->is_open()) {
        //emit start_xadc(QString::fromStdString(ip));
        m_calibration_module->start_xadc_sampling(QString::fromStdString(ip));
        if(m_daqMonitor && m_set_daq_monitor) {
            m_daqMonitor->startTimer();
        }
    }
}

void DaqServer::resetCounts()
{
    reported = false;
    n_effCount = 0;
    n_daqCount = 0;
    n_trigCount = 0;
    n_input_packets = 0;
    n_input_empty_packets = 0;
    n_received = 0;
    m_event_builder->resetCounts();
    m_calibration_module->reset_counts();
    emit updateCounts(0, 0, 0, 0, false);
}
int DaqServer::getCounts()
{
    return n_daqCount;
}
int DaqServer::getTrigCounts()
{
    return n_trigCount;
}

void DaqServer::handle_data()
{
    //DaqBuffer
    //if( ( (daq_counter % 100) == 0) ) {
    //    emit updateCounts(daq_counter, false);
    //}
    m_socket->async_receive_from(
        boost::asio::buffer(m_data_buffer), m_remote_endpoint,
        boost::bind(&DaqServer::gather_data, this,
        boost::asio::placeholders::error,
        boost::asio::placeholders::bytes_transferred)
    );
}

void DaqServer::toggle_occupancy_check(bool do_check)
{
    m_do_occupancy_check = do_check;
}

void DaqServer::gather_data(const boost::system::error_code error, std::size_t size_)
{
    if(!(*m_continue_gathering)) return;



    std::string ip_ = m_remote_endpoint.address().to_string();
    //for(auto& x : m_data_buffer) {
    //    if(x!=0x0)
    //    std::cout << std::hex << x << std::endl;
    //}
    //std::cout << "=====================" << std::endl;

    std::vector<uint32_t> incoming_;
    for(auto& x : m_data_buffer) {
        incoming_.push_back(x);
        if(x==0xffffffff) { break; }

    }

    m_full_event = 0;
    m_empty_event = 0;
    int header_size = 3;
    int event_size = 2;
    if(vmm_type()==2) {
        header_size = 2; 
        event_size = 2; 
    }
    else if(vmm_type()==3) {
        if(m_do_l0) {
            header_size = 3;
            event_size = 1;
        }
        else {
            header_size = 3;
            event_size = 2;
        }
    }
    


    if(incoming_.size() > header_size) {
        n_input_packets += (incoming_.size() - (header_size + 1)) / event_size; // remove (header + trailer) and divide by size of hit
        m_full_event = 1;
    }
    if(incoming_.size() == header_size) {
        n_input_empty_packets++;
        m_empty_event = 1;
    }

    if(n_input_empty_packets%100==0) {
        emit update_empty_count(n_input_empty_packets);
    }

    // occupancy
    double elapsed_time = m_run_time.elapsed() / 1000.0; // in seconds
    //if(elapsed_time - m_last_occ_check > 0.2) { // every second
    if(m_do_occupancy_check && rand()%100==0) {
        emit perform_occupancy_check(QString::fromStdString(ip_), m_full_event, m_empty_event);
        //m_last_occ_check = elapsed_time; 

    //    cout << "board delta: " << (*m_trigger_board_delta) << "  delta: " << (*m_trigger_delta) << endl;
    //}
    //if(m_do_occupancy_check && (elapsed_time - m_last_occ_check > 1)) {
        emit perform_trigger_delta_check((*m_trigger_board_delta), (*m_trigger_delta));
        m_last_occ_check = elapsed_time;
    }

    IPDataArray incoming_data = std::make_pair(ip_, incoming_);
    std::tuple<std::string, std::vector<uint32_t>, uint32_t> iptuple(ip_, incoming_, incoming_.size());
    IPDataArrayLength incoming_data_l = iptuple;
        //if(!m_buffer->add_to_queue(incoming_data)) {
        if(!m_buffer->add_to_queue(incoming_data_l)) {
            global_stream_lock.lock();
            std::cout << "failed to add to queue!" << std::endl;
            global_stream_lock.unlock();
        }
    n_received++;

    // keep listening (give the service more work)
    //caldelay
    if((*m_continue_gathering))
        handle_data();
/*
    if((*m_continue_gathering) || ((!(*m_continue_gathering) && m_current_caltype==CalibrationType::Time && n_cal_check<1000))) {
        handle_data();
        if(!(*m_continue_gathering)) {
            n_cal_check++;
            if(n_cal_check==1000) {
                while(m_buffer->has_data()) {
                    IPDataArray dummy;
                    m_buffer->dequeue(dummy);
                }
            }
        }
    }
*/
    n_effCount++;
    if((n_effCount>=n_pulse_limit) && (n_pulse_limit > 0) && !reported && m_do_eff) {
        //*m_continue_gathering = false;
        reported = true;
        //emit effCountReached();
        emit updateCounts(n_daqCount, n_trigCount, n_input_packets, n_input_empty_packets, true);

        //#warning NOT CALLING UPDATECALIBRATIONSTATE
        if(!m_use_pulser_calib) {
            emit updateCalibrationState();
            *m_continue_calibration_gathering = false;
        }
     //   return;
    }
}

//DaqBuffer
void DaqServer::read_data(int& daq_counter, int& trig_counter)
{
    //global_stream_lock.lock();
    //std::cout << "[" << boost::this_thread::get_id() << "] DaqServer::read_data    Waiting for buffered data..." << std::endl;
    //std::cout << "daq_count: " << daq_counter << " trig_count: " << trig_counter << std::endl;
    //global_stream_lock.unlock();


    //original
    IPDataArray incoming_element;
    IPDataArrayLength incoming_element_l;
    while(true) {

        // if stop signal given but still data in buffer, finish recording those events
        //if(m_io_service->stopped() && m_buffer->has_data()) {
        if((*m_continue_gathering) && m_buffer->has_data()) {
        //if(!m_io_service->stopped() && m_buffer->has_data()) {

            if(!m_is_calibration_run) {
                if( ( (daq_counter % 100) == 0) ) {
                    emit updateCounts(daq_counter, trig_counter, n_input_packets, n_input_empty_packets, false);
                }
            }

            if(m_buffer->dequeue(incoming_element_l)) {
                string ip_str = std::get<0>(incoming_element_l);
                vector<uint32_t> dgram = std::get<1>(incoming_element_l);
                uint32_t packet_length = std::get<2>(incoming_element_l);

                if(m_is_calibration_run && (*m_continue_calibration_gathering)) { 
                    m_calibration_module->decode_data(ip_str, dgram,
                            packet_length, boost::ref(daq_counter));
                    //m_calibration_module->decode_data(incoming_element.first,
                    //    incoming_element.second, boost::ref(daq_counter));
                }
                else if(!m_is_calibration_run){
                    m_event_builder->receive_event_data(ip_str, dgram,
                        packet_length, boost::ref(daq_counter), boost::ref(trig_counter));
                    //m_event_builder->receive_event_data(incoming_element.first,
                    //    incoming_element.second, boost::ref(daq_counter), boost::ref(trig_counter));
                }
                else {
                    //global_stream_lock.lock();
                    //std::cout << "DaqServer::read_data    Unable to handle incoming data packet!"
                    //    << "  cal? " << m_is_calibration_run << "  continue? "
                    //    << (*m_continue_calibration_gathering)
                    //        << std::endl;
                    //global_stream_lock.unlock();
                }
                // update the counter within the event decoding, when the trailer is hit
                //daq_counter++;
            }
            else {
                //std::cout << "failed to dequeue element!" << std::endl;
            }

            if( (daq_counter >= m_total_events_to_process) && (m_total_events_to_process>0 && !m_is_calibration_run)) {
                emit updateCounts(daq_counter, trig_counter, n_input_packets, n_input_empty_packets, true);
                emit eventCountReached();
                *m_continue_gathering = false;
            }
            else if( (daq_counter >= m_total_events_to_process) && m_is_calibration_run && !m_do_eff) {

                //caldelay
                n_cal_check=0;

                #warning jun 20 stop emit counts during calibration run
                //emit updateCounts(daq_counter, trig_counter, false);
                //#warning NOT CALLING UPDATECALIBRATION STATE
                #warning NOT CALLING UPDATE CALIBRATION STATE
                if(!m_use_pulser_calib) {
                    emit updateCounts(daq_counter, trig_counter, n_input_packets, n_input_empty_packets, false);
                    emit updateCalibrationState();
                    *m_continue_calibration_gathering = false;
                }
                
                //emit stopTimer();
                //boost::this_thread::sleep(boost::posix_time::milliseconds(300));
            }

        }
        //else if((*m_flushing)) {
        //    //if( ( (daq_counter % 100) == 0) ) {
        //    //    emit updateCounts(daq_counter, trig_counter, false);
        //    //}
        //    m_event_builder->flush_remaining_events(boost::ref(n_daqCount), boost::ref(n_trigCount));
        //    //if( (daq_counter >= m_total_events_to_process) && (m_total_events_to_process>0 && !m_is_calibration_run)) {
        //    //    emit updateCounts(daq_counter, trig_counter, true);
        //    //    emit eventCountReached();
        //    //    *m_continue_gathering = false;
        //    //}

        //}
        // if the user has pressed stop, just stop immediately (don't worry about what is in the buffer still)
        else if(!(*m_continue_gathering)) {
            break;
        }

    } // while

}

void DaqServer::stop_listening()
{
    if(dbg()) {
        m_sx << "[" << boost::this_thread::get_id() << "] DAQ socket shutting down";
        global_stream_lock.lock();
        msg()(m_sx,"DaqServer::stop_listening"); m_sx.str("");
        global_stream_lock.unlock();
    }
    ///DaqBuffer test
    //std::cout << "----------------------------------" << std::endl;
    //std::cout << "packets recorded   : " << n_daqCount << std::endl;
    //std::cout << "packets received   : " << n_received << std::endl;
    //std::cout << "----------------------------------" << std::endl;

    if(m_socket->is_open()) {
        m_socket->close();
    }
    boost::system::error_code ec;
    m_socket->shutdown(ip::udp::socket::shutdown_both, ec);

    //if(m_calibration_module && m_is_calibration_run) {
    //    //m_calibration_module->closeCalibrationSocket();
    //    if(m_daqMonitor && m_set_daq_monitor)
    //        m_daqMonitor->stopTimer();
    //}
    // DaqBuffer
    *m_continue_gathering = false;
    *m_continue_calibration_gathering = false;
}

void DaqServer::flush()
{
    if(!(n_daqCount > 0) || !(n_trigCount > 0)) return;
    *m_flushing = true;
    m_event_builder->flush_remaining_events(boost::ref(n_daqCount), boost::ref(n_trigCount));
    boost::this_thread::sleep(boost::posix_time::milliseconds(200));
    emit updateCounts(n_daqCount, n_trigCount, n_input_packets, n_input_empty_packets, true);
}

void DaqServer::stop_server()
{
    if(dbg()) {
        m_sx << "[" << boost::this_thread::get_id() << "] Server stopping and closing all working threads";
        global_stream_lock.lock();
        msg()(m_sx,"DaqServer::stop_server"); m_sx.str("");
        global_stream_lock.unlock();
    }

    if(do_flush && m_do_eff)
        flush();

    m_io_service->stop();
    if(!is_stopped()) {
        cout << "DaqServer::stop_server    IO service did not stop!" << endl;
    }
    m_thread_group.join_all();
}

bool DaqServer::is_stopped()
{
    return m_io_service->stopped();
}

void DaqServer::write_output()
{
    if(m_is_calibration_run) {
        m_calibration_module->write_output();
    }
    else {
        m_event_builder->write_output();
    };
}



