//vmm
#include "pulser_calibration.h"
#include "configuration_module.h"
#include "run_module.h"
#include "config_handler.h"
#include "bit_manip.h"

//std/stl
#include <iostream>
#include <bitset>
#include <iomanip>
#include <sstream>
#include <random>
#include <algorithm>
#include <iomanip>
using namespace std;

//boost
#include <boost/thread.hpp>


bool direct_qt = false;

///////////////////////////////////////////////////////////////////////////////
// PulserCalib
///////////////////////////////////////////////////////////////////////////////
PulserCalib::PulserCalib(QObject* parent) :
    QObject(parent),
    m_dbg(false),
    m_vmm_type(3),
    m_run_number(0),
    m_accept_data(new bool()),
    m_accept(true),
    m_config_module(nullptr),
    m_run_module(nullptr),
    m_config_handle(nullptr),
    m_current_loop_idx(0),
    m_calib_complete(false),
    move_to_next_test(false),
    m_current_gain(0),
    m_current_gain_val(0),
    m_current_tac(0),
    m_current_tac_val(0),
    m_current_peak_time(0),
    m_current_peak_time_val(0),
    m_current_threshold_dac(0),
    m_current_pulser_dac(0),
    m_current_tp_skew(0),
    m_current_tp_skew_ns(0)
{
    (*m_accept_data) = false;

    m_config_module = new Configuration();
    m_config_handle = new ConfigHandler();
    //m_socket_handler = new SocketHandler();

    m_loaded_boards.clear();
    m_loaded_chips.clear();
    m_loaded_channels.clear();

}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::LoadModulesForCalibration(Configuration& config, RunModule& run) //, SocketHandler& socket)
{
    //cout << "PulserCalib::LoadModulesForCalibration    Configuration = " << (&config)
    //        << ", RunModule = " << (&run) << endl;
    m_config_module = &config;
    m_run_module = &run;
    //(*m_socket_handler) = socket;

    //cout << "PulserCalib::LoadModulesForCalibration    [" << boost::this_thread::get_id() << "] socket = " << (&socket) << endl;
    //m_run_module->LoadSocket(&m_socket_handler);
    //m_config_module->LoadSocket(&m_socket_handler);


    connect(this, SIGNAL(set_acq_pulser(int)), m_run_module, SLOT(set_acq_pulser(int)));//, Qt::DirectConnection);
    connect(this, SIGNAL(set_clocks_pulser(int,int,int,int,int,int)), m_run_module, SLOT(set_clocks_pulser(int,int,int,int,int,int)));
    connect(this, SIGNAL(send_configuration()), m_config_module, SLOT(send_configuration_pulser()));
    connect(this, SIGNAL(calib_configuration_update(int, int)), m_run_module, SLOT(calib_configuration_update(int, int)));

}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::load_flags(bool* flags)
{
    m_accept_data = flags;
    //cout << "PulserCalib::load_flags = " << m_accept_data << endl;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::set_vmm_type(int type)
{
    m_vmm_type = type;
    //cout << "PulserCalib::set_vmm_type    Setting VMM type to " << type << endl;
    set_value_maps(type);
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::set_value_maps(int type)
{
    m_gain_vals.clear();
    m_tac_vals.clear();
    m_peak_time_vals.clear();

    vector<float> gains { 0.5, 1.0, 3.0, 4.5, 6.0, 9.0, 12.0, 16.0 }; // mV/fC
    vector<float> pts { 200, 100, 50, 25 }; // ns
    vector<float> tac_vmm2 { 125, 250, 500, 1000 }; // ns
    vector<float> tac_vmm3 { 60, 100, 350, 650 }; // ns

    for(int i = 0; i < (int)gains.size(); i++) {
        m_gain_vals[i] = gains.at(i);
    }
    for(int i = 0; i < (int)pts.size(); i++) {
        m_peak_time_vals[i] = pts.at(i);
    }
    if(type==2) {
        for(int i = 0; i < (int)tac_vmm2.size(); i++) {
            m_tac_vals[i] = tac_vmm2.at(i);
        }
    }
    else {
        for(int i = 0; i < (int)tac_vmm3.size(); i++) {
            m_tac_vals[i] = tac_vmm3.at(i);
        }
    }
}
///////////////////////////////////////////////////////////////////////////////
int PulserCalib::n_loop()
{
    return m_calibration_loop_holder.size();
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::calib_loop(int i, PulserCalibHolder& holder)
{
    bool status = true;
    try {
        holder.clear();
        holder = m_calibration_loop_holder.at(i);
    }
    catch(std::exception& e) {
        cout << "PulserCalibHolder::calib_loop    PulserCalibHolder object at index " << i
            << " could not be accessed (there are " << n_loop() << " PulserCalibHolders in total)" << endl;
        status = false;
    }
    return status;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::load_calibration_loop(PulserCalibDef definition)
{
    bool status = true;
    // check that things are ok
    status &= (definition.channel_high >= definition.channel_low);

    status &= (definition.gain_idx_stop >= definition.gain_idx_start);

    status &= (definition.peak_time_idx_stop >= definition.peak_time_idx_start);

    status &= (definition.tac_idx_stop >= definition.tac_idx_start);

    status &= (definition.tp_skew_stop >= definition.tp_skew_start);

    status &= (definition.threshold_dac_val_stop >= definition.threshold_dac_val_start);

    status &= (definition.pulser_dac_val_stop >= definition.pulser_dac_val_start);

    m_config_handle->setSelectedBoards(definition.board_selection);

    m_def = definition;
    cout << m_def.str() << endl;

    build_calibration_quota();

    build_calibration_loop();

    return status;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::load_run_properties(PulserCalibRunProperties props)
{
    m_props = props;

    // set the decoding type based on the current run properties
    int decoding_type = VMMDecoder::VMMEVENT;
    if(vmm_type()==2 && m_props.l0_decoding) {
        cout << "PulserCalib::load_run_properties    ERROR You have configured VMMs for L0 decoding but are "
            << "reading out VMM2, cannot start pulser run" << endl;
        return false;
    }

    if(vmm_type()==3) {
        if(props.l0_decoding) {
            decoding_type = VMMDecoder::VMMEVENTL0;
        }
        else {
            decoding_type = VMMDecoder::VMMEVENT;
        }
    }
    else if(vmm_type()==2) {
        decoding_type = VMMDecoder::VMM2EVENT;
    }
    m_decoder.set_type(decoding_type);

    cout << m_props.str() << endl;
    return initialize_output_file();
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::initialize_output_file()
{
    if(!init_tfile()) return false;

    if(!init_ttree()) return false;

    return true;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::init_tfile()
{
    stringstream rn;
    rn << std::setfill('0') << std::setw(4) << m_props.run_number;
    stringstream fname;
    fname << m_props.output_dir << "/";
    fname << "calib_run_" << rn.str() << ".root";

    m_root_file = new TFile(fname.str().c_str(), "UPDATE");
    if(m_root_file->IsZombie()) {
        cout << "PulserCalib::init_tfile    ERROR Calibration ROOT file '" << fname.str() << " unable to be created" << endl;
        delete m_root_file;
        return false;
    }
    m_root_file->cd();
    return true;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::init_ttree()
{
    m_tree = new TTree("calib", "calib");

    br_runNumber            = m_tree->Branch("runNumber", &m_props.run_number);
    br_gain                 = m_tree->Branch("gain", &m_current_gain_val);
    br_tacSlope             = m_tree->Branch("tacSlope", &m_current_tac_val);
    br_peakTime             = m_tree->Branch("peakTime", &m_current_peak_time_val);
    br_dacCounts            = m_tree->Branch("dacCounts", &m_current_threshold_dac);
    br_pulserCounts         = m_tree->Branch("pulserCounts", &m_current_pulser_dac);
    br_tpSkew               = m_tree->Branch("tpSkew", &m_current_tp_skew_ns);
    br_channelTrims         = m_tree->Branch("channelTrim", &m_props.channel_trims);
    br_subhysteresis        = m_tree->Branch("subhysteresis", &m_props.subhysteresis);
    br_neighbortrigger      = m_tree->Branch("neighbor_trigger", &m_props.neighbor_enable);
    br_nExpected            = m_tree->Branch("nExpected", &m_def.n_expected);

    br_chipId               = m_tree->Branch("chip", "std::vector<int>", &m_chipId);
    br_boardId              = m_tree->Branch("boardId", "std::vector<int>", &m_boardId);
    br_channelId            = m_tree->Branch("channel", "std::vector< vector<int> >", &m_channelId);
    br_pdo                  = m_tree->Branch("pdo", "std::vector< vector<int> >", &m_pdo);
    br_tdo                  = m_tree->Branch("tdo", "std::vector< vector<int> >", &m_tdo);
    br_evSize               = m_tree->Branch("eventSize", "std::vector<int>", &m_eventSize);
    br_thresh               = m_tree->Branch("threshold", "std::vector< vector<int> >", &m_threshold);
    br_flag                 = m_tree->Branch("flag", "std::vector< vector<int> >", &m_flag);
    br_bcid                 = m_tree->Branch("bcid", "std::vector< vector<int> >", &m_bcid);
    br_grayDecoded          = m_tree->Branch("grayDecoded", "std::vector< vector<int> >", &m_grayDecoded);
    br_neighborCalib        = m_tree->Branch("isNeighbor", "std::vector< vector<int> >", &m_neighbor_calib);

    return true;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::write_output()
{
    cout << "PulserCalib::write_output" << endl;
    bool write_ok = true;
    m_root_file->cd();
    m_tree->Write("", TObject::kOverwrite);
    if(!m_root_file->Write()) {
        write_ok = false;
        cout << "PulserCalib::write_output    ERROR Writing output calib ROOT file" << endl;
    }

    TNamed user_comment_branch("comments", m_props.run_comment.c_str());
    user_comment_branch.Write();

    m_root_file->Close();

    if(write_ok) {
        stringstream sx;
        sx << "PulserCalib::write_output    Pulser calibration run " << m_props.run_number << " stored in file: "
            << m_root_file->GetName();
        cout << sx.str() << endl;
    }
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::is_valid_tuple(PulserCalib::BoardChipChannelTuple tuple)
{
    return ( is_valid_board(std::get<0>(tuple)) && is_valid_chip(std::get<1>(tuple)) && is_valid_channel(std::get<2>(tuple)));
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::is_valid_board(int board)
{
    bool is_found = (std::find(m_loaded_boards.begin(), m_loaded_boards.end(), board) != m_loaded_boards.end());
    //bool is_found = false;
    //for(auto x : m_loaded_boards) {
    //    if(x==board) { is_found = true; break; }
    //}
    return is_found;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::is_valid_chip(int chip)
{
    bool is_found = (std::find(m_loaded_chips.begin(), m_loaded_chips.end(), chip) != m_loaded_chips.end());
    //bool is_found = false;
    //for(auto x : m_loaded_chips) {
    //    if(x==chip) { is_found = true; break; }
    //}
    return is_found;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::is_valid_channel(int channel)
{
    bool is_found = (std::find(m_loaded_channels.begin(), m_loaded_channels.end(), channel) != m_loaded_channels.end());
    //bool is_found = false;
    //for(auto x : m_loaded_channels) {
    //    if(x==channel) { is_found = true; break; }
    //}
    return is_found;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::build_calibration_quota()
{
    m_quota_map.clear();
    m_last_count_check.clear();
    m_static_count.clear();

    int n_samples_per_chan = m_def.n_samples_per_channel;

    std::vector<int> vmms;
    int def_mask = m_def.vmm_mask;
    int mask = 0x1;
    for(int i = 0; i < 8; i++) {
        if( def_mask & (mask << i) ) {
            vmms.push_back(i); 
        }
    }

    m_loaded_boards.clear();
    m_loaded_chips.clear();
    m_loaded_channels.clear();

    int ch_low = m_def.channel_low;
    int ch_high = m_def.channel_high;
    for(auto board_num : m_def.board_numbers) {
        if(!PulserCalib::is_valid_board(board_num)) m_loaded_boards.push_back(board_num);
        for(int ichan = 0; ichan < (int)vmms.size(); ichan++) {
            int vmm_number = vmms.at(ichan);
            if(!PulserCalib::is_valid_chip(vmm_number)) m_loaded_chips.push_back(vmm_number);
            for(int i = ch_low; i <= ch_high; i++) {
                if(!PulserCalib::is_valid_channel(i)) m_loaded_channels.push_back(i);

                PulserCalib::BoardChipChannelTuple tuple(board_num, vmm_number, i);
                if(!is_valid_tuple(tuple)) {
                // initialize the quota
                    m_quota_map[tuple] = 0;
                    m_last_count_check[tuple] = 0;
                    m_static_count[tuple] = 0;
                }
            } // i
        } // ch
    } // iboard

    n_unique_channels = 0;
    m_shutdown_list.clear();
    n_shutdown = 0;
    for(auto b : m_loaded_boards) {
        for(auto ch : m_loaded_chips) {
            for(auto c : m_loaded_channels) {
                n_unique_channels++;
            } //c
        } // ch
    } // b

    //cout << "loaded boards = ";
    //for(auto x : m_loaded_boards) cout << x << " ";
    //cout << endl;
    //cout << "loaded chips = ";
    //for(auto x : m_loaded_chips) cout << x << " ";
    //cout << endl;
    //cout << "loaded channels = ";
    //for(auto x : m_loaded_channels) cout << x << " ";
    //cout << endl;

    //for(auto bcct : m_quota_map) {
    //    PulserCalib::BoardChipChannelTuple tuple(bcct.first);
    //    int board = std::get<0>(bcct.first);
    //    int chip = std::get<1>(bcct.first);
    //    int chan = std::get<2>(bcct.first);
    //    cout << "quota : [board,chip,chan]=["<<board<<","<<chip<<","<<chan<<"] -> " << bcct.second << endl;
    //}

}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::above_quota(PulserCalib::BoardChipChannelTuple tuple)
{
    return (m_quota_map[tuple] >= m_def.n_samples_per_channel); 
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::build_calibration_loop()
{
    int vmm_mask = config_handle()->vmmMap().mask;
    int channel_low = m_def.channel_low; 
    int channel_high = m_def.channel_high;

    // gain
    int gain_start = m_def.gain_idx_start;
    int gain_stop = m_def.gain_idx_stop;
    int max_gain = 8;

    // peak time
    int pt_start = m_def.peak_time_idx_start;
    int pt_stop = m_def.peak_time_idx_stop;
    int max_pt = 4;

    // tac slope
    int tac_start = m_def.tac_idx_start;
    int tac_stop = m_def.tac_idx_stop;
    int tac_step = m_def.tac_idx_step;
    int max_tac = 4;

    // TP skew
    int tp_skew_start = m_def.tp_skew_start;
    int tp_skew_stop = m_def.tp_skew_stop;
    int tp_skew_step = m_def.tp_skew_step;
    m_current_tp_skew = tp_skew_start;
    m_current_tp_skew_ns = tp_skew_start * m_def.tp_time_per_step;

    // threshold DAC
    int sdt_val_start = m_def.threshold_dac_val_start;
    int sdt_val_stop = m_def.threshold_dac_val_stop;
    int sdt_val_step = m_def.threshold_dac_val_step;
    int max_sdt_val = 1023;

    // pulser DAC
    int sdp_val_start = m_def.pulser_dac_val_start;
    int sdp_val_stop = m_def.pulser_dac_val_stop;
    int sdp_val_step = m_def.pulser_dac_val_step;
    int max_sdp_val = 1023;


    // build the loop holders
    int calib_loop_idx = 0;
    m_calibration_loop_holder.clear();
    for(int gain = gain_start; gain <= gain_stop; gain++) {
        if(gain >= max_gain) break;
        for(int pt = pt_start; pt <= pt_stop; pt++) {
            if(pt >= max_pt) break;
            for(int tac = tac_start; tac <= tac_stop; tac += tac_step) {
                if(tac >= max_tac) break;
                for(int tp_skew = tp_skew_start; tp_skew <= tp_skew_stop; tp_skew += tp_skew_step) {
                    for(int sdt = sdt_val_start; sdt <= sdt_val_stop; sdt += sdt_val_step) {
                        if(sdt >= max_sdt_val) break;
                        for(int sdp = sdp_val_start; sdp <= sdp_val_stop; sdp += sdp_val_step) {
                            if(sdp >= max_sdp_val) break;
                                PulserCalibHolder pch(calib_loop_idx);
                                pch.gain_idx = gain;
                                pch.peak_time_idx = pt;
                                pch.tac_idx = tac;
                                pch.tp_skew = tp_skew;
                                pch.threshold_dac_val = sdt;
                                pch.pulser_dac_val = sdp;
                                m_calibration_loop_holder.push_back(pch);
                                calib_loop_idx++;
                        } // sdp
                    } // sdt
                } // tp_skew
            } // tac
        } // pt
     } // gain

    //for(auto x : m_calibration_loop_holder) {
    //    cout << x.str() << endl;
    //}
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::load_initial_config(GlobalSetting& global, VMMMap& vmmMap,
        std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq)
{
    m_def_global_settings = global;
    m_def_fpga_clocks_settings = clocks;
    m_def_vmm_map = vmmMap;
    m_def_channel_settings = channels;

    m_config_handle->LoadBoardConfiguration(global, vmmMap, channels);
    m_config_handle->LoadTDAQConfiguration(daq);
    m_config_handle->LoadFPGAClocksConfiguration(clocks);

    m_config_module->LoadConfig(*m_config_handle);

    //string fn = "PulserCalib::load_initial_config    ";
    //cout << fn << "GLOBAL    sdt_dac = " << config_handle()->globalSettings().sdt_dac << "   sdp_dac = " << config_handle()->globalSettings().sdp_dac << "   gain = " << config_handle()->globalSettings().sg << "  sbip = " << config_handle()->globalSettings().sbip << endl;
    //cout << fn << "VMMMAP    mask = " << std::hex << config_handle()->vmmMap().mask << std::dec << endl;
    //for(int i = 0; i < 10; i++) {
    //    cout << fn <<  "CHANNEL    [" << i << "]    st = " << config_handle()->channelSettings(i).st << "   sm = " << config_handle()->channelSettings(i).sm << endl;
    //}
    //cout << fn << "CLOCKS    cktp_skew_steps = " << config_handle()->clocksSettings().cktp_skew_steps << "   # = " << config_handle()->clocksSettings().cktp_max_number << endl; 

}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::start()
{
    if(m_calibration_loop_holder.size()==0) {
        cout << "PulserCalib::start    No calibration loop defined, cannot start pulser calib" << endl;
        return false;
    }

    // start time
    m_start_run_time = QTime::currentTime();

    m_trigger_events.clear();

    (*m_accept_data) = true;
    m_accept = true;
    m_current_loop_idx = 0;
    m_calib_complete = false;
    if(direct_qt) {
        m_run_module->ACQoff();
    }
    else {
        emit set_acq_pulser(0);
    }
    set_configuration(m_current_loop_idx);
    
    boost::this_thread::sleep(boost::posix_time::milliseconds(1000));

    if(direct_qt) {
        m_run_module->ACQon();
    }
    else {
        emit set_acq_pulser(1);
    }

    //for(const auto & pch : m_calibration_loop_holder) {
    //    m_run_module->ACQoff();
    //    set_configuration(pch);
    //    m_run_module->ACQon();
    //}
    return true;
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::move_to_next_calib_step()
{

    //cout << "PulserCalib::move_to_next_calib_step    setting ACQ off" << endl;
    if(direct_qt) {
        m_run_module->ACQoff();
    }
    else {
        emit set_acq_pulser(0);
    }

    int current = m_current_loop_idx;
    if(current == ((int)m_calibration_loop_holder.size()-1)) {
        //cout << "PulserCalib::move_to_next_calib_step    *** Pulser calibration done, time to take some kick ass data ***" << endl;
        m_calib_complete = true;

        stop_pulser_calibration();

        return false;
    }
    m_current_loop_idx++;
    //cout << "PulserCalib::move_to_next_calib_step    moving from calib loop idx " << current << " -> " << m_current_loop_idx << endl;
    set_configuration(m_current_loop_idx);
    m_accept = true;
    //move_to_next_test = false;
    boost::this_thread::sleep(boost::posix_time::milliseconds(300));

    build_calibration_quota();

    // clear counters
    n_shutdown = 0;
    m_shutdown_list.clear();
    if(direct_qt) {
        m_run_module->ACQon();
    }
    else {
        emit set_acq_pulser(1);
        emit set_acq_pulser(1);
        emit set_acq_pulser(1);
    }
    return true;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::stop_pulser_calibration()
{
    // write output files here
    if(direct_qt) {
        m_run_module->ACQoff();
        m_run_module->ACQoff();
        m_run_module->ACQoff();
    }
    else {
        emit set_acq_pulser(0);
        emit set_acq_pulser(0);
        emit set_acq_pulser(0);
    }

    double elapsed_time = m_start_run_time.elapsed() / 1000.; // seconds
    cout << "PulserCalib::stop_pulser_calibration    Pulser calibration elapsed time: " << std::setw(10) << elapsed_time << " s" << endl;
    emit pulser_calib_complete();
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::set_configuration(int loop_idx)
{
    //cout << "PulserCalib::set_configuration    [" << boost::this_thread::get_id() << "]    [" << (loop_idx+1) << " / " << m_calibration_loop_holder.size() << "]    Updating calibration configuration state" << endl;

    emit calib_configuration_update( (loop_idx+1), (int)m_calibration_loop_holder.size() );

    GlobalSetting global = m_def_global_settings;
    FPGAClocks fpga_clocks = m_def_fpga_clocks_settings;
    VMMMap vmm_map = m_def_vmm_map; 
    vector<Channel> channels = m_def_channel_settings;

    PulserCalibHolder pch = m_calibration_loop_holder.at(loop_idx);

    // set gain
    global.sg = pch.gain_idx;
    m_current_gain = pch.gain_idx;
    m_current_gain_val = m_gain_vals[pch.gain_idx];

    // set peak time
    global.st = pch.peak_time_idx;
    m_current_peak_time = pch.peak_time_idx;
    m_current_peak_time_val = m_peak_time_vals[pch.peak_time_idx];

    // set tac slope
    global.stc = pch.tac_idx;
    m_current_tac = pch.tac_idx;
    m_current_tac_val = m_tac_vals[pch.tac_idx];

    // tp skew
    fpga_clocks.cktp_skew_steps = pch.tp_skew;
    bool do_time = (pch.tp_skew != m_current_tp_skew);
    m_current_tp_skew = pch.tp_skew;
    m_current_tp_skew_ns = (pch.tp_skew * m_def.tp_time_per_step);

    // threshold dac
    global.sdt_dac = pch.threshold_dac_val;
    m_current_threshold_dac = pch.threshold_dac_val;

    // pulser dac
    global.sdp_dac = pch.pulser_dac_val;
    m_current_pulser_dac = pch.pulser_dac_val;

    // channels
    for(auto ch : channels) {
        ch.sc = 0;
        ch.sl = 0;
        ch.st = 0;
        ch.sth = 0;
        ch.sm = 0;
        ch.smx = 0;
    }
    for(int ch = 0; ch < 64; ch++) {
        if(ch >= m_def.channel_low && ch <= m_def.channel_high) {
            channels.at(ch).st = 1;
            channels.at(ch).sm = 0;
        }
        else {
            channels.at(ch).st = 0;
            channels.at(ch).sm = 1;
        }
    }
    config_handle()->LoadBoardConfiguration(global, vmm_map, channels);
    if(direct_qt) {
        m_config_module->SendConfig();
    }
    else {
        emit send_configuration();
    }

    if(do_time) {
        // add some time here to avoid sending the clock configuration while the
        // fpga is still clocking in the VMM SPI
        boost::this_thread::sleep(boost::posix_time::milliseconds(200));
        emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
            m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
        //    m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
        //    m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
        //    m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
        //    m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //emit set_clocks_pulser(m_props.cktk_max, m_props.ckbc_frequency, m_def.n_expected,
        //    m_current_tp_skew, m_props.cktp_period, m_props.cktp_width);
        //boost::this_thread::sleep(boost::posix_time::milliseconds(500));
    }

    //emit send_configuration();
    int time = 500;
    if(do_time) time = 300;
    boost::this_thread::sleep(boost::posix_time::milliseconds(time));
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::is_valid_sample(vector<uint32_t>& datagram)
{
    bool valid = true;
    if(vmm_type()==3) {
        if(datagram.size()>=2) {
            if(VMMDecoder::is_configuration_reply(datagram)) { valid = false; }
            valid = true;
        }
    }
    else if(vmm_type()==2) {
        bool found_non_empty = false;
        for(const auto & x : datagram) {
            if(!(x==0xffffffff) && !(x==0x0)) { found_non_empty = true; break; }
        }
        valid = found_non_empty;
    }
    return valid;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::decode_pulser_data(string& ip_string, vector<uint32_t>& datagram,
        int& counter)
{
    //cout << "PulserCalib::decode_pulser_data    " << __LINE__ << "   ";
    //for(auto x : datagram) {
    //    if(x!=0x0) {
    //        cout << " " << std::hex <<  x;
    //    }
    //}
    //cout << endl;
    //if(m_trigger_events.ready_to_read()) {
    if(m_trigger_events.n_triggers()) {
        fill_trigger_event();
    }

    /////////////////////////////////////////////////////////////////
    // first check where we are in the calibration loop
    /////////////////////////////////////////////////////////////////
    if(!m_accept && !m_calib_complete) { //return;

    //if(ready_for_next_step()) {
        //if(!move_to_next_test) {
        //    cout << "PulserCalib::decode_pulser_data    READY FOR NEXT STEP" << endl;
        //    move_to_next_test = true;
        //}
        if(!move_to_next_calib_step()) {
            m_accept = false;
        }

        //(*m_accept_data) = false;
    //    m_accept = false;
    //    return;
    }
    else if(m_calib_complete) {
        return;
    }

    if(ready_for_next_step()) { m_accept = false; return; }

    /////////////////////////////////////////////////////////////////
    // now start the decoding
    /////////////////////////////////////////////////////////////////

    // if we've got enough events, start storing
    size_t n_bytes_read = 0;

    // check and avoid configuration replies (fw sends them back to us on the data line for some reason)
    if(datagram.size()>=2) {
        if(!is_valid_sample(datagram) || datagram.size()==3) return;
    }


    ////////////////////////////////////////////
    // get the board ID from the packet IP
    ////////////////////////////////////////////
    string boardId = "";
    int board_id = -1;
    stringstream b_id;
    try {
        boardId = ip_string.substr(ip_string.find_last_of('.')+1);
        board_id = std::stoi(boardId);
    } // try
    catch(std::exception& e) {
        cout << "PulserCalib::decode_pulser_data    [" << boost::this_thread::get_id() << "]"
            << " Unable to get board id from board ip " << ip_string << endl;
        board_id = -1;
        boardId = "";
    }

    //cout << "PulserCalib::decode_pulser_data    [" << boost::this_thread::get_id() << "] data from board with IP " << ip_string << " -> board id = " << board_id << "  ";
    //for(auto x : datagram) cout << " " << std::hex << x;
    //cout << std::dec << endl;

    // reverse each of the 32-bit words
    vector<uint32_t> datagram_vector_tmp(datagram.begin(), datagram.end());
    vector<uint32_t> datagram_vector;
    for(const auto & data : datagram_vector_tmp) {
        datagram_vector.push_back(bits::endian_swap32()(data));
    }

    // decode header [header length = 2 for VMM2, 3 for VMM3]
    bool continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin(), datagram_vector.begin() + vmm_type()), true);

    unsigned int n_empty_data = 0;

    while(true) {
        if(!continue_decoding) break;

        size_t trigger_count_pos = 1; // take into account different data format for VMM type
        if(vmm_type()==2) trigger_count_pos = 0;
        uint32_t trigger_count = datagram_vector.at(trigger_count_pos);
        VMMHit vmm_hit;
        int n_step (m_props.l0_decoding ? 1 : 2);
        if(vmm_type()==2) n_step = 2;
        //cout << "PulserCalib::decode_data    n_step = " << n_step << "  vmm_type = " << vmm_type() << "  ";
        //for(auto x : datagram_vector) cout << " " << std::hex << x;
        //cout << endl;

        for(unsigned int i = vmm_type(); i < datagram_vector.size(); i+=n_step) {
            continue_decoding = m_decoder.decode(vector<uint32_t>(datagram_vector.begin() + i,
                datagram_vector.begin() + i + n_step), false);

            // if we see a period of empty packets, break
            if(n_step==1 && datagram_vector.at(i)==0x0) {
                n_empty_data++;
            }
            else if(n_step==2 && datagram_vector.at(i)==0x0 && datagram_vector.at(i+1)==0x0) {
                n_empty_data++;
            }
            if(n_empty_data >= 4) {
                continue_decoding = false;
            }

            if(!continue_decoding) break;

            vmm_hit.clear();

            vmm_hit.set_vmm_channel(m_decoder.vmm_channel());
            vmm_hit.set_pdo(m_decoder.pdo());
            vmm_hit.set_tdo(m_decoder.tdo());
            vmm_hit.set_graycode_bcid(m_decoder.bcid());
            vmm_hit.set_decoded_graycode_bcid(m_decoder.decoded_bcid());
            if(m_props.l0_decoding) {
                vmm_hit.set_relative_bcid(m_decoder.relbcid());
                vmm_hit.set_overflow_bit(m_decoder.vmm_overflow());
                vmm_hit.set_orbit_counter(m_decoder.orbit_counter());
            } // l0
            else {
                vmm_hit.set_flag(m_decoder.flag());
                vmm_hit.set_pass_threshold(m_decoder.pass_threshold());
            }

            n_bytes_read += n_step * 4;


            //// TEST
            //int channel = rand()%4;
            //vmm_hit.set_vmm_channel(channel);
            //board_id = 2;
            //if(rand()%2==0) board_id = 2;
            //else { board_id = 3; }
            //if(board_id == 1) {
            //    board_id = 2;
            //}

            vmm_hit.set_board_id(board_id);
            int32_t chip_no = m_decoder.chip_number();
            vmm_hit.set_chip_id(chip_no);
            //chip_no = rand()%2;
            //if(channel == 3 && chip_no == 1 && board_id == 2) continue; // testing dead channels
            //if(channel == 0 && chip_no == 0 && board_id == 2) continue;

            BoardChipChannelTuple tuple(board_id, chip_no, vmm_hit.get_vmm_channel());
            //print_quota();
            // maybe add these counter checks inside of fill_trigger_event()
            if(is_valid_tuple(tuple)) {
                m_quota_map[tuple]++;
                if(above_quota(tuple)) {
                    shutdown_channel(tuple);
                }
            }
            // store
            if(!tuple_is_shutdown(tuple))
                m_trigger_events.add_hit2(trigger_count, board_id, chip_no, vmm_hit);

            
        } // i
    } // while

    check_for_static_channels(board_id);

    /*

    // testing below
    int channel = rand()%4;
    //int channel = 0;
    //int channel = 0;
    //int chip = 0;
    int chip = rand()%2;
    board_id = 2;
    if(rand()%2==0) board_id = 2;
    else { board_id = 3; }
    //if(board_id == 1) {
    //    board_id = 2;
    //}

    BoardChipChannelTuple tuple(board_id, chip, channel);
    //cout << "PulserCalib::decode_pulser_data    [" << boost::this_thread::get_id() << "]  quota for [board,chip,channel]=["<<board_id<<","<<chip<<","<<channel<<"] = " << m_quota_map[tuple] << endl;
    print_quota();
    if(is_valid_tuple(tuple)) {
        m_quota_map[tuple]++;
        if(above_quota(tuple)) {
            shutdown_channel(tuple);
        }
    }
    */

}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::check_for_static_channels(int board_id)
{
    int current_count = 0;
    int last_count = 0;
    for(auto board : m_loaded_boards) {
        if(board != board_id) continue;
        for(auto chip : m_loaded_chips) {
            for(auto chan : m_loaded_channels) {
                BoardChipChannelTuple tuple(board, chip, chan);
                current_count = m_quota_map[tuple];
                last_count = m_last_count_check[tuple];
                stringstream sc;
                //sc << "SAME (" << m_static_count[tuple] << ")";
                //cout << "PulserCalib::check_for_static_channels    [board,chip,chan]=["<<board<<","<<chip<<","<<chan<<"]    current_count = " << current_count << "   last_count = " << last_count << "   " << ((current_count == last_count) ? sc.str() : "") << endl;
                if(!(current_count > last_count) && !(tuple_is_shutdown(tuple))) {
                    m_static_count[tuple]++;
                    if(m_static_count[tuple] > 20) {
                        //cout << "PulserCalib::check_for_static_channels    STATIC CHANNEL Shutting down static channel which has static count > 5 : [board,chip,channel]=["<<board<<","<<chip<<","<<chan<<"]" << endl;
                        shutdown_channel(tuple);
                    }
                }
                else {
                    // reset the static count
                    m_static_count[tuple] = 0;
                }
                m_last_count_check[tuple] = current_count;
            } // chan
        } // chip
    } // board

}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::fill_trigger_event()
{
    bool trig_ok = true;
    uint32_t trigger_to_readout = m_trigger_events.trigger_to_read(trig_ok);
    if(!trig_ok) return;

    map<BoardChipPair, vector<VMMHit> > hitgroup = m_trigger_events.get_trigger2(trigger_to_readout);

    if(hitgroup.size()==0) {
        cout << "PulserCalib::fill_trigger_event    WARNING Hit group empty for "
            << "trigger " << trigger_to_readout << " (0x" << std::hex << trigger_to_readout << std::dec << ")" << endl;
        return;
    }

    bool write = true; // for calib, do we really need a flag for writing?

    std::vector<int> board_id_tree;
    std::vector<int> chip_id_tree;
    std::vector<int> eventsize_tree;
    std::vector<int> trigcounter_tree;

    std::vector< std::vector<int> > vmm_channel_tree;
    std::vector< std::vector<int> > pdo_tree;
    std::vector< std::vector<int> > tdo_tree;
    std::vector< std::vector<int> > flag_tree;
    std::vector< std::vector<int> > threshold_tree;
    std::vector< std::vector<int> > bcid_tree;
    std::vector< std::vector<int> > graydecoded_tree;
    std::vector< std::vector<int> > relbcid_tree;
    std::vector< std::vector<int> > overflow_tree;
    std::vector< std::vector<int> > orbit_counter_tree;
    std::vector< std::vector<int> > neighbor_tree;

    std::vector<int> vmm_channel;
    std::vector<int> pdo;
    std::vector<int> tdo;
    std::vector<int> flag;
    std::vector<int> threshold;
    std::vector<int> bcid;
    std::vector<int> graydecoded;
    std::vector<int> relbcid;
    std::vector<int> overflow;
    std::vector<int> orbit_counter;
    std::vector<int> neighbor;

    size_t n_bytes_read = 0;
    for(const auto group : hitgroup) {
        BoardChipPair bcpair = group.first;

        if(write) {
            vmm_channel.clear();
            pdo.clear();
            tdo.clear();
            flag.clear();
            threshold.clear();
            bcid.clear();
            graydecoded.clear();
            relbcid.clear();
            overflow.clear();
            orbit_counter.clear();
            neighbor.clear();
        } // write

        std::vector<VMMHit> hits = hitgroup[bcpair];
        for(auto hit : hits) {
            if(write) {
                vmm_channel.push_back(hit.get_vmm_channel());
                pdo.push_back(hit.get_pdo());
                tdo.push_back(hit.get_tdo());
                bcid.push_back(hit.get_graycode_bcid());
                graydecoded.push_back(hit.get_decoded_graycode_bcid());
                if(m_props.l0_decoding) {
                    relbcid.push_back(hit.get_relative_bcid());
                    overflow.push_back(hit.get_overflow_bit());
                    orbit_counter.push_back(hit.get_orbit_counter());
                }
                else {
                    flag.push_back(hit.get_flag());
                    threshold.push_back(hit.get_pass_threshold());
                }
            } // write

            int multiplier = (m_props.l0_decoding ? 1 : 2);
            if(vmm_type()==2) multiplier = 2;
            n_bytes_read += multiplier * 4;
        } // hit

        if(n_bytes_read > 0 && write) {
            trigcounter_tree.push_back(trigger_to_readout);
            board_id_tree.push_back(bcpair.first);
            chip_id_tree.push_back(bcpair.second);
            eventsize_tree.push_back(n_bytes_read);

            vmm_channel_tree.push_back(vmm_channel);
            pdo_tree.push_back(pdo);
            tdo_tree.push_back(tdo);
            bcid_tree.push_back(bcid);
            graydecoded_tree.push_back(graydecoded);
            if(m_props.l0_decoding) {
                relbcid_tree.push_back(relbcid);
                overflow_tree.push_back(overflow);
                orbit_counter_tree.push_back(orbit_counter);
            }
            else {
                threshold_tree.push_back(threshold);
                flag_tree.push_back(flag);
            }
        } // n_bytes_read > 0
    } // group

    if(n_bytes_read > 0 && write) {
        m_chipId.clear();
        m_boardId.clear();
        m_triggerCounter.clear();
        m_eventSize.clear();

        m_channelId.clear();
        m_pdo.clear();
        m_tdo.clear();
        m_flag.clear();
        m_threshold.clear();
        m_bcid.clear();
        m_grayDecoded.clear();
        m_relbcid.clear();
        m_overflow.clear();
        m_orbit_count.clear();

        m_chipId = chip_id_tree;
        m_boardId = board_id_tree;
        m_triggerCounter = trigcounter_tree;
        m_eventSize = eventsize_tree;

        m_channelId = vmm_channel_tree;
        m_pdo = pdo_tree;
        m_tdo = tdo_tree;
        m_flag = flag_tree;
        m_threshold = threshold_tree;
        m_bcid = bcid_tree;
        m_grayDecoded = graydecoded_tree;
        m_relbcid = relbcid_tree;
        m_overflow = overflow_tree;
        m_orbit_count = orbit_counter_tree;

        /////////////////////////////////////////
        // fill branches
        /////////////////////////////////////////
        fill_event();
    }
    m_last_trigger_read = trigger_to_readout;
    m_trigger_events.remove_trigger(trigger_to_readout);
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::fill_event()
{
    bool write = true;
    if(write) {
        m_root_file->cd();
        m_tree->Fill();
    }
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::print_quota()
{
    int den = m_def.n_samples_per_channel;
    stringstream sx;

    sx << "PulserCalib::print_quota    [IDX=" << (m_current_loop_idx+1) << "/" << m_calibration_loop_holder.size() << "]";
    for(auto b : m_loaded_boards) {
        for(auto ch : m_loaded_chips) {
            for(auto c : m_loaded_channels) {
                BoardChipChannelTuple tuple(b,ch,c);
                sx << std::setw(7) << "["<<b<<" "<<ch<<" "<<c<<"] " << (float)(m_quota_map[tuple])/(float)(den);
            }
        }
    }
    sx << endl;
    cout << sx.str();
}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::tuple_is_shutdown(PulserCalib::BoardChipChannelTuple tuple)
{
    bool is_found =  (std::find(m_shutdown_list.begin(), m_shutdown_list.end(), tuple) != m_shutdown_list.end());
    return is_found;
}
///////////////////////////////////////////////////////////////////////////////
void PulserCalib::shutdown_channel(PulserCalib::BoardChipChannelTuple tuple)
{
    int board = std::get<0>(tuple);
    int chip = std::get<1>(tuple);
    int chan = std::get<2>(tuple);

    if(!tuple_is_shutdown(tuple)) {
        m_shutdown_list.push_back(tuple);
        n_shutdown++;
        //cout << "PulserCalib::shutdown_channel ******************************** " << endl;
        //cout << "PulserCalib::shutdown_channel    Shutting down [board,chip,channe;]=[" << board << "," << chip << "," << chan << "] -> has quota = " << m_quota_map[tuple] << endl;
        //cout << "PulserCalib::shutdown_channel ******************************** " << endl;
    }

}
///////////////////////////////////////////////////////////////////////////////
bool PulserCalib::ready_for_next_step()
{
    return (n_shutdown >= n_unique_channels);
}
///////////////////////////////////////////////////////////////////////////////
// PulserCalibDef
///////////////////////////////////////////////////////////////////////////////
void PulserCalibDef::clear()
{
    type = CalibrationType::Invalid;

    n_samples_per_channel = 1000;
    n_expected = -1;

    board_numbers.clear();
    board_selection = -1;
    vmm_mask = 0;
    channel_low = 0;
    channel_high = 63;

    gain_idx_start = 0;
    gain_idx_stop = 0;

    peak_time_idx_start = 0;
    peak_time_idx_stop = 0;

    tac_idx_start = 0;
    tac_idx_stop = 0;
    tac_idx_step = 1;

    tp_time_per_step = 1; // ns
    tp_skew_start = 0;
    tp_skew_stop = 0;
    tp_skew_step = 1;

    threshold_dac_val_start = 230;
    threshold_dac_val_stop = 230;
    threshold_dac_val_step = 50;

    pulser_dac_val_start = 300;
    pulser_dac_val_stop = 300;
    pulser_dac_val_step = 50;
}
///////////////////////////////////////////////////////////////////////////////
string PulserCalibDef::str()
{
    stringstream sx;

    stringstream bx;
    bx << "[ ";
    for(auto x : board_numbers) bx << x << " ";
    bx << "]";

    sx << "-------------------------------------------------------------------" << endl;
    sx << "PulserCalib    Calibration definition:" << endl
        << "PulserCalib     - Calibration type :         " << CalibrationType2Str(type) << endl
        << "PulserCalib     - Board selection :          " << bx.str() << endl
        << "PulserCalib     - VMM selection mask :       0x" << std::hex << vmm_mask << std::dec << endl
        << "PulserCalib     - # samples per channel :    " << n_samples_per_channel << endl
        << "PulserCalib     - # expected :               " << n_expected << endl
        << "PulserCalib     - VMM channels :             [" << channel_low << "," << channel_high << "]" << endl
        << "PulserCalib     - Gain index loop :          [" << gain_idx_start << "," << gain_idx_stop << "]" << endl
        << "PulserCalib     - Peak time index loop :     [" << peak_time_idx_start << "," << peak_time_idx_stop << "]" << endl
        << "PulserCalib     - TAC slope index loop :     [" << tac_idx_start << "," << tac_idx_stop << ":" << tac_idx_step << "]" << endl
        << "PulserCalib     - TP skew index loop   :     [" << tp_skew_start << "," << tp_skew_stop << ":" << tp_skew_step << "]" << endl
        << "PulserCalib     - Threshold DAC loop   :     [" << threshold_dac_val_start << "," << threshold_dac_val_stop << ":" << threshold_dac_val_step << "]" << endl
        << "PulserCalib     - Pulser DAC loop      :     [" << pulser_dac_val_start << "," << pulser_dac_val_stop << ":" << pulser_dac_val_stop << "]" << endl;
    sx << "-------------------------------------------------------------------" << endl;
    return sx.str();
}
///////////////////////////////////////////////////////////////////////////////
// PulserCalibRunProperties
///////////////////////////////////////////////////////////////////////////////
void PulserCalibRunProperties::clear()
{
    l0_decoding = false;
    filename = "";
    full_filename = "";
    output_dir = "./";
    run_comment = "";
    run_number = 0;
    subhysteresis = 0;
    neighbor_enable = 0;
    channel_trims.clear();
    for(int i = 0; i < 63; i++) {
        channel_trims.push_back(0);
    }
    cktk_max = 7;
    ckbc_frequency = 40;
    cktp_period = 30000;
    cktp_width = 4;
    
}
///////////////////////////////////////////////////////////////////////////////
string PulserCalibRunProperties::str()
{
    stringstream sx;
    sx << "------------------------------------------------------------" << endl;
    sx << " PulserCalibRunProperties" << endl
       << "  - l0 decoding :                " << l0_decoding << endl
       << "  - run comment :                " << run_comment << endl
       << "  - run number :                 " << run_number << endl
       << "  - output dir for files :       " << output_dir << endl
       << "  - full output name             " << full_filename << endl
       << "  - subhysteresis :              " << subhysteresis << endl
       << "  - neighbor enable :            " << neighbor_enable << endl;
    stringstream ct;
    ct << "[";
    for(int i = 0; i < 64; i++) {
        ct << " " << i << ":" << channel_trims.at(i);
    }
    ct << " ]";
    sx << "  - channel trims :               " << ct.str() << endl;
    sx << "------------------------------------------------------------" << endl;

    return sx.str();
}
///////////////////////////////////////////////////////////////////////////////
// PulserCalibHolder
///////////////////////////////////////////////////////////////////////////////
void PulserCalibHolder::clear()
{
    board_number = 0;
    vmm_mask = 0;
    gain_idx = 0;
    peak_time_idx = 0;
    tac_idx = 0;
    tp_skew = 0;
    threshold_dac_val = 230;
    pulser_dac_val = 300;
}
///////////////////////////////////////////////////////////////////////////////
string PulserCalibHolder::str()
{
    stringstream sx;
    sx << "PulserCalib    idx = " << idx() << " : peak time idx = " << peak_time_idx << "  threshold_dac_val = " << threshold_dac_val << "  pulser_dac_val = " << pulser_dac_val; 
    return sx.str();
}
