#include "data_handler.h"

//qt
#include <QString>
#include <QStringList>
#include <QDir>
#include <QFileInfo>
#include <QFileInfoList>
#include <QByteArray>
#include <QBitArray>

#include <sstream>
#include <string>
#include <iostream>
#include <fstream>
using namespace std;

//boost
#include <boost/filesystem.hpp>

//ROOT
#include "TROOT.h"
#include "TFile.h"

//nsw
#include "map_handler.h"
#include "OnlineMonTool.h"
#include "message_handler.h"
#include "pulser_calibration.h"



DataHandler::DataHandler(QObject* parent) :
    QObject(parent),
    m_dbg(false),
    m_vmm_type(3),
    m_initialized(false),
    m_output_dir(""),
    m_output_filename(""),
    m_output_fullfilename(""),
    n_total_events_to_process(0),
    m_tree_flush(0),
    m_current_run_number(0),
    m_writeRaw(false),
    m_writeNtuple(false),
    m_do_monitoring(false),
    m_is_calibration_run(false),
    m_server(NULL),
    m_monTool(NULL),
    m_monitoringSetup(false),
    m_mapHandler(NULL),
    times_updated(0)
{
    m_sx.str("");
}
void DataHandler::set_vmm_type(int type)
{
    m_vmm_type = type;
    m_server->set_vmm_type(type);
}
void DataHandler::testMon()
{
    // for testing dummy packets, the button on the GUI that activates this
    // method is typically disabled and hidden from the user
    m_monTool->testMon();
}

void DataHandler::LoadMessageHandler(MessageHandler& msg)
{
    m_msg = &msg;

    m_server->LoadMessageHandler(*m_msg);
}

void DataHandler::setDebug(bool dbg)
{
    m_dbg = dbg;
    m_server->setDebug(dbg);
    m_monTool->setDebug(dbg);
}

void DataHandler::setDoMonitoring(bool doit)
{
    if(!m_server) {
        m_sx << "ERROR server not initailized. You must call this function after DaqServer is set up.";
        msg()(m_sx, "DataHandler::setDoMonitoring"); m_sx.str("");
        m_writeRaw = false;
        m_writeNtuple = false;
        return;
    }
    m_server->setDoMonitoring(doit);
    m_writeNtuple = doit;
    m_writeRaw = doit;
}
void DataHandler::setCalibrationRun(bool doit)
{
    if(!m_server) {
        m_sx << "ERROR server not initialized. You must call this function after DaqServer is set up.";
        msg()(m_sx, "DataHandler::setCalibrationRun"); m_sx.str("");
        m_is_calibration_run = false;
        return;
    }
    m_server->setCalibrationRun(doit);
    m_is_calibration_run = doit;
}

void DataHandler::updateCalibrationState(double gain, int dacThreshold, int dacAmplitude,
            double tp_skew, int peakTime)
{
    m_server->updateCalibrationState(gain, dacThreshold, dacAmplitude,
                    tp_skew, peakTime);
    if(dbg()) {
        m_sx << " *** Updating calibration state *** \n"
            << "  > gain             : " << gain << " mV/fC\n"
            << "  > dac threshold    : " << dacThreshold << " cts\n"
            << "  > dac amplitude    : " << dacAmplitude << " cts\n"
            << "  > tp skew          : " << tp_skew << " ns\n"
            << "  > peak time        : " << peakTime << " ns";
        msg()(m_sx, "DataHandler::updateCalibrationState"); m_sx.str("");
    }
}
void DataHandler::setCalibrationChannel(int channel)
{

    if(dbg()) {
        m_sx << "Setting calibration channel to " << channel;
        msg()(m_sx, "DataHandler::setCalibrationChannel"); m_sx.str("");
    }
    m_server->setCalibrationChannel(channel);

}

void DataHandler::initialize()
{
    if(m_server) { 
        m_server->stop_listening();
        m_server->stop_server();
        delete m_server;
    }
    m_server = new DaqServer();
    m_server->initialize();
    connect(m_server, SIGNAL(eventCountReached()), this, SLOT(eventCountReached()), Qt::DirectConnection);
    connect(m_server, SIGNAL(effCountReached()), this, SLOT(effCountReached()), Qt::DirectConnection);
    connect(m_server, SIGNAL(updateCounts(int, int, unsigned int, unsigned int, bool)), this, SLOT(updateCounts(int, int, unsigned int, unsigned int, bool)), Qt::DirectConnection);
    connect(m_server, SIGNAL(update_empty_count(unsigned int)), this, SLOT(update_empty_count(unsigned int)));
    connect(m_server, SIGNAL(perform_occupancy_check(QString, int, int)),
                                        this, SLOT(occupancy_check(QString, int, int)));
    connect(m_server, SIGNAL(perform_trigger_delta_check(int, int)),
                                        this, SLOT(trigger_delta_check(int, int)));

    // move forward with calibration
    connect(m_server, SIGNAL(updateCalibrationState()), this, SLOT(updateCalibrationState()));//, Qt::DirectConnection);
    connect(m_server, SIGNAL(daqHangObserved()), this, SLOT(sendDaqHangSignal()));

    connect(this, SIGNAL(resetCountsSignal()), m_server, SLOT(resetCounts()));

    connect(m_server, SIGNAL(pulser_calib_complete_signal()), this, SLOT(pulser_calib_complete()));

    // monitoring
    m_monTool = new OnlineMonTool();
    m_monTool->setDebug(dbg());
    m_server->loadMonitoringTool(*m_monTool);
    // monitoring enable
    connect(this, SIGNAL(setup_monitoring(bool)), m_server, SLOT(recv_setup_monitoring(bool)));
    m_monitoringSetup = true;

    m_initialized = true;
}
void DataHandler::LoadModulesForCalibration(Configuration& config, RunModule& run) //, SocketHandler& socket)
{
    m_server->LoadModulesForCalibration(config, run); //, socket);
}
bool DataHandler::load_pulser_calib(PulserCalibDef def)
{
    return m_server->load_pulser_calib(def);
}
bool DataHandler::load_pulser_calib_run_properties(PulserCalibRunProperties props)
{
    props.run_number = m_current_run_number;
    props.output_dir = m_output_dir;
    props.filename = m_output_filename;
    props.full_filename = m_output_fullfilename;
    return m_server->load_pulser_calib_run_properties(props);
}
void DataHandler::send_initial_config_to_pulser_calib(GlobalSetting& global, VMMMap& vmmMap,
        std::vector<Channel>& channels, FPGAClocks& clocks, TriggerDAQ& daq)
{
    m_server->send_initial_config_to_pulser_calib(global, vmmMap, channels, clocks, daq);
}
void DataHandler::pulser_calib_complete()
{
    emit stop_pulser_calib_run();
}
void DataHandler::sendDaqHangSignal(){
    emit daqHangObserved();
}

void DataHandler::update_monitoring_rate(int rate)
{
    if(!m_server) return;
    m_server->update_monitoring_rate(rate);
}

bool DataHandler::loadMapping(std::string daq_xml_file)
{
    if(!initialized()) {
        msg()("DataHandler not yet initialized, cannot load mapping. Call 'initialize()'.", "DataHandler::loadMapping");
        return false;
    }

    if(m_mapHandler) {
        delete m_mapHandler;
    }
    m_mapHandler = new MapHandler();
    bool ok = m_mapHandler->loadDaqConfiguration(daq_xml_file);
    if(ok) {
        m_mapHandler->buildMapping();
        if(m_mapHandler->mapLoaded()) {
            m_server->loadMappingTool(*m_mapHandler);
        }
        else {
            ok = false;
            msg()("ERROR loading mapping", "DataHandler::loadMapping");
        }
    }

    m_mappingSetup = ok;

    return ok;
}
void DataHandler::disableMapping()
{
    if(m_mapHandler) {
        m_server->disableMapping();
    }
}

std::string DataHandler::getFirstIP()
{
    std::string out = "";
    if(mapHandler().mapLoaded()) {
        out = mapHandler().firstIP();
    }
    else {
        msg()("Map is not loaded, cannot get first IP", "DataHandler::getFirstIP");
    }
    return out;
}

int DataHandler::getNumberOfFecs()
{
    int nFecs = 0;
    if(mapHandler().mapLoaded()) {
        nFecs = mapHandler().numberOfBoards();
    }
    else {
        msg()("Map is not loaded, cannot get number of FECs", "DataHandler::getNumberOfFecs");
    }
    return nFecs;
}

void DataHandler::recv_send_monitor_config()
{
    sendMonitoringConfiguration();
}

void DataHandler::recv_setup_monitoring(bool enable_monitoring, bool send_config)
{
    m_do_monitoring = enable_monitoring;
    if(m_do_monitoring && send_config)
        sendMonitoringConfiguration();
    emit setup_monitoring(enable_monitoring);
}

void DataHandler::setupMonitoring(bool /*do_monitoring*/, std::string /*ip*/, int /*port*/, bool /*is_middle_of_run*/)
{
    msg()("This method is obsolete!","DataHandler::setupMonitoring");
    return;
    //if(!initialized()) {
    //    msg()("DataHandler not yet initialized, will not setup monitoring. Call 'initialize()'.", "DataHandler::setupMonitoring");
    //    m_monitoringSetup = false;
    //    return;
    //}

    //// if we are in the middle of a run, etc...
    ///*
    //if(is_middle_of_run) {
    //    if(m_monitoringSetup && do_monitoring) {
    //        m_server->stopMonitoring();
    //        if(m_monTool)
    //            delete m_monTool;
    //    }
    //}
    //*/

    //if(do_monitoring) {
    //    //if(!m_monitoringSetup) {
    //        //if(!m_monitoringSetup) {
    //        if(!m_monTool) {
    //            m_monTool = new OnlineMonTool();
    //            m_monTool->setMonitorAddress(ip, port);
    //            m_server->loadMonitoringTool(*m_monTool);
    //        }
    //        m_monTool->setDebug(dbg());
    //        //bool ok = m_monTool->initialize();
    //        bool ok = true;


    //        if(ok) {
    //            if(dbg()) msg()("Monitoring socket initialized", "DataHandler::setupMonitoring");
    //            m_do_monitoring = true;
    //            m_monitoringSetup = true;
    //            m_server->setMonitoringStatus(m_monitoringSetup);
    //            //if(is_middle_of_run && m_monTool->is_paused())
    //            //    m_monTool->restart();
    //        }
    //        else {
    //            m_do_monitoring = false;
    //            m_monitoringSetup = false;
    //            m_server->setMonitoringStatus(m_monitoringSetup);
    //            delete m_monTool;
    //        }
    //   // }
    //}
    //else {
    //    //if(m_monitoringSetup && is_middle_of_run) {
    //    //    m_monTool->pause();
    //    //    m_do_monitoring = false;
    //    //    m_monitoringSetup = false;
    //    //    m_server->setMonitoringStatus(m_monitoringSetup);
    //    //}
    //    if(m_monitoringSetup) {
    //        m_monTool->closeMonitoringSocket();
    //        m_do_monitoring = false;
    //        m_monitoringSetup = false;
    //        m_server->setMonitoringStatus(m_monitoringSetup);
    //    }
    //}
}
// ------------------------------------------------------------------------ //
bool DataHandler::sendMonitoringConfiguration()
{
    stringstream sx;
    if(!initialized()) {
        sx << "DataHandler not yet initialized, will not send monitoring configuration\n";
        sx << "Call 'initialize()'."; 
        msg()(sx, "DataHandler::sendMonitoringConfiguration"); sx.str("");
        return false;
    }

    bool send_ok = true;

    if(m_monitoringSetup && m_do_monitoring) {
        // get list of boards and their associated chips
        if(mapHandler().mapLoaded()) {
            std::vector<std::string> words_to_send;

            stringstream send_string;
            send_string << "config start";
            words_to_send.push_back(send_string.str()); send_string.str("");
    
            for(auto boardidmap : mapHandler().boardContents()) {
                for(auto boardid : boardidmap) {
                    send_string << "config " << boardid.first << " "; 
                    for(auto vmm : boardid.second) {
                        send_string << vmm << " ";
                    } // vmm
                    words_to_send.push_back(send_string.str()); send_string.str("");
                } // boardid
            } // boardidmap

            send_string << "config end";
            words_to_send.push_back(send_string.str()); send_string.str("");

            if(m_monTool) {
                // reset the io_service used by the monitoring tool if we have ended the run previously
                // config
                //if(m_server->is_stopped())
                //    m_server->setMonitoringStatus(m_monitoringSetup);
                m_monTool->sendConfiguration(words_to_send);
            }
            else {
                msg()("Monitor tool is null! Cannot send configuration.", "DataHandler::sendMonitoringConfiguration");
                send_ok = false;
            }
        } // maploaded
        else {
            msg()("ERROR Mapping is not loaded, cannot send monitoring configuration", "DataHandler::sendMonitoringConfiguration");
            send_ok = false;
        }
    } // monitoring setup
    else {
        stringstream ss;
        ss << "ERROR Monitoring configuration could not be setup properly!";
        msg()(ss,"DataHandler::sendMonitoringConfiguration"); ss.str("");
        send_ok = false;
    }

    return send_ok;
}
// ------------------------------------------------------------------------ //
void DataHandler::resetMonitoringConfiguration()
{
    if(m_monTool)
        m_monTool->resetMonitoringConfiguration();
}
// ------------------------------------------------------------------------ //
void DataHandler::setOutputDir(std::string out_dir)
{
    m_output_dir = out_dir;
    emit updateRunNumber(0);
}
// ------------------------------------------------------------------------ //
void DataHandler::set_cktp_limit(int n_pulses_expected)
{
    m_server->set_cktp_limit(n_pulses_expected);
}
// ------------------------------------------------------------------------ //
bool DataHandler::initializeRun(bool writeRaw, bool writeNtuple_, std::string output_dir, int run_number, int events_to_process, int tree_flush, bool do_calibration, bool do_L0_decoding, bool is_xadc)
{

    if(!initialized()) {
        msg()("DataHandler not initialized, will not initialize current run. Call 'initialize()'.", "DataHandler::initializeRun");
        return false;
    }

    m_writeRaw = writeRaw;
    //if(do_L0_decoding && m_writeRaw) {
    //    msg()("Not writing out RAW data for L0 decoding, this is not yet implemented", "DataHandler::initializeRun");
    //    m_writeRaw = false;
    //}
    m_writeNtuple = writeNtuple_;

    if(do_calibration) {
        m_is_calibration_run = true;
    }

    std::stringstream sx;

    //if(writeNtuple_)
    if(writeNtuple_ || writeRaw) {
        if(!setOutputFile(output_dir, run_number, do_calibration)) return false;

        if(m_current_run_number != run_number) {
            m_sx << "Re-setting run number to " << m_current_run_number;
            msg()(m_sx, "DataHandler::initializeRun"); m_sx.str("");
        }
    }

    if(dbg()) {
        m_sx << "Set to process " << events_to_process << " events for run " << m_current_run_number;
        msg()(m_sx, "DataHandler::initializeRun"); m_sx.str("");
    }

    n_total_events_to_process = events_to_process;
    m_tree_flush = tree_flush;


    if(writeNtuple_ && dbg()) {
        msg()("Will write output ntuple", "DataHandler::initializeRun");
    }

    if(!m_server->initializeRun(m_writeRaw, m_writeNtuple, m_output_fullfilename, m_current_run_number, events_to_process, tree_flush, do_calibration, do_L0_decoding, is_xadc)) return false;

    return true;
}

bool DataHandler::initializeCalibration(int vmm_id, int n_samples, int sample_period, CalibrationType type, bool skip_dead_channels, bool doL0)
{
    return m_server->initializeCalibration(vmm_id, n_samples, sample_period, type, skip_dead_channels, doL0);
}

void DataHandler::start_xadc_sampling(std::string ip)
{
    m_server->start_xadc_sampling(ip);
}
void DataHandler::fillRunProperties(int gain, int tacSlope, int peakTime, int dac_threshold,
            int dac_amplitude, int angle, int tp_skew, int ckbc)
{
    m_server->fillRunProperties(gain, tacSlope, peakTime, dac_threshold, dac_amplitude,
                    angle, tp_skew, ckbc);
}
void DataHandler::fillRunComment(std::string comment)
{
    m_server->fillRunComment(comment);
}

void DataHandler::startGathering()
{
    if(m_do_monitoring && m_monitoringSetup) {
        bool ok = sendMonitoringConfiguration();
        boost::this_thread::sleep(boost::posix_time::milliseconds(200));
        if(ok) {
            if(m_monTool->configurationSent() && !m_monTool->is_started()) {
                //m_server->startMonitoring();
                m_monTool->start(); // start monitoring heartbeat
            }
            else {
                stringstream sx;
                sx << "WARNING Monitoring configuration not acknowledged by OnlineMonTool. Will not start monitoring.";
                msg()(sx, "DataHandler::startGathering"); sx.str("");
            }
        }
    }
    m_server->listen();
}

bool DataHandler::begin_pulser_calibration()
{
    return m_server->begin_pulser_calibration();
}

void DataHandler::resetCounts()
{
    //emit resetCountsSignal();
    m_server->resetCounts();
}
int DataHandler::getCounts()
{
    return m_server->getCounts();
}

void DataHandler::updateCounts(int counts, int trig_counts,
                unsigned int input_packets, unsigned int input_empty_packets, bool fine)
{
    emit updateCountsSend(counts, trig_counts, input_packets, input_empty_packets, fine);
}
void DataHandler::update_empty_count(unsigned int empty_count)
{
    emit update_empty_count_send(empty_count);
}

void DataHandler::occupancy_check(QString ip, int full, int empty)
{
    emit occupancy_check_send(ip, full, empty);
}
void DataHandler::trigger_delta_check(int board, int delta)
{
    emit trigger_delta_check_send(board, delta);
}

void DataHandler::toggle_occupancy_check(bool do_check)
{
    m_server->toggle_occupancy_check(do_check);
}

void DataHandler::updateCalibrationState()
{
    emit updateCalibration();
}
//void DataHandler::loadCalibrationState(CalibrationState state)
void DataHandler::loadCalibrationState(int channel, int gain, int run_number, int tac_slope,
            int neighbor_trigger, int subhyst, int peak_time, int test_pulse_dac,
            int thresh_dac, std::vector<int> trims, double tp_skew, int n_expected_pulses)
{
    m_server->loadCalibrationState(channel, gain, run_number, tac_slope, neighbor_trigger,
                    subhyst, peak_time, test_pulse_dac, thresh_dac, trims, tp_skew, n_expected_pulses);
    //m_server->loadCalibrationState(state);
}

void DataHandler::eventCountReached()
{
    emit eventCountReachedSend();
}

void DataHandler::effCountReached()
{
    emit effCountReachedSend();
}

void DataHandler::endRun(bool do_write)
{
    m_server->stop_listening();

    //m_server->flush();
    if(do_write && (writeNtuple() || writeRaw())) {
        m_server->write_output();
    }

    m_server->stop_server();

    while(true) {
        if(m_server->is_stopped()) break;
        msg()("Waiting for DAQ server to stop...", "DataHandler::endRun");
   }
    // put this afterwards since IO service will stop the monitoring?
    if(m_do_monitoring) {
        //m_server->stopMonitoring();
        // this closes the socket that the monitoring tool uses and stops the internal timer
        m_monTool->stop();
    }
}

bool DataHandler::setOutputFile(std::string out_dir, int provided_run_number, bool do_calibration)
{
    stringstream sx;

    QString fullfilename = "";

    m_output_dir = "";

    if(out_dir == "") {
        msg()("ERROR Output directory is \"\"","DataHandler::setOutputFile");
        emit badOutputDir();
        return false;
    }

    bool exists = std::ifstream(out_dir).good();
    if(!exists) {
        msg()("ERROR Output directory is invalid","DataHandler::setOutputFile");
        emit badOutputDir();
        return false;
    }

    QString dirname = QString::fromStdString(out_dir);
    QString spacer = "";
    if(!dirname.endsWith("/")) spacer = "/";

    QString check_file = dirname + spacer;
    int run_number_from_dir = checkForExistingFiles(check_file.toStdString(), provided_run_number);
    if(run_number_from_dir < provided_run_number)
        m_current_run_number = provided_run_number; 
    else {
        m_current_run_number = run_number_from_dir;
    }
    emit updateRunNumber(m_current_run_number);
    //m_current_run_number = checkForExistingFiles(check_file.toStdString());
    //int run_number = m_current_run_number;

     if(do_calibration && m_writeNtuple)	{
        QString filename_init = "calib_run_%04d.root";
        const char* filename_formed = Form(filename_init.toStdString().c_str(), m_current_run_number);
        std::string filename_formed_str(filename_formed);
        QString full_name = dirname + spacer + QString::fromStdString(filename_formed_str);
	
        if(dbg()) {
            m_sx << "Setting output file: " << full_name.toStdString();
		    msg()(m_sx, "DataHandler::setOutputFile"); m_sx.str("");
        }
		m_output_dir = out_dir; 

        if(!checkRootFile(full_name.toStdString())) {
            m_sx << "Output ROOT file (" << full_name.toStdString() << ") unable to be created";
			msg()(m_sx,"DataHandler::setOutputFile"); m_sx.str("");
			emit badOutputDir();
			return false;
		}
		
        m_output_filename = filename_formed_str;
        m_output_fullfilename = full_name.toStdString();
	}
    else {
        QString filename_init = "run_%04d.root";
        const char* filename_formed = Form(filename_init.toStdString().c_str(), m_current_run_number);
        std::string filename_formed_str(filename_formed);
        QString full_name = dirname + spacer + QString::fromStdString(filename_formed_str);

        if(m_writeNtuple) {
            m_sx << "Setting output ntuple file: " << full_name.toStdString();
            msg()(m_sx, "DataHandler::setOutputFile"); m_sx.str("");

		    if(!checkRootFile(full_name.toStdString())) {
		    	m_sx << "Output ROOT file (" << full_name.toStdString() << ") unable to be created";
		    	msg()(m_sx,"DataHandler::setOutputFile"); m_sx.str("");
		    	emit badOutputDir();
		    	return false;
		    }
        }
        m_output_dir = out_dir;


		//QString filename_init = "run_%04d.root";
		//const char* filename_formed = Form(filename_init.toStdString().c_str(), m_current_run_number);
		//std::string filename_formed_str(filename_formed);
		//QString full_name = dirname + spacer + QString::fromStdString(filename_formed_str);

		//m_sx << "Setting output ntuple file: " << full_name.toStdString();
		//msg()(m_sx, "DataHandler::setOutputFile"); m_sx.str("");
		//m_output_dir = out_dir; 

		
		m_output_filename = filename_formed_str;
    	m_output_fullfilename = full_name.toStdString();
	}
    
    return true;
}

bool DataHandler::checkRootFile(std::string filename)
{
    TFile* tmpFile = new TFile(filename.c_str(), "UPDATE");
    if(tmpFile->IsZombie()) {
        delete tmpFile;
        return false;
    }
    tmpFile->Close();
    return true;
}
int DataHandler::checkForExistingFiles(std::string dirname, int provided_run_number)
{
    // we expect that output ntuples are nammed as
    // *run_XXXX.root
    bool ok;

    QStringList filters;
    filters << "*run_*.root";// << "*run_*.bin";
    QDir dir(QString::fromStdString(dirname));
    dir.setNameFilters(filters);

    int max_run = -1;
    QFileInfoList listOfFiles = dir.entryInfoList();
    if(listOfFiles.size()>0) {
        for(int i = 0; i < (int)listOfFiles.size(); i++) {
            QFileInfo fileInfo = listOfFiles.at(i);
            QString fname = fileInfo.fileName().split("/").last();
            QString number = fname.split("_").last();
            number.replace(".root","");
            int other_run = number.toInt(&ok, 10);
            if(other_run > max_run) max_run = other_run;
        } // i
    }

    QStringList filters2;
    filters2 << "*run_*.bin";
    QDir dir2(QString::fromStdString(dirname));
    dir2.setNameFilters(filters2);
    QFileInfoList listOfFiles2 = dir2.entryInfoList();
    if(listOfFiles2.size()>0) {
        for(int i = 0; i < (int)listOfFiles2.size(); i++) {
            QFileInfo fileInfo = listOfFiles2.at(i);
            QString fname = fileInfo.fileName().split("/").last();
            QString number = fname.split("_").last();
            number.replace(".bin","");
            int other_run = number.toInt(&ok, 10);
            if(other_run > max_run) max_run = other_run;
        } // i
    }


    if( (max_run >= m_current_run_number) && max_run >=0 ) {
        //m_current_run_number = max_run+1;
        return max_run+1;
    }
    else {
        //return m_current_run_number;
        return provided_run_number;
    }
}


//////////////////////////////////////////////////////////////
// misc Qt-based bit manipulation
//////////////////////////////////////////////////////////////
quint32 DataHandler::ValueToReplaceHEX32(QString hex, int bitToChange,
                                                bool newBitValue)
{
    bool ok;

    // initialize a 32 bit word = 0
    QBitArray commandToSend(32,false);
    QString bin, tmp;

    // convert input hex word to binary
    bin = tmp.number(hex.toUInt(&ok,16),2); 

    // copy old word
    for(int i = 0; i < bin.size(); i++) {
        QString bit = bin.at(i);
        commandToSend.setBit(32-bin.size()+i, bit.toUInt(&ok,10));
    } // i

    // now change the bit
    commandToSend.setBit(31-bitToChange, newBitValue);
    QByteArray byteArr = bitsToBytes(commandToSend);
    quint32 tmp32 = byteArr.toHex().toUInt(&ok,16);
    return tmp32; 
}
// ------------------------------------------------------------------------ //
QByteArray DataHandler::bitsToBytes(QBitArray bits)
{
    QByteArray outbytes;
    outbytes.resize(bits.count()/8);
    outbytes.fill(0);

    for(int b = 0; b < bits.count(); ++b)
        outbytes[b/8] = ( outbytes.at(b/8) | ( (bits[b]?1:0) << (7-(b%8))));
    return outbytes;
}
// ------------------------------------------------------------------------ //
QBitArray DataHandler::bytesToBits(QByteArray bytes)
{
    QBitArray outbits;
    outbits.resize(bytes.count()*8);

    for(int i = 0; i < bytes.count(); ++i) {
        for(int b = 0; b < 8; ++b) {
            outbits.setBit( i*8+b, bytes.at(i)&(1<<(7-b)) );
        } // b
    } // i
    return outbits;
} 
// ------------------------------------------------------------------------ //
QString DataHandler::QBitArrayToString(const QBitArray& array)
{
    uint value = 0;
    for(int i = 0; i < (int)array.size(); i++)
    {
        value <<= 1;
        value += (int)array.at(i);
    }
    QString str;
    str.setNum(value, 10);
    
    return str;
}
// ------------------------------------------------------------------------ //
quint32 DataHandler::reverse32(QString hex)
{
    bool ok;
    QBitArray received(32,false);
    QString bin, tmp;
    bin = tmp.number(hex.toUInt(&ok,16),2); // convert input to binary
    if(bin.size()>32) {
        cout << "DataHandler::reverse32    Input datagram is larger than 32 bits!" << endl;
        cout << "DataHandler::reverse32    >>> Exiting." << endl;
        exit(1);
    }
    // turn input array into QBitArray
    for(int i = 0; i < bin.size(); i++) {
        QString bit = bin.at(i);
        received.setBit(32-bin.size() + i, bit.toUInt(&ok,10)); // pad left with 0's
    } // i

    // now reverse
    QBitArray reversed(32, false);
    for(int j = 0; j < 32; j++) {
        reversed.setBit(31-j, received[j]);
    } // j

    // turn into QByteArray and return
    QByteArray reversed_byte = DataHandler::bitsToBytes(reversed);
    return reversed_byte.toHex().toUInt(&ok,16);
}
